/*************
 *
 * Filename:    gobiimgmgmt.c
 *
 * Purpose:     GOBI Image Management application
 *
 * Copyright: © 2013 Sierra Wireless Inc., all rights reserved
 *
 **************/
#define _SVID_SOURCE
#include "SWIWWANCMAPI.h"
#include "ra_gobidispmngmt.h"
#include "qmerrno.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/dir.h>
#include <syslog.h>

/****************************************************************
 *                    GLOBAL DATA
 ****************************************************************/


/* Folder info holding the image */
extern const CHAR *pImgFolderName[MAX_IMAGES];
extern BYTE imgFolderIdx[MAX_IMAGES+1];
extern BYTE folderIdx;
extern struct ImageList devImgList;


#define MC_SERIES      "MC77"
#define WP_SERIES      "WP71"
#define GOBI5K_SERIES  "MC73"
#define MC78_SERIES    "MC78"

#define FAIL                     1



/* firmware download */
static bool fwDwlComplete = false;
static bool dwlImage      = false;
//static CHAR imagePath[MAX_IMAGE_PATH];
char   *configFilePath = NULL;
struct ImageFinalList devImgFinalList[50];
struct ImageFinalList ModemImgFinalList[50];

/* ImageID String for Fail Safe AMSS image */
const char imgIddefAMSSImage[16] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                                     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
const ULONG imageIDLen = 0x10;

/* Image information table */
struct imgInfoTable imgDetailInfo[MAX_IMAGES_LEN] =
{
    { "D3600", 'S', eGOBI_IMG_CAR_SPRINT,        eGOBI_IMG_REG_NA,     eGOBI_IMG_TECH_CDMA },
    { "D3600", 'V', eGOBI_IMG_CAR_VERIZON,       eGOBI_IMG_REG_NA,     eGOBI_IMG_TECH_CDMA },
    { "D3600", 'C', eGOBI_IMG_CAR_CHINA_TELECOM, eGOBI_IMG_REG_ASIA,   eGOBI_IMG_TECH_CDMA },
    { "D3200", 'V', eGOBI_IMG_CAR_VODAFONE,      eGOBI_IMG_REG_NA,     eGOBI_IMG_TECH_UMTS },
    { "D3200", 'A', eGOBI_IMG_CAR_ATT,           eGOBI_IMG_REG_NA,     eGOBI_IMG_TECH_UMTS },
    { "D3200", 'L', eGOBI_IMG_CAR_TMOBILE,       eGOBI_IMG_REG_EU,     eGOBI_IMG_TECH_UMTS },
    { "D3200", 'G', eGOBI_IMG_CAR_GENERIC,       eGOBI_IMG_REG_GLOBAL, eGOBI_IMG_TECH_UMTS },
    { "D3200", 'H', eGOBI_IMG_CAR_TELEFONICA,    eGOBI_IMG_REG_EU,     eGOBI_IMG_TECH_UMTS },
    { "D3200", 'I', eGOBI_IMG_CAR_TELCOM_ITALIA, eGOBI_IMG_REG_NA,     eGOBI_IMG_TECH_UMTS },
    { "D3200", 'O', eGOBI_IMG_CAR_ORANGE,        eGOBI_IMG_REG_NA,     eGOBI_IMG_TECH_UMTS },
    { "D3200", 'U', eGOBI_IMG_CAR_GENERIC,       eGOBI_IMG_REG_GLOBAL, eGOBI_IMG_TECH_UMTS }
};

#define TRUE 1
#define FALSE 0
BOOL ra_IsGobiDevice(void)
{
    BYTE stringsize = 128;
    CHAR ModelId[stringsize];

    GetModelID( stringsize,
                ModelId );

    if (strstr(ModelId, MC_SERIES))

    {
        return false;
    }
    else
    {
        return true;
    }
}



/****************************************************************
*                    COMMON FUNCTIONS
****************************************************************/
/*
 * Name:     ra_ExtractPRIImgInfo
 *
 * Purpose:  Extract the Carrier, Technology, region and GPS information from
 *           the received build id.
 *
 * Return:   true  - If the image is successfully identified
 *           false - If the image is not identified
 *
 * Notes:    none
 */
bool ra_ExtractPRIImgInfo(
    char               *pBuildId,
    struct qmifwinfo_s *pImgInfo,
    int                 BuildIdLen )
{
    BYTE idx = 0;

    /* Extract the information from the Image Info table */
    for( idx = 0; idx < MAX_IMAGES; idx++ )
    {
        /* If there ia a match for technology string */
        if( NULL != strstr( pBuildId, imgDetailInfo[idx].pTechnology ) )
        {
            if( pBuildId[GOBI33_BUILD_ID_VER_INITIALS_UQCN] == '3' &&
                pBuildId[GOBI33_BUILD_ID_VER_INITIALS_UQCN + 1] > '0')
            {
                if ( pBuildId[BuildIdLen - GOBI33_CARRIER_INIT_OFFSET_FROM_END] ==
                      imgDetailInfo[idx].carrierIntials )
                {
                        pImgInfo->dev.g.Carrier = imgDetailInfo[idx].carrier;
                        pImgInfo->dev.g.Region =  imgDetailInfo[idx].region;
                        pImgInfo->dev.g.Technology = imgDetailInfo[idx].technology;
                        return true;
                }
            }
            /* If there is a match for carrier names's initial */
            else if( pBuildId[GOBI_BUILD_ID_CARRIER_INITIALS_OFFSET] ==
                         imgDetailInfo[idx].carrierIntials )
            {
                pImgInfo->dev.g.Carrier = imgDetailInfo[idx].carrier;
                pImgInfo->dev.g.Region =  imgDetailInfo[idx].region;
                pImgInfo->dev.g.Technology = imgDetailInfo[idx].technology;
                return true;
            }
        }
    }
    return false;
}

/*
 * Name:     ra_ExtractModemImgInfo
 *
 * Purpose:  Extract the Carrier, Technology, region and GPS information from
 *           the received build id.
 *
 * Return:   true  - If the image is successfully identified
 *           false - If the image is not identified
 *
 * Notes:    none
 */
bool ra_ExtractModemImgInfo(
    char               *pBuildId,
    struct qmifwinfo_s *pImgInfo,
    int                 BuildIdLen )
{
    BYTE idx = 0;

    /* Extract the information from the Image Info table */
    for( idx = 0; idx < MAX_IMAGES_LEN; idx++ )
    {
        /* If there ia a match for technology string */
        if( NULL != strstr( pBuildId, imgDetailInfo[idx].pTechnology ) )
        {
            struct sGetDeviceSeriesResult result;
            eGetDeviceSeries(&result);
            if( (result.uResult==eGOBI_DEV_SERIES_SIERRA_GOBI) &&
                pBuildId[SL9090_BUILD_ID_CARRIER_INITIALS_OFFSET] ==
                imgDetailInfo[idx].carrierIntials )
            {
                pImgInfo->dev.g.Carrier = imgDetailInfo[idx].carrier;
                pImgInfo->dev.g.Region =  imgDetailInfo[idx].region;
                pImgInfo->dev.g.Technology = imgDetailInfo[idx].technology;
                return true;
            }
            else
            {
                if( pBuildId[GOBI33_BUILD_ID_VER_INITIALS_AMSS] == '3' &&
                    pBuildId[GOBI33_BUILD_ID_VER_INITIALS_AMSS + 1] > '0')
                {
                    if ( pBuildId[BuildIdLen - GOBI33_CARRIER_INIT_OFFSET_FROM_END] ==
                      imgDetailInfo[idx].carrierIntials )
                    {
                        pImgInfo->dev.g.Carrier = imgDetailInfo[idx].carrier;
                        pImgInfo->dev.g.Region =  imgDetailInfo[idx].region;
                        pImgInfo->dev.g.Technology = imgDetailInfo[idx].technology;
                        return true;
                     }

                }
                /* If there is a match for carrier names's initial */
                else if( pBuildId[GOBI_BUILD_ID_CARRIER_INITIALS_OFFSET] ==
                         imgDetailInfo[idx].carrierIntials )
                {
                    pImgInfo->dev.g.Carrier = imgDetailInfo[idx].carrier;
                    pImgInfo->dev.g.Region =  imgDetailInfo[idx].region;
                    pImgInfo->dev.g.Technology = imgDetailInfo[idx].technology;
                    return true;
                }
            }
        }
    }
    return false;
}

/*
 * Name:     ra_IsFwNew
 *
 * Purpose:  Return the status of the PRI Image based on the rest of the images on the
 *           host
 *
 * param     [out] pPRIImgBuildID
 *            The build ID of the PRI image
 *
 * param     [out] pOldPRIImgBuildID
 *           The build ID used to compare whether the current Image is new
 *
 * return
 *      FW_DIFFERENT_CARRIER if the carrier image is not present
 *      FW_SAME_CARRIER_OLD if no new carrier of the same image is present
 *      FW_SAME_CARRIER_NEW if new carrier of the same image is present
 *
 * note
 *     none
 */
enum carrierImgStatus ra_IsFwNew( CHAR *pPRIImgBuildID,
                               CHAR *pOldPRIImgBuildID,
                               ULONG PRIBuildIdLen,
                               ULONG OldPRIBuildLen )
{
    CHAR PRIImgStr[100], OldPRIImgStr[100];
    CHAR *pPRIBuildIDVer, *pOldPRIBuildIDVer;
    struct qmifwinfo_s ImgInfo, OldImgInfo;
    CHAR BuildIdVerNew[MAX_VER_STR_LEN];
    CHAR BuildIdVerOld[MAX_VER_STR_LEN];
    BYTE count = 0, idx = 0, DigitsInOldVer = 0 ,DigitsInNewVer = 0;
    int diff = 0;
    CHAR DigitsAfterString[3] = {'\0','\0','\0'};

    strcpy( PRIImgStr, pPRIImgBuildID );
    strcpy( OldPRIImgStr, pOldPRIImgBuildID );

    /* Extract the image informations */
    if ( !ra_ExtractPRIImgInfo(pPRIImgBuildID, &ImgInfo, PRIBuildIdLen ) ||
         !ra_ExtractPRIImgInfo(pOldPRIImgBuildID, &OldImgInfo, OldPRIBuildLen ))
    {
        return FW_DIFFERENT_CARRIER;
    }

    /* Make sure that the carrier, technology and region are the same */
    if ((ImgInfo.dev.g.Carrier    != OldImgInfo.dev.g.Carrier) ||
        (ImgInfo.dev.g.Region     != OldImgInfo.dev.g.Region) ||
        (ImgInfo.dev.g.Technology != OldImgInfo.dev.g.Technology))
    {
        return FW_DIFFERENT_CARRIER;
    }

    /* Get the PRI Image version */
    pPRIBuildIDVer = strtok ( PRIImgStr, "-" );
    pPRIBuildIDVer = strtok ( NULL, "-" );
    pPRIBuildIDVer = strtok ( NULL, "-" );

    /* Get the old PRI Image version */
    pOldPRIBuildIDVer = strtok ( OldPRIImgStr, "-" );
    pOldPRIBuildIDVer = strtok ( NULL, "-" );
    pOldPRIBuildIDVer = strtok ( NULL, "-" );

    /* Check if AMSS iamge is Gobi3.3 or higher
     * (to allow Gobi5.5 down the line)
     */
    if( pPRIImgBuildID[GOBI33_BUILD_ID_VER_INITIALS_UQCN] == '3' &&
        pPRIImgBuildID[GOBI33_BUILD_ID_VER_INITIALS_UQCN + 1] > '0' )
    {
        if( pOldPRIImgBuildID[GOBI33_BUILD_ID_VER_INITIALS_UQCN] == '3' &&
            pOldPRIImgBuildID[GOBI33_BUILD_ID_VER_INITIALS_UQCN + 1] > '0' )
        {
            /* Read and store digits in NULL terminated Version string */
            for ( idx = 0; pOldPRIBuildIDVer[idx] != '\0'; idx++ )
            {
                if ( pOldPRIBuildIDVer[idx] >= '0' && pOldPRIBuildIDVer[idx] <= '9' )
                {
                    BuildIdVerOld[DigitsInOldVer] = pOldPRIBuildIDVer[idx];
                    DigitsInOldVer++;
                }
            }

            for ( idx = 0; pPRIBuildIDVer[idx] != '\0'; idx++ )
            {
                if ( pPRIBuildIDVer[idx] >= '0' && pPRIBuildIDVer[idx] <= '9')
                {
                    BuildIdVerNew[DigitsInNewVer] = pPRIBuildIDVer[idx];
                    DigitsInNewVer++;
                }
            }

            /* If digits in version string are not equal do Normalization */
            diff = DigitsInOldVer -DigitsInNewVer;
            if ( diff > 0 )
            {
                /* Save digits "zz" after YY string i.e. carrier string */
                DigitsAfterString[0] = BuildIdVerNew[DigitsInNewVer-2];
                DigitsAfterString[1] = BuildIdVerNew[DigitsInNewVer-1];

                for( count = 0; diff > count; count++)
                {
                    BuildIdVerNew[DigitsInNewVer -2 + count]= '0';
                }
                /* Terminate buffer with NULL */
                BuildIdVerNew[DigitsInNewVer-2 + count ]= '\0';
                /* Append digits "zz" in buffer after Normalization */
                strcat(BuildIdVerNew,DigitsAfterString);
            }

            if( diff < 0 )
            {
                /* To make diff a positive number */
                diff *= -1;
                DigitsAfterString[0] = BuildIdVerOld[DigitsInOldVer-2];
                DigitsAfterString[1] = BuildIdVerOld[DigitsInOldVer-1];
                for( count = 0; diff > count; count++)
                {
                    BuildIdVerOld[DigitsInOldVer -2 + count]= '0';
                }
                BuildIdVerOld[DigitsInOldVer-2 + count ]= '\0';
                strcat(BuildIdVerOld,DigitsAfterString);
            }

            /*
             * Check to see which of the FW is newer:
             * Only of AMSS Images are Gobi3.3 or higher
             */
            if (atol(BuildIdVerNew) > atol(BuildIdVerOld))
            {
                return FW_SAME_CARRIER_NEW;
            }
            else
            {
                return FW_SAME_CARRIER_OLD;
            }
        }
        else
        {
             /* Check to see which of the FW is newer */
             return FW_SAME_CARRIER_NEW;
        }
    }

    /* Check to see which of the FW is newer: Only if AMSS Images Gobi1.7 */
    if (atoi(pPRIBuildIDVer) > atoi(pOldPRIBuildIDVer))
    {
        return FW_SAME_CARRIER_NEW;
    }
    return FW_SAME_CARRIER_OLD;
}

/*
 * Name:     ra_IsFwModemImageNew
 *
 * Purpose:  Return the status of the Modem Image based on the rest of the images on the
 *           host
 *
 * param     [out] pModemImgBuildID
 *           The build ID of the Modem image
 *
 * param     [out] pOldModemImgBuildID
 *           The build ID used to compare whether the current Image is new
 *
 * return
 *      FW_DIFFERENT_CARRIER if the carrier image is not present
 *      FW_SAME_CARRIER_OLD if no new carrier of the same image is present
 *      FW_SAME_CARRIER_NEW if new carrier of the same image is present
 *
 * note
 *     none
 */
enum carrierImgStatus ra_IsFwModemImageNew( CHAR *pModemImgBuildID,
                                         CHAR *pOldModemImgBuildID,
                                         ULONG ModemBuildIdLen,
                                         ULONG OldModemBuildLen )
{
    CHAR ModemImgStr[100], OldModemImgStr[100];
    CHAR *pModemBuildIDVer, *pOldModemBuildIDVer;
    CHAR newModemBuildIDVer[10];
    CHAR oldModemBuildIDVer[10];
    BYTE offset, newImageCount = 0, oldImageCount = 0;
    struct qmifwinfo_s ImgInfo, OldImgInfo;
    CHAR BuildIdVerNew[MAX_VER_STR_LEN];
    CHAR BuildIdVerOld[MAX_VER_STR_LEN];
    BYTE count = 0, idx = 0, DigitsInOldVer = 0 ,DigitsInNewVer = 0;
    int diff = 0;
    CHAR DigitsAfterString[3] = {'\0','\0','\0'};
    struct sGetDeviceSeriesResult result;

    strcpy( ModemImgStr, pModemImgBuildID );
    strcpy( OldModemImgStr, pOldModemImgBuildID );

    /* Extract the image informations */
    if ( !ra_ExtractModemImgInfo(pModemImgBuildID, &ImgInfo,ModemBuildIdLen ) ||
         !ra_ExtractModemImgInfo(pOldModemImgBuildID, &OldImgInfo, OldModemBuildLen))
    {
        return FW_DIFFERENT_CARRIER;
    }

    /* Make sure that the carrier, technology and region are the same */
    if ((ImgInfo.dev.g.Carrier    != OldImgInfo.dev.g.Carrier) ||
        (ImgInfo.dev.g.Region     != OldImgInfo.dev.g.Region) ||
        (ImgInfo.dev.g.Technology != OldImgInfo.dev.g.Technology))
    {
        return FW_DIFFERENT_CARRIER;
    }

    /* Get the Modem Image version */
    pModemBuildIDVer = strtok ( ModemImgStr, "-" );
    pModemBuildIDVer = strtok ( NULL, "-" );
    pModemBuildIDVer = strtok ( NULL, "-" );

    /* Get the old Modem Image version */
    pOldModemBuildIDVer = strtok ( OldModemImgStr, "-" );
    pOldModemBuildIDVer = strtok ( NULL, "-" );
    pOldModemBuildIDVer = strtok ( NULL, "-" );
    
    eGetDeviceSeries(&result);
    if(result.uResult==eGOBI_DEV_SERIES_SIERRA_GOBI)
    {
        for( offset = VERSION_ID_OFFSET; offset <= strlen(pModemBuildIDVer); offset++)
        {
            newModemBuildIDVer[newImageCount] = pModemBuildIDVer[offset];
            newImageCount++;
        }
        for( offset = VERSION_ID_OFFSET; offset<= strlen(pOldModemBuildIDVer); offset++)
        {
            oldModemBuildIDVer[oldImageCount] = pOldModemBuildIDVer[offset];
            oldImageCount++;
        }
        if(atol(newModemBuildIDVer) > atol(oldModemBuildIDVer))
        {
            return FW_SAME_CARRIER_NEW;
        }
        else
        {
            return FW_SAME_CARRIER_OLD;
        }
    }
    else
    {
        /* Check if AMSS iamge is Gobi3.3 or higher
         * (to allow Gobi5.5 down the line)
         */
         if( pModemImgBuildID[GOBI33_BUILD_ID_VER_INITIALS_AMSS] == '3' &&
             pModemImgBuildID[GOBI33_BUILD_ID_VER_INITIALS_AMSS + 1] > '0' )
         {
             if( pOldModemImgBuildID[GOBI33_BUILD_ID_VER_INITIALS_AMSS] == '3' &&
                 pOldModemImgBuildID[GOBI33_BUILD_ID_VER_INITIALS_AMSS + 1] > '0' )
             {
                 /* Read and store digits in NULL terminated Version string */
                 for ( idx = 0; pOldModemBuildIDVer[idx] != '\0'; idx++ )
                 {
                    if ( pOldModemBuildIDVer[idx] >= '0' && pOldModemBuildIDVer[idx] <= '9' )
                    {
                        BuildIdVerOld[DigitsInOldVer] = pOldModemBuildIDVer[idx];
                        DigitsInOldVer++;
                    }
                 }

                 for ( idx = 0; pModemBuildIDVer[idx] != '\0'; idx++ )
                 {
                     if ( pModemBuildIDVer[idx] >= '0' && pModemBuildIDVer[idx] <= '9')
                     {
                         BuildIdVerNew[DigitsInNewVer] = pModemBuildIDVer[idx];
                         DigitsInNewVer++;
                     }
                 }

                 /* If digits in version string are not equal do Normalization */
                 diff = DigitsInOldVer -DigitsInNewVer;
                 if ( diff > 0 )
                 {
                     /* Save digits "zz" after YY string i.e. carrier string */
                     DigitsAfterString[0] = BuildIdVerNew[DigitsInNewVer-2];
                     DigitsAfterString[1] = BuildIdVerNew[DigitsInNewVer-1];

                     for( count = 0; diff > count; count++)
                     {
                         BuildIdVerNew[DigitsInNewVer -2 + count]= '0';
                     }
                     /* Terminate buffer with NULL */
                     BuildIdVerNew[DigitsInNewVer-2 + count ]= '\0';
                     /* Append digits "zz" in buffer after Normalization */
                     strcat(BuildIdVerNew,DigitsAfterString);
                 }

                 if( diff < 0 )
                 {
                     /* To make diff a positive number */
                     diff *= -1;
                     DigitsAfterString[0] = BuildIdVerOld[DigitsInOldVer-2];
                     DigitsAfterString[1] = BuildIdVerOld[DigitsInOldVer-1];
                     for( count = 0; diff > count; count++)
                     {
                         BuildIdVerOld[DigitsInOldVer -2 + count]= '0';
                     }
                     BuildIdVerOld[DigitsInOldVer-2 + count ]= '\0';
                     strcat(BuildIdVerOld,DigitsAfterString);
                 }

                 /*
                  * Check to see which of the FW is newer:
                  * Only of AMSS Images are Gobi3.3 or higher
                  */
                 if (atol(BuildIdVerNew) > atol(BuildIdVerOld))
                 {
                     return FW_SAME_CARRIER_NEW;
                 }
                 else
                 {
                     return FW_SAME_CARRIER_OLD;
                 }
             }
             else
             {
                  /* Check to see which of the FW is newer */
                  return FW_SAME_CARRIER_NEW;
             }
         }
         /* Check to see which of the FW is newer: Only if AMSS Images Gobi1.7 */
         if (atoi(pModemBuildIDVer) > atoi(pOldModemBuildIDVer))
         {
             return FW_SAME_CARRIER_NEW;
         }
         return FW_SAME_CARRIER_OLD;
    }
}

/* Name:    ra_GetFinalImageList
 *
 * Purpose: Gets the Final PRI image index from the list.
 *
 * param [out] pPriImgEntry
 *    The list of PRI images on the device
 *
 * return
 *     None
 *
 * note
 *      The list which needs to be displayed to the user is based on two
 *      conditions:
 *          - The Image is the latest Image (based on build ID version )
 *          - The Image is ACTIVE if not the latest
 */
void ra_GetFinalImageList( struct ImageIDEntries *pPriImgEntry )
{
    BYTE idx = 0, idx_old = 0;

    /* Clear the Image list structure */
    memset( (void *)&devImgFinalList, 0, sizeof(devImgFinalList) );

    /* Extract the information from each PRI image */
    for( idx = 0; idx < pPriImgEntry->imageIDSize; idx++ )
    {
        struct ImageIdElement *pImgElement = &pPriImgEntry->imageIDElement[idx];

        /* Set this by default to true */
        devImgFinalList[idx].IsLatest = true;

        if ( !ra_ExtractPRIImgInfo( pImgElement->buildID,
                                        &devImgFinalList[idx].imgInfo,
                                        pImgElement->buildIDLength ) )
        {
            fprintf( stderr, "Unknown Image : %s\n",
                              pPriImgEntry->imageIDElement[idx].buildID );
            devImgFinalList[idx].IsLatest = false;
            continue;
        }

        /*
         * Cross check with the other PRI Images and proceed to display only
         * when the FW is new or if the FW is currently active one on the host
         */
        for( idx_old = 0; idx_old < idx; idx_old++ )
        {
            struct ImageIdElement *pOldImgElement =
                         &pPriImgEntry->imageIDElement[idx_old];

            /* Check to see if we found a new FW, of the same carrier */
            switch( ra_IsFwNew( pImgElement->buildID,
                             pOldImgElement->buildID,
                             pImgElement->buildIDLength,
                             pOldImgElement->buildIDLength) )
            {
                case FW_SAME_CARRIER_NEW:
                     /* Change the status if this is not an executing image */
                    if( idx_old != pPriImgEntry->executingImage ) {
                        devImgFinalList[idx_old].IsLatest = false;
                    }
                    break;
                case FW_SAME_CARRIER_OLD:
                    /* Change the status if this is not an executing image */
                    if( idx != pPriImgEntry->executingImage ) {
                        devImgFinalList[idx].IsLatest = false;
                    }
                    break;
                case FW_DIFFERENT_CARRIER:
                default:
                    break;
            }

            /*
             * Break from the loop if the current Image is neither the latest
             * nor the active one on the host
             */
            if( !devImgFinalList[idx].IsLatest )
                break;
        }
    }
}

/*
 * Name:    ra_GetFinalModemImageList
 *
 * Purpose: Gets the Final PRI image index from the list.
 *
 * param    [out] pPriImgEntry
 *          The list of PRI images on the device
 *
 * return
 *          None
 *
 * note
 *      The list which needs to be displayed to the user is based on two
 *      conditions:
 *          - The Image is the lastest Image (based on build ID version )
 *          - The Image is ACTIVE if not the latest
 */
static void ra_GetFinalModemImageList( struct ImageIDEntries *pModemImgEntry )
{
    BYTE idx = 0, idx_old = 0;

    /* Clear the Image list structure */
    memset( (void *)&ModemImgFinalList, 0, sizeof(ModemImgFinalList) );

    /* Extract the information from each PRI image */
    for( idx = 0; idx < pModemImgEntry->imageIDSize; idx++ )
    {
        struct ImageIdElement *pImgElement = &pModemImgEntry->imageIDElement[idx];

        /* Set this by default to true */
        ModemImgFinalList[idx].IsLatest = true;

        if ( !ra_ExtractModemImgInfo( pImgElement->buildID,
                                   &ModemImgFinalList[idx].imgInfo,
                                   pImgElement->buildIDLength) )
        {
            /* Do not print unknown Image for fail safe image of SL9090 and Gobi3k v3.3*/
            if( 'X' != pImgElement->buildID[SL9090_BUILD_ID_CARRIER_INITIALS_OFFSET] &&
                'X' != pImgElement->buildID[GOBI_BUILD_ID_CARRIER_INITIALS_OFFSET] )
            {
                fprintf( stderr, "Unknown Image : %s\n", pModemImgEntry->imageIDElement[idx].buildID );
            }
            ModemImgFinalList[idx].IsLatest = false;
            continue;
        }

        /* Check if the image ID is zero*/
        if( strncmp( (const char *)pImgElement->imageID,
                     imgIddefAMSSImage,
                     imageIDLen ) == 0 )
        {
            /* Avoid setting the fail safe AMSS image as latest image */
            ModemImgFinalList[idx].IsLatest = false;
            continue;
        }

        /*
         * Cross check with the other PRI Images and proceed to display only
         * when the FW is new or if the FW is currently active one on the host
         */
        for( idx_old = 0; idx_old < idx; idx_old++ )
        {
            struct ImageIdElement *pOldImgElement =
                         &pModemImgEntry->imageIDElement[idx_old];
            /* Check if the image ID is zero*/
            if( strncmp( (const char *)pOldImgElement->imageID,
                        imgIddefAMSSImage,
                        imageIDLen ) == 0 )
            {
                /* Ignore the fail safe AMSS image for cross checking for latest image */
                continue;
            }
            /* Check to see if we found a new FW, of the same carrier */
            switch( ra_IsFwModemImageNew( pImgElement->buildID,
                                       pOldImgElement->buildID,
                                       pImgElement->buildIDLength,
                                       pOldImgElement->buildIDLength ) )
            {
                case FW_SAME_CARRIER_NEW:
                     /* Change the status if not latest image */
                        ModemImgFinalList[idx_old].IsLatest = false;
                    break;
                case FW_SAME_CARRIER_OLD:
                    /* Change the status if this is not latest image */
                        ModemImgFinalList[idx].IsLatest = false;
                    break;
                case FW_DIFFERENT_CARRIER:
                default:
                    break;
            }

            /*
             * Break from the loop if the current Image is neither the latest
             * nor the active one on the host
             */
            if( !ModemImgFinalList[idx].IsLatest )
                break;
        }
    }
}

/*
 * Name:    ra_Scanfile
 *
 * Purpose: Scans carrierImageOverride file line by line
 *
 * param   [in] filename
 *          File Path
 *
 * param   [in] Buffer
 *          Buffer to store carrierImageOverride file content.
 *
 * return
 *         The carrier substring as present in the carrierImageOverride file.
 *
 * note
 *         None
 */
bool ra_Scanfile( char *filename, char *buffer )
{
    FILE *file = fopen ( filename, "r" );
    char *pch;
    if ( file != NULL )
    {
        /* Assume Maximum line length is not more than 100 characters */
        char line [ 100 ];

        /* Read a line */
        while ( fgets ( line, sizeof line, file ) != NULL )
        {
            /* Discard the content if read line is comment */
            pch = strstr(line,"#");
            if( !pch )
            {
                strcat(buffer, line);
            }
        }
        /* Close file */
        fclose ( file );
        return true;
    }
    else
    {
        return false;
    }
}

/*
 * Name:    ra_IsAptModemImagePresent
 *
 * Purpose: Check whether the modem image has a version according to the carrierImageOverride file
 *
 * param    [in] carrier
 *          The carrier of the Modem image
 *
 * param    [in] version
 *          The version of the Modem image
 *
 * return
 *          true if function succeed
 *          false if function failed
 *
 * note
 *          None
 */
bool ra_IsAptModemImagePresent( BYTE modemImdIdx, ULONG carrier, ULONG version )
{
    char buffer[1000];
    char *pVersion, *pCarrier;
    bool nRet = false,fileScan = false;
    struct sGetDeviceSeriesResult result;

    /* Initialize path for carrierImageOverride.txt file
     * to point at directory where we fire build for application */
    eGetDeviceSeries(&result);
    if(result.uResult==eGOBI_DEV_SERIES_SIERRA_GOBI)
    {
        configFilePath = "./carrierImageOverrideSierraGobi.txt.";
    }
    else
    {
        configFilePath = "./carrierImageOverride.txt";
    }

    /* Scan file line by line */
    fileScan = ra_Scanfile( configFilePath, buffer );

    /*
     * If file scan fails, Default to normal image switching.
     * This could be a case when file is not present in make directory
     */
    if( !fileScan )
    {
        /* If modem image at present index is latest? */
        if ( !ModemImgFinalList[modemImdIdx].IsLatest)
        {
            /* Check for other images present in modem */
            return false;
        }
        else
        {
            /* If modem Image latest activate this */
            return true;
        }

    }

    /* If the file is present, check for the configure file for compatibility */
    if ( fileScan )
    {
        pCarrier = ra_GetCarrierString(carrier);

        /* If the carrier information is present in the file, extract version */
        if( pCarrier )
        {
            pVersion = strstr(buffer, pCarrier);

            if( pVersion )
            {
                strtok(pVersion, "=");
                pVersion = strtok(NULL, "\n");
                nRet = true;
            }
            else
            {
                /* If modem image at present index is latest? */
                if ( !ModemImgFinalList[modemImdIdx].IsLatest)
                {
                    /* Check for other images present in modem */
                    return false;
                }
                else
                {
                    /* If modem Image latest activate this */
                    return true;
                }
            }
        }

        /*
         * Once the version is extracted, compare with the version of the
         * image on the device
         */
        if ( nRet )
        {
            if ( (ULONG)atol(pVersion) == version )
            {
                nRet = true;
            }
            else
            {
                nRet = false;
            }
        }
    /* If the file is not present select the first image found, hence true */
    }
    else
    {
        nRet = true;
    }
    return nRet;
}

/* Name:    ra_CheckModemImage
 *
 * Purpose: Check whether the corresponding (with respect to the PRI image in the
 *          mentioned imageIndex) modem image is present
 *
 * param    [in] imageIndex
 *          The image index of the PRI image$
 *
 * param    [in] modemImdIdx,
 *          The modem image index of the Modem image
 *
 * param    [in] pModemImgBuildID
 *          The build ID of the Modem image
 *
 * return
 *          true if function succeed
 *          false if function failed
 *
 * note
 *          None
 */
bool ra_CheckModemImage( BYTE imageIndex, BYTE modemImdIdx, CHAR *pModemImgBuildID, ULONG BuildIDlen )
{
    struct qmifwinfo_s ModemImgInfo, *pPRIImgInfo;
    CHAR *pModemBuildIDVer, *pNewModemBuildIDVer = NULL;
    CHAR ModemImgStr[100];

    strcpy( ModemImgStr, pModemImgBuildID );
    pPRIImgInfo = &devImgFinalList[imageIndex].imgInfo;
    ra_ExtractModemImgInfo( pModemImgBuildID, &ModemImgInfo, BuildIDlen );

    /* Get the modem build ID Image version */
    pModemBuildIDVer = strtok ( ModemImgStr, "-" );
    pModemBuildIDVer = strtok ( NULL, "-" );
    pModemBuildIDVer = strtok ( NULL, "-" );

    /*
     * If the carrier, region and technology are the same; check whether modem
     * image has a similar or higher version present in the configuration file
     */
    if (((ModemImgInfo.dev.g.Technology == eGOBI_IMG_TECH_UMTS) &&
         (ModemImgInfo.dev.g.Carrier    == eGOBI_IMG_CAR_GENERIC) &&
         (ModemImgInfo.dev.g.Technology == pPRIImgInfo->dev.g.Technology)) ||
        ((ModemImgInfo.dev.g.Technology == eGOBI_IMG_TECH_CDMA) &&
         (ModemImgInfo.dev.g.Carrier    == pPRIImgInfo->dev.g.Carrier) &&
         (ModemImgInfo.dev.g.Region     == pPRIImgInfo->dev.g.Region) &&
         (ModemImgInfo.dev.g.Technology == pPRIImgInfo->dev.g.Technology)))
    {
         /* Is it a SL9090 */
        struct sGetDeviceSeriesResult result;
        eGetDeviceSeries(&result);
        if(result.uResult==eGOBI_DEV_SERIES_SIERRA_GOBI)
        {
            pNewModemBuildIDVer = pModemBuildIDVer + VERSION_ID_OFFSET;
            return ra_IsAptModemImagePresent ( modemImdIdx,
                     pPRIImgInfo->dev.g.Carrier, atol(pNewModemBuildIDVer));
        }
        else
        {
            if( pModemImgBuildID[GOBI33_BUILD_ID_VER_INITIALS_AMSS] == '3' &&
                pModemImgBuildID[GOBI33_BUILD_ID_VER_INITIALS_AMSS +1 ] > '0')
            {
                /* If modem image at present index is latest? */
                if ( !ModemImgFinalList[modemImdIdx].IsLatest)
                {
                    /* Check for other images present in modem */
                    return false;
                }
                else
                {
                    /* If modem Image latest activate this */
                    return true;
                }
             }
             else
             {
                 return ra_IsAptModemImagePresent ( modemImdIdx,
                                                 pPRIImgInfo->dev.g.Carrier,
                                                 atoi(pModemBuildIDVer));
             }
       }
    }
    return false;
}
/* Name:    ra_PrintDeviceModel
 *
 * Purpose: Prints the model of the device connected.
 *
 * param    None
 *
 * return
 *          None
 *
 * note
 *          None
 */
void ra_PrintDeviceModel( void )
{
    BYTE stringsize = 128;
    CHAR ModelId[stringsize];

    GetModelID( stringsize,
                ModelId );
    fprintf( stderr, "Device Connected: %s\n", ModelId );
}

/*************************************************************************
 * Option 1 - Display information about host images.
 ************************************************************************/
/*
 * Name:     ra_GetImgInformation
 *
 * Purpose:  Retrieve the Image information located at the specified path.
 *
 * Return:   None
 *
 * Notes:    none
 */
static ULONG ra_GetImgInformation(
    CHAR                *pImagePath,
    const CHAR          *pImgFolderName,
    struct qmifwinfo_s  *pImgInfo )
{
    CHAR  completeImagePath[MAX_IMAGE_PATH];

    /* Intialize the image path buffer */
    memset( completeImagePath, 0, MAX_IMAGE_PATH );

    if( (strlen((CHAR *)pImagePath) + strlen((const CHAR *)pImgFolderName)) >=  MAX_IMAGE_PATH)
    {
            fprintf( stderr, "Image Folder Path Too Long\n" );
            return FAIL;
    }

    /* Copy the path and file name into the image path buffer */
    strcpy( completeImagePath, (const char *)pImagePath);
    strcat( completeImagePath, pImgFolderName );
    strcat( completeImagePath, "/" );

    /* Get the Image information */
    return SLQSGetImageInfo( completeImagePath,
                             pImgInfo);
}

/*************************************************************************
 * Option 2 - Display information about device images.
 ************************************************************************/
/*
 * Name:     ra_ListDeviceImages
 *
 * Purpose:  Display the list of carrier images loaded in the device.
 *
 * Return:   true  - If the image listing is successful
 *           false - If the image listing fails.
 *
 * Notes:    none
 */
bool ra_ListDeviceImages( BOOL MDMImgMatch )
{
    struct ImageIDEntries *pPriImgEntry, *pModemImgEntry = NULL;
    ULONG            resultCode, size = 0;
    BYTE             imgListIdx = 0;
    BYTE             activeImageFound = eACTIVE_IMG_DIS, idx = 0;
    struct ImageIDEntries *pActModemImg = NULL;
    BYTE modemImdIdx = 0;
    int modemImageFound = false;;

    size = sizeof( devImgList );

    /* Clear the Image list structure */
    memset( (void *)&devImgList, 0, size );

    /* Retrieve the information about the images loaded in device, SDK function call */
    resultCode = GetStoredImages( &size, (BYTE *)&devImgList );

    /* Return if we receive an error */
    if( SUCCESS != resultCode )
    {
        fprintf( stderr, "ListDeviceImages failed\n"\
                         "Failure code : %u\n", resultCode );
        return false;
    }

    /* Figure out the index of PRI image type in the received image list */
    if( !ra_GetPRIImageIdx( &imgListIdx ) )
    {
        fprintf(stderr,"\tFAIL SAFE Image Active!!\n\n" );
        return false;
    }

    /* Store the pointer to PRI and MODEM image list */
    pPriImgEntry   = &( devImgList.imageIDEntries[imgListIdx] );
    pModemImgEntry = &( devImgList.imageIDEntries[ ( !imgListIdx) ] );

    /* Get final image list of UQCN(PRI) Images */
    ra_GetFinalImageList( pPriImgEntry );

    /* Get Final list of AMSS(MODEM) Images */
    ra_GetFinalModemImageList( pModemImgEntry );

    int w10=-10, w20=-20;
    fprintf( stderr,
             "%3$*1$s%4$*2$s%5$*2$s%6$*2$s%7$*2$s\n",
             w10, w20, "Id", "Carrier", "Technology", "Region", "Status" );

    /* Extract the information from each PRI image */
    for( idx = 0; idx < pPriImgEntry->imageIDSize; idx++ )
    {
        /* Consider image as inactive by default */
        activeImageFound = eACTIVE_IMG_DIS;

        /*  If fail safe image is active, do not consider PRI image */
        if( FAIL_SAFE_IMG_IDX != pModemImgEntry->executingImage )
        {
             /* If the current image is executing image */
            if( idx == pPriImgEntry->executingImage )
            {
                /* Set the value to display the image as active */
                activeImageFound = eACTIVE_IMG;
            }
            else
            {
                activeImageFound = eACTIVE_IMG_DIS;
            }
        }


        if ( devImgFinalList[idx].IsLatest )
        {
            /* Check if Modem Image match is requested before displaying */
            if( TRUE == MDMImgMatch )
            {
                /* Store the pointer to Modem image list */
                pActModemImg = &( devImgList.imageIDEntries[!imgListIdx] );

                /* Check if the corresponding MODEM image exist on device */
                for( modemImdIdx = 0;
                     modemImdIdx < pActModemImg->imageIDSize;
                     modemImdIdx++ )
                {
                     CHAR *pbuildID = pActModemImg->imageIDElement[modemImdIdx].buildID;

                     /* If the corresponding MODEM image is found */
                     if ( ra_CheckModemImage( idx, modemImdIdx, pbuildID,
                                           pActModemImg->imageIDElement[modemImdIdx].buildIDLength ))
                     {
                         modemImageFound = true;
                         break;
                     }
                }

                /* If corresponding MODEM image does not exist on the device, return */
                if( !modemImageFound )
                {
                    continue;
                }

                /* Reset the flag to false */
                modemImageFound = false;
            }

            fprintf( stderr, "%2$*1$d", w10, idx + 1);
            /* Display the image information */
            ra_DisplayImageInfo( activeImageFound, &devImgFinalList[idx].imgInfo );
        }
    }

    /* If no image is active, then fail safe image will be executing */
    if( ( UNKNOWN_EXECUTING_IMG_IDX == pPriImgEntry->executingImage ) ||
        ( FAIL_SAFE_IMG_IDX == pModemImgEntry->executingImage ) )
    {
        fprintf( stderr, "\tFAIL SAFE Image Active!!\n\n" );
    }
    return true;
}

/*
 * Name:     ra_ListHostImages
 *
 * Purpose:  Display the list of carrier images available for download at the
 *           path specified by user.
 *
 * Return:   true      - If the image listing is success
 *           ENTER_KEY - If <ENTER> key is pressed.
 *
 * Notes:    none
 */
static bool ra_ListHostImages(CHAR *InputImagePath)
{
    CHAR completeImagePath[MAX_IMAGE_PATH];
    while(1)
    {
        /* query user for path to directory containing carrier image folders */
        if(false==ra_ValidateImagePath( InputImagePath ))
            return false;

        /* Display the headings */
        int w10=-10, w20=-20;
        fprintf( stderr,
                 "%3$*1$s%4$*2$s%5$*2$s%6$*1$s%7$*1$s%8$*2$s\n",
                 w10, w20, "Id", "Carrier", "Technology", "Region", "Version", "GPS Support" );

        memset( completeImagePath, 0, MAX_IMAGE_PATH );
        strcpy( completeImagePath, InputImagePath);
        int offset = strlen(completeImagePath);

        BYTE idx;
        for ( idx = 0 ; idx < folderIdx; idx++ )
        {
            /* Copy the path and file name into the image path buffer */
            sprintf( &completeImagePath[offset], "%c",imgFolderIdx[idx] );

            /* get the image info */
            struct qmifwinfo_s imgInfo;
            ULONG rc = SLQSGetImageInfoMC83xx( completeImagePath, &imgInfo );
            if ( SUCCESS == rc )
            {
                /* Display image information in verbose mode */
                fprintf( stderr, "%2$*1$c", w10, imgFolderIdx[idx] );
                ra_DisplayImageInfo( eACTIVE_IMG_NO_DIS, &imgInfo );
            }
        }
        folderIdx = 0;

        /* if image listing is displayed for downloading the image, exit */
        if( dwlImage )
        {
            dwlImage = false;
            return true;
        }
    }
}



/****************************************************************
*                       FUNCTIONS
****************************************************************/
/*
 * Name:     ra_DevStateChgCbk
 *
 * Purpose:  Device State change callback
 *
 * Return:   None
 *
 * Notes:    none
 */
void ra_DevStateChgCbk(eDevState devstatus)
{
    /* If device is ready to communicate */
    if( devstatus ==  DEVICE_STATE_READY )
    {
        fprintf( stderr, "\nDevice Ready\n" );

        /* Unsubscribe from the callback */
        SetDeviceStateChangeCbk(NULL);
        fwDwlComplete = true;
    }
}

/*
 * Name:     ra_FirmwareDwlCbk
 *
 * Purpose:  Firmware download completion callback
 *
 * Return:   None
 *
 * Notes:    none
 */
void ra_FirmwareDwlCbk(ULONG status)
{
    if (0 == status)
    {
        fprintf( stderr, "\nFirmware Download Completed" );
    }
    else if (eQCWWAN_ERR_SWIIM_FIRMWARE_NOT_DOWNLOADED == status)
    {
        fprintf( stderr, "\nFirmware Not Downloaded" );
    }
    /* set firmware complete to true */
    fwDwlComplete = true;

    /* Unsubscribe from the callback */
    SetFwDldCompletionCbk(NULL);
}


/*************************************************************************
 * Option 6 - Select a carrier image resident on device as the new active image.
 ************************************************************************/
/*
 * Name:     ra_ActivateSelectedImage
 *
 * Purpose:  List all the images present on the device and then activate the
 *           image on the device selected by the user.
 *
 * Return:   none
 *
 * Notes:    none
 */
bool ra_ActivateSelectedImage(BYTE inputindex)
{
    struct PrefImageList prefImageList;
    struct ImageIdElement *pActPRIImg = NULL;
    struct ImageIDEntries *pActModemImg = NULL;
    ULONG resultCode = 0, imageListSize = 0, imageTypeSize = 0;
    int modemImageFound = false;
    BYTE modemImdIdx = 0, imageIndex = 0, priImageIdx = 0;
    BYTE imageType[IMG_BUF_SIZE];

    /* If we fail to list the device images, return */
    if( !ra_ListDeviceImages(TRUE) )
    {
        return false;
    }

    while(1)
    {
        /* Receive the user input */
        imageIndex = ra_GetUserImageId( eDEL_IMG, NULL, inputindex );

        /* Decrement the image index selected by the user */
        imageIndex--;

        /* Get the PRI image index from the Image list */
        if( !ra_GetPRIImageIdx( &priImageIdx ) )
        {
            fprintf(stderr,"\tFAIL SAFE Image Active!!\n\n" );
            return false;
        }

        /* Store the pointer to PRI image list */
        pActPRIImg = &( devImgList.imageIDEntries[priImageIdx].\
                          imageIDElement[imageIndex] );

        /* Fill the PrefImageList structure with PRI image info */
        prefImageList.listSize = 1;
        prefImageList.listEntries[0].imageType = IMG_TYPE_PRI;
        memcpy( (void *)prefImageList.listEntries[0].imageId,
                (void *)pActPRIImg->imageID,
                GOBI_MBN_IMG_ID_STR_LEN );

        prefImageList.listEntries[0].buildIdLength = pActPRIImg->buildIDLength;
        if( 0 != pActPRIImg->buildIDLength )
        {
            strcpy( prefImageList.listEntries[0].buildId,
                    pActPRIImg->buildID );
        }

        /* Store the pointer to Modem image list */
        pActModemImg = &( devImgList.imageIDEntries[!priImageIdx] );

        /* Check if the corresponding MODEM image exist on device */
        for( modemImdIdx = 0;
             modemImdIdx < pActModemImg->imageIDSize;
             modemImdIdx++ )
        {
             CHAR *pbuildID = pActModemImg->imageIDElement[modemImdIdx].buildID;
             #ifdef DBG
             fprintf(stderr, "Modem image index number:%d\n ", modemImdIdx);
             #endif
             /* If the corresponding MODEM image is found */
             if ( ra_CheckModemImage( imageIndex, modemImdIdx, pbuildID,
                                   pActModemImg->imageIDElement[modemImdIdx].buildIDLength ))
             {
                 modemImageFound = true;
                 break;
             }
        }

        /* If corresponding MODEM image does not exist on the device, return */
        if( !modemImageFound )
        {
            fprintf( stderr,"Failed to activate selected image\n"\
                            "Failure reason : Corresponding MODEM image does not exist on the device\n");
            //continue;
            return false;
        }

        /* Reset the flag to false */
        modemImageFound = false;

        /* MODEM image exist, retrieve the information */
        prefImageList.listSize++;
        prefImageList.listEntries[1].imageType = IMG_TYPE_MODEM;
        memcpy( (void *)prefImageList.listEntries[1].imageId,
                (void *)pActModemImg->imageIDElement[modemImdIdx].imageID,
                GOBI_MBN_IMG_ID_STR_LEN );
        prefImageList.listEntries[1].buildIdLength = pActModemImg->imageIDElement[modemImdIdx].buildIDLength;
        if( 0 != pActModemImg->imageIDElement[modemImdIdx].buildIDLength )
        {
            strcpy( prefImageList.listEntries[1].buildId,
                    pActModemImg->imageIDElement[modemImdIdx].buildID );
        }

        imageListSize = sizeof( prefImageList );
        imageTypeSize = IMG_BUF_SIZE;
        memset( (void *)imageType, 0, imageTypeSize );

        /* Set the preference for selected image in device */
        resultCode = SetImagesPreference( imageListSize,
                                          (BYTE *)&prefImageList,
                                          0,
                                          0xFF,
                                          &imageTypeSize,
                                          imageType );

        if( SUCCESS != resultCode )
        {
            fprintf( stderr, "Failed to activate selected image\n"\
                             "Failure Code : %u\n", resultCode );
            //continue;
            return false;
        }

        /* Subscribe to Device State Change callback */
        resultCode = SetFwDldCompletionCbk(ra_FirmwareDwlCbk);
        if( SUCCESS != resultCode )
        {
            fprintf( stderr, "FwDwldCompletionCallback failed.\n"\
                             "Failure Code : %u\n", resultCode );
            return false;
        }

        /* Reset the device to activate the image */
        resultCode = SetPower( RESET_MODEM );
        if( SUCCESS != resultCode )
        {
            fprintf( stderr, "Failed to reset modem\n"\
                             "Failure Code : %u\n", resultCode );
            //continue;
            return false;
        }

        /* Keep displaying "." until selected image gets activated */
        fprintf( stderr, "Activating Selected Image. . .");
        fwDwlComplete = false;
        while( !fwDwlComplete )
        {
            fprintf( stderr, " .");
            sleep(1);
        }

        fprintf( stderr, "\nRetrieved Image list from the device\n");

        /* If we fail to list the device images, return */
        if( !ra_ListDeviceImages(TRUE) )
        {
            return false;
        }

        imageIndex = ( devImgList.imageIDEntries[priImageIdx].executingImage );
        pActPRIImg = &( devImgList.imageIDEntries[priImageIdx].\
                          imageIDElement[imageIndex] );

        /* Check if the selected image gets activated */
        if( SUCCESS == strcmp( pActPRIImg->buildID,
                               prefImageList.listEntries[0].buildId ) )
        {
            fprintf( stderr, "Selected Image activated successfully\n");
            return false;
        }
        else
        {
           fprintf( stderr, "Failed to activate selected image\n");

           #ifdef DBG
           fprintf( stderr, "Selected build id = %s\n",prefImageList.listEntries[0].buildId);
           fprintf( stderr, "Activated build id = %s\n",pActPRIImg->buildID);
           #endif
           return false;
        }
    }
}


/*************************************************************************
 * Option 4 - Download carrier image to the device.
 ************************************************************************/
/*
 * Name:     DownloadImage
 *
 * Purpose:  List all the images present on the device, prompt the user for the
 *           image path on the host, display their information, prompt again for
 *           the index of the image to be downloaded and then download that
 *           image to the device.
 *
 * Return:   none
 *
 * Notes:    none
 */
bool ra_FirmwareDownloader_Gobi(CHAR *InputImagePath)

{
    int              dwlImageId = 0;
    bool             returnValue = false;
    ULONG            resultCode = 0;
    ULONG            firmwareId = 0;
    struct qmifwinfo_s imgInformation;
    CHAR completeImagePath[MAX_IMAGE_PATH];
    struct sGetDeviceSeriesResult result;

    memset( completeImagePath, 0, MAX_IMAGE_PATH );
    strcpy( completeImagePath, InputImagePath);
        
        
    fprintf( stderr, "List Of images present on the device:\n" );
    
    eGetDeviceSeries(&result);
    
    if( (eGOBI_DEV_SERIES_G3K !=result.uResult)  && 
        (result.uResult!=eGOBI_DEV_SERIES_SIERRA_GOBI))
        return false;       

    ra_PrintDeviceModel();

    /* If we fail to list the device images, return */
    if( !ra_ListDeviceImages(TRUE) )
    {
        return false;
    }

    /* Prompt the user for path of the imges located on host and display the
     * information
     */
    dwlImage    = true;
    
    
    returnValue = ra_ListHostImages(completeImagePath);

    /* if the user has pressed <enter> key, return to the main menu */
    if( !returnValue )
    {
        dwlImage = false;
        return false;
    }
    

    /* Prompt the user to select an index of the image to be downloaded to the
     * device from above displayed list
     */
    dwlImageId = ra_GetUserImageId( eDWL_IMG, NULL, 1 );

    /* if image id provide by user is greater than 4, decrement it by 1 */
    if( 4 < dwlImageId )
    {
        dwlImageId--;
    }

    /* Get the image information about downloading image */
    resultCode = ra_GetImgInformation( completeImagePath,
                                    pImgFolderName[dwlImageId],
                                    &imgInformation );
    if ( SUCCESS == resultCode )
    {
        fprintf( stderr, "Image Listing SUCCESS for %s%s\n", completeImagePath, pImgFolderName[dwlImageId] );
        /* Display the headings */
        int w10=-10, w20=-20;
        fprintf( stderr,
                 "%3$*1$s%4$*2$s%5$*2$s%6$*1$s%7$*1$s%8$*2$s\n",
                 w10, w20, "Id", "Carrier", "Technology", "Region", "Version", "GPS Support" );

        if( 4 < dwlImageId )
        {
            fprintf( stderr, "%2$*1$d", w10, dwlImageId + 1 );
        }
        else
        {
            fprintf( stderr, "%2$*1$d", w10, dwlImageId );
        }

    }
    else
    {
        fprintf( stderr, "Image Listing FAILED for %s%s\n", completeImagePath, pImgFolderName[dwlImageId] );
        fprintf( stderr, "Failure Code : %u\n", resultCode );
        return false;
    }

    /* Save the firmware id for the image being downloaded */
    firmwareId = imgInformation.dev.g.FirmwareID;

    /* Copy the path and file name into the image path buffer */
    strcat( completeImagePath, pImgFolderName[dwlImageId] );
    strcat( completeImagePath, "/" );

    fprintf( stderr, "Path for the image to download : %s\n", completeImagePath );

    /* Reaching here means, MBN image is found at specified path */
    /* Subscribe to Device State Change callback */
    resultCode = SetDeviceStateChangeCbk(ra_DevStateChgCbk);
    if( SUCCESS != resultCode )
    {
        fprintf( stderr, "REGISTRATION FAILED - Device State Change Callback\n"\
                         "Failure Code : %u\n", resultCode );
        return false;
    }

    /* Subscribe to Firmware Download completion callback */
    resultCode = SetFwDldCompletionCbk(ra_FirmwareDwlCbk);
    if( SUCCESS != resultCode )
    {
        fprintf( stderr, "REGISTRATION FAILED - Firmware Download Completion Callback\n"\
                         "Failure Code : %u\n", resultCode );
        return false;
    }

    /* Start downloading the firmware */
    resultCode = UpgradeFirmware2k( completeImagePath );
    if( SUCCESS != resultCode )
    {
        fprintf( stderr, "Firmware Download Failed\n"\
                         "Failure Code : %u\n", resultCode );

        /* De register device state change and firmware download completion
         * callback.
         */
        SetFwDldCompletionCbk( NULL );
        SetDeviceStateChangeCbk( NULL );
        return false;
    }

    /* Keep displaying "." until fimrware downloads complete */
    fprintf( stderr, "Downloading Firmware. . .");
    fwDwlComplete = false;
    while( !fwDwlComplete )
    {
        fprintf( stderr, " .");
        sleep(1);
    }

    memset( (void *)&imgInformation, 0, sizeof( imgInformation ) );

    /* Get the information about the currently running image on device */
    resultCode = SLQSGetFirmwareInfo( &imgInformation );

    /* Check if the firmware download is success */
    if( firmwareId == imgInformation.dev.g.FirmwareID )
    {
        fprintf( stderr, "Firmware Upgrade Success!!!\n" );
        return true;
    }
    else
    {
        #ifdef DBG
        fprintf( stderr, "Selected FW Id : %lx\n Current  FW Id : %lx\n",
                         firmwareId, imgInformation.dev.g.FirmwareID );
        #endif

        fprintf( stderr, "Firmware Upgrade Failed!!!\n" );
        return false;
    }

    fprintf( stderr, "\nList Of images present on the device:\n" );

    /* If we fail to list the device images, return */
    if( !ra_ListDeviceImages(TRUE) )
    {
        return false;
    }

}

