/*************
 *
 * Filename:    main.c
 *
 * Purpose:     Main module of the Sierra Wireless Air Vantage Device Management Tree
 *              adapter library.  This file contains the start-up and shut-down code
 *              for the adapter.
 *
 * Copyright: © 2012 Sierra Wireless Inc., all rights reserved
 *
 **************/

#include "SWIWWANCMAPI.h"
#include "qmerrno.h"
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include "main.h"
#include "devtree_v2.h"


/****************************************************************
*                       DEFINES
****************************************************************/
#define RA_DEBUG    /* Comment this out to turn off debug printf() calls */
#define SUCCESS            0
#define DEV_NODE_SZ        256
#define DEV_KEY_SZ         16


/****************************************************************
*                       TYPEDEFS
****************************************************************/

/* Device information structure */
typedef struct
{
  CHAR deviceNode[DEV_NODE_SZ];
  CHAR deviceKey[DEV_KEY_SZ];
}
device_info_t;


/****************************************************************
 *                    GLOBAL DATA
 ****************************************************************/

/* device connectivity */
static device_info_t DeviceList[1] = { { {'\0'}, {'\0'} } };

/* macros*/
#define rcprint(s, u) syslog(LOG_USER, "%s: rc = 0x%lX, %s", s, u)


/* ***************************************************************
*                       INTERNAL FUNCTIONS
****************************************************************/

/*
 * Name:     StartSDK
 *
 * Purpose:  It starts the SDK by setting the SDK path, enumerates the device
 *           and connects to the SDK.
 *
 * Return:   SUCCESS on successfully starting SDK, else error code
 *
 * Notes:    none
 */
static ULONG StartSDK()
{
    ULONG rc = 0;
    BYTE  devicesSize = 1;

    /* Set SDK image path */
    if( eQCWWAN_ERR_NONE != (rc = SetSDKImagePath("bin/slqssdk")) )
    {
        return rc;
    }

    /* Establish APP<->SDK IPC */
    if( eQCWWAN_ERR_NONE != (rc = SLQSStart_AVAgent(0)) )
    {
        return rc;
    }

    /* Can enumerate and connect only if device is in Application mode */
    if ( main_GetDeviceMode() == DEVICE_STATE_READY )
    {
        /* Enumerate the device */
        rc = QCWWAN2kEnumerateDevices( &devicesSize, (BYTE *)DeviceList );

        /* Return if there is an error */
        if( SUCCESS != rc )
        {
            return rc;
        }

#ifdef RA_DEBUG
    fprintf( stderr,  "#devices: %d\ndeviceNode: %s\ndeviceKey: %s\n",
            devicesSize,
            DeviceList->deviceNode,
            DeviceList->deviceKey );
#endif

        /* Connect to the SDK */
        rc = QCWWANConnect( DeviceList->deviceNode, DeviceList->deviceKey );

        if (SUCCESS != rc)
        {
            return rc;
        }
    }

    return rc;
}


/** Library constructor function.
 *
 * This gets called when the library is first loaded by the process.
 * 
 * It starts up the SDK and prepares for the exchange of Device Management Tree
 * variables with Air Vantage.
 */
static void __attribute__ ((constructor)) ConstructLibrary
(
    void
)
{
    static const char* deviceModeStr = "unknown";

    StartSDK();
    
    /* delay for SDK to initialize */
    sleep(2);

    switch(main_GetDeviceMode())
    {
        case DEVICE_STATE_READY:
            deviceModeStr = "ready";
            break;
            
        case DEVICE_STATE_BOOT:
            deviceModeStr = "boot and hold";
            break;

        case DEVICE_STATE_DISCONNECTED:
            deviceModeStr = "disconnected";
            break;

        default:
            break;
    };

    fprintf( stderr, "\n[DevTree] Running with device in %s mode\n", deviceModeStr);
}


/* ***************************************************************
*  FUNCTIONS EXPORTED TO OTHER MODULES IN THIS ADAPTER LIB
****************************************************************/

/**
 * Fetch the current operating mode of the modem device.
 * 
 * @return The device mode (see qaGobiApiDcs.h).
 */
eDevState main_GetDeviceMode
(
    void
)
{
    /**
     * @todo    Consider optimizing by registering for callbacks from the SLQS
     *          when the device changes modes, and cache the device mode at that time.
     */

    BYTE deviceMode;
    ULONG resultCode = SLQSGetDeviceMode (&deviceMode);
    if (resultCode != eQCWWAN_ERR_NONE)
    {
        fprintf(stderr, "SLQSGetDeviceMode() failed (returned %u).\n", resultCode);
        deviceMode = DEVICE_STATE_DISCONNECTED;
    }
    
    return deviceMode;
}



/* ***************************************************************
*               DEVICE MANAGEMENT TREE FUNCTIONS
****************************************************************/

/** Get Device ID.
 * 
 * This function is used to fetch the device ID that will be used to identify this
 * device to the Air Vantage server.
 * 
 * The device ID is a string of ANSI characters, usually (but not necessarily) the IMEI.
 * 
 * @return Result code.  Anything other than DT2_RESULT_OK indicates a failure, in which
 *         case the string buffer contents are undefined. 
 */
dt2_ResultCode_t dt2_GetDeviceId
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */ 
)
{
    return dt2_GetImei(stringPtr, buffSize);
}
