/*************
 *
 * Filename:    cellular.c
 *
 * Purpose:     Module of the Sierra Wireless Air Vantage Device Management Tree
 *              adapter library that implements the cellular sub-tree parameters
 *              (i.e., the parameters under "system.cellular".
 *
 * Copyright: © 2012 Sierra Wireless Inc., all rights reserved
 *
 **************/

#include "SWIWWANCMAPI.h"
#include "qmerrno.h"
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include "main.h"
#include "devtree_v2.h"

#include <pthread.h>

#define nMaxStrLen              0xFF
#define TRUE 1
#define FALSE 0
#define GPSPOSVALIDMASK 0x00000001
#define GPSALTVALIDMASK 0x00000002
#define FSN_STRING_LENGTH 14
#define CDMA_1xRTT        1
#define CDMA_1xEVDO       2
#define GSM_RI            4
#define UMTS_RI           5
#define LTE_RI            8

/* Check the current device mode */
#define DEV_CHECK \
switch (main_GetDeviceMode()) \
{ \
    case DEVICE_STATE_DISCONNECTED: \
        return DT2_RESULT_DEVICE_NOT_FOUND; \
    case DEVICE_STATE_BOOT: \
        return DT2_RESULT_WRONG_MODE; \
    case DEVICE_STATE_READY: \
        break; \
} 

union {
    double d;
    unsigned long long ull;
} u;

extern BOOL qaRetrieveWDSSessionIDs(
    BYTE   pdpIndex,
    ULONG *pV4sessionId,
    ULONG *pV6sessionId );


int iInitParasedPositionDataMutex();
double dGetLongitude();
double dGetLatitude();
void SetLatitude(double dCachedLatitude);
void SetLongitude(double dCachedLongitude);


int getWirelessTech( )
{
    qaQmiServingSystemParam ServingSystemInfo;

    ULONG nRet;
    BYTE i;
    ULONG technology = 0;

    /* Check if device connected is CDMA */
    nRet = SLQSGetServingSystem( &ServingSystemInfo );

    if ( nRet == 0 && ServingSystemInfo.ServingSystem.numRadioInterfaces > 0)
    {
        /* report the higher technology if several technogies were reported */
        for (i = 0; i < ServingSystemInfo.ServingSystem.numRadioInterfaces; i++)
        {
            if ( technology < ServingSystemInfo.ServingSystem.radioInterface[i])
            {
                technology = ServingSystemInfo.ServingSystem.radioInterface[i];
            }
        }
        //fprintf(stdout, "%s: technology is %u\n", __func__, technology);
        return technology;
    }

    return -1;
}

BOOL isGSMDevice( )
{
    qaQmiServingSystemParam ServingSystemInfo;

    ULONG nRet;

    /* Check if device connected is CDMA */
    nRet = SLQSGetServingSystem( &ServingSystemInfo );

    if ( nRet == 0 && ServingSystemInfo.ServingSystem.numRadioInterfaces > 0)
    {
        switch( ServingSystemInfo.ServingSystem.radioInterface[0] )
        {
            case 4: /* GSM*/
            case 5: /* UMTS */
            case 8: /* LTE */
                return TRUE;
            break;
        }
    }

    return FALSE;
}
/** Get APN Override.
 * 
 * This function is used to fetch the current APN override setting.  This is a writeable
 * string that can be used to override the default APN to be used by the device.
 * 
 * The APN is a string of ANSI characters.
 * 
 * @return Result code.  Anything other than DT2_RESULT_OK indicates a failure, in which
 *         case the string buffer contents are undefined. 
 */
dt2_ResultCode_t dt2_GetApnOverride
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */ 
)
{

    DEV_CHECK
    
    /* Make sure the size isn't large enough to overflow a BYTE. */
    if (buffSize > 0xFF)
    {
        buffSize = 0xFF;
    }
    
    /** @todo   Figure out the real requirements for this.  For now, I'm just using
     *          the APN in the default profile for UMTS.
     */

    const ULONG profileType = 0;  /* 0 = UMTS */
    
    ULONG pdpType;
    ULONG ipAddress;
    ULONG primaryDNS;
    ULONG secondaryDNS;
    ULONG authentication;
    CHAR  name[30];  /* Don't care about this, so this is an arbitrary length. */
    BYTE  nameSize = sizeof(name) - 1;
    CHAR* apnNamePtr = stringPtr;
    BYTE  apnSize = buffSize;
    CHAR  userName[30]; /* Don't care about this, so this is an arbitrary length. */
    BYTE  userNameSize = sizeof(userName) - 1;

    ULONG rCode = GetDefaultProfile(profileType,
                                    &pdpType,
                                    &ipAddress,
                                    &primaryDNS,
                                    &secondaryDNS,
                                    &authentication,
                                    nameSize,
                                    &name[0],
                                    apnSize,
                                    &apnNamePtr[0],
                                    userNameSize,
                                    &userName[0]);

    if (rCode == eQCWWAN_ERR_NONE)
    {
        fprintf(stderr,"Received Profile Details are as follows:\n");
        fprintf(stderr," PDPType: %x\n",pdpType);
        fprintf(stderr," IPAddress: %x\n",ipAddress);
        fprintf(stderr," PrimaryDNS: %x\n",primaryDNS);
        fprintf(stderr," SecondaryDNS: %x\n",secondaryDNS);
        fprintf(stderr," Authentication: %x\n",authentication);
        fprintf(stderr," Name: %s\n",name);
        fprintf(stderr," APNName: %s\n",apnNamePtr);
        fprintf(stderr," Username: %s\n",userName);
        
        return DT2_RESULT_OK;
    }
    else
    {
        return DT2_RESULT_ERROR;
    }
}


/** Set APN Override.
 * 
 * This function is used to set the current APN override setting.  This is a writeable
 * string that can be used to override the default APN to be used by the device.
 * 
 * The APN is a string of ANSI characters.
 * 
 * @return Result code.  Anything other than DT2_RESULT_OK indicates a failure. 
 */
dt2_ResultCode_t dt2_SetApnOverride
(
    const char* stringPtr       /**< Pointer to the new value. */
)
{

    DEV_CHECK

    /** @todo   Figure out what the real requirements are for this.
     *          For now, I'm just setting the default UMTS APN.
     */
    ULONG rCode = SetDefaultProfile(0,      // Profile type (0 = UMTS)
                                    NULL,   // PDPType
                                    NULL,   // IPAddress
                                    NULL,   // PrimaryDNS
                                    NULL,   // SecondaryDNS
                                    NULL,   // Authentication
                                    NULL,   // Name
                                    (char*)stringPtr, // APNName
                                    NULL,   // Username,
                                    NULL ); // Password

    if (rCode != eQCWWAN_ERR_NONE)
    {
        fprintf(stderr,
                "SetDefaultProfile() failed for APN name '%s' (returned %u).\n",
                stringPtr,
                rCode);
        return DT2_RESULT_ERROR;
    }
    else
    {
        return DT2_RESULT_OK;
    }
}


/** Get APN.
 * 
 * This function is used to fetch the current APN being used by the device.  This is a
 * read-only variable.
 * 
 * The APN is a string of ANSI characters.
 * 
 * @return Result code.  Anything other than DT2_RESULT_OK indicates a failure, in which
 *         case the string buffer contents are undefined.
 */
dt2_ResultCode_t dt2_GetApn
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */ 
)
{
    DEV_CHECK
    
    /** @todo   Figure out the real requirements.  For now, just return the
     *          APN override value.
     */

    return dt2_GetApnOverride(stringPtr, buffSize);
}


/** Get IMEI.
 * 
 * This function is used to fetch the device's IMEI.  This is a read-only variable.
 * 
 * The IMEI is a string of ANSI characters.
 * 
 * @return Result code.  Anything other than DT2_RESULT_OK indicates a failure, in which
 *         case the string buffer contents are undefined.
 */
dt2_ResultCode_t dt2_GetImei
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */ 
)
{
    DEV_CHECK

    char esn[30] = "";
    char imei[30] = "";
    char meid[30] = "";

    ULONG resultCode;

    resultCode = GetSerialNumbers(  (BYTE)sizeof(esn),  /* BYTE     esnSize, */
            esn,                /* CHAR*    pESNString, */
            (BYTE)sizeof(imei), /* BYTE     imeiSize, */
            imei,               /* CHAR*    pIMEIString, */
            (BYTE)sizeof(meid), /* BYTE     meidSize, */
            meid );             /* CHAR*    pMEIDString */
    if (resultCode == eQCWWAN_ERR_NONE)
    {
        stringPtr[0] = '\0';
        strncat(stringPtr, imei, buffSize - 1); /* strncat always null terminates */
        return DT2_RESULT_OK;
    }
    else
    {
        fprintf(stderr, "GetSerialNumbers() failed (returned %u).\n", resultCode);
        return DT2_RESULT_ERROR;
    }
}

ULONG GetServingTechology( BYTE *technology)
{
    qaQmiServingSystemParam ServingSystemInfo;
    BYTE i;
    ULONG retval;

    /* initialize the technology */
    *technology = 0;

    retval = SLQSGetServingSystem( &ServingSystemInfo );
    if ( eQCWWAN_ERR_NONE != retval )
    {
        fprintf(stdout, "SLQSGetServingSystem Unsuccessful (error = %u)\n", retval);
        return retval;
    }
    if ( ServingSystemInfo.ServingSystem.numRadioInterfaces > 0)
    {
        /* report the higher technology if several technogies were reported */
        for (i = 0; i < ServingSystemInfo.ServingSystem.numRadioInterfaces; i++)
        {
            if ( *technology < ServingSystemInfo.ServingSystem.radioInterface[i])
            {
                *technology = ServingSystemInfo.ServingSystem.radioInterface[i];
            }
        }
    }
 
    return retval;
}

/** Get RSSI.
 * 
 * Fetches the current cellular signal strength (RSSI).
 * 
 * The RSSI is a signed integer value from -150 to 0 dBm.
 * 
 * @return Result code.  Anything other than DT2_RESULT_OK indicates a failure, in which
 *         case the value buffer contents are undefined. 
 */
dt2_ResultCode_t dt2_GetRssi
(
    int*    valuePtr        /**< Pointer to the buffer into which the value will be copied. */
)
{
    DEV_CHECK

    INT8 rssi = -128;    /* dBm */
    INT8 techSpecificRssi;
    
    CDMASSInfo  cdmaSSInfo;
    HDRSSInfo   hdrSSInfo;
    INT8        gsmSSInfo;
    CDMASSInfo  wcdmaSSInfo;
    LTESSInfo   lteSSInfo;
    BYTE        technology;



    nasGetSigInfoResp resp;
    memset(&resp,0,sizeof(nasGetSigInfoResp));
    resp.pCDMASSInfo  = &cdmaSSInfo;
    resp.pHDRSSInfo   = &hdrSSInfo;
    resp.pGSMSSInfo   = &gsmSSInfo;
    resp.pWCDMASSInfo = &wcdmaSSInfo;
    resp.pLTESSInfo   = &lteSSInfo;

    ULONG retCode = SLQSNasGetSigInfo( &resp );

    if ( eQCWWAN_ERR_NONE != retCode )
    {
        fprintf(stdout, "SLQSNasGetSigInfo Unsuccessful (error = %u)\n", retCode);
        return DT2_RESULT_ERROR;
    }

    fprintf(stdout, "SLQSNasGetSigInfo Successful\n");

    retCode = GetServingTechology(&technology);
    if ( eQCWWAN_ERR_NONE != retCode )
    {
        return DT2_RESULT_ERROR;
    }

    techSpecificRssi = -128;
    switch (technology)
    {
        case CDMA_1xRTT:
            fprintf(stdout, "CDMA SS Information:\n");

            techSpecificRssi = resp.pCDMASSInfo->rssi;
            if (techSpecificRssi > rssi)
            {
                rssi = techSpecificRssi;
            }
            fprintf(stdout, " RSSI  : %d (%d dBm)\n", resp.pCDMASSInfo->rssi, techSpecificRssi);
            fprintf(stdout, " ECIO  : %d\n", resp.pCDMASSInfo->ecio);
            break;
        case CDMA_1xEVDO:
            fprintf(stdout, "HDR SS Information:\n");
            techSpecificRssi = (signed char)(resp.pHDRSSInfo->rssi);
            if (techSpecificRssi > rssi)
            {
                rssi = techSpecificRssi;
            }
            fprintf(stdout, " RSSI  : %d (%d dBm)\n", resp.pHDRSSInfo->rssi, techSpecificRssi);
            fprintf(stdout, " ECIO  : %d\n", resp.pHDRSSInfo->ecio);
            fprintf(stdout, " SINR  : %d\n", resp.pHDRSSInfo->sinr);
            fprintf(stdout, " IO    : %d\n", resp.pHDRSSInfo->io);
            break;
        case GSM_RI:
            fprintf(stdout, "GSM SS Information:\n");
            techSpecificRssi = (signed char)(*resp.pGSMSSInfo);
            if (techSpecificRssi > rssi)
            {
                rssi = techSpecificRssi;
            }
            fprintf(stdout, " RSSI  : %d (%d dBm)\n", *(resp.pGSMSSInfo), techSpecificRssi);
            break;
        case UMTS_RI:
            fprintf(stdout, "WCDMA SS Information:\n");
            techSpecificRssi = (signed char)(resp.pWCDMASSInfo->rssi);
            if (techSpecificRssi > rssi)
            {
                rssi = techSpecificRssi;
            }
            fprintf(stdout, " RSSI  : %d (%d dBm)\n", resp.pWCDMASSInfo->rssi, techSpecificRssi);
            fprintf(stdout, " ECIO  : %d\n", resp.pWCDMASSInfo->ecio);
            break;
        case LTE_RI:
            fprintf(stdout, "LTE SS Information:\n");
            techSpecificRssi = (signed char)(resp.pLTESSInfo->rssi);
            if (techSpecificRssi > rssi)
            {
                rssi = techSpecificRssi;
            }
            fprintf(stdout, " RSSI  : %d (%d dBm)\n", resp.pLTESSInfo->rssi, techSpecificRssi);
            fprintf(stdout, " RSRQ  : %d\n", resp.pLTESSInfo->rsrq);
            fprintf(stdout, " RSRP  : %d\n", resp.pLTESSInfo->rsrp);
            fprintf(stdout, " SNR   : %d\n", resp.pLTESSInfo->snr);            
            break;
    }
    
    *valuePtr = rssi;
    
    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetGsmEcio
(
    double*    valuePtr        /**< Pointer to the buffer into which the value will be copied. */
)
{
    DEV_CHECK

    SHORT ecio = -150;    /* dBm */
    SHORT techSpecificEcio;
    
    CDMASSInfo  cdmaSSInfo;
    HDRSSInfo   hdrSSInfo;
    INT8        gsmSSInfo;
    CDMASSInfo  wcdmaSSInfo;
    LTESSInfo   lteSSInfo;
    BYTE        technology;

    nasGetSigInfoResp resp;

    memset(&resp,0,sizeof(nasGetSigInfoResp));
    resp.pCDMASSInfo  = &cdmaSSInfo;
    resp.pHDRSSInfo   = &hdrSSInfo;
    resp.pGSMSSInfo   = &gsmSSInfo;
    resp.pWCDMASSInfo = &wcdmaSSInfo;
    resp.pLTESSInfo   = &lteSSInfo;

    ULONG retCode = SLQSNasGetSigInfo( &resp );

    if ( eQCWWAN_ERR_NONE != retCode )
    {
        fprintf(stdout, "SLQSNasGetSigInfo Unsuccessful (error = %u)\n", retCode);
        return DT2_RESULT_ERROR;
    }

    fprintf(stdout, "SLQSNasGetSigInfo Successful\n");

    retCode = GetServingTechology(&technology);
    if ( eQCWWAN_ERR_NONE != retCode )
    {
        return DT2_RESULT_ERROR;
    }

    techSpecificEcio = -128;
    switch (technology)
    {
        case CDMA_1xRTT:
            fprintf(stdout, "CDMA SS Information:\n");
            techSpecificEcio = (signed char)(resp.pCDMASSInfo->ecio);
            if (techSpecificEcio > ecio)
            {
                ecio = techSpecificEcio;
            }
            fprintf(stdout, " RSSI  : %d (%d dBm)\n", resp.pCDMASSInfo->rssi, techSpecificEcio);
            fprintf(stdout, " ECIO  : %d\n", resp.pCDMASSInfo->ecio);
            break;
        case CDMA_1xEVDO:
            fprintf(stdout, "HDR SS Information:\n");
            techSpecificEcio = (signed char)(resp.pHDRSSInfo->ecio);
            if (techSpecificEcio > ecio)
            {
                ecio = techSpecificEcio;
            }
            fprintf(stdout, " RSSI  : %d (%d dBm)\n", resp.pHDRSSInfo->rssi, techSpecificEcio);
            fprintf(stdout, " ECIO  : %d\n", resp.pHDRSSInfo->ecio);
            fprintf(stdout, " SINR  : %d\n", resp.pHDRSSInfo->sinr);
            fprintf(stdout, " IO    : %d\n", resp.pHDRSSInfo->io);
            break;
        case GSM_RI:
            fprintf(stdout, "GSM SS Information:\n");
            techSpecificEcio = (signed char)(*resp.pGSMSSInfo);
            if (techSpecificEcio > ecio)
            {
                ecio = techSpecificEcio;
            }
            fprintf(stdout, " RSSI  : %d (%d dBm)\n", *(resp.pGSMSSInfo), techSpecificEcio);
            break;
        case UMTS_RI:
            fprintf(stdout, "WCDMA SS Information:\n");
            techSpecificEcio = (signed char)(resp.pWCDMASSInfo->ecio);
            if (techSpecificEcio > ecio)
            {
                ecio = techSpecificEcio;
            }
            fprintf(stdout, " RSSI  : %d (%d dBm)\n", resp.pWCDMASSInfo->rssi, techSpecificEcio);
            fprintf(stdout, " ECIO  : %d\n", resp.pWCDMASSInfo->ecio);
            break;
    }

    /*
       ECIO value representing negative 0.5 dBm increments, i.e.,
       2 means -1 dBm (14 means -7 dBm, 63 means -31.5 dBm).
       - 0xFFFF - Not Available
       */
    *valuePtr = ecio * -0.5;
    
    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetTech
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */ 
)
{

    DEV_CHECK

    char    *strTech = "Unknown";
    ULONG                   nRet;
    qaQmiServingSystemParam ServingSystemInfo;
    int technology = 0;
    BYTE i;

    nRet = SLQSGetServingSystem( &ServingSystemInfo );

    if ( eQCWWAN_ERR_NONE != nRet )
    {
        fprintf(stdout, "SLQSGetServingSystem Unsuccessful (error = %u)\n", nRet);
        return DT2_RESULT_ERROR;
    }

    fprintf(stdout, "SLQSGetServingSystem Successful\n");

    if ( nRet == 0 && ServingSystemInfo.ServingSystem.numRadioInterfaces > 0)
    {
        /* report the higher technology if several technogies were reported */
        for (i = 0; i < ServingSystemInfo.ServingSystem.numRadioInterfaces; i++)
        {
            if ( technology < ServingSystemInfo.ServingSystem.radioInterface[i])
            {
                technology = ServingSystemInfo.ServingSystem.radioInterface[i];
            }
        }  
        fprintf(stdout, "RF Tech %d\n", technology);
        switch( technology )
        {
            case 0:
                strTech = "None";
                break;
            case 1:
                strTech = "1X";
                break;
            case 2:
                strTech = "EV-DO";
                break;
            case 3:
                strTech = "AMPS";
                break;
            case 4:
                strTech = "GSM";
                break;
            case 5:
                strTech = "UMTS";
                break;
            case 8:
                strTech = "LTE";
                break;
        }
    }

    memset(stringPtr, 0, buffSize);
    strncpy(stringPtr, strTech, buffSize);

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetPktRcvd
(
    double*    valuePtr        /**< Pointer to the buffer into which the value will be copied. */
)
{
    DEV_CHECK

    double pkt_rcvd = 0;
    ULONG  TXPacketSuccesses = 0;
    ULONG  RXPacketSuccesses = 0;
    ULONG  TXPacketErrors = 0;
    ULONG  RXPacketErrors = 0;
    ULONG  TXPacketOverflows = 0;
    ULONG  RXPacketOverflows = 0;
    BYTE   instance = 0;

    ULONG                   nRet;

    nRet = GetPacketStatus (
            &TXPacketSuccesses,
            &RXPacketSuccesses,
            &TXPacketErrors,
            &RXPacketErrors,
            &TXPacketOverflows,
            &RXPacketOverflows,
            instance );

    if ( nRet == 0 )
    {
        pkt_rcvd = RXPacketSuccesses;
    }

    *valuePtr = pkt_rcvd;
    
    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetPktSent
(
    double*    valuePtr        /**< Pointer to the buffer into which the value will be copied. */
)
{
    DEV_CHECK

    double pkt_sent = 0;
    ULONG  TXPacketSuccesses = 0;
    ULONG  RXPacketSuccesses = 0;
    ULONG  TXPacketErrors = 0;
    ULONG  RXPacketErrors = 0;
    ULONG  TXPacketOverflows = 0;
    ULONG  RXPacketOverflows = 0;
    BYTE   instance = 0;

    ULONG                   nRet;

    nRet = GetPacketStatus (
            &TXPacketSuccesses,
            &RXPacketSuccesses,
            &TXPacketErrors,
            &RXPacketErrors,
            &TXPacketOverflows,
            &RXPacketOverflows,
            instance );

    if ( nRet == 0 )
    {
        pkt_sent = TXPacketSuccesses;
    }

    *valuePtr = pkt_sent;
    
    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetOperator
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */ 
)
{
    DEV_CHECK 

    WORD  pMCC;
    WORD  pMNC;
    CHAR  pName[nMaxStrLen];
    WORD  pSID;
    WORD  pNID;

    char    *strOperator = "Unknown";
    ULONG                   nRet;

    fprintf(stdout, "buffSize %zx\n", buffSize);

    nRet = GetHomeNetwork(&pMCC,
            &pMNC,
            nMaxStrLen,
            &pName[0],
            &pSID,
            &pNID);

    fprintf(stdout, "GetHomeNetwork rtn %x\n", nRet);

    if ( nRet == 0 )
    {
        strOperator = pName;
    }

    memset(stringPtr, 0, buffSize);
    strncpy(stringPtr, strOperator, buffSize);

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetIP
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */ 
)
{
    DEV_CHECK

    ULONG ipv4;
    ULONG v4sessionId;
    ULONG v6sessionId;
    struct WdsRunTimeSettings lRunTimeSettings;

    char    strIP[16] = "99.99.99.99";
    ULONG                   nRet;

    fprintf(stdout, "buffSize %zx\n", buffSize);

    memset(&lRunTimeSettings, 0, sizeof(lRunTimeSettings));
    lRunTimeSettings.rts.pIPAddressV4 = &ipv4;
    lRunTimeSettings.v4sessionId = &v4sessionId;
    lRunTimeSettings.v6sessionId = &v6sessionId;

    nRet = qaRetrieveWDSSessionIDs(0,
            lRunTimeSettings.v4sessionId,
            lRunTimeSettings.v6sessionId);
    fprintf(stdout, "get session id rtn %x sid %x\n", nRet,
            v4sessionId);

    lRunTimeSettings.v6sessionId = NULL;
    nRet = SLQSGetRuntimeSettings( &lRunTimeSettings );
    fprintf(stdout, "SLQSGetRuntimeSettings rtn %x\n", nRet);

    if ( nRet == 0 )
    {
        sprintf( strIP, "%d.%d.%d.%d",
                (unsigned int)(ipv4 >> 24) & 0xFF,
                (unsigned int)(ipv4 >> 16) & 0xFF,
                (unsigned int)(ipv4 >> 8) & 0xFF,
                (unsigned int)ipv4 & 0xFF );
    }

    memset(stringPtr, 0, buffSize);
    strncpy(stringPtr, strIP, buffSize);

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetRoamStatus
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */ 
)
{

    DEV_CHECK

    ULONG pRegistrationState;
    ULONG pCSDomain;
    ULONG pPSDomain;
    ULONG pRAN;
    BYTE  pRadioIfacesSize;
    BYTE  pRadioIfaces[nMaxStrLen];
    ULONG pRoaming;
    WORD  pMCC;
    WORD  pMNC;
    CHAR  pName[nMaxStrLen];

    char    *strRoaming = "Unknown";
    ULONG                   nRet;

    fprintf(stdout, "buffSize %zx\n", buffSize);

    nRet = GetServingNetwork(&pRegistrationState,
            &pCSDomain,
            &pPSDomain,
            &pRAN,
            &pRadioIfacesSize,
            &pRadioIfaces[0],
            &pRoaming,
            &pMCC,
            &pMNC,
            nMaxStrLen,
            &pName[0]);

    fprintf(stdout, "API rtn %x\n", nRet);

    if ( nRet == 0 )
    {
        if (pRoaming == 0x00)
            strRoaming = "Roaming";
        else if (pRoaming == 0x01)
            strRoaming = "Home";
        else
            strRoaming = "Operator defined";
    }

    memset(stringPtr, 0, buffSize);
    strncpy(stringPtr, strRoaming, buffSize);

    return DT2_RESULT_OK;
}

//TODO share with rssi
dt2_ResultCode_t dt2_GetRsrp
(
    int*    valuePtr        /**< Pointer to the buffer into which the value will be copied. */
)
{
    DEV_CHECK

    SHORT rsrp = -150;    /* dBm */
    SHORT techSpecificRsrp;

    LTESSInfo   lteSSInfo;

    nasGetSigInfoResp resp;

    memset(&resp, 0, sizeof(resp));
    resp.pLTESSInfo   = &lteSSInfo;

    ULONG retCode = SLQSNasGetSigInfo( &resp );

    if ( eQCWWAN_ERR_NONE != retCode )
    {
        fprintf(stdout, "SLQSNasGetSigInfo Unsuccessful (error = %u)\n", retCode);
        return DT2_RESULT_ERROR;
    }

    fprintf(stdout, "SLQSNasGetSigInfo Successful\n");

    fprintf(stdout, "LTE SS Information:\n");
    techSpecificRsrp= -150;
    techSpecificRsrp = resp.pLTESSInfo->rsrp;
    if (techSpecificRsrp > rsrp)
    {
        rsrp = techSpecificRsrp;
    }

    fprintf(stdout, " RSRQ  : %x\n", resp.pLTESSInfo->rsrq);
    fprintf(stdout, " RSRP  : %x\n", resp.pLTESSInfo->rsrp);

    *valuePtr = rsrp *10;

    return DT2_RESULT_OK;
}

//TODO share with rssi
dt2_ResultCode_t dt2_GetRsrq
(
    int*    valuePtr        /**< Pointer to the buffer into which the value will be copied. */
)
{
    DEV_CHECK

    INT8 rsrq = -128;    /* dBm */
    INT8 techSpecificRsrq;

    LTESSInfo   lteSSInfo;

    nasGetSigInfoResp resp;

    memset(&resp, 0, sizeof(resp));
    resp.pLTESSInfo   = &lteSSInfo;

    ULONG retCode = SLQSNasGetSigInfo( &resp );

    if ( eQCWWAN_ERR_NONE != retCode )
    {
        fprintf(stdout, "SLQSNasGetSigInfo Unsuccessful (error = %u)\n", retCode);
        return DT2_RESULT_ERROR;
    }

    fprintf(stdout, "SLQSNasGetSigInfo Successful\n");

    fprintf(stdout, "LTE SS Information:\n");
    techSpecificRsrq= -128;
    techSpecificRsrq = resp.pLTESSInfo->rsrq;
    if (techSpecificRsrq > rsrq)
    {
        rsrq = techSpecificRsrq;
    }
    fprintf(stdout, " RSRQ  : %d\n", resp.pLTESSInfo->rsrq);
    fprintf(stdout, " RSRP  : %d\n", resp.pLTESSInfo->rsrp);

    *valuePtr = rsrq * 10;

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetRadioTemp
(
    int*    valuePtr        /**< Pointer to the buffer into which the value will be copied. */
)
{
    DEV_CHECK

    swiModemStatusResp resp;
    int temp = -273;

    memset(&resp, 0, sizeof(resp));
    ULONG nRet = SLQSNasSwiModemStatus( &resp );

    if ( eQCWWAN_ERR_NONE == nRet )
    {
        fprintf( stdout, "SLQSNasSwiModemStatus Successful \n" );
        temp = resp.commonInfo.temperature;
    }

    *valuePtr = temp;

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetBytesRcvd
(
    double*    valuePtr        /**< Pointer to the buffer into which the value will be copied. */
)
{
    DEV_CHECK

    double bytes_rcvd = 0;
    ULONGLONG  TXTotalBytes;
    ULONGLONG  RXTotalBytes;
    BYTE   instance = 0;

    ULONG                   nRet;

    nRet = GetByteTotals (
            &TXTotalBytes,
            &RXTotalBytes,
            instance );

    if ( nRet == 0 )
    {
        bytes_rcvd = RXTotalBytes;
    }

    *valuePtr = bytes_rcvd;
    
    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetBytesSent
(
    double*    valuePtr        /**< Pointer to the buffer into which the value will be copied. */
)
{
    DEV_CHECK

    double bytes_sent = 0;
    ULONGLONG  TXTotalBytes;
    ULONGLONG  RXTotalBytes;
    BYTE   instance = 0;

    ULONG                   nRet;

    nRet = GetByteTotals (
            &TXTotalBytes,
            &RXTotalBytes,
            instance );

    if ( nRet == 0 )
    {
        bytes_sent = TXTotalBytes;
    }

    *valuePtr = bytes_sent;
    
    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetCdmaEcio
(
    double*    valuePtr        /**< Pointer to the buffer into which the value will be copied. */
)
{
    DEV_CHECK

    ULONG             nRet = 0;
    CDMASSInfo        cdmaSigInfo;
    nasGetSigInfoResp nasSigInfo;

    memset(&nasSigInfo,0,sizeof(nasGetSigInfoResp));
    nasSigInfo.pCDMASSInfo  = &cdmaSigInfo;
    nasSigInfo.pHDRSSInfo   = NULL;
    nasSigInfo.pGSMSSInfo   = NULL;
    nasSigInfo.pWCDMASSInfo = NULL;
    nasSigInfo.pLTESSInfo   = NULL;

    /* Check if device is GSM */
    if( TRUE == isGSMDevice() )
    {
        fprintf(stdout, "\nGet CDMA Ecio, return error since it is 3GPP.\n");
        *valuePtr = 0;
        return DT2_RESULT_OK;
    }


    nRet = SLQSNasGetSigInfo( &nasSigInfo );

    if ( eQCWWAN_ERR_NONE != nRet )
    {
        fprintf(stdout, "SLQSNasGetSigInfo Unsuccessful (error = %u)\n", nRet);
        return DT2_RESULT_ERROR;
    }

    fprintf(stdout, "SLQSNasGetSigInfo Successful\n");

    if ( nRet == 0 )
    {
        /*
           ECIO value representing negative 0.5 dBm increments, i.e.,
           2 means -1 dBm (14 means -7 dBm, 63 means -31.5 dBm).
           - 0xFFFF - Not Available
           */
            *valuePtr = cdmaSigInfo.ecio * -0.5;
    }
    
    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetCdmaOperator
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */ 
)
{
    DEV_CHECK

    WORD  pMCC;
    WORD  pMNC;
    CHAR  pName[nMaxStrLen];
    WORD  pSID;
    WORD  pNID;

    char    *strOperator = "Unknown";
    ULONG   nRet;

    /* Check if device is GSM */
    if( TRUE == isGSMDevice() )
    {
        return DT2_RESULT_OK;
    }

    fprintf(stdout, "buffSize %zx\n", buffSize);

    nRet = GetHomeNetwork( &pMCC,
                           &pMNC,
                           nMaxStrLen,
                           &pName[0],
                           &pSID,
                           &pNID );

    fprintf(stdout, "GetHomeNetwork rtn %x\n", nRet);

    if ( nRet == 0 )
    {
        strOperator = pName;
    }

    memset(stringPtr, 0, buffSize);
    strncpy(stringPtr, strOperator, buffSize);

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetCdmaPnOff
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */ 
)
{
    DEV_CHECK

    CDMAInfo                cdmaResp;
    nasCellLocationInfoResp CellLocInfo;
    char  PNOffset[10] = "Unknown";
    ULONG nRet;

    /* Check if device is GSM */
    if( TRUE == isGSMDevice() )
    {
        return DT2_RESULT_OK;
    }

    fprintf(stdout, "buffSize %zx\n", buffSize);

    memset(&CellLocInfo, 0, sizeof(nasCellLocationInfoResp));
    CellLocInfo.pCDMAInfo = &cdmaResp;

    nRet = SLQSNasGetCellLocationInfo( &CellLocInfo );

    fprintf(stdout, "SLQSNasGetCellLocationInfo rtn %x\n", nRet);

    if ( nRet == 0 )
    {
        if( 0xFFFF != cdmaResp.refpn )
        {
            sprintf(PNOffset, "%d", cdmaResp.refpn );
        }
    }

    memset(stringPtr, 0, buffSize);
    strncpy(stringPtr, PNOffset, buffSize);

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetCdmaSid
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */
)
{
    DEV_CHECK

    CDMAInfo                cdmaResp;
    nasCellLocationInfoResp CellLocInfo;
    char  SID[10] = "Unknown";
    ULONG nRet;

    /* Check if device is GSM */
    if( TRUE == isGSMDevice() )
    {
        return DT2_RESULT_OK;
    }

    fprintf(stdout, "buffSize %zx\n", buffSize);

    memset(&CellLocInfo, 0, sizeof(nasCellLocationInfoResp));
    CellLocInfo.pCDMAInfo = &cdmaResp;

    nRet = SLQSNasGetCellLocationInfo( &CellLocInfo );

    fprintf(stdout, "SLQSNasGetCellLocationInfo rtn %x\n", nRet);

    if ( nRet == 0 )
    {
        if( 0xFFFF != cdmaResp.sid )
        {
            sprintf(SID, "%d", cdmaResp.sid );
        }
    }

    memset(stringPtr, 0, buffSize);
    strncpy(stringPtr, SID, buffSize);

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetCdmaNid
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */
)
{
    DEV_CHECK

    CDMAInfo                cdmaResp;
    nasCellLocationInfoResp CellLocInfo;
    char  NID[10] = "Unknown";
    ULONG nRet;

    /* Check if device is GSM */
    if( TRUE == isGSMDevice() )
    {
        return DT2_RESULT_OK;
    }

    fprintf(stdout, "buffSize %zx\n", buffSize);

    memset(&CellLocInfo, 0, sizeof(nasCellLocationInfoResp));
    CellLocInfo.pCDMAInfo = &cdmaResp;

    nRet = SLQSNasGetCellLocationInfo( &CellLocInfo );

    fprintf(stdout, "SLQSNasGetCellLocationInfo rtn %x\n", nRet);

    if ( nRet == 0 )
    {
        if( 0xFFFF != cdmaResp.nid )
        {
            sprintf(NID, "%d", cdmaResp.nid );
        }
    }

    memset(stringPtr, 0, buffSize);
    strncpy(stringPtr, NID, buffSize);

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetGsmCellId
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */ 
)
{
    DEV_CHECK

    GERANInfo               geranInfo;
    UMTSInfo                umtsInfo;
    nasCellLocationInfoResp CellLocInfo;
    char                    CellID[20] = "Unknown";
    qaQmiServingSystemParam ServingSystemInfo;
    ULONG                   retCode;

    fprintf(stdout, "buffSize %zx\n", buffSize);

    /* Check if device connected is GSM */
    retCode = SLQSGetServingSystem( &ServingSystemInfo );

    fprintf(stdout, "SLQSGetServingSystem rtn %x\n", retCode);

    if ( retCode == 0 && ServingSystemInfo.ServingSystem.numRadioInterfaces > 0)
    {
        switch( ServingSystemInfo.ServingSystem.radioInterface[0] )
        {
            case 0: /* NONE*/
            case 1: /* 1X */
            case 2: /* EV-D0*/
            case 3: /* AMPS */
                return DT2_RESULT_OK;
            break;
        }
    }

    memset(&CellLocInfo, 0, sizeof(nasCellLocationInfoResp));
    CellLocInfo.pGERANInfo = &geranInfo;
    CellLocInfo.pUMTSInfo  = &umtsInfo;

    retCode = SLQSNasGetCellLocationInfo( &CellLocInfo );

    fprintf(stdout, "SLQSNasGetCellLocationInfo rtn %x\n", retCode);

    if ( retCode == 0 )
    {
        if( 0xFFFF != geranInfo.cellID )
        {
            sprintf( CellID, "%u", geranInfo.cellID );
        }

        if( 0xFFFF != umtsInfo.cellID )
        {
            sprintf( CellID, "%d", umtsInfo.cellID );
        }
    }

    memset(stringPtr, 0, buffSize);
    strncpy(stringPtr, CellID, buffSize);

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetIccid
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */
)
{
    DEV_CHECK

    ULONG nRet = 0;
    BYTE  ICCIDlen = 32;
    char  ICCID[ICCIDlen];

    strcpy(ICCID, "Unknown");
    fprintf(stdout, "buffSize %zx\n", buffSize);

     nRet = UIMGetICCID( ICCIDlen, ICCID );

     fprintf(stdout, "UIMGetICCID rtn %x\n", nRet);

     if ( nRet != 0 )
     {
         memset(stringPtr, 0, buffSize);
         strncpy(stringPtr, "SIM ERROR", buffSize);
         return DT2_RESULT_OK;
     }

     memset(stringPtr, 0, buffSize);
     strncpy(stringPtr, ICCID, buffSize);

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetImsi
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */
)
{
    DEV_CHECK

    ULONG nRet = 0;
    BYTE  IMSIlen = 32;
    char  IMSI[IMSIlen];

    strcpy(IMSI, "Unknown");
    fprintf(stdout, "buffSize %zx\n", buffSize);

     nRet = GetIMSI( IMSIlen, IMSI );

     fprintf(stdout, "GetIMSI rtn %x\n", nRet);

     if ( nRet != 0 )
     {
         memset(stringPtr, 0, buffSize);
         strncpy(stringPtr, "SIM ERROR", buffSize);
         return DT2_RESULT_OK;
     }

     memset(stringPtr, 0, buffSize);
     strncpy(stringPtr, IMSI, buffSize);

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetPhoneNum
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */
)
{
    DEV_CHECK

    ULONG nRet = 0;
    BYTE  VoiceNumlen = 32;
    CHAR  VoiceNum[VoiceNumlen];
    BYTE  MinSz = 16;
    CHAR  Min[MinSz];

    strcpy(VoiceNum, "Unknown");
    strcpy(Min, "Unknown");

    fprintf(stdout, "buffSize %zx\n", buffSize);

     nRet = GetVoiceNumber( VoiceNumlen, VoiceNum, MinSz, Min );

     fprintf(stdout, "GetVoiceNumber rtn %x\n", nRet);

     if ( nRet != 0 )
     {
         memset(stringPtr, 0, buffSize);
         strncpy(stringPtr, "Unknown", buffSize);

         return DT2_RESULT_OK;
     }

     memset(stringPtr, 0, buffSize);
     strncpy(stringPtr, VoiceNum, buffSize);

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetFwVer
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */
)
{
    DEV_CHECK

    ULONG nRet = 0;
    BYTE  amssSize = nMaxStrLen;
    CHAR  AMSString[nMaxStrLen];
    BYTE  bootSize = nMaxStrLen;
    CHAR  bootString[nMaxStrLen];
    BYTE  priSize = nMaxStrLen;
    CHAR  PRIString[nMaxStrLen];
    CHAR  fwVer[nMaxStrLen];

    strcpy(AMSString, "Unknown");

    fprintf(stdout, "buffSize %zx\n", buffSize);

    nRet = GetFirmwareRevisions(amssSize,
            &AMSString[0],
            bootSize,
            &bootString[0],
            priSize,
            &PRIString[0]);

    fprintf(stdout, "GetFirmwareRevisions rtn %x\n", nRet);

    if ( nRet != 0 )
    {
        memset(stringPtr, 0, buffSize);
        strncpy(stringPtr, AMSString, buffSize);

        return DT2_RESULT_OK;
    }

    sscanf(AMSString, "%[^' ']", fwVer);
    memset(stringPtr, 0, buffSize);
    strncpy(stringPtr, fwVer, buffSize);

    return DT2_RESULT_OK;
}

static void
at_write(int fd, char *wbuf, int wlen, char* rbuf)
{
    int bytes_avail;
    write(fd, wbuf, wlen);
    usleep(100e3); 

    bytes_avail = 0;
    ioctl(fd, FIONREAD, &bytes_avail);

    read(fd, rbuf, bytes_avail);
}

static int
get_partno(char* partno)
{
    ULONG nRet;
    struct DcsUsbPortNames UsbPortNames;
    int rtn;
    char l_skuid[10];
    char l_partno[10];
    char buf[512];
    char *pch;
    int fd;

    char LOCK[] = "at!entercnd=\"\"\r";
    char PRI[] = "at!priid?\r";
    static char s_partno[10] = {0};

    memset(&UsbPortNames, 0, sizeof(UsbPortNames));
    nRet = SLQSGetUsbPortNames( &UsbPortNames );
    if (nRet)
    {
       fprintf(stdout, "SLQSGtUsbPortNames returned %d\n", nRet);
       return -1;
    }

    fprintf(stdout, "opening %s\n", UsbPortNames.AtCmdPort);
    fd = open(UsbPortNames.AtCmdPort, O_RDWR);
    if (fd < 0)
    {
        /* use the part number previously collected */
        strcpy(partno, s_partno);
        return 0;
    }

    //lock the modem to avoid multi PRIs listed
    at_write(fd, LOCK, sizeof(LOCK), buf);

    at_write(fd, PRI, sizeof(PRI), buf);

    close(fd);

    pch = strtok(buf, "\n");
    while(pch)
    {
        rtn = sscanf(pch, "Carrier PRI: %[^'_']_%[^'_']", l_skuid, l_partno);
        if (rtn == 2)
        {
            strcpy(partno, l_partno);
            /* store the part number into the static variable which can be used if the 
               AT port is unable to access */
            strcpy(s_partno, partno);
            return 0;
        }
        pch = strtok(NULL, "\n");
    }

    return -2;
}

dt2_ResultCode_t dt2_GetFwName
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */
)
{
    DEV_CHECK;
    char partno[10];

    ULONG nRet = 0;

    fprintf(stdout, "buffSize %zx\n", buffSize);

    struct qmifwinfo_s  fwInfo;
    nRet = SLQSGetFirmwareInfo( &fwInfo );

    fprintf(stdout, "SLQSGetFirmwareInfo rtn %x\n", nRet);

    if ( nRet != 0 )
    {
        memset(stringPtr, 0, buffSize);
        strncpy(stringPtr, "Unknown", buffSize);

        return DT2_RESULT_OK;
    }

    //get partno from AT
    if (get_partno(partno) != 0)
    {
        memset(stringPtr, 0, buffSize);
        strncpy(stringPtr, "Unknown", buffSize);

        return DT2_RESULT_OK;
    }

    memset(stringPtr, 0, buffSize);
    snprintf(stringPtr, buffSize, "%s_%s_%s", 
            fwInfo.dev.s.modelid_str,
            fwInfo.dev.s.sku_str,
            partno
            );

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetPinStatus
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */
)
{
    DEV_CHECK

    ULONG nRet = 0;
    ULONG id = 1;
    ULONG PinStatus;
    ULONG PinVerifyRetriesLeft;
    ULONG PinBlockRetriesLeft;
    char  status[255];

    fprintf(stdout, "buffSize %zx\n", buffSize);

    nRet = UIMGetPINStatus( id,
                            &PinStatus,
                            &PinVerifyRetriesLeft,
                            &PinBlockRetriesLeft );

    fprintf(stdout, "UIMGetPINStatus rtn %x\n", nRet);

    if ( nRet != 0 )
    {
        memset(stringPtr, 0, buffSize);
        strncpy(stringPtr, "SIM ERROR", buffSize);
        return DT2_RESULT_OK;
    }

    switch( PinStatus )
    {
        case 0:
            sprintf( status, "%s", "PIN not initialized" );
            break;

        case 1:
            sprintf( status,
                     "%s: %u",
                     "PIN not verified - Retries left",
                     PinVerifyRetriesLeft );
            break;

        case 2:
            sprintf( status,
                    "%s",
                    "PIN Verified ");
            break;

        case 3:
            sprintf( status,
                    "%s",
                    "PIN Disabled ");
            break;

        case 4:
            sprintf( status,
                     "%s: %u",
                     "PIN Blocked - Unblock Retries left",
                     PinBlockRetriesLeft );
            break;

        case 5:
            sprintf( status,
                     "%s",
                     "PIN is permanently blocked" );
            break;

        case 6:
            sprintf( status,
                     "%s",
                     "PIN is unblocked" );
            break;

        case 7:
            sprintf( status,
                     "%s",
                     "PIN is changed" );
            break;

        default:
            sprintf( status,
                     "%s",
                     "Unknown Status" );
            break;
    }
    memset(stringPtr, 0, buffSize);
    strncpy(stringPtr, status, buffSize);

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetAltitude
(
    double*   valuePtr      /**< Pointer to the buffer into which the value will be copied. */
)
{
    DEV_CHECK

    ULONG        nRet = 0;
    GPSStateInfo tempGPSStateInfo;
    tempGPSStateInfo.Altitude = 0; /* Initialize default value */

    nRet = SLQSGetGPSStateInfo( &tempGPSStateInfo );

    if ( nRet == 0 )
    {
        /*
           For Altitude data to be valid, validity mask must have the
           following bit set - 0x00000002
           */
        if( 0x00 != (tempGPSStateInfo.ValidMask & GPSALTVALIDMASK) )
        {
            *valuePtr = tempGPSStateInfo.Altitude;
        }
    }

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetLatitude
(
    double*   valuePtr      /**< Pointer to the buffer into which the value will be copied. */
)
{
    DEV_CHECK

    *valuePtr = dGetLatitude();

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetLongitude
(
    double*   valuePtr      /**< Pointer to the buffer into which the value will be copied. */
)
{
    DEV_CHECK

    *valuePtr = dGetLongitude();

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetDateTime
(
    double*   valuePtr      /**< Pointer to the buffer into which the time stamp will be copied. */
)
{
    DEV_CHECK

    ULONG     nRet = 0;
    ULONGLONG timeStamp;
    ULONG     timeSource;
    ULONGLONG EpochTimeStamp;

    /* EPOCH value for Jan 6, 1980( GPS EPOCH starting time ) */
    ULONGLONG GPSEpochZero = 315964800;

    nRet = GetNetworkTime( &timeStamp, &timeSource );

    if ( nRet == 0 )
    {
        /* Current time stamp is in GPS epoch. Convert to epoch. */
        EpochTimeStamp = ((timeStamp * 1.25)/1000) + GPSEpochZero;

        /* Convert to milliseconds as per AVMS documentation */
        *valuePtr = ( EpochTimeStamp * 1000 );
    }

    return DT2_RESULT_OK;
}

static int get_serialnum(char* fsnno)
{
    ULONG nRet;
    struct DcsUsbPortNames UsbPortNames;
    char buf[512];
    char *pch;
    int fd;
    char ati[] = "ati\r";
    static char s_fsnno[15] = {0};

    memset(&UsbPortNames, 0, sizeof(UsbPortNames));
    nRet = SLQSGetUsbPortNames( &UsbPortNames );
    if (nRet)
    {
       fprintf(stdout, "SLQSGtUsbPortNames returned %d\n", nRet);
       return -1;
    }

    fprintf(stdout, "opening %s\n", UsbPortNames.AtCmdPort);
    fd = open(UsbPortNames.AtCmdPort, O_RDWR);
    if (fd < 0)
    {
        /* use the fsn number previously collected */
        strcpy(fsnno, s_fsnno);
        return 0;
    }

    /* write ati command to get the output, FSN number is part of the output */
    at_write(fd, ati, sizeof(ati), buf);

    close(fd);

    pch = strstr(buf, "FSN");
    if ( pch != NULL)
    {
        memcpy(buf, pch+5, FSN_STRING_LENGTH);
        buf[FSN_STRING_LENGTH] = '\0';
        strcpy(fsnno, buf);
        /* store the fsn number into the static variable which can be used if the 
           AT port is unable to access */
        strcpy(s_fsnno, fsnno);
        return 0;
    }
    else
    {
        fprintf(stdout, "cannot locate FSN string!\n");
    }

    return -2;
}


dt2_ResultCode_t dt2_GetSerialNumber
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */
)
{
    DEV_CHECK;

    ULONG nRet = 0;

    FactorySequenceNumber fsn;
    char buf[32] = {0};

    fprintf(stdout, "buffSize %zx\n", buffSize);
    memset(stringPtr, 0, buffSize);

    nRet = SLQSSwiGetFSN(&fsn);

    fprintf(stdout, "SLQSSwiGetFSN rtn %x\n", nRet);

    if ( nRet != 0 )
    {
        /* if QMI command return failure, try AT command */
        nRet = get_serialnum(buf);
        if (nRet != 0)
        {
           /* if AT command failed too, just set it to Unknown */
           strncpy(stringPtr, "Unknown", buffSize);
        }
        else
        {
            strncpy(stringPtr, buf, sizeof(buf));
        }
        return DT2_RESULT_OK;
    }

    strncpy(stringPtr, (const char*)fsn.FSNumber, buffSize);

    return DT2_RESULT_OK;
}

struct _sParsedPostionData{
    double dLongitude;
    double dLatitude;
    pthread_mutex_t mutexUpdate;
    int iInit;
}sParsedPostionData={0,0,PTHREAD_MUTEX_INITIALIZER,0};

void QmiPdsSetGPSCB(double dLongitude,double dLatitude,
       BYTE session_status, ULONG pos_src )

{
    UNUSEDPARAM(pos_src);
    /* set 0s lat/long when GPS is still in progress
     * portal then display not located instead of Carrier's default location
     */
    if (session_status == 0) //success
    {
        SetLatitude(dLatitude);
        SetLongitude(dLongitude);
    }
    else
    {
        SetLatitude(0);
        SetLongitude(0);
    }
}


ULONG iInitPdsLocationData(){
    if(sParsedPostionData.iInit==1)
    {
        return eQCWWAN_ERR_NONE;
    }
    ULONG rc = SetGPSCallback(&QmiPdsSetGPSCB );
    if(rc)
    {
        return eQCWWAN_ERR_INTERNAL;
    }
    
    if(sParsedPostionData.iInit == 0)
    {
        if(iInitParasedPositionDataMutex()!=0)
        {
            return eQCWWAN_ERR_INTERNAL;
        }
      sParsedPostionData.iInit = 1;
    }
    
    return eQCWWAN_ERR_NONE;
}

int iInitParasedPositionDataMutex()
{
    int iRet=0;
    iRet = pthread_mutex_init(& sParsedPostionData.mutexUpdate,NULL);
    return iRet;
}

double dGetLatitude(){
    double dCachedLatitude = 0;
    if(sParsedPostionData.iInit == 0)
    {
        if(iInitPdsLocationData()!=eQCWWAN_ERR_NONE)
        {
            return 0;
        }
      sParsedPostionData.iInit = 1;
    }
    pthread_mutex_lock( &sParsedPostionData.mutexUpdate);
    dCachedLatitude = sParsedPostionData.dLatitude;
    pthread_mutex_unlock( &sParsedPostionData.mutexUpdate);
    return dCachedLatitude;
}

double dGetLongitude(){
    double dCachedLongitude = 0;
    if(sParsedPostionData.iInit == 0)
    {
        if(iInitPdsLocationData()!=eQCWWAN_ERR_NONE)
        {
            return 0;
        }
        sParsedPostionData.iInit = 1;
    }
    pthread_mutex_lock( &sParsedPostionData.mutexUpdate);
    dCachedLongitude = sParsedPostionData.dLongitude;
    pthread_mutex_unlock( &sParsedPostionData.mutexUpdate);
    return dCachedLongitude;
}



void SetLatitude(double dCachedLatitude){
    if(sParsedPostionData.iInit == 0)
    {
        if(iInitPdsLocationData()!=eQCWWAN_ERR_NONE)
        {
            return ;
        }
        sParsedPostionData.iInit = 1;
    }
    pthread_mutex_lock( &sParsedPostionData.mutexUpdate);
    sParsedPostionData.dLatitude = dCachedLatitude;
    pthread_mutex_unlock( &sParsedPostionData.mutexUpdate);
    return ;
}

void SetLongitude(double dCachedLongitude){
    if(sParsedPostionData.iInit == 0)
    {
        if(iInitPdsLocationData()!=eQCWWAN_ERR_NONE)
        {
            return ;
        }
        sParsedPostionData.iInit = 1;
    }
    pthread_mutex_lock( &sParsedPostionData.mutexUpdate);
    sParsedPostionData.dLongitude = dCachedLongitude ;
    pthread_mutex_unlock( &sParsedPostionData.mutexUpdate);
    return ;
}

