/*************
 *
 * Filename: qmwds.c
 *
 * Purpose:  QMI Wireless Device Service ( internal )
 *
 * Copyright: © 2014 Sierra Wireless Inc., all rights reserved
 *
 **************/

/*-------------
  include files
 --------------*/
//#include "qm/qmqmisvc.h"
#include "aa/aaglobal.h"
#include "qm/qmerrno.h"
#include "am/amudefs.h"
#include "ci/ciudefs.h"
#include "er/erudefs.h"
#include "im/imudefs.h"
#include "mm/mmudefs.h"
#include "os/swi_ossdk.h"
#include "qm/qmidefs.h"
#include "qm/qmfms.h"
#include "qm/qmqmisvc.h"
#include "pi/piudefs.h"
#include "sl/sludefs.h"
#include "stdio.h"

/*-------------
  Local storage
 --------------*/
#define IP_FAMILY_V4 4
#define IP_FAMILY_V6 6
#define DEFAULTBYTEVALUE 0xFF
#define DEFAULTLONGVALUE 0xFFFFFFFF
#define INDEX_IPV4 0
#define INDEX_IPV6 1
#define DEFAULTSESSIONID 0

#define DBG_WDSINTERNAL
#ifdef DBG_WDSINTERNAL
#include <syslog.h>
#endif

/* QMI FMS service response structure */
local struct qmqmisvcresponse qmiwdsresp;

/*--------------------
  Forward declarations
 ---------------------*/
local void qm_wds_set_session_id(swi_uint8 *, swi_int32 );
local void qm_wds_get_instance_id(swi_uint8 *, swi_int32 );
local void qm_wds_get_session_id(swi_uint8 *, swi_int32 );

/* Internal Wireless Device Service request handler table */
local void(*wdshandlertbl[])(swi_uint8 *preq, swi_int32) =
{
    qm_wds_set_session_id,
    qm_wds_get_instance_id,
    qm_wds_get_session_id,
};

/*************
 *
 * Name:    qm_wds_set_sessID_unpack_tlv_sessID
 *
 * Purpose: Unpack session ID TLV Value
 *
 * Parms:   psource - source data
 *          pdest   - destination buffer
 *
 * Return:  eQCWWAN_ERR_NONE
 *
 * Abort:   none
 *
 * Notes:
 *
 **************/
local enum eQCWWANError
qm_wds_set_sessID_unpack_tlv_sessID(
    swi_uint8 *psource,
    swi_uint8 *pdest )
{
    struct qm_qmi_response_tlvs_values *pin;
    struct qm_wds_set_session_id_request_tlv_values *pval;

    pin =   (struct qm_qmi_response_tlvs_values *)pdest;
    pval =  (struct qm_wds_set_session_id_request_tlv_values *)
                &pin->tlvvalues.qmwdstlvs.set_session_id_req;

    return GetLong( psource, &pval->sessionID);
}

/*************
 *
 * Name:    qm_wds_set_sessID_unpack_tlv_InstID
 *
 * Purpose: Unpack instance ID TLV Value
 *
 * Parms:   psource - source data
 *          pdest   - destination buffer
 *
 * Return:  eQCWWAN_ERR_NONE
 *
 * Abort:   none
 *
 * Notes:
 *
 **************/
local enum eQCWWANError
qm_wds_set_sessID_unpack_tlv_InstID(
    swi_uint8 *psource,
    swi_uint8 *pdest )
{
    struct qm_qmi_response_tlvs_values *pin;
    struct qm_wds_set_session_id_request_tlv_values *pval;

    pin =   (struct qm_qmi_response_tlvs_values *)pdest;
    pval =  (struct qm_wds_set_session_id_request_tlv_values *)
                &pin->tlvvalues.qmwdstlvs.set_session_id_req;

    return GetByte( psource, &pval->instanceID);
}

/*************
 *
 * Name:    qm_wds_set_sessionid_request_unpack
 *
 * Purpose: Unpack WDS set session ID request
 *
 * Parms:   (IN)    preq        - incoming request packet
 *          (OUT)   preqtlvs    - destination data containing request
 *                                parameters.
 *
 * Return:  eQCWWAN_ERR_NONE if successfully unpacked request
 *          eQCWWAN_ERR_MEMORY if failed to unpack request
 *
 * Abort:   none
 *
 * Notes:
 *
 **************/
local enum eQCWWANError
qm_wds_set_sessionid_request_unpack(
    swi_uint8 *preq,
    struct qm_qmi_response_tlvs_values *preqtlvs )
{
    struct qmTlvUnpackerItem map[] =
    {
        { eTLV_WDS_INTERNAL_SET_SESSIONID,
          qm_wds_set_sessID_unpack_tlv_sessID },

        { eTLV_WDS_INTERNAL_SET_INSTANCEID,
          qm_wds_set_sessID_unpack_tlv_InstID },

        { eTLV_TYPE_INVALID, NULL }
    };

    return qmunpack( preq,
                     (swi_uint8 *)preqtlvs,
                     map,
                     eQMI_WDS_INT_SET_SESSIONID );
}

/*************
 *
 * Name:    qm_wds_set_sessionid_send_response
 *
 * Purpose: Pack and send a WDS get session ID response
 *
 * Parms:   msgid               - QMI WDS( internal ) service message id
 *          (OUT)   prsptlvs    - response structure
 *
 * Return:  eQCWWAN_ERR_NONE
 *
 * Abort:   none
 *
 * Notes:
 *
 **************/
local void
qm_wds_set_sessionid_send_response(
    enum   eQMIMessageWDSInternal      msgid,
    struct qm_qmi_response_tlvs_values *prsptlvs )
{
    struct qmTlvBuilderItem map[] =
    {
        {   eTLV_RESULT_CODE,
            qm_result_code_tlv_pack },

        {   eTLV_TYPE_INVALID,
            NULL }
    };

    /* construct and send response to the application */
    qm_qmisvc_send_response( &qmiwdsresp,
                             prsptlvs,
                             map,
                             eQMI_SVC_WDS,
                             msgid,
                             eQMIRES );
}

/*************
 *
 * Name:    qm_wds_set_session_id
 *
 * Purpose: Store the session ID corresponding to instance ID
 *
 * Parms:   preq - request packet pointer
 *          activeIPFamily - Current IP Family preference
 *
 * Return:  none
 *
 * Abort:   none
 *
 * Notes:   Mapping between session ID and instance ID prevents session
 *          handles from being lost if API process crashes.
 *
 **************/
local void
qm_wds_set_session_id(
    swi_uint8 *preq,
    swi_int32 activeIPFamily )
{
    BYTE      instID = DEFAULTBYTEVALUE;
    swi_int16 ipFam;

    struct qm_qmi_response_tlvs_values reqtlvs; /* for unpacking request */
    struct qm_qmi_response_tlvs_values rsptlvs; /* for packing response */

    /* initialize result tlv */
    rsptlvs.qmiresult.result = QMI_RC_SUCCESS;
    rsptlvs.qmiresult.error  = eQCWWAN_ERR_NONE;

    enum eQCWWANError rc = qm_wds_set_sessionid_request_unpack(preq, &reqtlvs);
    if( rc != eQCWWAN_ERR_NONE )
    {
        /* request TLV extraction failed */
        rsptlvs.qmiresult.result = QMI_RC_FAILURE;
        rsptlvs.qmiresult.error  = rc;

        char errmsg[100];
        snprintf(errmsg, sizeof(errmsg),
                 "%s:%d request TLV extraction failed",
                 (char *)__func__, __LINE__);
        erAbort(errmsg, (swi_ulong)rc );
    }

    struct qmtcb *pcb = qmgetcbp();

    /* map session ID and instance */
    instID = reqtlvs.tlvvalues.qmwdstlvs.set_session_id_req.instanceID;
    if( IP_FAMILY_V4 == activeIPFamily )
        ipFam = INDEX_IPV4;
    else
        ipFam = INDEX_IPV6;

    pcb->wds_data.sessnIDMap[instID][ipFam] =
        reqtlvs.tlvvalues.qmwdstlvs.set_session_id_req.sessionID;

#ifdef DBG_WDSINTERNAL
    swi_uint8 instIDCtr = 0;
    syslog( LOG_DEBUG, "~~~~%s~~~\n", __func__ );
    for( instIDCtr = 0; instIDCtr < QM_MAX_QMI_INST_SUPPORTED; instIDCtr++ )
    {
        syslog( LOG_DEBUG, "PID: %x, SIDv4: %lu, SIDv6: %lu\n",
                instIDCtr,
                pcb->wds_data.sessnIDMap[instIDCtr][0],
                pcb->wds_data.sessnIDMap[instIDCtr][1]);
    }
#endif

    qm_wds_set_sessionid_send_response( (enum eQMIMessageWDSInternal)*preq,
                                         &rsptlvs );
}

/*************
 *
 * Name:    qm_wds_get_Instance_unpack_tlv_sessID
 *
 * Purpose: Unpack session ID TLV Value
 *
 * Parms:   psource - source data
 *          pdest   - destination buffer
 *
 * Return:  eQCWWAN_ERR_NONE
 *
 * Abort:   none
 *
 * Notes:
 *
 **************/
local enum eQCWWANError
qm_wds_get_Instance_unpack_tlv_sessID(
    swi_uint8 *psource,
    swi_uint8 *pdest )
{
    struct qm_qmi_response_tlvs_values *pin;
    struct qm_wds_get_instance_id_request_tlv_values *pval;

    pin =   (struct qm_qmi_response_tlvs_values *)pdest;
    pval =  (struct qm_wds_get_instance_id_request_tlv_values *)
                &pin->tlvvalues.qmwdstlvs.get_instance_id_req;

    return GetLong( psource, &pval->sessionID);
}

/*************
 *
 * Name:    qm_wds_get_instance_id_req_unpack
 *
 * Purpose: Unpack WDS get instance ID request
 *
 * Parms:   (IN)    preq        - incoming request packet
 *          (OUT)   preqtlvs    - destination data containing request
 *                                parameters.
 *
 * Return:  eQCWWAN_ERR_NONE if successfully unpacked request
 *          eQCWWAN_ERR_MEMORY if failed to unpack request
 *
 * Abort:   none
 *
 * Notes:
 *
 **************/
local enum eQCWWANError
qm_wds_get_instance_id_req_unpack(
    swi_uint8 *preq,
    struct qm_qmi_response_tlvs_values *preqtlvs )
{
    struct qmTlvUnpackerItem map[] =
    {
        { eTLV_WDS_INTERNAL_GETINSTANCE_SESSIONID,
          qm_wds_get_Instance_unpack_tlv_sessID },

        { eTLV_TYPE_INVALID, NULL }
    };

    return qmunpack( preq,
                     (swi_uint8 *)preqtlvs,
                     map,
                     eQMI_WDS_INT_GET_INSTANCEID );
}

/*************
 *
 * Name:    qm_wds_get_instanceid_instid_tlv_pack
 *
 * Purpose: pack instance ID TLV Value
 *
 * Parms:   (IN)    psrc    - src data
 *          (OUT)   pdest   - destination buffer
 *
 * Return:  success: eQCWWAN_ERR_NONE
 *          failure: eQCWWAN_ERR_MEMORY
 *
 * Abort:   none
 *
 * Notes:
 *
 **************/
local enum eQCWWANError
qm_wds_get_instanceid_instid_tlv_pack(
    swi_uint8 *pdest,
    swi_uint8 *psrc )
{
    struct qm_qmi_response_tlvs_values *pin;
    struct qm_wds_get_instance_id_response_tlv_values *ptlv;

    pin = (struct qm_qmi_response_tlvs_values *)psrc;

    ptlv = (struct qm_wds_get_instance_id_response_tlv_values *)
           &pin->tlvvalues.qmwdstlvs.get_instance_id_resp;

    return PutByte( pdest,
                    ptlv->instanceID );
}

/*************
 *
 * Name:    qm_wds_get_instanceid_IPFamily_tlv_pack
 *
 * Purpose: Pack IP Family ID TLV Value
 *
 * Parms:   (IN)    psrc    - src data
 *          (OUT)   pdest   - destination buffer
 *
 * Return:  success: eQCWWAN_ERR_NONE
 *          failure: eQCWWAN_ERR_MEMORY
 *
 * Abort:   none
 *
 * Notes:
 *
 **************/
local enum eQCWWANError
qm_wds_get_instanceid_IPFamily_tlv_pack(
    swi_uint8 *pdest,
    swi_uint8 *psrc )
{
    struct qm_qmi_response_tlvs_values *pin;
    struct qm_wds_get_instance_id_response_tlv_values *ptlv;

    pin = (struct qm_qmi_response_tlvs_values *)psrc;

    ptlv = (struct qm_wds_get_instance_id_response_tlv_values *)
           &pin->tlvvalues.qmwdstlvs.get_instance_id_resp;

    return PutByte( pdest,
                    ptlv->IPFamily );
}

/*************
 *
 * Name:    qm_wds_get_instanceid_send_response
 *
 * Purpose: Pack and send a WDS get instance ID response
 *
 * Parms:           msgid       - QMI DCS service message id
 *          (OUT)   prsptlvs    - response structure
 *
 * Return:  eQCWWAN_ERR_NONE
 *
 * Abort:   none
 *
 * Notes:
 *
 **************/
local void
qm_wds_get_instanceid_send_response(
    enum   eQMIMessageWDSInternal      msgid,
    struct qm_qmi_response_tlvs_values *prsptlvs )
{
    struct qmTlvBuilderItem map[] =
    {
        { eTLV_RESULT_CODE,
          &qm_result_code_tlv_pack },

        { eTLV_WDS_INTERNAL_GETINSTANCE_INSTID,
          &qm_wds_get_instanceid_instid_tlv_pack },

        { eTLV_WDS_INTERNAL_GETINSTANCE_IPFAM,
          &qm_wds_get_instanceid_IPFamily_tlv_pack },

        { eTLV_TYPE_INVALID,
          NULL } /* sentinel signifies last item in map */
    };

    /* construct and send response to the application */
    qm_qmisvc_send_response( &qmiwdsresp,
                             prsptlvs,
                             map,
                             eQMI_SVC_WDS,
                             msgid,
                             eQMIRES );
}

/*************
 *
 * Name:    qm_wds_get_instance_id
 *
 * Purpose: Get the instance ID corresponding to session handle
 *
 * Parms:   preq - request packet pointer
 *          activeIPFamily - Current IP Family preference
 *
 * Return:  none
 *
 * Abort:   none
 *
 * Notes:   Mapping between session ID and instance ID prevents session
 *          handles from being lost if API process crashes.
 *
 **************/
local void
qm_wds_get_instance_id(
    swi_uint8 *preq,
    swi_int32 activeIPFamily )
{
    UNUSEDPARAM(activeIPFamily);
    swi_ulong sessnID = DEFAULTLONGVALUE;
    swi_uint8  IDctr = 0x0;

    struct qm_qmi_response_tlvs_values reqtlvs; /* for unpacking request */
    struct qm_qmi_response_tlvs_values rsptlvs; /* for packing response */

    /* initialize result tlv */
    rsptlvs.qmiresult.result = QMI_RC_SUCCESS;
    rsptlvs.qmiresult.error  = eQCWWAN_ERR_NONE;
    rsptlvs.tlvvalues.qmwdstlvs.get_instance_id_resp.IPFamily = DEFAULTBYTEVALUE;
    rsptlvs.tlvvalues.qmwdstlvs.get_instance_id_resp.instanceID = DEFAULTBYTEVALUE;

    enum eQCWWANError rc = qm_wds_get_instance_id_req_unpack(preq, &reqtlvs);
    if( rc != eQCWWAN_ERR_NONE )
    {
        /* request TLV extraction failed */
        rsptlvs.qmiresult.result = QMI_RC_FAILURE;
        rsptlvs.qmiresult.error  = rc;

        char errmsg[100];
        snprintf(errmsg, sizeof(errmsg),
                 "%s:%d request TLV extraction failed",
                 (char *)__func__, __LINE__);
        erAbort(errmsg, (swi_ulong)rc );
    }

    struct qmtcb *pcb = qmgetcbp();

    /* Retrieve instance, ip family from session ID */
    sessnID = reqtlvs.tlvvalues.qmwdstlvs.get_instance_id_req.sessionID;

    for( IDctr = 0; IDctr < QM_MAX_QMI_INST_SUPPORTED; IDctr++ )
    {
        if( sessnID == pcb->wds_data.sessnIDMap[IDctr][INDEX_IPV4] )
        {
            rsptlvs.tlvvalues.qmwdstlvs.get_instance_id_resp.IPFamily = IP_FAMILY_V4;
            rsptlvs.tlvvalues.qmwdstlvs.get_instance_id_resp.instanceID = IDctr;
            break;
        }
        else if( sessnID == pcb->wds_data.sessnIDMap[IDctr][INDEX_IPV6] )
        {
            rsptlvs.tlvvalues.qmwdstlvs.get_instance_id_resp.IPFamily = IP_FAMILY_V6;
            rsptlvs.tlvvalues.qmwdstlvs.get_instance_id_resp.instanceID = IDctr;
            break;
        }
    }

#ifdef DBG_WDSINTERNAL
    swi_uint8 instIDCtr = 0;
    syslog( LOG_DEBUG, "~~~~%s~~~\n", __func__ );
    for( instIDCtr = 0; instIDCtr < QM_MAX_QMI_INST_SUPPORTED; instIDCtr++ )
    {
        syslog( LOG_DEBUG, "PID: %x, SIDv4: %lu, SIDv6: %lu\n",
                instIDCtr,
                pcb->wds_data.sessnIDMap[instIDCtr][0],
                pcb->wds_data.sessnIDMap[instIDCtr][1]);
    }
#endif
    qm_wds_get_instanceid_send_response( (enum eQMIMessageWDSInternal)*preq,
                                         &rsptlvs );
}

/*************
 *
 * Name:    qm_wds_get_sessID_unpack_tlv_InstID
 *
 * Purpose: Unpack instance ID TLV Value
 *
 * Parms:   psource - source data
 *          pdest   - destination buffer
 *
 * Return:  eQCWWAN_ERR_NONE
 *
 * Abort:   none
 *
 * Notes:
 *
 **************/
local enum eQCWWANError
qm_wds_get_sessID_unpack_tlv_InstID(
    swi_uint8 *psource,
    swi_uint8 *pdest )
{
    struct qm_qmi_response_tlvs_values *pin;
    struct qm_wds_get_session_id_request_tlv_values *pval;

    pin =  (struct qm_qmi_response_tlvs_values *)pdest;
    pval = (struct qm_wds_get_session_id_request_tlv_values *)
            &pin->tlvvalues.qmwdstlvs.get_session_id_req;

    return GetByte( psource, &pval->instanceID );
}

/*************
 *
 * Name:    qm_wds_get_session_id_req_unpack
 *
 * Purpose: Unpack WDS get session ID request
 *
 * Parms:   (IN)    preq        - incoming request packet
 *          (OUT)   preqtlvs    - destination data containing request
 *                                parameters.
 *
 * Return:  eQCWWAN_ERR_NONE if successfully unpacked request
 *          eQCWWAN_ERR_MEMORY if failed to unpack request
 *
 * Abort:   none
 *
 * Notes:
 *
 **************/
local enum eQCWWANError
qm_wds_get_session_id_req_unpack(
    swi_uint8 *preq,
    struct qm_qmi_response_tlvs_values *preqtlvs )
{
    struct qmTlvUnpackerItem map[] =
    {
        { eTLV_WDS_INTERNAL_GETSESSIONID_INST,
          qm_wds_get_sessID_unpack_tlv_InstID },

        { eTLV_TYPE_INVALID, NULL }
    };

    return qmunpack( preq,
                     (swi_uint8 *)preqtlvs,
                     map,
                     eQMI_WDS_INT_GET_SESSIONID );
}

/*************
 *
 * Name:    qm_wds_get_instanceid_sessidv4_tlv_pack
 *
 * Purpose: Pack Session ID v4 TLV Value
 *
 * Parms:   (IN)    psrc    - src data
 *          (OUT)   pdest   - destination buffer
 *
 * Return:  success: eQCWWAN_ERR_NONE
 *          failure: eQCWWAN_ERR_MEMORY
 *
 * Abort:   none
 *
 * Notes:
 *
 **************/
local enum eQCWWANError
qm_wds_get_instanceid_sessidv4_tlv_pack(
    swi_uint8 *pdest,
    swi_uint8 *psrc )
{
    struct qm_qmi_response_tlvs_values *pin;
    struct qm_wds_get_session_id_response_tlv_values *ptlv;

    pin = (struct qm_qmi_response_tlvs_values *)psrc;

    ptlv = (struct qm_wds_get_session_id_response_tlv_values *)
           &pin->tlvvalues.qmwdstlvs.get_session_id_resp;

    return PutLong( pdest,
                    ptlv->sessionIDv4 );
}

/*************
 *
 * Name:    qm_wds_get_instanceid_sessidv6_tlv_pack
 *
 * Purpose: Pack Session ID v6 TLV Value
 *
 * Parms:   (IN)    psrc    - src data
 *          (OUT)   pdest   - destination buffer
 *
 * Return:  success: eQCWWAN_ERR_NONE
 *          failure: eQCWWAN_ERR_MEMORY
 *
 * Abort:   none
 *
 * Notes:
 *
 **************/
local enum eQCWWANError
qm_wds_get_instanceid_sessidv6_tlv_pack(
    swi_uint8 *pdest,
    swi_uint8 *psrc )
{
    struct qm_qmi_response_tlvs_values *pin;
    struct qm_wds_get_session_id_response_tlv_values *ptlv;

    pin = (struct qm_qmi_response_tlvs_values *)psrc;

    ptlv = (struct qm_wds_get_session_id_response_tlv_values *)
           &pin->tlvvalues.qmwdstlvs.get_session_id_resp;

    return PutLong( pdest,
                    ptlv->sessionIDv6 );
}

/*************
 *
 * Name:    qm_wds_get_sessionid_send_response
 *
 * Purpose: Pack and send a WDS get session ID response
 *
 * Parms:           msgid       - QMI WDS service message id
 *          (OUT)   prsptlvs    - response structure
 *
 * Return:  eQCWWAN_ERR_NONE
 *
 * Abort:   none
 *
 * Notes:
 *
 **************/
local void
qm_wds_get_sessionid_send_response(
    enum   eQMIMessageWDSInternal      msgid,
    struct qm_qmi_response_tlvs_values *prsptlvs )
{
    struct qmTlvBuilderItem map[] =
    {
        { eTLV_RESULT_CODE,
          &qm_result_code_tlv_pack },

        { eTLV_WDS_INTERNAL_GETSESSIONID_SESSIDv4,
          &qm_wds_get_instanceid_sessidv4_tlv_pack },

        { eTLV_WDS_INTERNAL_GETSESSIONID_SESSIDv6,
          &qm_wds_get_instanceid_sessidv6_tlv_pack },

        { eTLV_TYPE_INVALID,
          NULL } /* sentinel signifies last item in map */
    };

    /* construct and send response to the application */
    qm_qmisvc_send_response( &qmiwdsresp,
                             prsptlvs,
                             map,
                             eQMI_SVC_WDS,
                             msgid,
                             eQMIRES );
}

/*************
 *
 * Name:    qm_wds_get_session_id
 *
 * Purpose: Get the session ID's corresponding to instance id's
 *
 * Parms:   preq - request packet pointer
 *          activeIPFamily - Current IP Family preference
 *
 * Return:  none
 *
 * Abort:   none
 *
 * Notes:   Mapping between session ID and instance ID prevents session
 *          handles from being lost if API process crashes.
 *
 **************/
local void qm_wds_get_session_id(swi_uint8 *preq, swi_int32 activeIPFamily )
{
    UNUSEDPARAM( activeIPFamily);

    swi_uint8  instID = DEFAULTBYTEVALUE;
    swi_ulong sessIDv4 = DEFAULTSESSIONID;
    swi_ulong sessIDv6 = DEFAULTSESSIONID;

    struct qm_qmi_response_tlvs_values reqtlvs; /* for unpacking request */
    struct qm_qmi_response_tlvs_values rsptlvs; /* for packing response */

    /* initialize result tlv */
    rsptlvs.qmiresult.result = QMI_RC_SUCCESS;
    rsptlvs.qmiresult.error  = eQCWWAN_ERR_NONE;
    rsptlvs.tlvvalues.qmwdstlvs.get_session_id_resp.sessionIDv4 = DEFAULTSESSIONID;
    rsptlvs.tlvvalues.qmwdstlvs.get_session_id_resp.sessionIDv6 = DEFAULTSESSIONID;

    enum eQCWWANError rc = qm_wds_get_session_id_req_unpack(preq, &reqtlvs);
    if( rc != eQCWWAN_ERR_NONE )
    {
        /* request TLV extraction failed */
        rsptlvs.qmiresult.result = QMI_RC_FAILURE;
        rsptlvs.qmiresult.error  = rc;

        char errmsg[100];
        snprintf(errmsg, sizeof(errmsg),
                 "%s:%d request TLV extraction failed",
                 (char *)__func__, __LINE__);
        erAbort(errmsg, (swi_ulong)rc );
    }

    struct qmtcb *pcb = qmgetcbp();

    /* Retrieve instance, ip family from session ID */
    instID = reqtlvs.tlvvalues.qmwdstlvs.get_session_id_req.instanceID;

    if ( instID >= QM_MAX_QMI_INST_SUPPORTED )
    {
        sessIDv4 = DEFAULTSESSIONID;
        sessIDv6 = DEFAULTSESSIONID;
    }
    else
    {
        sessIDv4 = pcb->wds_data.sessnIDMap[instID][INDEX_IPV4];
        sessIDv6 = pcb->wds_data.sessnIDMap[instID][INDEX_IPV6];
    }

    rsptlvs.tlvvalues.qmwdstlvs.get_session_id_resp.sessionIDv4 = sessIDv4;
    rsptlvs.tlvvalues.qmwdstlvs.get_session_id_resp.sessionIDv6 = sessIDv6;

#ifdef DBG_WDSINTERNAL
    swi_uint8 instIDCtr = 0;
    syslog( LOG_DEBUG, "~~~~%s~~~\n", __func__ );
    for( instIDCtr = 0; instIDCtr < QM_MAX_QMI_INST_SUPPORTED; instIDCtr++ )
    {
        syslog( LOG_DEBUG, "PID: %x, SIDv4: %lu, SIDv6: %lu\n",
                instIDCtr,
                pcb->wds_data.sessnIDMap[instIDCtr][0],
                pcb->wds_data.sessnIDMap[instIDCtr][1]);
    }
#endif

    qm_wds_get_sessionid_send_response( (enum eQMIMessageWDSInternal)*preq,
                                         &rsptlvs );
}

/*************
 *
 * Name:    qm_wds_handler
 *
 * Purpose: Internal Wireless Device Service (WDS) request handler
 *
 * Parms:   preq           - request packet pointer
 *          activeIPFamily - Current IP family preference
 *
 * Return:  none
 *
 * Abort:   none
 *
 * Notes:  pseudo WDS QMI service managed by the SDK in order to provide
 *         additional support for data connections.
 *
 **************/
package void
qm_wds_handler(
    swi_uint8 *preq,
    swi_int32 activeIPFamily )
{
    struct qmtcb *tcbp = qmgetcbp();

    enum eQMIMessageWDSInternal request = tcbp->wds_data.msgID - eQMI_WDS_INT_ENUM_BEGIN;

    qmiwdsresp.ctlflgs = eQMIRES;

    if( (swi_uint8)request < sizeof(wdshandlertbl)/sizeof(wdshandlertbl[0]) )
    {
        wdshandlertbl[*preq]( preq, activeIPFamily );
    }
    else
    {
        qmshortresp( eQCWWAN_ERR_INTERNAL, tcbp->qmwdata.qmipcchannel );
    }
}

/*************
 *
 * Name:    qmwdsintinit
 *
 * Purpose: Initialize internal WDS QMI service
 *
 * Parms:   none
 *
 * Return:  none
 *
 * Abort:   none
 *
 * Notes:
 *
 **************/
package void qmwdsintinit()
{
    /* Initialize QMI response structure */
    qmiwdsresp.svc = eQMI_SVC_WDS;
    qmiwdsresp.instanceid = 0;
}
