/*
 * \ingroup wds
 *
 * \file    qaWdsSWISetDataLoopback.c
 *
 * \brief   Contains Packing and UnPacking routines for the 
 *           eQMI_WDS_SWI_SET_DATA_LOOPBACK message.
 *
 * Copyright: © 2015 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */
#include "SwiDataTypes.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaGobiApiWds.h"
#include "qaWdsSWISetDataLoopback.h"

/*****************************************************************************
 * Request handling
 *****************************************************************************/
/*
 * This function packs the Set Data Loopback Mode to the QMI message
 * SDU.
 *
 * \param  pBuf   [OUT] - Pointer to storage into which the packed
 *                        data will be placed by this function.
 *
 * \param  pParam [IN]  - Pointer to structure containing data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError BuildTlvSetDataLoopbackModeReq( BYTE *pBuf, BYTE *pParam )
{
    WDSSetLoopbackData *pReq = (WDSSetLoopbackData *)pParam;
     if(pReq==NULL)
        return eQCWWAN_ERR_NONE;
    /* Insert Loopback Mode */
    return PutByte( pBuf, *pReq->pLoopbackMode );
}

/*
 * This function packs the Set Data Loopback Multiplier to the QMI message
 * SDU.
 *
 * \param  pBuf   [OUT] - Pointer to storage into which the packed
 *                        data will be placed by this function.
 *
 * \param  pParam [IN]  - Pointer to structure containing data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError BuildTlvSetDataLoopbackMultiplierReq( BYTE *pBuf, BYTE *pParam )
{
    WDSSetLoopbackData *pReq = (WDSSetLoopbackData *)pParam;
    if(pReq==NULL)
        return eQCWWAN_ERR_NONE;
    /* Insert Loopback Multiplier */
    return PutByte( pBuf, *pReq->pLoopbackMultiplier );
}



/* Request handling */

/*
 * This function packs the QMI WDS Set Data Loopback Request parameters
 * to the QMI message SDU
 *
 * \param  pParamField - Pointer to storage into which the packed
 *                       data will be placed by this function.
 *
 * \return eQCWWAN_ERR_NONE, on success
 *
 * \sa     qaGobiApiWds.h for remaining parameter descriptions.
 *
 */

enum eQCWWANError PkQmiWdsSWISetDataLoopbackReq(
    WORD                 *pMlength,
    BYTE                 *pParamField,
    WDSSetLoopbackData *pQmiWdsSWISetDataLoopboackReq)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_SWI_SET_DATA_LOOPBACK_MODE,
            &BuildTlvSetDataLoopbackModeReq },
        { eTLV_SWI_SET_DATA_LOOPBACK_MULTIPLIER,
            &BuildTlvSetDataLoopbackMultiplierReq },
        { eTLV_TYPE_INVALID,                 NULL } /* Important. Sentinel.
                                                     * Signifies last item in map
                                                     */
    };
    eRCode = qmbuild( pParamField,
                      (BYTE *)pQmiWdsSWISetDataLoopboackReq,
                      map,
                      eQMI_WDS_SWI_SET_DATA_LOOPBACK,
                      pMlength );
    return eRCode;
}

/*****************************************************************************
 * Response handling
 *****************************************************************************/
/*
 * This function unpacks the  QMI WDS SWI Set Current Channel Rate response message to a user-
 * provided response structure.
 *
 * \param  pMdmResp           - Pointer to packed response from the modem.
 *
 * \param  pApiResp           - Pointer to storage to unpack into.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UpkQmiWdsSWISetDataLoopbackReq(
    BYTE                                *pMdmResp,
    struct sQmiWdsSWISetDataLoopboackResp *pApiResp )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
     /* The unpack of the Callback contains only result code */
    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,  &qmUnpackTlvResultCode },
        { eTLV_TYPE_INVALID, NULL } /* Important. Sentinel.
                                     * Signifies last item in map.
                                     */
    };

    eRCode = qmunpackresp( pMdmResp,
                           (BYTE *)pApiResp,
                           map,
                           eQMI_WDS_SWI_SET_DATA_LOOPBACK );
    return eRCode;
}

