/*
 * \ingroup wds
 *
 * \file    qaWdsSWISetDHCPv4ClientConfig.c
 *
 * \brief   Contains Packing and UnPacking routines for the 
 *           eQMI_WDS_SWI_SET_DHCPV4_CLIENT_CONFIG message.
 *
 * Copyright: © 2015 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */
#include "SwiDataTypes.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaGobiApiWds.h"
#include "qaWdsSWISetDHCPv4ClientConfig.h"

/*****************************************************************************
 * Request handling
 *****************************************************************************/

/*
 * Packs the DHCPV4 Client profileType and profileID field to the
 * QMI message SDU
 *
 * \param   pBuf    [OUT] - Pointer to storage into which the packed
 *                          data will be placed by this function.
 *
 * \param   pParam  [IN]  - Pointer to structure containing data for this TLV.
 *
 * \return  eQCWWAN_ERR_NONE on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError BuildTlvSetDHCPv4ClientProfileId( BYTE *pBuf, BYTE *pParam )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    WdsDHCPv4ProfileId *pProfileId = ((WdsDHCPv4Config *)pParam)->pProfileId;

    if(pProfileId==NULL)
        return eQCWWAN_ERR_INVALID_ARG;

    /* Insert Profile Type */
    eRCode = PutByte( pBuf, pProfileId->profileType );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    /* Insert Profile ID */
    eRCode = PutByte( pBuf, pProfileId->profileId );

    return eRCode;
}

/*
 * This function packs the DHCPV4 Client HW Config to the QMI message
 * SDU.
 *
 * \param  pBuf   [OUT] - Pointer to storage into which the packed
 *                        data will be placed by this function.
 *
 * \param  pParam [IN]  - Pointer to structure containing data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError BuildTlvSetDHCPv4ClientHWConfigReq( BYTE *pBuf, BYTE *pParam )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    WdsDHCPv4HWConfig *pHwConfigReq = ((WdsDHCPv4Config *)pParam)->pHwConfig;

    if(pHwConfigReq==NULL)
        return eQCWWAN_ERR_NONE;

    /* Insert HW Type */
    eRCode = PutByte( pBuf, pHwConfigReq->hwType );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    /* Insert chaddr length */
    eRCode = PutByte( pBuf, pHwConfigReq->chaddrLen );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    /* Insert chaddr */
    eRCode = PutStream( pBuf, pHwConfigReq->chaddr, sizeof(pHwConfigReq->chaddr));

    return eRCode;
}

/*
 * This function packs the Set DHCP Client V4 Request Options to the QMI message
 * SDU.
 *
 * \param  pBuf   [OUT] - Pointer to storage into which the packed
 *                        data will be placed by this function.
 *
 * \param  pParam [IN]  - Pointer to structure containing data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError BuildTlvSetDHCPv4ClientReqOptionListReq( BYTE *pBuf, BYTE *pParam )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE i = 0;
    WdsDHCPv4Option *pOption;
    WdsDHCPv4OptionList *pOptionListReq = ((WdsDHCPv4Config *)pParam)->pRequestOptionList;

    if ( pOptionListReq==NULL )
        return eQCWWAN_ERR_NONE;

    if ( (pOptionListReq->numOpt > 0) && (pOptionListReq->pOptList == NULL) )
        return eQCWWAN_ERR_INVALID_ARG;

    /* Insert number of options */
    eRCode = PutByte( pBuf, pOptionListReq->numOpt );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    for( i = 0; i < pOptionListReq->numOpt; i++ )
    {
        pOption = pOptionListReq->pOptList + i;

        /* Insert option code */
        eRCode = PutByte( pBuf, pOption->optCode );
        if( eQCWWAN_ERR_NONE != eRCode )
            return eRCode;

        /* Insert option value length */
        eRCode = PutByte( pBuf, pOption->optValLen );
        if( eQCWWAN_ERR_NONE != eRCode )
            return eRCode;

        /* Insert option value */
        if (pOption->optValLen > 0)
        {
          eRCode = PutStream( pBuf, pOption->optVal, pOption->optValLen);
          if( eQCWWAN_ERR_NONE != eRCode )
              return eRCode;
        }
    }

    return eRCode;
}

/* Request handling */

/*
 * This function packs the QMI WDS Set Data Loopback Request parameters
 * to the QMI message SDU
 *
 * \param  pParamField - Pointer to storage into which the packed
 *                       data will be placed by this function.
 *
 * \return eQCWWAN_ERR_NONE, on success
 *
 * \sa     qaGobiApiWds.h for remaining parameter descriptions.
 *
 */

enum eQCWWANError PkQmiWdsSWISetDHCPv4ClientConfigReq(
    WORD            *pMlength,
    BYTE            *pParamField,
    WdsDHCPv4Config *pQmiWdsSWISetDHCPv4ClientConfigReq)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_SWI_SET_DHCPV4_CLIENT_PROFILE_ID,      &BuildTlvSetDHCPv4ClientProfileId },
        { eTLV_SWI_SET_DHCPV4_CLIENT_HW_CONFIG,       &BuildTlvSetDHCPv4ClientHWConfigReq },
        { eTLV_SWI_SET_DHCPV4_CLIENT_REQ_OPTION_LIST, &BuildTlvSetDHCPv4ClientReqOptionListReq },
        { eTLV_TYPE_INVALID,                 NULL } /* Important. Sentinel.
                                                     * Signifies last item in map
                                                     */
    };
    eRCode = qmbuild( pParamField,
                      (BYTE *)pQmiWdsSWISetDHCPv4ClientConfigReq,
                      map,
                      eQMI_WDS_SWI_SET_DHCPV4_CLIENT_CONFIG,
                      pMlength );
    return eRCode;
}

/*****************************************************************************
 * Response handling
 *****************************************************************************/
/*
 * This function unpacks the  QMI WDS SWI Set Current Channel Rate response message to a user-
 * provided response structure.
 *
 * \param  pMdmResp           - Pointer to packed response from the modem.
 *
 * \param  pApiResp           - Pointer to storage to unpack into.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UPkQmiWdsSWISetDHCPv4ClientConfigReq(
    BYTE                                       *pMdmResp,
    struct sQmiWdsSWISetDHCPv4ClientConfigResp *pApiResp )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
     /* The unpack of the Callback contains only result code */
    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,  &qmUnpackTlvResultCode },
        { eTLV_TYPE_INVALID, NULL } /* Important. Sentinel.
                                     * Signifies last item in map.
                                     */
    };

    eRCode = qmunpackresp( pMdmResp,
                           (BYTE *)pApiResp,
                           map,
                           eQMI_WDS_SWI_SET_DHCPV4_CLIENT_CONFIG );
    return eRCode;
}

