/*
 * \ingroup wds
 *
 * \file    qaWdsSWIRMTransferStatistics.c
 *
 * \brief   Contains Packing and UnPacking routines for the eQMI_WDS_SET_EVENT
 *          message.
 *
 * Copyright: © 2015 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */
#include "SwiDataTypes.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaGobiApiWds.h"
#include "qaWdsSWIRMTransferStatistics.h"

/*****************************************************************************
 * Request handling
 *****************************************************************************/
/*
 * This function packs the RM Transfter Statistics to the QMI message
 * SDU.
 *
 * \param  pBuf   [OUT] - Pointer to storage into which the packed
 *                        data will be placed by this function.
 *
 * \param  pParam [IN]  - Pointer to structure containing data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError BuildTlvRMTrasnferStatistics(BYTE *pBuf, BYTE *pParam)
{
    swiRMTrasnferStaticsReq *pReq = (swiRMTrasnferStaticsReq *)pParam;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    /* Add TLV data */
    eRCode = PutByte( pBuf, pReq->bResetStatistics );
    if ( eQCWWAN_ERR_NONE == eRCode )
    {
     eRCode= PutLong( pBuf, pReq->ulMask );
    }
    
    return eRCode;
}

/*
 * This function packs the SLQS Set RM Transfer Statistics parameters to the QMI
 * message SDU
 *
 * \param  pMlength     [OUT] - Total length of built message.
 *
 * \param  pParamField  [OUT] - Pointer to storage into which the packed
 *                              data will be placed by this function.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 * \sa qaWdsSWIRMTransferStatistics.h for remaining parameter descriptions.
 */
enum eQCWWANError PkQmiWdsSWIRMTrasnferStaticsReq(
    WORD                 *pMlength,
    BYTE                 *pParamField,
    swiRMTrasnferStaticsReq *pSetEventReportReq )
{
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_RM_TRANSFER_STATISTICS_INDICATOR, &BuildTlvRMTrasnferStatistics }, 
        { eTLV_TYPE_INVALID,                 NULL } /* Important. Sentinel.
                                                     * Signifies last item in map
                                                     */
    };

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    eRCode = qmbuild( pParamField,
                      (BYTE *)pSetEventReportReq,
                      map,
                      eQMI_WDS_SWI_RM_TRANSFER_STATISTICS,
                      pMlength );
    return eRCode;
}

/*****************************************************************************
 * Response handling
 *****************************************************************************/
/*
 * This function unpacks the  QMI WDS SWI RM Transfer Statics response message to a user-
 * provided response structure.
 *
 * \param  pMdmResp           - Pointer to packed response from the modem.
 *
 * \param  pApiResp           - Pointer to storage to unpack into.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UpkQmiWdsSWIRMTrasnferStaticsReq(
    BYTE                                *pMdmResp,
    struct QmiWdsRMTransferStatisticsResp *pApiResp )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
     /* The unpack of the Callback contains only result code */
    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,  &qmUnpackTlvResultCode },
        { eTLV_TYPE_INVALID, NULL } /* Important. Sentinel.
                                     * Signifies last item in map.
                                     */
    };

    eRCode = qmunpackresp( pMdmResp,
                           (BYTE *)pApiResp,
                           map,
                           eQMI_WDS_SWI_RM_TRANSFER_STATISTICS );
    return eRCode;
}



