/*
 * \ingroup wds
 *
 * \file    qaWdsSWIGetDataLoopback.c
 *
 * \brief   Contains Packing and UnPacking routines for the 
 *           eQMI_WDS_SWI_GET_DATA_LOOPBACK message.
 *
 * Copyright: © 2015 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */
#include "SwiDataTypes.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaGobiApiWds.h"
#include "qaWdsSWIGetDataLoopback.h"

/* Response handling */

/*
 * This function unpacks the QMI WDS Get Data Loopback Mode from the
 * QMI response message to a user provided response structure
 *
 * \param   pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param   pResp    - Pointer to structure containing storage
 *                           to return data for this TLV.
 *
 * \return  eQCWWAN_ERR_NONE, on success
 */

enum eQCWWANError UnpackTlvPacketQmiWdsSWIGetDataLoopbackMode(
BYTE *pTlvData, BYTE *pResp)
{
    struct QmiWdsSWIGetDataLoopBackResp *lResp =
                       (struct QmiWdsSWIGetDataLoopBackResp *)pResp;
    WDSGetLoopbackData *data = NULL;

    BYTE u8Temp= 0;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    if(pResp==NULL)
        return eQCWWAN_ERR_INVALID_ARG;
    
    if(lResp->sGetDataLoopbackData==NULL)
        return eQCWWAN_ERR_INVALID_ARG;
    data = (*lResp).sGetDataLoopbackData;

    eRCode = GetByte(pTlvData,&u8Temp);
    if(eRCode == eQCWWAN_ERR_NONE)
    {
        data->ByteLoopbackMode = u8Temp;
    }
    return eRCode;
}

/*
 * This function unpacks the QMI WDS Get Data Loopback Multiplier from the
 * QMI response message to a user provided response structure
 *
 * \param   pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param   pResp    - Pointer to structure containing storage
 *                           to return data for this TLV.
 *
 * \return  eQCWWAN_ERR_NONE, on success
 */

enum eQCWWANError UnpackTlvPacketQmiWdsSWIGetDataLoopbackMultiplier(
BYTE *pTlvData, BYTE *pResp)
{
    
    struct QmiWdsSWIGetDataLoopBackResp *lResp =
                       (struct QmiWdsSWIGetDataLoopBackResp *)pResp;
    WDSGetLoopbackData *data = NULL;
    
    BYTE u8Temp= 0;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    
    if(pResp==NULL)
        return eQCWWAN_ERR_INVALID_ARG;
    
    if(lResp->sGetDataLoopbackData==NULL)
        return eQCWWAN_ERR_INVALID_ARG;
    data = (*lResp).sGetDataLoopbackData;
    
    eRCode = GetByte(pTlvData,&u8Temp);
    
    if(eRCode == eQCWWAN_ERR_NONE)
    {
        data->ByteLoopbackMultiplier = u8Temp;
    }
    return eRCode;
    
}



/* Request handling */

/*
 * This function packs the QMI WDS Get Data Loopback parameters
 * to the QMI message SDU
 *
 * \param  pParamField - Pointer to storage into which the packed
 *                       data will be placed by this function.
 *
 * \return eQCWWAN_ERR_NONE, on success
 *
 * \sa     qaGobiApiWds.h for remaining parameter descriptions.
 *
 */

enum eQCWWANError PkQmiWdsSWIGetDataLoopbackReq(
    WORD                 *pMlength,
    BYTE                 *pParamField )
{
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_TYPE_INVALID,                 NULL } /* Important. Sentinel.
                                                     * Signifies last item in map
                                                     */
    };

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    eRCode = qmbuild( pParamField,
                      (BYTE *)NULL,
                      map,
                      eQMI_WDS_SWI_GET_DATA_LOOPBACK,
                      pMlength );
    return eRCode;
}

/*****************************************************************************
 * Response handling
 *****************************************************************************/
/*
 * This function unpacks the  QMI WDS SWI Get Current Channel Rate response message to a user-
 * provided response structure.
 *
 * \param  pMdmResp           - Pointer to packed response from the modem.
 *
 * \param  pApiResp           - Pointer to storage to unpack into.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UpkQmiWdsSWIGetDataLoopbackReq(
    BYTE                                *pMdmResp,
    struct QmiWdsSWIGetDataLoopBackResp *pApiResp )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
     /* The unpack of the Callback contains only result code */
    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_SWI_GET_DATA_LOOPBACK_MODE,
            &UnpackTlvPacketQmiWdsSWIGetDataLoopbackMode},
        { eTLV_SWI_GET_DATA_LOOPBACK_MULTIPLIER,
            &UnpackTlvPacketQmiWdsSWIGetDataLoopbackMultiplier},
        { eTLV_RESULT_CODE,  &qmUnpackTlvResultCode },
        { eTLV_TYPE_INVALID, NULL } /* Important. Sentinel.
                                     * Signifies last item in map.
                                     */
    };

    eRCode = qmunpackresp( pMdmResp,
                           (BYTE *)pApiResp,
                           map,
                           eQMI_WDS_SWI_GET_DATA_LOOPBACK );
    return eRCode;
}

