/*
 * \ingroup wds
 *
 * \file qaWdsSLQSGetDUNCallInfo.c
 *
 * \brief Contains Packing and UnPacking routines for the
 *        QMI_WDS_GET_DUN_CALL_INFO message.
 *
 * Copyright: © 2014 Sierra Wireless, Inc. all rights reserved
 *
 */
/* include files */

#include "aa/aaglobal.h"
#include "SwiDataTypes.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "amudefs.h"

#include "qaQmiBasic.h"
#include "qaWdsSLQSGetDUNCallInfo.h"

/*****************************************************************************
 * Request handling
 ******************************************************************************/

/*
 * This function packs the SLQSGetDUNCallInfo mask field to the QMI message
 * SDU
 *
 * \param  pBuf   - Pointer to storage into which the packed data will be
 *                  placed by this function.
 *
 * \param  pParam - Pointer to structure containing data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success
 */
local enum eQCWWANError BuildTlvReqInfo(BYTE *pBuf, BYTE *pParam )
{
    getDUNCallInfoReq *pReq = (getDUNCallInfoReq *)pParam;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Add Mask field */
    eRCode = PutLong(pBuf, pReq->Mask);

    return eRCode;
}

/*
 * This function packs the SLQSGetDUNCallInfo Connection Status Indicator
 * field to the QMI message SDU
 *
 * \param  pBuf   - Pointer to storage into which the packed data will be
 *                  placed by this function.
 *
 * \param  pParam - Pointer to structure containing data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success
 */
local enum eQCWWANError BuildConnStatInd(BYTE *pBuf, BYTE *pParam )
{
    getDUNCallInfoReq *pReq = (getDUNCallInfoReq *)pParam;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    if( NULL == pReq->pReportConnStatus )
    {
        return eQCWWAN_ERR_NONE;
    }

    /* Add Connection Status Indicator */
    eRCode = PutByte( pBuf, *(pReq->pReportConnStatus) );
    return eRCode;
}

/*
 * This function packs the SLQSGetDUNCallInfo Transfer Status Indicator
 * field to the QMI message SDU
 *
 * \param  pBuf   - Pointer to storage into which the packed data will be
 *                  placed by this function.
 *
 * \param  pParam - Pointer to structure containing data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success
 */
local enum eQCWWANError BuildTsferStatInd(BYTE *pBuf, BYTE *pParam )
{
    getDUNCallInfoReq *pReq = (getDUNCallInfoReq *)pParam;
    TransferStatInd   *pTransReq = pReq->pTransferStatInd;

    enum eQCWWANError   eRCode = eQCWWAN_ERR_NONE;

    if( NULL == pTransReq )
    {
        return eQCWWAN_ERR_NONE;
    }

    /* Add Stats Period */
    eRCode = PutByte( pBuf, pTransReq->StatsPeriod );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    /* Add Stats Mask */
    eRCode = PutLong( pBuf, pTransReq->StatsMask );
    return eRCode;
}

/*
 * This function packs the SLQSGetDUNCallInfo Dormancy Status Indicator
 * field to the QMI message SDU
 *
 * \param  pBuf   - Pointer to storage into which the packed data will be
 *                  placed by this function.
 *
 * \param  pParam - Pointer to structure containing data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success
 */
local enum eQCWWANError BuildDormStatInd(BYTE *pBuf, BYTE *pParam )
{
    getDUNCallInfoReq *pReq = (getDUNCallInfoReq *)pParam;

    enum eQCWWANError   eRCode = eQCWWAN_ERR_NONE;

    if( NULL == pReq->pReportDormStatus )
    {
        return eQCWWAN_ERR_NONE;
    }

    /* Add Dormancy Status */
    eRCode = PutByte( pBuf, *pReq->pReportDormStatus );
    return eRCode;
}

/*
 * This function packs the SLQSGetDUNCallInfo Current Data Bearer Technology
 * Indicator field to the QMI message SDU
 *
 * \param  pBuf   - Pointer to storage into which the packed data will be
 *                  placed by this function.
 *
 * \param  pParam - Pointer to structure containing data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success
 */
local enum eQCWWANError BuildDataBearTech(BYTE *pBuf, BYTE *pParam )
{
    getDUNCallInfoReq *pReq = (getDUNCallInfoReq *)pParam;

    enum eQCWWANError   eRCode = eQCWWAN_ERR_NONE;

    if( NULL == pReq->pReportDataBearerTech )
    {
        return eQCWWAN_ERR_NONE;
    }

    /* Add Data Bearer Technology */
    eRCode = PutByte( pBuf, *pReq->pReportDataBearerTech );
    return eRCode;
}

/*
 * This function packs the SLQSGetDUNCallInfo Channel Rate
 * Indicator field to the QMI message SDU
 *
 * \param  pBuf   - Pointer to storage into which the packed data will be
 *                  placed by this function.
 *
 * \param  pParam - Pointer to structure containing data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success
 */
local enum eQCWWANError BuildChanRateInd(BYTE *pBuf, BYTE *pParam )
{
    getDUNCallInfoReq *pReq = (getDUNCallInfoReq *)pParam;

    enum eQCWWANError   eRCode = eQCWWAN_ERR_NONE;

    if( NULL == pReq->pReportChannelRate )
    {
        return eQCWWAN_ERR_NONE;
    }

    /* Add Channel Rate Indicator */
    eRCode = PutByte( pBuf, *pReq->pReportChannelRate );
    return eRCode;
}

/*
 * This function packs the SLQSGetDUNCallInfo parameters
 * to the QMI message SDU
 *
 * \param  pParamField [OUT] - Pointer to storage into which the packed
 *                             data will be placed by this function.
 *
 * \param  pMlength    [OUT] - Total length of built message.
 *
 * \param  pReq        [IN]  - Structure containing request parameters.
 *
 * \return eQCWWAN_ERR_NONE, on success
 *
 * \sa     qaGobiApiWds.h for remaining parameter descriptions.
 *
 */
enum eQCWWANError PkQmiWdsSlqsGetDUNCallInfo(
    WORD               *pMlength,
    BYTE               *pParamField,
    getDUNCallInfoReq  *pReq )
{
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_REQUEST_INFO,      &BuildTlvReqInfo },
        { eTLV_CONNECT_STAT_IND,  &BuildConnStatInd },
        { eTLV_TRANSFER_STAT_IND, &BuildTsferStatInd },
        { eTLV_DORM_STAT_IND,     &BuildDormStatInd },
        { eTLV_DATA_BEAR_TECH,    &BuildDataBearTech },
        { eTLV_CHANNEL_RATE_IND,  &BuildChanRateInd },
        { eTLV_TYPE_INVALID,      NULL }  /* Important. Sentinel.
                                           * Signifies last item in map.
                                           */
    };
    enum eQCWWANError eRCode;

    eRCode = qmbuild( pParamField,
                      (BYTE *)pReq,
                      map,
                      eQMI_WDS_GET_DUN_CALL_INFO,
                      pMlength );
    return eRCode;
}

/*****************************************************************************
 * Response handling
 ******************************************************************************/

/*
 * This function unpacks the connection status from the
 * QMI response message to a user provided response structure
 *
 * \param pTlvData [IN/OUT] - Pointer to TLV data from which to unpack.
 *
 * \param pResp    [OUT]    - Pointer to structure containing storage
 *                            to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success
 *
 */
local enum eQCWWANError UnpackTlvConnStat( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiWdsSlqsGetDUNCallInfoResp *lResp =
        (struct QmiWdsSlqsGetDUNCallInfoResp *)pResp;
    getDUNCallInfoResp *pCaInfo = lResp->pGetDUNCallInfoResp;
    ConnectionStatus *pConnStat = pCaInfo->pConnectionStatus;
    enum   eQCWWANError eRCode  = eQCWWAN_ERR_NONE;

    if ( NULL == pConnStat )
        return eRCode;

    /* Extract modem connection status */
    eRCode = GetByte( pTlvData, &(pConnStat->MDMConnStatus) );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    /* Extract modem call duration */
    eRCode = GetLongLong( pTlvData, &(pConnStat->MDMCallDuration) );
    return eRCode;
}

/*
 * This function unpacks the last modem call end reason from the
 * QMI response message to a user provided response structure
 *
 * \param pTlvData [IN/OUT] - Pointer to TLV data from which to unpack.
 *
 * \param pResp    [OUT]    - Pointer to structure containing storage
 *                            to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success
 *
 */
local enum eQCWWANError UnpackTlvLastMDMCallEndRsn( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiWdsSlqsGetDUNCallInfoResp *lResp =
        (struct QmiWdsSlqsGetDUNCallInfoResp *)pResp;
    getDUNCallInfoResp *pCaInfo = lResp->pGetDUNCallInfoResp;
    enum   eQCWWANError eRCode  = eQCWWAN_ERR_NONE;

    if ( NULL == pCaInfo->pCallEndReason )
        return eRCode;

    /* Extract Call End Reason */
    eRCode = GetWord( pTlvData, pCaInfo->pCallEndReason );
    return eRCode;
}

/*
 * This function unpacks the TX bytes OK count from the
 * QMI response message to a user provided response structure
 *
 * \param pTlvData [IN/OUT] - Pointer to TLV data from which to unpack.
 *
 * \param pResp    [OUT]    - Pointer to structure containing storage
 *                            to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success
 *
 */
local enum eQCWWANError UnpackTlvTXBYTESOKCNT( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiWdsSlqsGetDUNCallInfoResp *lResp =
        (struct QmiWdsSlqsGetDUNCallInfoResp *)pResp;
    getDUNCallInfoResp *pCaInfo = lResp->pGetDUNCallInfoResp;
    enum   eQCWWANError eRCode  = eQCWWAN_ERR_NONE;

    if ( NULL == pCaInfo->pTXOKBytesCount )
        return eRCode;

    /* Extract TX bytes OK count */
    eRCode = GetLongLong( pTlvData, pCaInfo->pTXOKBytesCount );
    return eRCode;
}

/*
 * This function unpacks the RX bytes OK count from the
 * QMI response message to a user provided response structure
 *
 * \param pTlvData [IN/OUT] - Pointer to TLV data from which to unpack.
 *
 * \param pResp    [OUT]    - Pointer to structure containing storage
 *                            to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success
 *
 */
local enum eQCWWANError UnpackTlvRXBYTESOKCNT( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiWdsSlqsGetDUNCallInfoResp *lResp =
        (struct QmiWdsSlqsGetDUNCallInfoResp *)pResp;
    getDUNCallInfoResp *pCaInfo = lResp->pGetDUNCallInfoResp;
    enum   eQCWWANError eRCode  = eQCWWAN_ERR_NONE;

    if ( NULL == pCaInfo->pRXOKBytesCount )
        return eRCode;

    /* Extract RX bytes OK count */
    eRCode = GetLongLong( pTlvData, pCaInfo->pRXOKBytesCount );
    return eRCode;
}

/*
 * This function unpacks the dormancy status from the
 * QMI response message to a user provided response structure
 *
 * \param pTlvData [IN/OUT] - Pointer to TLV data from which to unpack.
 *
 * \param pResp    [OUT]    - Pointer to structure containing storage
 *                            to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success
 *
 */
local enum eQCWWANError UnpackTlvDormStat( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiWdsSlqsGetDUNCallInfoResp *lResp =
        (struct QmiWdsSlqsGetDUNCallInfoResp *)pResp;
    getDUNCallInfoResp *pCaInfo = lResp->pGetDUNCallInfoResp;
    enum   eQCWWANError eRCode  = eQCWWAN_ERR_NONE;

    if ( NULL == pCaInfo->pDormancyStatus )
        return eRCode;

    /* Extract dormancy status */
    eRCode = GetByte( pTlvData, pCaInfo->pDormancyStatus );
    return eRCode;
}

/*
 * This function unpacks the data bearer technology from the
 * QMI response message to a user provided response structure
 *
 * \param pTlvData [IN/OUT] - Pointer to TLV data from which to unpack.
 *
 * \param pResp    [OUT]    - Pointer to structure containing storage
 *                            to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success
 *
 */
local enum eQCWWANError UnpackTlvDataBearerTech( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiWdsSlqsGetDUNCallInfoResp *lResp =
        (struct QmiWdsSlqsGetDUNCallInfoResp *)pResp;
    getDUNCallInfoResp *pCaInfo = lResp->pGetDUNCallInfoResp;
    enum   eQCWWANError eRCode  = eQCWWAN_ERR_NONE;

    if ( NULL == pCaInfo->pDataBearerTech )
        return eRCode;

    /* Extract data bearer technology*/
    eRCode = GetByte( pTlvData, pCaInfo->pDataBearerTech );
    return eRCode;
}

/*
 * This function unpacks the channel rate from the
 * QMI response message to a user provided response structure
 *
 * \param pTlvData [IN/OUT] - Pointer to TLV data from which to unpack.
 *
 * \param pResp    [OUT]    - Pointer to structure containing storage
 *                            to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success
 *
 */
local enum eQCWWANError UnpackTlvChanRate( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiWdsSlqsGetDUNCallInfoResp *lResp =
        (struct QmiWdsSlqsGetDUNCallInfoResp *)pResp;
    getDUNCallInfoResp *pCaInfo = lResp->pGetDUNCallInfoResp;
    ChannelRate        *pCRate  = pCaInfo->pChannelRate;
    enum   eQCWWANError eRCode  = eQCWWAN_ERR_NONE;

    if ( NULL == pCRate )
        return eRCode;

    /* Extract current channel TX rate */
    eRCode = GetLong( pTlvData, &(pCRate->CurrChanTxRate) );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    /* Extract current channel TX rate */
    eRCode = GetLong( pTlvData, &(pCRate->CurrChanRxRate) );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    /* Extract Max channel TX rate */
    eRCode = GetLong( pTlvData, &(pCRate->MaxChanTxRate) );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    /* Extract Max channel RX rate */
    eRCode = GetLong( pTlvData, &(pCRate->MaxChanRxRate) );
    return eRCode;
}

/*
 * This function unpacks the Last Call Tx Bytes OK from the
 * QMI response message to a user provided response structure
 *
 * \param pTlvData [IN/OUT] - Pointer to TLV data from which to unpack.
 *
 * \param pResp    [OUT]    - Pointer to structure containing storage
 *                            to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success
 *
 */
local enum eQCWWANError UnpackTlvLastCallTXBytesOK( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiWdsSlqsGetDUNCallInfoResp *lResp =
        (struct QmiWdsSlqsGetDUNCallInfoResp *)pResp;
    getDUNCallInfoResp *pCaInfo = lResp->pGetDUNCallInfoResp;
    enum   eQCWWANError eRCode  = eQCWWAN_ERR_NONE;

    if ( NULL == pCaInfo->pLastCallTXOKBytesCnt )
        return eRCode;

    /* Extract Last Call Tx Bytes OK */
    eRCode = GetLongLong( pTlvData, pCaInfo->pLastCallTXOKBytesCnt );
    return eRCode;
}

/*
 * This function unpacks the Last Call Rx Bytes OK from the
 * QMI response message to a user provided response structure
 *
 * \param pTlvData [IN/OUT] - Pointer to TLV data from which to unpack.
 *
 * \param pResp    [OUT]    - Pointer to structure containing storage
 *                            to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success
 *
 */
local enum eQCWWANError UnpackTlvLastCallRXBytesOK( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiWdsSlqsGetDUNCallInfoResp *lResp =
        (struct QmiWdsSlqsGetDUNCallInfoResp *)pResp;
    getDUNCallInfoResp *pCaInfo = lResp->pGetDUNCallInfoResp;
    enum   eQCWWANError eRCode  = eQCWWAN_ERR_NONE;

    if ( NULL == pCaInfo->pLastCallRXOKBytesCnt )
        return eRCode;

    /* Extract Last Call Rx Bytes OK */
    eRCode = GetLongLong( pTlvData, pCaInfo->pLastCallRXOKBytesCnt );
    return eRCode;
}

/*
 * This function unpacks the Call Active Duration from the
 * QMI response message to a user provided response structure
 *
 * \param pTlvData [IN/OUT] - Pointer to TLV data from which to unpack.
 *
 * \param pResp    [OUT]    - Pointer to structure containing storage
 *                            to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success
 *
 */
local enum eQCWWANError UnpackTlvCallActivDuration( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiWdsSlqsGetDUNCallInfoResp *lResp =
        (struct QmiWdsSlqsGetDUNCallInfoResp *)pResp;
    getDUNCallInfoResp *pCaInfo = lResp->pGetDUNCallInfoResp;
    enum   eQCWWANError eRCode  = eQCWWAN_ERR_NONE;

    if ( NULL == pCaInfo->pMdmCallDurationActive )
        return eRCode;

    /* Extract modem call duration active */
    eRCode = GetLongLong( pTlvData, pCaInfo->pMdmCallDurationActive );
    return eRCode;
}

/*
 * This function unpacks the Last Call Data Bearer Technology from the
 * QMI response message to a user provided response structure
 *
 * \param pTlvData [IN/OUT] - Pointer to TLV data from which to unpack.
 *
 * \param pResp    [OUT]    - Pointer to structure containing storage
 *                            to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success
 *
 */
local enum eQCWWANError UnpackTlvLastCallDataBearerTech( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiWdsSlqsGetDUNCallInfoResp *lResp =
        (struct QmiWdsSlqsGetDUNCallInfoResp *)pResp;
    getDUNCallInfoResp *pCaInfo = lResp->pGetDUNCallInfoResp;
    enum   eQCWWANError eRCode  = eQCWWAN_ERR_NONE;

    if ( NULL == pCaInfo->pLastCallDataBearerTech )
        return eRCode;

    /* Extract last call data bearer technology */
    eRCode = GetByte( pTlvData, pCaInfo->pLastCallDataBearerTech );
    return eRCode;
}

/*
 * This function unpacks the SLQSGetDUNCallInfo response message to a
 * user-provided response structure.
 *
 * \param  pMdmResp     [IN]  - Pointer to packed response from the modem.
 *
 * \param  pProfileType [IN]  - Profile type of the profile to be fetched.
 *
 * \param  pApiResp     [OUT] - Pointer to storage to unpack into.
 *
 * \return eQCWWAN_ERR_NONE, on success
 */
enum eQCWWANError UpkQmiWdsSlqsGetDUNCallInfo(
    BYTE                                *pMdmResp,
    struct QmiWdsSlqsGetDUNCallInfoResp *pApiResp)
{
    enum   eQCWWANError eRCode;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,          &qmUnpackTlvResultCode },
        { eTLV_CONNECTIONSTATUS,     &UnpackTlvConnStat },
        { eTLV_LASTMDMCALLENDREASON, &UnpackTlvLastMDMCallEndRsn },
        { eTLV_TXBYTESOKCNT,         &UnpackTlvTXBYTESOKCNT },
        { eTLV_RXBYTESOKCNT,         &UnpackTlvRXBYTESOKCNT },
        { eTLV_DORMSTATUS,           &UnpackTlvDormStat },
        { eTLV_DATABEARERTECH,       &UnpackTlvDataBearerTech },
        { eTLV_CHANNRATE,            &UnpackTlvChanRate },
        { eTLV_LASTCALLTXBYTESOK,    &UnpackTlvLastCallTXBytesOK },
        { eTLV_LASTCALLRXBYTESOK,    &UnpackTlvLastCallRXBytesOK },
        { eTLV_CALLACTIVEDURATION,   &UnpackTlvCallActivDuration },
        { eTLV_LASTCALLDATABEARTECH, &UnpackTlvLastCallDataBearerTech },
        { eTLV_TYPE_INVALID,         NULL } /* Important. Sentinel.
                                             * Signifies last item
                                             * in map
                                             */
    };

    eRCode = qmunpackresp( pMdmResp,
                           (BYTE *)pApiResp,
                           map,
                           eQMI_WDS_GET_DUN_CALL_INFO );
    return eRCode;
}
