/*
 * \ingroup wds
 *
 * \file qaWdsSLQSGetCurrDataSystemStat.c
 *
 * \brief Contains Packing and UnPacking routines for the
 *        QMI_WDS_SLQS_GET_CURR_DATA_SYSTEM_STAT message.
 *
 * Copyright: © 2014 Sierra Wireless, Inc. all rights reserved
 *
 */
/* include files */

#include "aa/aaglobal.h"
#include "SwiDataTypes.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "amudefs.h"

#include "qaQmiBasic.h"
#include "qaWdsSLQSGetCurrDataSystemStat.h"

/*****************************************************************************
 * Request handling
 ******************************************************************************/

/*
 * This function packs the SLQSGetCurrDataSystemStat parameters
 * to the QMI message SDU
 *
 * \param  pParamField [OUT] - Pointer to storage into which the packed
 *                             data will be placed by this function.
 *
 * \param  pMlength    [OUT] - Total length of built message.
 *
 * \return eQCWWAN_ERR_NONE, on success
 *
 * \sa     qaGobiApiWds.h for remaining parameter descriptions.
 *
 */
enum eQCWWANError PkQmiWdsSLQSGetCurrDataSystemStat(
    WORD               *pMlength,
    BYTE               *pParamField )
{
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_TYPE_INVALID, NULL }  /* Important. Sentinel.
                                      * Signifies last item in map.
                                      */
    };
    enum eQCWWANError eRCode;

    eRCode = qmbuild( pParamField,
                      NULL,
                      map,
                      eQMI_WDS_GET_CURR_DATA_SYS_STAT,
                      pMlength );
    return eRCode;
}

/*****************************************************************************
 * Response handling
 ******************************************************************************/

/*
 * This function unpacks the data system status from the
 * QMI response message to a user provided response structure
 *
 * \param pTlvData [IN/OUT] - Pointer to TLV data from which to unpack.
 *
 * \param pResp    [OUT]    - Pointer to structure containing storage
 *                            to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success
 *
 */
local enum eQCWWANError UnpackTlvDataSystemStat( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiWdsSLQSGetCurrDataSystemStat *lResp =
        (struct QmiWdsSLQSGetCurrDataSystemStat *)pResp;
    CurrDataSysStat *pCurrDataSysStat = lResp->pCurrDataSysStat;
    CurrNetworkInfo *lTemp;

    enum eQCWWANError eRCode  = eQCWWAN_ERR_NONE;
    BYTE bTemp = 0;
    BYTE lCtr = 0;

    if ( NULL == pCurrDataSysStat->pPrefNetwork ||
         NULL == pCurrDataSysStat->pNetworkInfoLen )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Extract preferred network */
    eRCode = GetByte( pTlvData, pCurrDataSysStat->pPrefNetwork );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    /* Extract network information length */
    eRCode = GetByte( pTlvData, &bTemp );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    /* Check buffer undersize error */
    if( bTemp > *(pCurrDataSysStat->pNetworkInfoLen) )
        return eQCWWAN_ERR_BUFFER_SZ;

    *(pCurrDataSysStat->pNetworkInfoLen) = bTemp;

    /* Fetch Network Information */
    for( lCtr = 0; lCtr < *(pCurrDataSysStat->pNetworkInfoLen); lCtr++ )
    {
        lTemp = &(pCurrDataSysStat->pCurrNetworkInfo[lCtr]);

        /* Extract network type */
        eRCode = GetByte( pTlvData, &lTemp->NetworkType );
        if( eQCWWAN_ERR_NONE != eRCode )
            return eRCode;

        /* Extract RAT mask */
        eRCode = GetLong( pTlvData, &lTemp->RATMask );
        if( eQCWWAN_ERR_NONE != eRCode )
            return eRCode;

        /* Extract SO mask */
        eRCode = GetLong( pTlvData, &lTemp->SOMask );
        if( eQCWWAN_ERR_NONE != eRCode )
            return eRCode;
    }

    return eRCode;
}

/*
 * This function unpacks the SLQSGetCurrDataSystemStat response message to a
 * user-provided response structure.
 *
 * \param  pMdmResp     [IN]  - Pointer to packed response from the modem.
 *
 * \param  pApiResp     [OUT] - Pointer to storage to unpack into.
 *
 * \return eQCWWAN_ERR_NONE, on success
 */
enum eQCWWANError UpkQmiWdsSLQSGetCurrDataSystemStat(
    BYTE                                   *pMdmResp,
    struct QmiWdsSLQSGetCurrDataSystemStat *pApiResp)
{
    enum   eQCWWANError eRCode;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,      &qmUnpackTlvResultCode },
        { eTLV_DATASYSTEMSTATUS, &UnpackTlvDataSystemStat },
        { eTLV_TYPE_INVALID,     NULL } /* Important. Sentinel.
                                         * Signifies last item
                                         * in map
                                         */
    };

    eRCode = qmunpackresp( pMdmResp,
                           (BYTE *)pApiResp,
                           map,
                           eQMI_WDS_GET_CURR_DATA_SYS_STAT );
    return eRCode;
}
