/*
 * \ingroup wds
 *
 * \file qaWdsIntSLQSGetSessionID.c
 *
 * \brief  Contains Packing and UnPacking routines for the
 *         QMI_WDS_INT_GET_SESSIONID message.
 *
 * Copyright: © 2014 Sierra Wireless, Inc., all rights reserved
 *
 */

/* include files */

#include "SwiDataTypes.h"
#include "sludefs.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaGobiApiWds.h"
#include "qaWdsIntSLQSGetSessionID.h"

/*****************************************************************************
 * Request handling
 ******************************************************************************/
/*
 * This function packs the iSLQSGetSessionID instance ID field to
 * the QMI message SDU
 *
 * \param   pBuf    [OUT] - Pointer to storage into which the packed
 *                          data will be placed by this function.
 *
 * \param   pParam  [IN]  - Pointer to structure containing data for this TLV.
 *
 * \return  eQCWWAN_ERR_NONE, on success
 */
enum eQCWWANError BuildTlvInstID(BYTE *pBuf, BYTE *pParam)
{
    struct QmiWdsSLQSGetSessionIDReq *pReq =
                (struct QmiWdsSLQSGetSessionIDReq *)pParam;

    /* Add TLV data */
    return PutByte( pBuf, pReq->InstanceID );
}

/*
 * This function packs the iSLQSGetSessionID parameters
 * to the QMI message SDU
 *
 * \param  pParamField [OUT] - Pointer to storage into which the packed
 *                             data will be placed by this function.
 *
 * \param  pMlength    [OUT] - Total length of built message.
 *
 * \param  sessionID   [OUT] - session ID to pack.
 *
 * \return eQCWWAN_ERR_NONE, on success
 *
 * \sa     qaGobiApiWds.h for remaining parameter descriptions.
 */
enum eQCWWANError PkQmiWdsIntGetSessnID(
    WORD  *pMlength,
    BYTE  *pParamField,
    BYTE  InstanceID )
{
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_WDS_INTERNAL_GETSESSIONID_INST, &BuildTlvInstID },
        { eTLV_TYPE_INVALID,                   NULL }  /* Important. Sentinel.
                                                        * Signifies last item
                                                        * in map. */
    };

    struct QmiWdsSLQSGetSessionIDReq req;
    enum   eQCWWANError              eRCode;

    slmemset((char *)&req, 0, sizeof (struct QmiWdsSLQSGetSessionIDReq));
    req.InstanceID = InstanceID;

    eRCode = qmbuild( pParamField,
                      (BYTE *)&req,
                      map,
                      eQMI_WDS_INT_GET_SESSIONID,
                      pMlength );
    return eRCode;
}

/*****************************************************************************
 * Response handling
 ******************************************************************************/

/*
 * This function unpacks the iSLQSGetSessionID Session ID v4 from the
 * QMI response message to a user provided response structure
 *
 * \param     pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param     pResp    [OUT]     - Pointer to structure containing storage
 *                                 to return data for this TLV.
 *
 * \return    eQCWWAN_ERR_NONE, on success
 */
enum eQCWWANError UnpackTlvSessIDv4( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiWdsSLQSGetSessionIDResp *pLresp =
                                   (struct QmiWdsSLQSGetSessionIDResp *)pResp;
    GetSessionIDResp *lResp = pLresp->pGetSessionIDResp;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    ULONG             lTemp;

    if ( NULL == lResp->pSessionIDv4 )
        return eQCWWAN_ERR_MEMORY;

    /* Extract the IPv4 session ID */
    eRCode = GetLong( pTlvData, &lTemp );
    if( eQCWWAN_ERR_NONE == eRCode )
        *(lResp->pSessionIDv4) = lTemp;

    return eRCode;
}

/*
 * This function unpacks the iSLQSGetSessionID Session ID v6 from the
 * QMI response message to a user provided response structure
 *
 * \param     pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param     pResp    [OUT]     - Pointer to structure containing storage
 *                                 to return data for this TLV.
 *
 * \return    eQCWWAN_ERR_NONE, on success
 */
enum eQCWWANError UnpackTlvSessIDv6( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiWdsSLQSGetSessionIDResp *pLresp =
                                   (struct QmiWdsSLQSGetSessionIDResp *)pResp;
    GetSessionIDResp *lResp = pLresp->pGetSessionIDResp;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    ULONG             lTemp;

    if ( NULL == lResp->pSessionIDv6 )
        return eQCWWAN_ERR_MEMORY;

    /* Extract the IPv6 session ID */
    eRCode = GetLong( pTlvData, &lTemp );
    if( eQCWWAN_ERR_NONE == eRCode )
        *(lResp->pSessionIDv6) = lTemp;

    return eRCode;
}

/*
 * This function unpacks the iSLQSGetSessionID response message to a
 * user-provided response structure.
 *
 * \param     pMdmResp   [IN]  - Pointer to packed response from the modem.
 *
 * \param     pApiResp   [OUT] - Pointer to storage to unpack into.
 *
 * \return    eQCWWAN_ERR_NONE, on success
 */
enum eQCWWANError UpkQmiWdsIntGetSessnID(
    BYTE                              *pMdmResp,
    struct QmiWdsSLQSGetSessionIDResp *pApiResp )
{
    enum eQCWWANError eRCode;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,                        &qmUnpackTlvResultCode },
        { eTLV_WDS_INTERNAL_GETSESSIONID_SESSIDv4, &UnpackTlvSessIDv4 },
        { eTLV_WDS_INTERNAL_GETSESSIONID_SESSIDv6, &UnpackTlvSessIDv6 },
        { eTLV_TYPE_INVALID,                     NULL } /* Important. Sentinel.
                                                         * Signifies last item
                                                         *  in map.
                                                         */
    };
    eRCode = qmunpackresp( pMdmResp,
                           (BYTE *)pApiResp,
                           map,
                           eQMI_WDS_INT_GET_SESSIONID );
    return eRCode;
}
