/*
 * \ingroup wds
 *
 * \file qaWdsIntSLQSGetInstanceID.c
 *
 * \brief  Contains Packing and UnPacking routines for the
 *         QMI_WDS_INT_GET_INSTANCEID message.
 *
 * Copyright: © 2014 Sierra Wireless, Inc., all rights reserved
 *
 */

/* include files */

#include "SwiDataTypes.h"
#include "sludefs.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaGobiApiWds.h"
#include "qaWdsIntSLQSGetInstanceID.h"

/*****************************************************************************
 * Request handling
 ******************************************************************************/
/*
 * This function packs the iSLQSGetInstanceID session ID field to
 * the QMI message SDU
 *
 * \param   pBuf    [OUT] - Pointer to storage into which the packed
 *                          data will be placed by this function.
 *
 * \param   pParam  [IN]  - Pointer to structure containing data for this TLV.
 *
 * \return  eQCWWAN_ERR_NONE, on success
 */
enum eQCWWANError BuildTlvSessID(BYTE *pBuf, BYTE *pParam)
{
    struct QmiWdsSLQSGetInstIDReq *pReq =
                (struct QmiWdsSLQSGetInstIDReq *)pParam;

    /* Add TLV data */
    return PutLong( pBuf, pReq->sessionID );
}

/*
 * This function packs the iSLQSGetInstanceID parameters
 * to the QMI message SDU
 *
 * \param  pParamField [OUT] - Pointer to storage into which the packed
 *                             data will be placed by this function.
 *
 * \param  pMlength    [OUT] - Total length of built message.
 *
 * \param  sessionID   [OUT] - session ID to pack.
 *
 * \return eQCWWAN_ERR_NONE, on success
 *
 * \sa     qaGobiApiWds.h for remaining parameter descriptions.
 */
enum eQCWWANError PkQmiWdsIntGetInstID(
    WORD  *pMlength,
    BYTE  *pParamField,
    ULONG sessionID )
{
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_WDS_INTERNAL_GETINSTANCE_SESSIONID, &BuildTlvSessID },
        { eTLV_TYPE_INVALID,                       NULL }  /* Important.
                                                            * Sentinel.
                                                            * Signifies last
                                                            * item in map. */
    };

    struct QmiWdsSLQSGetInstIDReq req;
    enum   eQCWWANError           eRCode;

    slmemset((char *)&req, 0, sizeof (struct QmiWdsSLQSGetInstIDReq));
    req.sessionID = sessionID;

    eRCode = qmbuild( pParamField,
                      (BYTE *)&req,
                      map,
                      eQMI_WDS_INT_GET_INSTANCEID,
                      pMlength );
    return eRCode;
}

/*****************************************************************************
 * Response handling
 ******************************************************************************/

/*
 * This function unpacks the iSLQSGetInstanceID Instance ID from the
 * QMI response message to a user provided response structure
 *
 * \param     pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param     pResp    [OUT]     - Pointer to structure containing storage
 *                                 to return data for this TLV.
 *
 * \return    eQCWWAN_ERR_NONE, on success
 */
enum eQCWWANError UnpackTlvInstID( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiWdsSLQSGetInstIDResp *pLresp =
                                   (struct QmiWdsSLQSGetInstIDResp *)pResp;
    GetInstIDResp *lResp = pLresp->pGetInstIDResp;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE              lTemp;

    if ( NULL == lResp->pInstanceID )
        return eQCWWAN_ERR_MEMORY;

    /* Extract the Instance ID */
    eRCode = GetByte( pTlvData, &lTemp );
    if( eQCWWAN_ERR_NONE == eRCode )
        *(lResp->pInstanceID) = lTemp;
    return eRCode;
}

/*
 * This function unpacks the iSLQSGetInstanceID IP family from the
 * QMI response message to a user provided response structure
 *
 * \param     pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param     pResp    [OUT]     - Pointer to structure containing storage
 *                                 to return data for this TLV.
 *
 * \return    eQCWWAN_ERR_NONE, on success
 */
enum eQCWWANError UnpackTlvIPFamily( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiWdsSLQSGetInstIDResp *pLresp =
                                   (struct QmiWdsSLQSGetInstIDResp *)pResp;
    GetInstIDResp *lResp = pLresp->pGetInstIDResp;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE              lTemp;

    if ( NULL == lResp->pIPFamily )
        return eRCode;

    /* Extract the IP Family */
    eRCode = GetByte( pTlvData, &lTemp );
    if(eQCWWAN_ERR_NONE == eRCode)
        *(lResp->pIPFamily) = lTemp;
    return eRCode;
}

/*
 * This function unpacks the iSLQSGetInstanceID response message to a
 * user-provided response structure.
 *
 * \param     pMdmResp   [IN]  - Pointer to packed response from the modem.
 *
 * \param     pApiResp   [OUT] - Pointer to storage to unpack into.
 *
 * \return    eQCWWAN_ERR_NONE, on success
 */
enum eQCWWANError UpkQmiWdsIntGetInstID(
    BYTE                           *pMdmResp,
    struct QmiWdsSLQSGetInstIDResp *pApiResp)
{
    enum eQCWWANError eRCode;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,                     &qmUnpackTlvResultCode },
        { eTLV_WDS_INTERNAL_GETINSTANCE_INSTID, &UnpackTlvInstID },
        { eTLV_WDS_INTERNAL_GETINSTANCE_IPFAM,  &UnpackTlvIPFamily },
        { eTLV_TYPE_INVALID,                    NULL } /* Important. Sentinel.
                                                        * Signifies last item
                                                        * in map.
                                                        */
    };
    eRCode = qmunpackresp( pMdmResp,
                           (BYTE *)pApiResp,
                           map,
                           eQMI_WDS_INT_GET_INSTANCEID );
    return eRCode;
}
