/*
 * \ingroup uim
 *
 * \file    qaUimSLQSUimSwitchSlot.c
 *
 * \brief   Contains Packing and UnPacking routines for the
 *          QMI_UIM_SWITCH_SLOT message.
 *
 * Copyright:  2016 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */
#include "SwiDataTypes.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaUimSLQSUimSwitchSlot.h"

/******************************************************************************
 * Request handling
 ******************************************************************************/
/*
 * This function packs the Logical Slot parameter to the QMI message SDU.
 *
 * \param  pBuf   [OUT] - Pointer to storage into which the packed
 *                        data will be placed by this function.
 *
 * \param  pParam [IN]  - Pointer to structure containing data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError BuildTlvLogicalSlot( BYTE *pBuf, BYTE *pParam )
{
    UIMSwitchSlotReq *pReq = ( UIMSwitchSlotReq *)pParam;

    if ( NULL == pReq )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    return PutByte( pBuf, pReq->bLogicalSlot);
}

/*
 * This function packs Physical Slot parameter to the QMI message SDU.
 *
 * \param  pBuf   [OUT] - Pointer to storage into which the packed
 *                        data will be placed by this function.
 *
 * \param  pParam [IN]  - Pointer to structure containing data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError BuildTlvPhysicalSlot( BYTE *pBuf, BYTE *pParam )
{
    UIMSwitchSlotReq *pReq = ( UIMSwitchSlotReq *)pParam;

    if ( NULL == pReq )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    return PutLong( pBuf, pReq->ulPhysicalSlot );
}

/*
 * This function packs the Switch Slot request parameters to the QMI message SDU.
 *
 * \param  pMlength     [OUT] - Total length of built message.
 *
 * \param  pParamField  [OUT] - Pointer to storage into which the packed
 *                              data will be placed by this function.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 * \sa qaUimSLQSUimSwitchSlot.h for remaining parameter descriptions.
 */
enum eQCWWANError PkQmiUimSlqsSwitchSlot(
    WORD            *pMlength,
    BYTE            *pParamField,
    UIMSwitchSlotReq *pReq )
{
    /* There is no mandatory TLV and hence the map is empty */
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_UIM_SWITCH_SLOT_LOGICAL_SLOT, &BuildTlvLogicalSlot },
        { eTLV_UIM_SWITCH_SLOT_PHYICAL_SLOT, &BuildTlvPhysicalSlot },
        { eTLV_TYPE_INVALID,    NULL } /* Important. Sentinel.
                                        * Signifies last item in map
                                        */
    };

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    eRCode = qmbuild( pParamField,
                      (BYTE *)pReq,
                      map,
                      eQMI_UIM_SWITCH_SLOT,
                      pMlength );
    return eRCode;
}

/******************************************************************************
 * Response handling
 ******************************************************************************/
/*
 * This function unpacks the Switch Slot response message to a
 * user-provided response structure.
 *
 * \param  pMdmResp           - Pointer to packed response from the modem.
 *
 * \param  pApiResp           - Pointer to storage to unpack into.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UpkQmiUimSlqsSwitchSlot(
    BYTE                           *pMdmResp,
    QmiUimSLQSSwitchSlotResp *pApiResp )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,  &qmUnpackTlvResultCode },
        { eTLV_TYPE_INVALID, NULL } /* Important. Sentinel.
                                     * Signifies last item in map.
                                     */
    };

    eRCode = qmunpackresp( pMdmResp,
                           (BYTE *)pApiResp,
                           map,
                           eQMI_UIM_SWITCH_SLOT );
    return eRCode;
}
