/*
 * \ingroup uim
 *
 * \file    qaUimSLQSUimPowerUp.c
 *
 * \brief   Contains Packing and UnPacking routines for the
 *          QMI_UIM_POWER_UP message.
 *
 * Copyright:  2015 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */
#include "SwiDataTypes.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaUimSLQSUimPowerUp.h"

/******************************************************************************
 * Request handling
 ******************************************************************************/
/*
 * This function packs the Power Up Slot parameter to the QMI message SDU.
 *
 * \param  pBuf   [OUT] - Pointer to storage into which the packed
 *                        data will be placed by this function.
 *
 * \param  pParam [IN]  - Pointer to structure containing data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError BuildTlvPowerUpSlot( BYTE *pBuf, BYTE *pParam )
{
    UIMPowerUpReq *pReq = ( UIMPowerUpReq *)pParam;

    if ( NULL == pReq )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Insert Slot param */
    return PutByte( pBuf, pReq->slot );
}

/*
 * This function packs the Hot Swap Switch parameter to the QMI message SDU.
 *
 * \param  pBuf   [OUT] - Pointer to storage into which the packed
 *                        data will be placed by this function.
 *
 * \param  pParam [IN]  - Pointer to structure containing data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError BuildTlvIgnoreHotSwapSwitch( BYTE *pBuf, BYTE *pParam )
{
    UIMPowerUpReq *pReq = ( UIMPowerUpReq *)pParam;

    if ( NULL == pReq )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }
    if(pReq->pIgnoreHotSwapSwitch==NULL)
    {
        return eQCWWAN_ERR_NONE;
    }

    /* Insert Ignore Hot Swap Switch param */
    return PutByte( pBuf, *pReq->pIgnoreHotSwapSwitch );
}

/*
 * This function packs the SLQS Power Up request parameters to the QMI message SDU.
 *
 * \param  pMlength     [OUT] - Total length of built message.
 *
 * \param  pParamField  [OUT] - Pointer to storage into which the packed
 *                              data will be placed by this function.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 * \sa qaUimSLQSUimPowerUp.h for remaining parameter descriptions.
 */
enum eQCWWANError PkQmiUimSlqsPowerUp(
    WORD            *pMlength,
    BYTE            *pParamField,
    UIMPowerUpReq *pUIMPowerUpReq )
{
    /* There is no mandatory TLV and hence the map is empty */
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_POWER_UP_SLOT, &BuildTlvPowerUpSlot },
        { eTLV_POWER_UP_IGNORE_HOT_SWAP_SWITCH, &BuildTlvIgnoreHotSwapSwitch },
        { eTLV_TYPE_INVALID,    NULL } /* Important. Sentinel.
                                        * Signifies last item in map
                                        */
    };

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    eRCode = qmbuild( pParamField,
                      (BYTE *)pUIMPowerUpReq,
                      map,
                      eQMI_UIM_POWER_UP,
                      pMlength );
    return eRCode;
}

/******************************************************************************
 * Response handling
 ******************************************************************************/
/*
 * This function unpacks the SLQS Power Up response message to a
 * user-provided response structure.
 *
 * \param  pMdmResp           - Pointer to packed response from the modem.
 *
 * \param  pApiResp           - Pointer to storage to unpack into.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UpkQmiUimSlqsPowerUp(
    BYTE                           *pMdmResp,
    struct QmiUimSLQSPowerUpResp *pApiResp )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,  &qmUnpackTlvResultCode },
        { eTLV_TYPE_INVALID, NULL } /* Important. Sentinel.
                                     * Signifies last item in map.
                                     */
    };

    eRCode = qmunpackresp( pMdmResp,
                           (BYTE *)pApiResp,
                           map,
                           eQMI_UIM_POWER_UP );
    return eRCode;
}
