/*
 * \ingroup uim
 *
 * \file    qaUimSLQSUimGetSlotsStatus.c
 *
 * \brief   Contains Packing and UnPacking routines for the
 *          QMI_UIM_GET_SLOTS_STATUS message.
 *
 * Copyright:  2016 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */
#include "SwiDataTypes.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaGobiApiUim.h"
#include "qaUimSLQSUimGetSlotsStatus.h"

/******************************************************************************
 * Response handling
 ******************************************************************************/
/*
 * This function unpacks the Slots Status Values from the QMI response
 * message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError UnpackSlotsStatus(
    BYTE *pTlvData,
    BYTE *pResp )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    UIMGetSlotsStatusResp *lResp = NULL;
    BYTE bCount  =0;
    
    /* Check For Invalid Parameter */
    if ( NULL == pResp )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }
    

    lResp = ((QmiUimSLQSGetSlotsStatusResp *)pResp)->pGetSlotsStatusResp;

    if(lResp == NULL)
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    if(lResp->pNumberOfPhySlot==NULL)
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Copy the card result values into the List */
    eRCode = GetByte( pTlvData, lResp->pNumberOfPhySlot );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    for(bCount=0;bCount<*lResp->pNumberOfPhySlot;bCount++)
    {
        BYTE bICCIDCount = 0;
        eRCode = GetLong( pTlvData, &lResp->pUimSlotsStatus->uimSlotStatus[bCount].uPhyCardStatus );
        if ( eQCWWAN_ERR_NONE != eRCode )
        {
            return eRCode;
        }
        eRCode = GetLong( pTlvData, &lResp->pUimSlotsStatus->uimSlotStatus[bCount].uPhySlotStatus );
        if ( eQCWWAN_ERR_NONE != eRCode )
        {
            return eRCode;
        }
        eRCode = GetByte( pTlvData, &lResp->pUimSlotsStatus->uimSlotStatus[bCount].bLogicalSlot );
        if ( eQCWWAN_ERR_NONE != eRCode )
        {
            return eRCode;
        }
        eRCode = GetByte( pTlvData, &lResp->pUimSlotsStatus->uimSlotStatus[bCount].bICCIDLength );
        if ( eQCWWAN_ERR_NONE != eRCode )
        {
            return eRCode;
        }
        for(bICCIDCount=0;bICCIDCount<lResp->pUimSlotsStatus->uimSlotStatus[bCount].bICCIDLength;bICCIDCount++)
        {
            eRCode = GetByte( pTlvData, &lResp->pUimSlotsStatus->uimSlotStatus[bCount].bICCID[bICCIDCount] );
            if ( eQCWWAN_ERR_NONE != eRCode )
            {
                return eRCode;
            }
        }
            
    }
    return eRCode;
}


/*
 * This function packs the Get Slots Status request parameters to the QMI message SDU.
 *
 * \param  pMlength     [OUT] - Total length of built message.
 *
 * \param  pParamField  [OUT] - Pointer to storage into which the packed
 *                              data will be placed by this function.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 * \sa qaUimSLQSUimGetSlotsStatus.h for remaining parameter descriptions.
 */
enum eQCWWANError PkQmiUimSlqsGetSlotsStatus(
    WORD            *pMlength,
    BYTE            *pParamField )
{
    /* There is no mandatory TLV and hence the map is empty */
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_TYPE_INVALID,    NULL } /* Important. Sentinel.
                                        * Signifies last item in map
                                        */
    };

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    eRCode = qmbuild( pParamField,
                      NULL,
                      map,
                      eQMI_UIM_GET_SLOTS_STATUS,
                      pMlength );
    return eRCode;
}

/******************************************************************************
 * Response handling
 ******************************************************************************/
/*
 * This function unpacks the Get Slots Status response message to a
 * user-provided response structure.
 *
 * \param  pMdmResp           - Pointer to packed response from the modem.
 *
 * \param  pApiResp           - Pointer to storage to unpack into.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UpkQmiUimSlqsGetSlotsStatus(
    BYTE                           *pMdmResp,
    QmiUimSLQSGetSlotsStatusResp *pApiResp )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,  &qmUnpackTlvResultCode },
        { eTLV_UIM_GET_SLOTS_STATUS_PHY_SLOT_STATUS,
            &UnpackSlotsStatus},
        { eTLV_TYPE_INVALID, NULL } /* Important. Sentinel.
                                     * Signifies last item in map.
                                     */
    };

    eRCode = qmunpackresp( pMdmResp,
                           (BYTE *)pApiResp,
                           map,
                           eQMI_UIM_GET_SLOTS_STATUS );
    return eRCode;
}
