/**************
 *
 *  Filename: qaGobiApiNasTest.c
 *
 *  Purpose:  Contains functions implementing specific tests for
 *            Network Access Service (NAS)
 *            called by the main() routine in qatest.c
 *
 * Copyright: © 2011-2014 Sierra Wireless, Inc., all rights reserved
 *
 **************/

/* Linux definitions */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* include files */
#include "SwiDataTypes.h"
#include "qmerrno.h"

#include "qaGobiApiNas.h"
#include "qaGobiApiNasTest.h"
#include "qatesthelper.h"
#include "qatestproto.h"

/*******************
    TEST DATA
 *******************/

/* Constants */
#define NETREG       0x00
#define NETSELMCC    123
#define NETSELMNC    456

/* Global Variables */
local BYTE      EmerMode    = 0x00;
local WORD      ModePref    = 0x20;//UMTS:0x08 TD-SCDMA=0x20
local ULONGLONG bandPref    = ~0;///33554432 CDMA US 1700 Band
local WORD      PRLPref     = 0x3FFF;
local WORD      RoamPref    = 0xFF;
local ULONGLONG LTEBandPref = 0x08;
local BYTE      chgDuration = 0x01;
local BYTE      MNCIncPCS   = 0x01;

local ULONG     srvDomPref  = eSYS_SRV_DOMAIN_PS_ONLY;
local ULONG     GWAcqOrder  = 0x00;
local BYTE      AcqOrder[2]       = {0x01,0x02};
local ULONGLONG TdscdmaBandPref   = 0x11;
local ULONG     SrvRegRestriction = 0x01;


local struct netSelectionPref NetSelPref = { NETREG, NETSELMCC, NETSELMNC };
local struct acqOrderPref    AcqOrderPref = { 2, AcqOrder };
local struct CSGID           CSGIDobj = { 9, 11, 1, 2, 0x09};

local BYTE      GetEmerMode;
local WORD      GetModePref;
local ULONGLONG GetBandPref;
local WORD      GetPRLPref;
local WORD      GetRoamPref;
local ULONGLONG GetLTEBandPref;
local BYTE      GetNetSelPref;
local ULONG     GetSrvDomPref;
local ULONG     GetGWAcqOrder;

/*  Parameters for SLQSNasIndicationRegisterExt() API */
local BYTE setInd   = 0x01;
local BYTE unsetInd = 0x00;

/*******************
  TEST FUNCTIONS
 *******************/
/*
 * Name:     doprintsysInfoCommon
 *
 * Purpose:  This is a wrapper for printing System Information that is common
 *
 * Parms:    fp          - pointer to FILE
 *           pSys        - pointer to sysInfoCommon
 *
 * Return:   None
 *
 * Notes:    None.
 *
 */
void doprintsysInfoCommon( FILE *fp, sysInfoCommon *pSys )
{
    FILE *fpTmp;
    fpTmp = fp;
    fprintf( fpTmp,"Service Domain Valid   : %x\n",pSys->srvDomainValid );
    fprintf( fpTmp,"Service Domain         : %x\n",pSys->srvDomain );
    fprintf( fpTmp,"Service Capability Valid: %x\n",pSys->srvCapabilityValid );
    fprintf( fpTmp,"Service Capability     : %x\n",pSys->srvCapability );
    fprintf( fpTmp,"Roam Status Valid      : %x\n",pSys->roamStatusValid );
    fprintf( fpTmp,"Roam Status            : %x\n",pSys->roamStatus );
    fprintf( fpTmp,"Forbidden System Valid : %x\n",pSys->isSysForbiddenValid );
    fprintf( fpTmp,"Is Forbidden System    : %x\n",pSys->isSysForbiddenValid );
}

/*
 * Name:     doGetHomeNetwork
 *
 * Purpose:  Perform the tests that call the GetHomeNetwork() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */

void doGetHomeNetwork( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    WORD  pMCC;
    WORD  pMNC;
    BYTE  nameSize;
    CHAR  pName[nMaxStrLen];
    WORD  pSID;
    WORD  pNID;

    FILE  *fp = NULL;

    fp = tfopen("../../nas/test/results/gethomenetwork.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_GET_HOME_NETWORK_TESTCASE_NUM)
        {
            nameSize    = GetHomeNetworkTestCases[tCaseNum].nameSize;

            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_GET_HOME_NETWORK_TESTCASE_NUM,
                       GetHomeNetworkTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "nameSize :%d\n", nameSize);

            nRet = GetHomeNetwork(&pMCC,
                                  &pMNC,
                                  nameSize,
                                  &pName[0],
                                  &pSID,
                                  &pNID);

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "pMCC         : %d\n", pMCC);
                fprintf(fp, "pMNC         : %d\n", pMNC);
                fprintf(fp, "pName        : %s\n", pName);
                fprintf(fp, "pSID         : %d\n", pSID);
                fprintf(fp, "pNID         : %d\n", pNID);
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doGetServingSystem
 *
 * Purpose:  Perform the tests that call the GetServingSystem() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doGetServingSystem( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    ULONG pRegistrationState;
    ULONG pCSDomain;
    ULONG pPSDomain;
    ULONG pRAN;
    BYTE  pRadioIfacesSize;
    BYTE  pRadioIfaces[nMaxStrLen];
    ULONG pRoaming;
    WORD  pMCC;
    WORD  pMNC;
    BYTE  nameSize;
    CHAR  pName[nMaxStrLen];
    BYTE  count;
    FILE  *fp = NULL;

    fp = tfopen("../../nas/test/results/getservingsystem.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_GET_SERVING_SYSTEM_TESTCASE_NUM)
        {
            pRadioIfacesSize    = GetServingSystemTestCases[tCaseNum].pRadioIfacesSize;
            nameSize            = GetServingSystemTestCases[tCaseNum].nameSize;

            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_GET_SERVING_SYSTEM_TESTCASE_NUM,
                       GetServingSystemTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "pRadioIfacesSize :%d\n", pRadioIfacesSize);
            fprintf(fp, "nameSize :%d\n", nameSize);

            nRet = GetServingNetwork(&pRegistrationState,
                                     &pCSDomain,
                                     &pPSDomain,
                                     &pRAN,
                                     &pRadioIfacesSize,
                                     &pRadioIfaces[0],
                                     &pRoaming,
                                     &pMCC,
                                     &pMNC,
                                     nameSize,
                                     &pName[0]);

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "pRegistrationState  : %u\n", pRegistrationState);
                fprintf(fp, "pCSDomain  : %u\n", pCSDomain);
                fprintf(fp, "pPSDomain : %u\n", pPSDomain);
                fprintf(fp, "pRAN  : %u\n", pRAN);
                fprintf(fp, "pRadioIfacesSize  : %d\n", pRadioIfacesSize);
                for(count = 0; count < pRadioIfacesSize; count++)
                    fprintf(fp, "pRadioIfaces  : %d\n", pRadioIfaces[count]);
                fprintf(fp, "pRoaming  : %u\n", pRoaming);
                fprintf(fp, "pMCC  : %d\n", pMCC);
                fprintf(fp, "pMNC  : %d\n", pMNC);
                fprintf(fp, "pName  : %s\n", pName);
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doGetSignalStrength
 *
 * Purpose:  Perform the tests that call the GetSignalStrength() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doGetSignalStrengths( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    ULONG pArraySizes;
    INT8  pSignalStrengths[nMaxStrLen];
    ULONG pRadioInterfaces[nMaxStrLen];
    BYTE  count;
    FILE  *fp = NULL;

    fp = tfopen("../../nas/test/results/getsignalstrength.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_GET_SIGNAL_STRENGTHS_TESTCASE_NUM)
        {
            pArraySizes    = GetSignalStrengthsTestCases[tCaseNum].pArraySizes;

            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_GET_SIGNAL_STRENGTHS_TESTCASE_NUM,
                       GetSignalStrengthsTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "pArraySizes :%u\n", pArraySizes);

            nRet = GetSignalStrengths(&pArraySizes,
                                      &pSignalStrengths[0],
                                      &pRadioInterfaces[0]);

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                for(count = 0; count < pArraySizes; count++)
                {
                    fprintf(fp, "pSignalStrengths[%d] : %d\n",
                            count, pSignalStrengths[count]);
                    fprintf(fp, "pRadioInterfaces[%d] : %u\n",
                            count, pRadioInterfaces[count]);
                }
            }
            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doInitNetworkReg
 *
 * Purpose:  Perform the tests that call the InitiateNetworkRegistration() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doInitNetworkReg( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    ULONG regType;
    WORD  mcc;
    WORD  mnc;
    ULONG rat;
    FILE  *fp = NULL;

    fp = tfopen("../../nas/test/results/initnetworkreg.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_INIT_NETWORK_REG_NUM)
        {
            regType = InitNetworkRegTestCases[tCaseNum].regType;
            mcc     = InitNetworkRegTestCases[tCaseNum].mcc;
            mnc     = InitNetworkRegTestCases[tCaseNum].mnc;
            rat     = InitNetworkRegTestCases[tCaseNum].rat;

            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_INIT_NETWORK_REG_NUM,
                       InitNetworkRegTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "regType :%u\n", regType);
            fprintf(fp, "mcc :%d\n", mcc);
            fprintf(fp, "mnc :%d\n", mnc);
            fprintf(fp, "rat :%u\n", rat);

            nRet = InitiateNetworkRegistration(regType,
                                               mcc,
                                               mnc,
                                               rat);

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "Network Registration Initiated\n");
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doPerformNetworkRegistration
 *
 * Purpose:  Perform the tests that call the PerformNetworkRegistration() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doPerformNetworkScan( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    BYTE  pInstanceSize;
    struct pInstance
    {
        WORD     pMCC;
        WORD     pMNC;
        ULONG    pInUse;
        ULONG    pRoaming;
        ULONG    pForbidden;
        ULONG    pPreferred;
        CHAR     pDesription[255];
    } pInstances[nMaxStrLen];
    //BYTE  pInstances[nMaxStrLen];
    BYTE  count;
    FILE  *fp = NULL;

    fp = tfopen("../../nas/test/results/performnetworkscan.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_PERFORM_NETWORK_SCAN_TESTCASE_NUM)
        {
            pInstanceSize = PerformNetworkScanTestCases[tCaseNum].pInstanceSize;

            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_PERFORM_NETWORK_SCAN_TESTCASE_NUM,
                       PerformNetworkScanTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "pInstanceSize :%d\n", pInstanceSize);

            nRet = PerformNetworkScan(&pInstanceSize,
                                      (BYTE *)&pInstances[0]);

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                for(count = 0; count < pInstanceSize; count++)
                {
                    fprintf(fp, "pMCC[%d] : %d\n",
                            count, pInstances[count].pMCC);
                    fprintf(fp, "pMNC[%d] : %d\n",
                            count, pInstances[count].pMNC);
                    fprintf(fp, "pInUse[%d] : %u\n",
                            count, pInstances[count].pInUse);
                    fprintf(fp, "pRoaming[%d] : %u\n",
                            count, pInstances[count].pRoaming);
                    fprintf(fp, "pForbidden[%d] : %u\n",
                            count, pInstances[count].pForbidden);
                    fprintf(fp, "pPreferred[%d] : %u\n",
                            count, pInstances[count].pPreferred);
                    fprintf(fp, "pDesription[%d] : %s\n",
                            count, pInstances[count].pDesription);
                }
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doGetServingNetworkCaps
 *
 * Purpose:  Perform the tests that call the GetServingNetworkCaps() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doGetServingNetworkCaps( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    BYTE  pDataCapsSize;
    BYTE  pDataCaps[nMaxStrLen];
    FILE  *fp = NULL;
    BYTE  count;

    fp = tfopen("../../nas/test/results/getservingnetworkcaps.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_GET_SERVING_NETWORK_CAPS_TESTCASE_NUM)
        {
            pDataCapsSize = GetServingNetworkCapsTestCases[tCaseNum].DataCapsSize;

            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_GET_SERVING_NETWORK_CAPS_TESTCASE_NUM,
                       GetServingNetworkCapsTestCases[tCaseNum].desc);

            nRet = GetServingNetworkCapabilities( &pDataCapsSize,
                                                  &pDataCaps[0] );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "pDataCapsSize  : %d\n", pDataCapsSize);
                for(count = 0; count < pDataCapsSize; count++)
                    fprintf(fp, "pDataCaps  : %u\n", pDataCaps[count]);
            }
            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doSetNetworkPreference
 *
 * Purpose:  Perform the tests that call the SetNetworkPreference() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSetNetworkPreference( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    FILE  *fp = NULL;

    fp = tfopen("../../nas/test/results/setnetworkpreference.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_SET_NETWORK_PREFERENCE_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_SET_NETWORK_PREFERENCE_TESTCASE_NUM,
                       SetNetworkPreferenceTestCases[tCaseNum].desc );

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "techpref :%u\n",
                    SetNetworkPreferenceTestCases[tCaseNum].techpref);
            fprintf(fp, "duration :%u\n",
                    SetNetworkPreferenceTestCases[tCaseNum].duration);

            nRet = SetNetworkPreference(
                              SetNetworkPreferenceTestCases[tCaseNum].techpref,
                              SetNetworkPreferenceTestCases[tCaseNum].duration );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "Network Preference Set\n");
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doGetNetworkPreference
 *
 * Purpose:  Perform the tests that call the GetNetworkPreference() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doGetNetworkPreference( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    ULONG pTechnologyPref;
    ULONG pDuration;
    ULONG pPersistentTechnologyPref;
    FILE  *fp = NULL;

    fp = tfopen("../../nas/test/results/getnetworkpreference.txt", "w");

    if (fp)
    {
        while ( tCaseNum < MAX_GET_NETWORK_PREFERENCE_TESTCASE_NUM )
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_GET_NETWORK_PREFERENCE_TESTCASE_NUM,
                       GetNetworkPreferenceTestCases[tCaseNum].desc);

            nRet = GetNetworkPreference( &pTechnologyPref,
                                         &pDuration,
                                         &pPersistentTechnologyPref );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "pTechnologyPref            : %u\n",
                            pTechnologyPref);
                fprintf(fp, "pDuration                  : %u\n", pDuration);
                fprintf(fp, "pPersistentTechnologyPref  : %u \n",
                            pPersistentTechnologyPref);
            }
            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doGetRFInfo
 *
 * Purpose:  Perform the tests that call the GetRFInfo() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doGetRFInfo( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    BYTE  count;
    FILE  *fp = NULL;
    BYTE  instanceSize = nMaxStrLen;
    struct RFBandInfoElements RFBandInfo[nMaxStrLen];

    fp = tfopen("../../nas/test/results/getrfinfo.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_GET_RF_INFO_TESTCASE_NUM)
        {
            memset(RFBandInfo, 0, sizeof(RFBandInfo));

            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_GET_RF_INFO_TESTCASE_NUM,
                       GetRFInfoTestCases[tCaseNum].desc);

            nRet = GetRFInfo( &instanceSize, RFBandInfo );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "Instance Size  : %d\n", instanceSize);
                for(count = 0; count < instanceSize; count++)
                {
                    fprintf(fp, "radioInterface[%d] : %d\n",
                            count, RFBandInfo[count].radioInterface);
                    fprintf(fp, "activeBandClass[%d] : %d\n",
                            count, RFBandInfo[count].activeBandClass);
                    fprintf(fp, "activeChannel[%d] : %d\n",
                            count, RFBandInfo[count].activeChannel);
                }
            }
            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doInitiateDomainAttach
 *
 * Purpose:  Perform the tests that call the InitiateDomainAttach() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doInitiateDomainAttach( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    FILE  *fp = NULL;

    fp = tfopen("../../nas/test/results/initdomainattach.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_INITIATE_DOMAIN_ATTACH_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_INITIATE_DOMAIN_ATTACH_TESTCASE_NUM,
                       InitiateDomainAttachTestCases[tCaseNum].desc );

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "action :%u\n",
                    InitiateDomainAttachTestCases[tCaseNum].action);

            nRet = InitiateDomainAttach(
                              InitiateDomainAttachTestCases[tCaseNum].action );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "Domain Attach Initiated\n");
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doGetACCOLC
 *
 * Purpose:  Perform the tests that call the GetACCOLC() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doGetACCOLC( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    BYTE  ACCOLC;
    FILE  *fp = NULL;

    fp = tfopen("../../nas/test/results/getaccolc.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_GET_ACCOLC_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_GET_ACCOLC_TESTCASE_NUM,
                       GetACCOLCTestCases[tCaseNum].desc);

            nRet = GetACCOLC(&ACCOLC);

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "ACCOLC  : %d\n", ACCOLC);
            }
            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doSetACCOLC
 *
 * Purpose:  Perform the tests that call the SetACCOLC() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSetACCOLC( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    CHAR  SPC[6];
    FILE  *fp = NULL;

    fp = tfopen("../../nas/test/results/setaccolc.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_SET_ACCOLC_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_SET_ACCOLC_TESTCASE_NUM,
                       SetACCOLCTestCases[tCaseNum].desc );

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "spc :%s\n",
                    SetACCOLCTestCases[tCaseNum].SPC);
            fprintf(fp, "accolc :%d\n",
                    SetACCOLCTestCases[tCaseNum].accolc);
            strcpy(SPC, SetACCOLCTestCases[tCaseNum].SPC);

            nRet = SetACCOLC( &SPC[0],
                              SetACCOLCTestCases[tCaseNum].accolc );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "ACCOLC Set\n");
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doSetCDMANetworkParameters
 *
 * Purpose:  Perform the tests that call the SetCDMANetworkParameters() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSetCDMANetworkParameters( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    FILE  *fp = NULL;
    BYTE  forcerev0;
    BYTE  customSCP;
    ULONG protocol;
    ULONG broadcast;
    ULONG application;
    ULONG roaming;
    fp = tfopen("../../nas/test/results/setCDMAnetworkparameters.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_SET_CDMA_NETWORK_PARAMETERS_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_SET_CDMA_NETWORK_PARAMETERS_TESTCASE_NUM,
                       SetCDMANetworkParametersTestCases[tCaseNum].desc );

            forcerev0   = SetCDMANetworkParametersTestCases[tCaseNum].forcerev0;
            customSCP   = SetCDMANetworkParametersTestCases[tCaseNum].customSCP;
            protocol    = SetCDMANetworkParametersTestCases[tCaseNum].protocol;
            broadcast   = SetCDMANetworkParametersTestCases[tCaseNum].broadcast;
            application = SetCDMANetworkParametersTestCases[tCaseNum].application;
            roaming     = SetCDMANetworkParametersTestCases[tCaseNum].roaming;

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "spc :%s\n",
                    SetCDMANetworkParametersTestCases[tCaseNum].spc);
            fprintf(fp, "forcerev0 :%d\n",
                    SetCDMANetworkParametersTestCases[tCaseNum].forcerev0);
            fprintf(fp, "customSCP :%d\n",
                    SetCDMANetworkParametersTestCases[tCaseNum].customSCP);
            fprintf(fp, "protocol :%u\n",
                    SetCDMANetworkParametersTestCases[tCaseNum].protocol);
            fprintf(fp, "broadcast :%u\n",
                    SetCDMANetworkParametersTestCases[tCaseNum].broadcast);
            fprintf(fp, "application :%u\n",
                    SetCDMANetworkParametersTestCases[tCaseNum].application);
            fprintf(fp, "roaming :%u\n",
                    SetCDMANetworkParametersTestCases[tCaseNum].roaming);

            nRet = SetCDMANetworkParameters(
                   (CHAR *)&SetCDMANetworkParametersTestCases[tCaseNum].spc[0],
                         &forcerev0,
                         &customSCP,
                         &protocol,
                         &broadcast,
                         &application,
                         &roaming );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "CDMA Network Parameters Set\n");
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doGetCDMANetworkParameters
 *
 * Purpose:  Perform the tests that call the GetCDMANetworkParameters() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doGetCDMANetworkParameters( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    BYTE  pSCI;
    BYTE  pSCM;
    BYTE  pRegHomeSID;
    BYTE  pRegForeignSID;
    BYTE  pRegForeignNID;
    BYTE  pForceRev0;
    BYTE  pCustomSCP;
    ULONG pProtocol;
    ULONG pBroadcast;
    ULONG pApplication;
    ULONG pRoaming;
    FILE  *fp = NULL;

    fp = tfopen("../../nas/test/results/getcdmanetworkparameters.txt", "w");

    if (fp)
    {
        while ( tCaseNum < MAX_GET_CDMA_NETWORK_PARAMETERS_TESTCASE_NUM )
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_GET_CDMA_NETWORK_PARAMETERS_TESTCASE_NUM,
                       GetCDMANetworkParametersTestCases[tCaseNum].desc);

            nRet = GetCDMANetworkParameters( &pSCI,
                                             &pSCM,
                                             &pRegHomeSID,
                                             &pRegForeignSID,
                                             &pRegForeignNID,
                                             &pForceRev0,
                                             &pCustomSCP,
                                             &pProtocol,
                                             &pBroadcast,
                                             &pApplication,
                                             &pRoaming );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "pSCI           : %d\n", pSCI);
                fprintf(fp, "pSCM           : %d\n", pSCM);
                fprintf(fp, "pRegHomeSID    : %d\n", pRegHomeSID);
                fprintf(fp, "pRegForeignSID : %d\n", pRegForeignSID);
                fprintf(fp, "pRegForeignNID : %d\n", pRegForeignNID);
                fprintf(fp, "pCustomSCP     : %d\n", pCustomSCP);
                fprintf(fp, "pProtocol      : %u\n", pProtocol);
                fprintf(fp, "pBroadcast     : %u\n", pBroadcast);
                fprintf(fp, "pApplication   : %u\n", pApplication);
                fprintf(fp, "pRoaming       : %u\n", pRoaming);
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doGetANAAAAuthStatus
 *
 * Purpose:  Perform the tests that call the GetANAAAAuthenticationStatus() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doGetANAAAAuthStatus( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    ULONG pStatus;
    FILE  *fp = NULL;

    fp = tfopen("../../nas/test/results/getanaaaauthstatus.txt", "w");

    if (fp)
    {
        while ( tCaseNum < MAX_GET_ANAAA_AUTH_TESTCASE_NUM )
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_GET_ANAAA_AUTH_TESTCASE_NUM,
                       GetANAAAAuthStatusTestCases[tCaseNum].desc);

            nRet = GetANAAAAuthenticationStatus( &pStatus );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "pStatus : %u\n", pStatus);
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doGetSLQSServingSystem
 *
 * Purpose:  Perform the tests that call the SLQSGetServingSystem() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSLQSGetServingSystem( void )
{
    ULONG                   nRet;
    FILE                    *fp = NULL;
    qaQmiServingSystemParam ServingSystemInfo;

    /* Define the test cases to be executed */
    SLQSGetServingSystemTestCase_t SLQSGetServingSystemTestCases[] =
    {
        { eQCWWAN_ERR_INVALID_ARG, 1, NULL, "Invalid Test Case - Response structure NULL"},

        { eQCWWAN_ERR_NONE, 2, &ServingSystemInfo, "Valid test Case "},
    };

    fp = tfopen( "../../nas/test/results/slqsgetservingsysteminfo.txt", "w" );
    if ( fp )
    {
        BYTE tCaseNum     = 0;
        BYTE ncases = sizeof( SLQSGetServingSystemTestCases ) /
                      sizeof( SLQSGetServingSystemTestCases[0] );

        while( tCaseNum < ncases )
        {
            SLQSGetServingSystemTestCase_t *pd =
                &SLQSGetServingSystemTestCases[tCaseNum];

            /* print test data */
            fprintf( fp,"\n\nTest Case %d / %d : %s\n",
                     (tCaseNum + 1),
                     ncases,
                     pd->desc );

            nRet = SLQSGetServingSystem( pd->pServingSystemInfo );

            /* Display result code and text */
            doprintreason( fp, nRet );
            if (!nRet)
            {
                int    index = 0;
                int    ctr   = 0;
                fprintf( fp,"Serving System Information \n" );
                fprintf( fp, "Registration state: 0x%x\n",         ServingSystemInfo.ServingSystem.regState );
                fprintf( fp, "CS Attach State: 0x%x\n",            ServingSystemInfo.ServingSystem.csAttachState );
                fprintf( fp, "PS Attach State: 0x%x\n",            ServingSystemInfo.ServingSystem.psAttachState );
                fprintf( fp, "Selected Network: 0x%x\n",           ServingSystemInfo.ServingSystem.selNetwork );
                fprintf( fp, "Number of Radio Interfaces: 0x%x\n", ServingSystemInfo.ServingSystem.numRadioInterfaces );
                ctr = ServingSystemInfo.ServingSystem.numRadioInterfaces;
                for( index = 0; index < ctr; index++ )
                {
                    fprintf( fp, "Radio Interface in use : 0x%x\n", ServingSystemInfo.ServingSystem.radioInterface[index] );
                }
                fprintf( fp,"\n" );
                fprintf( fp, "Roaming Indicator value: 0x%x\n",     ServingSystemInfo.roamIndicatorVal );

                fprintf( fp, "\n Data Service capability \n" );
                fprintf( fp, "Data Capability List Length: 0x%x\n", ServingSystemInfo.DataSrvCapabilities.dataCapabilitiesLen );
                ctr = ServingSystemInfo.DataSrvCapabilities.dataCapabilitiesLen;
                for( index = 0; index < ctr; index++ )
                {
                    fprintf( fp, "Data Capability : 0x%x\n", ServingSystemInfo.DataSrvCapabilities.dataCapabilities[index] );
                }

                fprintf( fp, "\n Current PLMN Information \n" );
                fprintf( fp, "Mobile Country Code: 0x%x\n",        ServingSystemInfo.CurrentPLMN.MCC );
                fprintf( fp, "Mobile Country Code: 0x%x\n",        ServingSystemInfo.CurrentPLMN.MNC );
                fprintf( fp, "Network Description Length: 0x%x\n", ServingSystemInfo.CurrentPLMN.netDescrLength );
                ctr = ServingSystemInfo.CurrentPLMN.netDescrLength;
                fprintf( fp, "Network Description : ");
                for( index = 0; index < ctr; index++ )
                {
                    fprintf( fp, "%c",  (char) ServingSystemInfo.CurrentPLMN.netDescr[index] );
                }
                fprintf( fp, "\n");

                fprintf( fp, "\n CDMA System ID \n" );
                fprintf( fp, "System ID: 0x%x\n",                  ServingSystemInfo.SystemID );
                fprintf( fp, "Network ID: 0x%x\n",                 ServingSystemInfo.NetworkID );

                fprintf( fp, "\n CDMA Base Station Information \n" );
                fprintf( fp, "Base ID: 0x%x\n",                    ServingSystemInfo.BasestationID );
                fprintf( fp, "Base Station latitude: 0x%x\n",     ServingSystemInfo.BasestationLatitude );
                fprintf( fp, "Base Station longitude: 0x%x\n",    ServingSystemInfo.BasestationLongitude );

                fprintf( fp, "\n Roaming Indicator List \n" );
                fprintf( fp, "Number of Instances: 0x%x\n",        ServingSystemInfo.RoamingIndicatorList.numInstances );
                ctr = ServingSystemInfo.RoamingIndicatorList.numInstances;
                for( index = 0; index < ctr; index++ )
                {
                    fprintf( fp, "Radio Interface : 0x%x\n",   ServingSystemInfo.RoamingIndicatorList.radioInterface[index] );
                    fprintf( fp, "Roaming Indicator : 0x%x\n", ServingSystemInfo.RoamingIndicatorList.roamIndicator[index] );
                }

                fprintf( fp,"\n" );
                fprintf( fp, "Default Roaming Indicator: 0x%x\n",  ServingSystemInfo.defaultRoamInd );

                fprintf( fp, "\n 3GPP2 Time Zone \n" );
                fprintf( fp, "Leap Seconds: 0x%x\n",               ServingSystemInfo.Gpp2TimeZone.leapSeconds );
                fprintf( fp, "Local Time Offset: 0x%x\n",          ServingSystemInfo.Gpp2TimeZone.localTimeOffset );
                fprintf( fp, "Daylight Saving indicator: 0x%x\n",  ServingSystemInfo.Gpp2TimeZone.daylightSavings );

                fprintf( fp,"\n" );
                fprintf( fp, "CDMA P_Rev in use : 0x%x\n",         ServingSystemInfo.CDMA_P_Rev );

                fprintf( fp, "3GPP Time Zone : 0x%x\n",            ServingSystemInfo.GppTimeZone );

                fprintf( fp, "3GPP NW Daylight Saving Adjustment : 0x%x\n", ServingSystemInfo.GppNetworkDSTAdjustment );

                fprintf( fp, "3GPP Location Area Code : 0x%x\n",    ServingSystemInfo.Lac );

                fprintf( fp, "Cell ID : 0x%x\n",                   ServingSystemInfo.CellID );

                fprintf( fp, "Concurrent Service Info : 0x%x\n",    ServingSystemInfo.concSvcInfo );

                fprintf( fp, "PRL Indicator : 0x%x\n",              ServingSystemInfo.PRLInd );

                fprintf( fp, "Dual Transfer Mode Ind : 0x%x\n",    ServingSystemInfo.DTMInd );

                fprintf( fp, "\n Detailed Service Information \n" );
                fprintf( fp, "Service Status: 0x%x\n",              ServingSystemInfo.DetailedSvcInfo.srvStatus );
                fprintf( fp, "Service Capability: 0x%x\n",          ServingSystemInfo.DetailedSvcInfo.srvCapability );
                fprintf( fp, "HDR service status: 0x%x\n",         ServingSystemInfo.DetailedSvcInfo.hdrSrvStatus );
                fprintf( fp, "HDR hybrid information: 0x%x\n",     ServingSystemInfo.DetailedSvcInfo.hdrHybrid );
                fprintf( fp, "Forbidden system information: 0x%x\n", ServingSystemInfo.DetailedSvcInfo.isSysForbidden );

                fprintf( fp, "\n CDMA System Info Ext \n" );
                fprintf( fp, "MCC: 0x%x\n",                         ServingSystemInfo.CDMASystemInfoExt.MCC );
                fprintf( fp, "IMSI_11_12: 0x%x\n",                  ServingSystemInfo.CDMASystemInfoExt.imsi_11_12 );

                fprintf( fp,"\n" );
                fprintf( fp, "HDR Personality: 0x%x\n",             ServingSystemInfo.hdrPersonality );
                fprintf( fp, "TAC info for LTE: 0x%x\n",            ServingSystemInfo.trackAreaCode );

                fprintf( fp, "\n Call Barring Status \n" );
                fprintf( fp, "CS Bar Status: 0x%x\n",               ServingSystemInfo.CallBarStatus.csBarStatus );
                fprintf( fp, "PS Bar Status: 0x%x\n",               ServingSystemInfo.CallBarStatus.psBarStatus );
            }
            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doSLQSSetBandPreference
 *
 * Purpose:  Perform the tests that call the SLQSSetBandPreference() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSLQSSetBandPreference( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    FILE  *fp      = NULL;

    fp = tfopen("../../nas/test/results/slqssetbandpreference.txt", "w");

    if (fp)
    {
        while ( tCaseNum < MAX_SLQS_SET_BAND_PREFERENCE_TESTCASE_NUM )
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_SLQS_SET_BAND_PREFERENCE_TESTCASE_NUM,
                       SLQSSetBandPreferenceTestCases[tCaseNum].desc );

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "band preference :%llx\n",
                    SLQSSetBandPreferenceTestCases[tCaseNum].bandpreference);

            nRet = SLQSSetBandPreference(
                    SLQSSetBandPreferenceTestCases[tCaseNum].bandpreference );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "System Band Preference Set\n");
            }

            tCaseNum++;
        }
    }

    if (fp)
    {
        tfclose(fp);
    }
}

/*
 * Name:     SLQSNasIndicationRegister
 *
 * Purpose:  Perform the tests that call the SLQSNasIndicationRegister() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSLQSNasIndicationRegister( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    FILE  *fp      = NULL;

    fp = tfopen("TestReport/slqsNasIndicationRegister.txt", "a");

    if (fp)
    {
        while ( tCaseNum < MAX_SLQS_INDICATION_REGISTER_TESTCASE_NUM )
        {
            const SLQSIndicationRegisterTestCase_t *pParams =
                                &(SLQSIndicationRegisterTestCases[tCaseNum]);

            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_SLQS_INDICATION_REGISTER_TESTCASE_NUM,
                       SLQSIndicationRegisterTestCases[tCaseNum].desc );

            nRet = SLQSNasIndicationRegister( pParams->systemSelectionInd,
                                              pParams->DDTMInd,
                                              pParams->servingSystemInd );

            fprintf(fp, "Return Code: %u\n", nRet);
            doprintreason( fp, nRet );

            if (!nRet)
            {
                fprintf(fp, "NAS Indication Register Success\n");
            }
            tCaseNum++;
        }
    }
    if (fp)
    {
        tfclose(fp);
    }
}

/*
 * Name:     doSLQSGetSignalStrength
 *
 * Purpose:  Perform the tests that call the SLQSGetSignalStrength() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSLQSGetSignalStrength( void )
{
    BYTE   tCaseNum = 0;
    ULONG  nRet;
    USHORT idx = 0, listLen = 0;
    FILE   *fp = NULL;
    struct slqsSignalStrengthInfo *pSigStrenInfo;

    fp = tfopen( "../../nas/test/results/slqsgetsignalstrength.txt", "w" );

    if (fp)
    {
        while ( tCaseNum < MAX_SLQS_GET_SIGNAL_STRENGTH_TESTCASE_NUM )
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                        ( tCaseNum + 1 ),
                        MAX_SLQS_GET_SIGNAL_STRENGTH_TESTCASE_NUM,
                        SLQSGetSignalStrengthTestCases[tCaseNum].desc );

            pSigStrenInfo = SLQSGetSignalStrengthTestCases[tCaseNum].pSignalInfo;

            /* Mask all the bits of "Request Mask" to retrieve complete info */
            if ( NULL != pSigStrenInfo)
            {
                signalTestInfo.signalStrengthReqMask = 0x1F;
                fprintf(fp, "Parameter Set:\n" );
                fprintf(fp, "ReqMask : 0x%x\n", signalTestInfo.signalStrengthReqMask );
                nRet = SLQSGetSignalStrength( SLQSGetSignalStrengthTestCases
                                              [tCaseNum].pSignalInfo );

                /* Display result code and text */
                doprintreason ( fp, nRet );

                if (!nRet)
                {
                    fprintf( fp, "\nSignal Strength Information:\n" );

                    /* Display the current signal strength and radio inteface */
                    fprintf( fp, "Current Signal Strength : %d\n",
                            signalTestInfo.rxSignalStrengthList[0].rxSignalStrength );
                    fprintf( fp, "Current Radio Interface : %d\n\n",
                            signalTestInfo.rxSignalStrengthList[0].radioIf );

                    /* Display Receive Signal Strength list */
                    fprintf( fp, "Receive Signal Strength List:\n" );
                    listLen = signalTestInfo.rxSignalStrengthListLen;
                    fprintf( fp, "List's Length : %d\n", listLen );
                    if( 0 != listLen )
                    {             
                        for( idx = 0; idx < listLen; idx++ )
                        {
                            fprintf( fp, "Receive Signal Strength[%d]: %d\n",
                                          idx,
                                          pSigStrenInfo->rxSignalStrengthList[idx].
                                          rxSignalStrength );
                            fprintf( fp, "Radio Interface[%d]        : %d\n",
                                          idx,
                                          pSigStrenInfo->rxSignalStrengthList[idx].
                                          radioIf );
                        }
                    }
                    fprintf( fp, "\n" );
                    idx = 0;
                    /* Display ECIO list */
                    fprintf( fp, "ECIO List:\n" );
                    listLen = signalTestInfo.ecioListLen;
                    fprintf( fp, "List's Length : %d\n", listLen );
                    if( 0 != listLen )
                    {
                        for( idx = 0; idx < listLen; idx++ )
                        {
                            fprintf( fp, "ECIO[%d]            : %d\n",
                                          idx,
                                          signalTestInfo.ecioList[idx].ecio );
                            fprintf( fp, "Radio Interface[%d] : %d\n",
                                          idx,
                                          signalTestInfo.ecioList[idx].radioIf );
                        }
                    }
                    fprintf( fp, "\n" );
                    idx = 0;

                    fprintf( fp, "IO   : %d\n", signalTestInfo.Io );
                    fprintf( fp, "SINR : %d\n\n", signalTestInfo.sinr );

                    /* Display Error Rate list */
                    fprintf( fp, "Error Rate List:\n" );
                    listLen = signalTestInfo.errorRateListLen;
                    fprintf( fp, "List's Length : %d\n", listLen );
                    if( 0 != listLen )
                    {
                        for( idx = 0; idx < listLen; idx++ )
                        {
                            fprintf( fp, "Error Rate[%d]      : %d\n",
                                          idx,
                                          signalTestInfo.errorRateList[idx].errorRate );
                            fprintf( fp, "Radio Interface[%d] : %d\n",
                                          idx,
                                          signalTestInfo.errorRateList[idx].radioIf );
                        }
                    }
                        fprintf( fp, "\n" );

                        /* Display RSRQ Information */
                        fprintf( fp, "RSRQ Information\n" );
                        fprintf( fp, "RSRQ            : %d\n",
                                      pSigStrenInfo->rsrqInfo.rsrq );
                        fprintf( fp, "Radio Interface : %d\n",
                                      pSigStrenInfo->rsrqInfo.radioIf );
                        /* Display LTE SNR information */
                        fprintf( fp, "LTE SNR  : %d\n", pSigStrenInfo->ltesnr);
                        /* Display LTE RSRP information*/
                        fprintf( fp, "LTE RSRP : %d\n\n", pSigStrenInfo->ltersrp);
                }
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doSLQSPerformNetworkScan
 *
 * Purpose:  Perform the tests that call the SLQSPerformNetworkScan() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSLQSPerformNetworkScan( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    BYTE  count;
    FILE  *fp = NULL;
    slqsNetworkScanInfo lnetworkInfo;
    ULONG scanResult;

    fp = tfopen("../../nas/test/results/slqsperformnetworkscan.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_SLQS_PERFORM_NETWORK_SCAN_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       ( tCaseNum + 1 ),
                       MAX_SLQS_PERFORM_NETWORK_SCAN_TESTCASE_NUM,
                       SLQSPerformNetworkScanTestCases[tCaseNum].desc);

            /* Initialize the structure to be sent to API */
            /* Initialize the Network Instances */
            lnetworkInfo.pNetworkInfoInstances =
               SLQSPerformNetworkScanTestCases[tCaseNum].pNetworkInfoInstances;

            /* Initialize the Network Info Structure */
            lnetworkInfo.pNetworkInfo =
               SLQSPerformNetworkScanTestCases[tCaseNum].pNetworkInfo;

            /* Initialize the RAT Instances */
            lnetworkInfo.pRATInstances =
               SLQSPerformNetworkScanTestCases[tCaseNum].pRATInstances;

            /* Initialize the RAT Info Structure */
            lnetworkInfo.pRATInfo =
               SLQSPerformNetworkScanTestCases[tCaseNum].pRATInfo;

            lnetworkInfo.pPCSDigitInstances =
               SLQSPerformNetworkScanTestCases[tCaseNum].pPCSDigitInstances;
            lnetworkInfo.pPCSDigitInfo =
               SLQSPerformNetworkScanTestCases[tCaseNum].pPCSDigitInfo;
            lnetworkInfo.pScanResult = &scanResult;

            fprintf(fp, "Parameter Set:\n");

            nRet = SLQSPerformNetworkScan( &lnetworkInfo );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet && lnetworkInfo.pNetworkInfoInstances
                      && lnetworkInfo.pNetworkInfo )
            {
                fprintf(fp, "SLQSPerformNetworkScan() Successful \n" );
                fprintf(fp, "NetworkInfoInstances from network : %d\n",
                        *( lnetworkInfo.pNetworkInfoInstances ));
                for(count = 0; count < *( lnetworkInfo.pNetworkInfoInstances ); count++)
                {
                    fprintf(fp, "MCC[%d]        : %d\n",
                          count, lnetworkInfo.pNetworkInfo[count].MCC);
                    fprintf(fp, "MNC[%d]        : %d\n",
                          count, lnetworkInfo.pNetworkInfo[count].MNC);
                    fprintf(fp, "InUse[%d]      : %u\n",
                          count, lnetworkInfo.pNetworkInfo[count].InUse);
                    fprintf(fp, "Roaming[%d]    : %u\n",
                          count, lnetworkInfo.pNetworkInfo[count].Roaming);
                    fprintf(fp, "Forbidden[%d]  : %u\n",
                          count, lnetworkInfo.pNetworkInfo[count].Forbidden);
                    fprintf(fp, "Preferred[%d]  : %u\n",
                          count, lnetworkInfo.pNetworkInfo[count].Preferred);
                    fprintf(fp, "Desription[%d] : %s\n",
                          count, lnetworkInfo.pNetworkInfo[count].Description);
                }
            }
            if (!nRet && lnetworkInfo.pRATInstances
                      && lnetworkInfo.pRATInfo )
            {
                fprintf(fp, "RATInstances from network : %d\n",
                                            *( lnetworkInfo.pRATInstances ));
                for (count = 0; count < *(lnetworkInfo.pRATInstances); count++)
                {
                    fprintf(fp, "RAT-MCC[%d]         : %d\n",
                             count, lnetworkInfo.pRATInfo[count].MCC);
                    fprintf(fp, "RAT-MNC[%d]         : %d\n",
                             count, lnetworkInfo.pRATInfo[count].MNC);
                    fprintf(fp, "RAT-Technology[%d]  : %x\n",
                             count, lnetworkInfo.pRATInfo[count].RAT);
                }
            }

            if (!nRet && lnetworkInfo.pPCSDigitInstances
                      && lnetworkInfo.pPCSDigitInfo )
            {
                fprintf(fp, "PCSInfoInstances from network : %d\n",
                                            *( lnetworkInfo.pPCSDigitInstances ));
                for (count = 0; count < *(lnetworkInfo.pPCSDigitInstances); count++)
                {
                    fprintf(fp, "MCC[%d]         : %d\n",
                             count, lnetworkInfo.pPCSDigitInfo[count].MCC);
                    fprintf(fp, "MNC[%d]         : %d\n",
                             count, lnetworkInfo.pPCSDigitInfo[count].MNC);
                    fprintf(fp, "PCS-Digit-included[%d]  : %x\n",
                             count, lnetworkInfo.pPCSDigitInfo[count].includes_pcs_digit);
                }
            }

            if (!nRet && lnetworkInfo.pScanResult)
            {
                fprintf(fp, "network scan result  : %d\n", *lnetworkInfo.pScanResult);
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doSLQSSetSysSelectionPref
 *
 * Purpose:  Perform the tests that call the SLQSSetSysSelectionPref() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSLQSSetSysSelectionPref( void )
{
    ULONG               nRet;
    FILE                *fp = NULL;
    sysSelectPrefParams SysSelectPrefParams;
    CHAR                selOption[OPTION_LEN];
    unsigned int        len    = 0;
    CHAR                *pEndOfLine = NULL;
    BYTE                bRadioAccessTechnologyGSM       = eNAS_RADIO_IF_GSM;
    BYTE                bRadioAccessTechnologyUMTS      = eNAS_RADIO_IF_UMTS;
    BYTE                bRadioAccessTechnologyLTE       = eNAS_RADIO_IF_LTE;
    BYTE                bRadioAccessTechnologyTDSCDMA   = eNAS_RADIO_IF_TDSCDMA;
    /* Define the test cases to be executed */
    SLQSSetSysSelectionPrefTestCase_t SLQSSetSysSelPrefTestCases[] =
    {
        { eQCWWAN_ERR_INVALID_ARG, 1, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, "Invalid Test Case - All params NULL"},

        { eQCWWAN_ERR_NONE, 2, &EmerMode, NULL, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, "Valid test Case - Emergency Mode set"},

        { eQCWWAN_ERR_NONE, 3, NULL, &ModePref, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, "Valid test Case - Mode Preference set"},

        { eQCWWAN_ERR_NONE, 4, NULL, NULL, &bandPref, NULL, NULL, NULL, NULL,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, "Valid test Case - Band Preference set"},

        { eQCWWAN_ERR_NONE, 5, NULL, NULL, NULL, &PRLPref, NULL, NULL, NULL,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, "Valid test Case - CDMA PRL Preference set"},

        { eQCWWAN_ERR_NONE, 6, NULL, NULL, NULL, NULL, &RoamPref, NULL, NULL,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, "Valid Test Case - Roaming Preference set"},

        { eQCWWAN_ERR_NONE, 7, NULL, NULL, NULL, NULL, NULL, &LTEBandPref, NULL,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, "Valid Test Case - LTE Band Preference set"},

        { eQCWWAN_ERR_NONE, 8, NULL, NULL, NULL, NULL, NULL, NULL, &NetSelPref,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, "Valid Test Case - Network Selection Preference set"},

        { eQCWWAN_ERR_NONE, 9, &EmerMode, NULL, NULL, NULL, NULL, NULL, NULL,
          &chgDuration, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, "Valid Test Case - Change duration set and 1 parameter set"},

        { eQCWWAN_ERR_NONE, 10, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
          &chgDuration, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, "Valid Test Case - Only Change duration set"},

        { eQCWWAN_ERR_NONE, 11, NULL, NULL, NULL, NULL, NULL, NULL, &NetSelPref,
          NULL, &MNCIncPCS, NULL, NULL, NULL, NULL, NULL, NULL, NULL, "Valid Test Case - MNC Includes PCS Digit Status and Net sel Pref set"},

        { eQCWWAN_ERR_NONE, 12, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
          NULL, &MNCIncPCS, NULL, NULL, NULL, NULL, NULL, NULL, NULL, "Valid Test Case - Only MNC Includes PCS Digit Status set"},

        { eQCWWAN_ERR_NONE, 13, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, &srvDomPref, NULL, NULL, NULL, NULL, NULL, NULL, "Valid Test Case - Service Domain Preference set"},

        { eQCWWAN_ERR_NONE, 14, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, NULL, &GWAcqOrder, NULL, NULL, NULL, NULL, NULL, "Valid Test Case - GSM/WCDMA Acquisition Order Pref. set"},

        { eQCWWAN_ERR_NONE, 15, &EmerMode, &ModePref, &bandPref, &PRLPref, &RoamPref, &LTEBandPref,
          &NetSelPref, &chgDuration, &MNCIncPCS, &srvDomPref, &GWAcqOrder, &TdscdmaBandPref, &AcqOrderPref,
          &SrvRegRestriction, &CSGIDobj, NULL,
          "Valid Test Case - All parameters set"},

        { eQCWWAN_ERR_NONE, 16, &EmerMode, &ModePref, NULL, &PRLPref, &RoamPref, &LTEBandPref,
          &NetSelPref, &chgDuration, &MNCIncPCS, &srvDomPref, &GWAcqOrder, &TdscdmaBandPref, &AcqOrderPref,
          &SrvRegRestriction, &CSGIDobj, NULL,
          "Valid Test Case - All parameters except bandPref set"},

        { eQCWWAN_ERR_NONE, 17, &EmerMode, &ModePref, &bandPref, &PRLPref, &RoamPref, NULL,
          &NetSelPref, &chgDuration, &MNCIncPCS, &srvDomPref, &GWAcqOrder,&TdscdmaBandPref, &AcqOrderPref,
          &SrvRegRestriction, &CSGIDobj, NULL,
          "Valid Test Case - All parameters except LTE Band Pref set"},
          
        { eQCWWAN_ERR_NONE, 18, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, &bRadioAccessTechnologyGSM, "Valid Test Case - Radio Access Technology GSM set"},
          
        { eQCWWAN_ERR_NONE, 19, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, &bRadioAccessTechnologyUMTS, "Valid Test Case - Radio Access Technology UMTS set"},

        { eQCWWAN_ERR_NONE, 20, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, &bRadioAccessTechnologyLTE, "Valid Test Case - Radio Access Technology LTE set"},

        { eQCWWAN_ERR_NONE, 21, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, &bRadioAccessTechnologyTDSCDMA, "Valid Test Case - Radio Access Technology TDSCDMA set"},

        { eQCWWAN_ERR_NONE, 22, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, &bRadioAccessTechnologyGSM, "Valid Test Case - Radio Access Technology GSM set"},

    };
    
    fprintf( stderr,"\nPlease review all parameters before you run this test case... \n"
                    "Wrong band, mode preference and other information set to the module may cause the "
                    "module not work properly.\n");
    fprintf( stderr,"\nDo you still want to run this test case?:(y/n) " );

    while(1)
    {
        /* Receive the input from the user */
        fgets( selOption, (OPTION_LEN), stdin );

        /* If '/n' character is not read, there are more characters in input
         * stream. Clear the input stream.
         */
        pEndOfLine = strchr( selOption, ENTER_KEY );
        if( NULL == pEndOfLine )
        {
            FlushStdinStream();
        }

        len = strlen( selOption );

        if((selOption[0] == 'Y' || selOption[0] == 'y' || selOption[0] == 'N' || selOption[0] == 'n') && (len < 3 ))
        {
            break;
        }
        else
        {
            fprintf( stderr,"\nPlease Enter the correct option (y/n) " );
            continue;
         }
    }
    switch( selOption[0] ){
    case 'Y':
    case 'y':
    fp = tfopen( "../../nas/test/results/slqssetsysselectionpref.txt", "w" );
    if ( fp )
    {
        BYTE tCaseNum     = 0;
        BYTE ncases = sizeof( SLQSSetSysSelPrefTestCases ) /
                      sizeof( SLQSSetSysSelPrefTestCases[0] );

        while( tCaseNum < ncases )
        {
            SLQSSetSysSelectionPrefTestCase_t *pd =
                &SLQSSetSysSelPrefTestCases[tCaseNum];

            /* print test data */
            fprintf( fp,"\n\nTest Case %d / %d : %s\n",
                     (tCaseNum + 1),
                     ncases,
                     pd->desc );
            fprintf( fp,"\nParameters set for System Selection Preference:\n" );
            IFPRINTF( fp, "Emergency Mode: 0x%x\n", pd->pEmerMode );
            IFPRINTF( fp, "Mode Preference: 0x%x\n", pd->pModePref );
            IFPRINTF( fp, "Band Preference: 0x%llx\n", pd->pBandPref );
            IFPRINTF( fp, "CDMA PRL Preference : 0x%x\n", pd->pPRLPref );
            IFPRINTF( fp, "Roaming Preference : 0x%x\n", pd->pRoamPref );
            IFPRINTF( fp, "LTE Band Preference : 0x%llx\n", pd->pLTEBandPref );
            if( pd->pNetSelPref)
            {
                fprintf( fp, "\nNetwork Selection Preference :\n");

                fprintf( fp, "Network Registration : 0x%x\n",
                          pd->pNetSelPref->netReg);
                fprintf( fp, "MCC : 0x%d\n", pd->pNetSelPref->mcc );
                fprintf( fp, "MNC : 0x%d\n", pd->pNetSelPref->mnc );
            }
            IFPRINTF( fp, "\nChange Duration : 0x%x\n", pd->pChgDuration );
            IFPRINTF( fp, "MNC Includes PCS Digit : 0x%x\n", pd->pMNCIncPCSDigStat );
            IFPRINTF( fp, "Service Domain Preference : 0x%x\n", pd->pSrvDomainPref );
            IFPRINTF( fp, "GSM - WCDMA Acquisition Order Preference : 0x%x\n",
                                                              pd->pGWAcqOrderPref );
            IFPRINTF( fp, "\nTDSCDMA band Preference : 0x%llx\n", pd->pTdscdmaBandPref);
            if( pd->pAcqOrderPref )
            {
                fprintf( fp, "\nAcquisition Order Preference :\n");
                BYTE i = 0;

                fprintf( fp, "acq order len : 0x%x\n", pd->pAcqOrderPref->acqOrdeLen);

                for ( i = 0; i < pd->pAcqOrderPref->acqOrdeLen; i++ )
                {
                    fprintf( fp, "acq order: 0x%d\n", pd->pAcqOrderPref->pAcqOrder[i] );
                }
            }
            IFPRINTF( fp, "\nsrv reg restriction : 0x%x\n", pd->pSrvRegRestriction );

            if( pd->pCSGID )
            {
                fprintf( fp, "\nCGSID:\n");
                fprintf( fp, "Network Registration : 0x%d\n", pd->pCSGID->mcc);
                fprintf( fp, "MCC : 0x%d\n", pd->pCSGID->mcc );
                fprintf( fp, "MNC : 0x%d\n", pd->pCSGID->mnc );
                fprintf( fp, "MNC : 0x%d\n", pd->pCSGID->mncPcsDigits );
                fprintf( fp, "MNC : 0x%d\n", pd->pCSGID->id );
                fprintf( fp, "MNC : 0x%d\n", pd->pCSGID->rat );
            }

            SysSelectPrefParams.pEmerMode    = pd->pEmerMode;
            SysSelectPrefParams.pModePref    = pd->pModePref;
            SysSelectPrefParams.pBandPref    = pd->pBandPref;
            SysSelectPrefParams.pPRLPref     = pd->pPRLPref;
            SysSelectPrefParams.pRoamPref    = pd->pRoamPref;
            SysSelectPrefParams.pLTEBandPref = pd->pLTEBandPref;
            SysSelectPrefParams.pNetSelPref  = pd->pNetSelPref;
            if( SysSelectPrefParams.pNetSelPref )
            {
                SysSelectPrefParams.pNetSelPref->netReg =
                                               pd->pNetSelPref->netReg;
                SysSelectPrefParams.pNetSelPref->mcc =
                                               pd->pNetSelPref->mcc;
                SysSelectPrefParams.pNetSelPref->mnc =
                                               pd->pNetSelPref->mnc;
            }
            SysSelectPrefParams.pChgDuration      = pd->pChgDuration;
            SysSelectPrefParams.pMNCIncPCSDigStat = pd->pMNCIncPCSDigStat;
            SysSelectPrefParams.pSrvDomainPref    = pd->pSrvDomainPref;
            SysSelectPrefParams.pGWAcqOrderPref   = pd->pGWAcqOrderPref;
            SysSelectPrefParams.pTdscdmaBandPref  = pd->pTdscdmaBandPref;
            SysSelectPrefParams.pAcqOrderPref     = pd->pAcqOrderPref;
            if( SysSelectPrefParams.pAcqOrderPref )
            {
                struct acqOrderPref *pAcqOrderPref =
                                (struct acqOrderPref *)SysSelectPrefParams.pAcqOrderPref;

                pAcqOrderPref->acqOrdeLen = pd->pAcqOrderPref->acqOrdeLen;
                pAcqOrderPref->pAcqOrder  = pd->pAcqOrderPref->pAcqOrder;
            }

            SysSelectPrefParams.pSrvRegRestriction = pd->pSrvRegRestriction;
            SysSelectPrefParams.pCSGID = pd->pCSGID;

            if( SysSelectPrefParams.pCSGID )
            {
                struct CSGID *pCSGID = (struct CSGID *)SysSelectPrefParams.pCSGID;
                pCSGID->mcc = pd->pCSGID->mcc;
                pCSGID->mnc = pd->pCSGID->mnc;
                pCSGID->mncPcsDigits = pd->pCSGID->mncPcsDigits;
                pCSGID->id = pd->pCSGID->id;
                pCSGID->rat = pd->pCSGID->rat;
            }

            SysSelectPrefParams.pRAT= pd->pRAT;
            if(pd->pRAT)
            {
                fprintf( fp, "Radio Access Technology : " );
                switch(*pd->pRAT)
                {
                    case eNAS_RADIO_IF_GSM:
                        fprintf( fp, "GSM" );
                        break;
                    case eNAS_RADIO_IF_UMTS:
                        fprintf( fp, "UMTS" );
                        break;
                    case eNAS_RADIO_IF_LTE:
                        fprintf( fp, "LTE");
                        break;
                    case eNAS_RADIO_IF_TDSCDMA:
                        fprintf( fp, "TDSCDMA" );
                        break;
                    default:
                        fprintf( fp, "Unknown\n" );
                        break;
                }
                fprintf( fp, "\n" );
            }
            
            nRet = SLQSSetSysSelectionPref( &SysSelectPrefParams );

            /* Display result code and text */
            doprintreason( fp, nRet );
            tCaseNum++;
        }
    }
    break;
    }
    if (fp)
        tfclose(fp);
}

/*
 * Name:     doSLQSGetSysSelectionPref
 *
 * Purpose:  Perform the tests that call the SLQSGetSysSelectionPref() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSLQSGetSysSelectionPref( void )
{
    ULONG             nRet;
    FILE              *fp = NULL;
    sysSelectPrefInfo SysSelectPrefInfo;

    /* Define the test cases to be executed */
    SLQSGetSysSelectionPrefTestCase_t SLQSGetSysSelPrefTestCases[] =
    {
        { eQCWWAN_ERR_INVALID_ARG, 1, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, "Invalid Test Case - All params NULL"},

        { eQCWWAN_ERR_NONE, 2, &GetEmerMode, NULL, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, "Valid test Case - Emergency Mode set"},

        { eQCWWAN_ERR_NONE, 3, NULL, &GetModePref, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, "Valid test Case - Mode Preference set"},

        { eQCWWAN_ERR_NONE, 4, NULL, NULL, &GetBandPref, NULL, NULL, NULL, NULL,
          NULL, NULL, "Valid test Case - Band Preference set"},

        { eQCWWAN_ERR_NONE, 5, NULL, NULL, NULL, &GetPRLPref, NULL, NULL, NULL,
          NULL, NULL, "Valid test Case - CDMA PRL Preference set"},

        { eQCWWAN_ERR_NONE, 6, NULL, NULL, NULL, NULL, &GetRoamPref, NULL, NULL,
          NULL, NULL, "Valid Test Case - Roaming Preference set"},

        { eQCWWAN_ERR_NONE, 7, NULL, NULL, NULL, NULL, NULL, &GetLTEBandPref, NULL,
          NULL, NULL, "Valid Test Case - LTE Band Preference set"},

        { eQCWWAN_ERR_NONE, 8, NULL, NULL, NULL, NULL, NULL, NULL, &GetNetSelPref,
          NULL, NULL, "Valid Test Case - Network Selection Preference set"},

        { eQCWWAN_ERR_NONE, 9, NULL, NULL, NULL, NULL, NULL, NULL, NULL, &GetSrvDomPref,
          NULL, "Valid Test Case - Service Domain Preference set"},

        { eQCWWAN_ERR_NONE, 10, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, &GetGWAcqOrder,
          "Valid Test Case - GSM/WCDMA Acquisition Order Pref. set"},

        { eQCWWAN_ERR_NONE, 11, &GetEmerMode, &GetModePref, &GetBandPref, &GetPRLPref,
          &GetRoamPref, &GetLTEBandPref, &GetNetSelPref, &GetSrvDomPref, &GetGWAcqOrder,
          "Valid Test Case - All parameters set"},

    };

    fp = tfopen( "../../nas/test/results/slqsgetsysselectionpref.txt", "w" );
    if ( fp )
    {
        BYTE tCaseNum     = 0;
        BYTE ncases = sizeof( SLQSGetSysSelPrefTestCases ) /
                      sizeof( SLQSGetSysSelPrefTestCases[0] );

        while( tCaseNum < ncases )
        {
            SLQSGetSysSelectionPrefTestCase_t *pd =
                &SLQSGetSysSelPrefTestCases[tCaseNum];

            /* print test data */
            fprintf( fp,"\n\nTest Case %d / %d : %s\n",
                     (tCaseNum + 1),
                     ncases,
                     pd->desc );

            SysSelectPrefInfo.pEmerMode       = pd->pEmerMode;
            SysSelectPrefInfo.pModePref       = pd->pModePref;
            SysSelectPrefInfo.pBandPref       = pd->pBandPref;
            SysSelectPrefInfo.pPRLPref        = pd->pPRLPref;
            SysSelectPrefInfo.pRoamPref       = pd->pRoamPref;
            SysSelectPrefInfo.pLTEBandPref    = pd->pLTEBandPref;
            SysSelectPrefInfo.pNetSelPref     = pd->pNetSelPref;
            SysSelectPrefInfo.pSrvDomainPref  = pd->pSrvDomainPref;
            SysSelectPrefInfo.pGWAcqOrderPref = pd->pGWAcqOrderPref;

            nRet = SLQSGetSysSelectionPref( &SysSelectPrefInfo );

            /* Display result code and text */
            doprintreason( fp, nRet );
            if (!nRet)
            {
                fprintf( fp,"System Selection Preferences \n" );
                IFPRINTF( fp, "Emergency Mode: 0x%x\n",               pd->pEmerMode );
                IFPRINTF( fp, "Mode Preference: 0x%x\n",              pd->pModePref );
                IFPRINTF( fp, "Band Preference: 0x%llx\n",            pd->pBandPref );
                IFPRINTF( fp, "CDMA PRL Preference : 0x%x\n",         pd->pPRLPref );
                IFPRINTF( fp, "Roaming Preference : 0x%x\n",          pd->pRoamPref );
                IFPRINTF( fp, "LTE Band Preference : 0x%llx\n",       pd->pLTEBandPref );
                IFPRINTF( fp, "Network Selection Preference: 0x%x\n", pd->pNetSelPref );
                IFPRINTF( fp, "Service Domain Preference : 0x%x\n",  pd->pSrvDomainPref );
                IFPRINTF( fp, "GSM - WCDMA Acquisition Order Preference : 0x%x\n",
                                                                      pd->pGWAcqOrderPref );
            }
            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doSLQSNasGetSysInfo
 *
 * Purpose:  Perform the tests that call the SLQSNasGetSysInfo() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSLQSNasGetSysInfo( void )
{
    local SrvStatusInfo      cdmassi;
    local SrvStatusInfo      hdrssi;
    local GSMSrvStatusInfo   gsmssi;
    local GSMSrvStatusInfo   wcdmassi;
    local GSMSrvStatusInfo   ltessi;
    local CDMASysInfo        cdmasi;
    local HDRSysInfo         hdrsi;
    local GSMSysInfo         gsmsi;
    local WCDMASysInfo       wcdmasi;
    local LTESysInfo         ltesi;
    local AddCDMASysInfo     addcdmasi;
    local WORD               addhdrsi;
    local AddSysInfo         addgsmsi;
    local AddSysInfo         addwcdmasi;
    local WORD               addltesi;
    local CallBarringSysInfo gsmcbsi;
    local CallBarringSysInfo wcdmacbsi;
    local BYTE               ltevssi;
    local BYTE               gsmcdsi;
    local BYTE               wcdmacdsi;

    /* Define the test cases to be executed */
    SLQSNasGetSysInfoTestCase_t SLQSNasGetSysInfoTestCases[] =
    {
        { eQCWWAN_ERR_NONE, 1, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
        NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
        NULL, NULL, "Valid Test Case - All params NULL"},

        { eQCWWAN_ERR_NONE, 2, &cdmassi, &hdrssi, &gsmssi, &wcdmassi, &ltessi,
        &cdmasi, &hdrsi, &gsmsi, &wcdmasi, &ltesi, &addcdmasi, &addhdrsi,
        &addgsmsi, &addwcdmasi, &addltesi, &gsmcbsi, &wcdmacbsi, &ltevssi,
        &gsmcdsi, &wcdmacdsi, "Valid Test Case - All params Provided "},
    };

    nasGetSysInfoResp resp;
    WORD              lC;
    FILE *fp = tfopen( "../../nas/test/results/slqsnasgetsysinfo.txt", "w" );
    if ( fp )
    {
        BYTE tCaseNum = 0;
        BYTE ncases   = sizeof( SLQSNasGetSysInfoTestCases ) /
                        sizeof( SLQSNasGetSysInfoTestCases[0] );

        while( tCaseNum < ncases )
        {
            SLQSNasGetSysInfoTestCase_t *pd =
                &SLQSNasGetSysInfoTestCases[tCaseNum++];

            /* print test data */
            fprintf( fp,"\n\nTest Case %d / %d : %s\n",
                     tCaseNum,
                     ncases,
                     pd->desc );

            resp.pCDMASrvStatusInfo        = pd->pCDMASrvStatusInfo;
            resp.pHDRSrvStatusInfo         = pd->pHDRSrvStatusInfo;
            resp.pGSMSrvStatusInfo         = pd->pGSMSrvStatusInfo;
            resp.pWCDMASrvStatusInfo       = pd->pWCDMASrvStatusInfo;
            resp.pLTESrvStatusInfo         = pd->pLTESrvStatusInfo;
            resp.pCDMASysInfo              = pd->pCDMASysInfo;
            resp.pHDRSysInfo               = pd->pHDRSysInfo;
            resp.pGSMSysInfo               = pd->pGSMSysInfo;
            resp.pWCDMASysInfo             = pd->pWCDMASysInfo;
            resp.pLTESysInfo               = pd->pLTESysInfo;
            resp.pAddCDMASysInfo           = pd->pAddCDMASysInfo;
            resp.pAddHDRSysInfo            = pd->pAddHDRSysInfo;
            resp.pAddGSMSysInfo            = pd->pAddGSMSysInfo;
            resp.pAddWCDMASysInfo          = pd->pAddWCDMASysInfo;
            resp.pAddLTESysInfo            = pd->pAddLTESysInfo;
            resp.pGSMCallBarringSysInfo    = pd->pGSMCallBarringSysInfo;
            resp.pWCDMACallBarringSysInfo  = pd->pWCDMACallBarringSysInfo;
            resp.pLTEVoiceSupportSysInfo   = pd->pLTEVoiceSupportSysInfo;
            resp.pGSMCipherDomainSysInfo   = pd->pGSMCipherDomainSysInfo;
            resp.pWCDMACipherDomainSysInfo = pd->pWCDMACipherDomainSysInfo;

            ULONG nRet = SLQSNasGetSysInfo( &resp );

            /* Display result code and text */
            doprintreason( fp, nRet );
            if ( nRet != eQCWWAN_ERR_NONE )
            {
                fprintf( fp,"SLQS NAS Get Sys Information Unsuccessful \n" );
                continue;
            }

            fprintf( fp,"SLQS NAS Get Sys Information Successful \n" );
            if ( resp.pCDMASrvStatusInfo )
            {
                fprintf( fp,"Service Status(CDMA) : %x\n",resp.pCDMASrvStatusInfo->srvStatus );
                fprintf( fp,"Is Preferred(CDMA)   : %x\n",resp.pCDMASrvStatusInfo->isPrefDataPath );
            }
            if ( resp.pHDRSrvStatusInfo )
            {
                fprintf( fp,"Service Status(HDR) : %x\n",resp.pHDRSrvStatusInfo->srvStatus );
                fprintf( fp,"Is Preferred(HDR)   : %x\n",resp.pHDRSrvStatusInfo->isPrefDataPath );
            }
            if ( resp.pGSMSrvStatusInfo )
            {
                fprintf( fp,"Service Status(GSM)      : %x\n",resp.pGSMSrvStatusInfo->srvStatus );
                fprintf( fp,"True Service Status(GSM) : %x\n",resp.pGSMSrvStatusInfo->trueSrvStatus );
                fprintf( fp,"Is Preferred(GSM)        : %x\n",resp.pGSMSrvStatusInfo->isPrefDataPath );
            }
            if ( resp.pWCDMASrvStatusInfo )
            {
                fprintf( fp,"Service Status(WCDMA)     : %x\n",resp.pWCDMASrvStatusInfo->srvStatus );
                fprintf( fp,"True Service Status(WCDMA): %x\n",resp.pWCDMASrvStatusInfo->trueSrvStatus );
                fprintf( fp,"Is Preferred(WCDMA)       : %x\n",resp.pWCDMASrvStatusInfo->isPrefDataPath );
            }
            if ( resp.pLTESrvStatusInfo )
            {
                fprintf( fp,"Service Status(LTE)      : %x\n",resp.pLTESrvStatusInfo->srvStatus );
                fprintf( fp,"True Service Status(LTE) : %x\n",resp.pLTESrvStatusInfo->trueSrvStatus );
                fprintf( fp,"Is Preferred(LTE)        : %x\n",resp.pLTESrvStatusInfo->isPrefDataPath );
            }
            if ( resp.pCDMASysInfo )
            {
                fprintf( fp,"\nCDMA SYSTEM INFORMATION ----\n");
                CDMASysInfo *t = resp.pCDMASysInfo;
                doprintsysInfoCommon( fp, &t->sysInfoCDMA );
                fprintf( fp,"System PRL Valid    : %x\n",t->isSysPrlMatchValid );
                fprintf( fp,"System PRL          : %x\n",t->isSysPrlMatch );
                fprintf( fp,"P_Rev Valid         : %x\n",t->pRevInUseValid );
                fprintf( fp,"P_Rev In Use        : %x\n",t->pRevInUse );
                fprintf( fp,"BS P_Rev Valid      : %x\n",t->bsPRevValid );
                fprintf( fp,"P_Rev In Use        : %x\n",t->bsPRev );
                fprintf( fp,"CCS_supp Valid      : %x\n",t->ccsSupportedValid );
                fprintf( fp,"CCS_supp            : %x\n",t->ccsSupported );
                fprintf( fp,"System Id           : %x\n",t->systemID );
                fprintf( fp,"Network Id          : %x\n",t->networkID );
                fprintf( fp,"BS Info Valid       : %x\n",t->bsInfoValid );
                fprintf( fp,"Base ID             : %x\n",t->baseId );
                fprintf( fp,"Base Latitude       : %x\n",t->baseLat );
                fprintf( fp,"Base Longitude      : %x\n",t->baseLong );
                fprintf( fp,"Packet Zone Valid   : %x\n",t->packetZoneValid );
                fprintf( fp,"Packet Zone         : %x\n",t->packetZone );
                fprintf( fp,"Network ID Valid    : %x\n",t->networkIdValid );
                fprintf( fp,"MCC info            : " );
                for ( lC = 0 ; lC < PLMN_LENGTH ; lC++ )
                {
                    if(t->MCC[lC] != 0xFF)
                        fprintf( fp,"%c",t->MCC[lC] );
                }
                fprintf( fp,"\n" );
                fprintf( fp,"MNC info            : " );
                for ( lC = 0 ; lC < PLMN_LENGTH ; lC++ )
                {
                    if(t->MNC[lC] != 0xFF)
                        fprintf( fp,"%c",t->MNC[lC] );
                }
                fprintf( fp,"\n" );
            }

            if ( resp.pHDRSysInfo )
            {
                fprintf( fp,"\nHDR SYSTEM INFORMATION ----\n");
                HDRSysInfo *t = resp.pHDRSysInfo;
                doprintsysInfoCommon( fp, &t->sysInfoHDR );
                fprintf( fp,"System PRL Valid    : %x\n",t->isSysPrlMatchValid );
                fprintf( fp,"System PRL          : %x\n",t->isSysPrlMatch );
                fprintf( fp,"Personality Valid   : %x\n",t->hdrPersonalityValid );
                fprintf( fp,"Personality         : %x\n",t->hdrPersonality );
                fprintf( fp,"Active Prot Valid   : %x\n",t->hdrActiveProtValid );
                fprintf( fp,"Active Protocol     : %x\n",t->hdrActiveProt );
                fprintf( fp,"IS-856 Sys Valid    : %x\n",t->is856SysIdValid );
                fprintf( fp,"IS-856 system ID    : " );
                for ( lC = 0 ; lC < SLQS_SYSTEM_ID_SIZE ; lC++ )
                {
                    fprintf( fp,"%x ",t->is856SysId[lC] );
                }
                fprintf( fp,"\n" );
            }
            if ( resp.pGSMSysInfo )
            {
                fprintf( fp,"\nGSM SYSTEM INFORMATION ----\n");
                GSMSysInfo *t = resp.pGSMSysInfo;
                doprintsysInfoCommon( fp, &t->sysInfoGSM );
                fprintf( fp,"LAC Valid           : %x\n",t->lacValid );
                fprintf( fp,"LAC                 : %x\n",t->lac );
                fprintf( fp,"Cell ID Valid       : %x\n",t->cellIdValid );
                fprintf( fp,"Cell ID             : %x\n",t->cellId );
                fprintf( fp,"Reg Rej Info Valid  : %x\n",t->regRejectInfoValid );
                fprintf( fp,"Reject Srvc Domain  : %x\n",t->rejectSrvDomain );
                fprintf( fp,"Reject Cause        : %x\n",t->rejCause );
                fprintf( fp,"Network Id Valid    : %x\n",t->networkIdValid );
                fprintf( fp,"MCC info            : " );
                for ( lC = 0 ; lC < PLMN_LENGTH ; lC++ )
                {
                    fprintf( fp,"%c",t->MCC[lC] );
                }
                fprintf( fp,"\n" );
                fprintf( fp,"MNC info            : " );
                for ( lC = 0 ; lC < PLMN_LENGTH ; lC++ )
                {
                    if(t->MNC[lC]!=0xFF)
                        fprintf( fp,"%c",t->MNC[lC] );
                }
                fprintf( fp,"\n" );
                fprintf( fp,"EGPRS Support Valid : %x\n",t->egprsSuppValid );
                fprintf( fp,"EGPRS Support       : %x\n",t->egprsSupp );
                fprintf( fp,"DTM Support Valid   : %x\n",t->dtmSuppValid );
                fprintf( fp,"DTM Support         : %x\n",t->dtmSupp );
            }
            if ( resp.pWCDMASysInfo )
            {
                fprintf( fp,"\nWCDMA SYSTEM INFORMATION ----\n");
                WCDMASysInfo *t = resp.pWCDMASysInfo;
                doprintsysInfoCommon( fp, &t->sysInfoWCDMA );
                fprintf( fp,"LAC Valid           : %x\n",t->lacValid );
                fprintf( fp,"LAC                 : %x\n",t->lac );
                fprintf( fp,"Cell ID Valid       : %x\n",t->cellIdValid );
                fprintf( fp,"Cell ID             : %x\n",t->cellId );
                fprintf( fp,"Reg Rej Info Valid  : %x\n",t->regRejectInfoValid );
                fprintf( fp,"Reject Srvc Domain  : %x\n",t->rejectSrvDomain );
                fprintf( fp,"Reject Cause        : %x\n",t->rejCause );
                fprintf( fp,"Network Id Valid    : %x\n",t->networkIdValid );
                fprintf( fp,"MCC info            : " );
                for ( lC = 0 ; lC < PLMN_LENGTH ; lC++ )
                {
                    fprintf( fp,"%c",t->MCC[lC] );
                }
                fprintf( fp,"\n" );
                fprintf( fp,"MNC info            : " );
                for ( lC = 0 ; lC < PLMN_LENGTH ; lC++ )
                {
                    if(t->MNC[lC]!=0xFF)
                        fprintf( fp,"%c",t->MNC[lC] );
                }
                fprintf( fp,"\n" );
                fprintf( fp,"HS Call Status Valid: %x\n",t->hsCallStatusValid );
                fprintf( fp,"HS Call Status      : %x\n",t->hsCallStatus );
                fprintf( fp,"HS Ind Valid        : %x\n",t->hsIndValid );
                fprintf( fp,"HS Indication       : %x\n",t->hsInd );
                fprintf( fp,"PSC Valid           : %x\n",t->pscValid );
                fprintf( fp,"Primary Scrambling Code : %x\n",t->psc );
            }
            if ( resp.pLTESysInfo )
            {
                fprintf( fp,"\nLTE SYSTEM INFORMATION ----\n");
                LTESysInfo *t = resp.pLTESysInfo;
                doprintsysInfoCommon( fp, &t->sysInfoLTE );
                fprintf( fp,"LAC Valid           : %x\n",t->lacValid );
                fprintf( fp,"LAC                 : %x\n",t->lac );
                fprintf( fp,"Cell ID Valid       : %x\n",t->cellIdValid );
                fprintf( fp,"Cell ID             : %x\n",t->cellId );
                fprintf( fp,"Reg Rej Info Valid  : %x\n",t->regRejectInfoValid );
                fprintf( fp,"Reject Srvc Domain  : %x\n",t->rejectSrvDomain );
                fprintf( fp,"Reject Cause        : %x\n",t->rejCause );
                fprintf( fp,"Network Id Valid    : %x\n",t->networkIdValid );
                fprintf( fp,"MCC info            : " );
                for ( lC = 0 ; lC < PLMN_LENGTH ; lC++ )
                {
                    fprintf( fp,"%c",t->MCC[lC] );
                }
                fprintf( fp,"\n" );
                fprintf( fp,"MNC info            : " );
                for ( lC = 0 ; lC < PLMN_LENGTH ; lC++ )
                {
                    if(t->MNC[lC]!=0xFF)
                        fprintf( fp,"%c",t->MNC[lC] );
                }
                fprintf( fp,"\n" );
                fprintf( fp,"TAC Valid           : %x\n",t->tacValid );
                fprintf( fp,"Tracking Area Code  : %x\n",t->tac );
            }
            if ( resp.pAddCDMASysInfo )
            {
                fprintf( fp, "Geo Sys Idx(CDMA) : %x\n", resp.pAddCDMASysInfo->geoSysIdx );
                fprintf( fp, "Reg Period (CDMA) : %x\n", resp.pAddCDMASysInfo->regPrd );
            }
            IFPRINTF( fp, "Geo Sys Idx(HDR) : %x\n", resp.pAddHDRSysInfo );
            if ( resp.pAddGSMSysInfo )
            {
                fprintf( fp, "Geo Sys Idx(GSM) : %x\n", resp.pAddGSMSysInfo->geoSysIdx );
                fprintf( fp, "Cell Br Cap(GSM) : %x\n", resp.pAddGSMSysInfo->cellBroadcastCap );
            }
            if ( resp.pAddWCDMASysInfo )
            {
                fprintf( fp, "Geo Sys Idx(WCDMA) : %x\n", resp.pAddWCDMASysInfo->geoSysIdx );
                fprintf( fp, "Cell Br Cap(WCDMA) : %x\n", resp.pAddWCDMASysInfo->cellBroadcastCap );
            }
            IFPRINTF( fp, "Geo Sys Idx(LTE) : %x\n", resp.pAddLTESysInfo );
            if ( resp.pGSMCallBarringSysInfo )
            {
                fprintf( fp, "CS Bar Status(GSM): %x\n", resp.pGSMCallBarringSysInfo->csBarStatus );
                fprintf( fp, "PS Bar Status(GSM): %x\n", resp.pGSMCallBarringSysInfo->psBarStatus );
            }
            if ( resp.pWCDMACallBarringSysInfo )
            {
                fprintf( fp, "CS Bar Status(WCDMA): %x\n", resp.pWCDMACallBarringSysInfo->csBarStatus );
                fprintf( fp, "PS Bar Status(WCDMA): %x\n", resp.pWCDMACallBarringSysInfo->psBarStatus );
            }
            IFPRINTF( fp, "Voice Supp on LTE: %x\n", resp.pLTEVoiceSupportSysInfo );
            IFPRINTF( fp, "GSM Cipher Domain: %x\n", resp.pGSMCipherDomainSysInfo );
            IFPRINTF( fp, "WCDMA Cipher Domain: %x\n", resp.pWCDMACipherDomainSysInfo );
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doSLQSNasSwiModemStatus
 *
 * Purpose:  Perform the tests that call the SLQSNasSwiModemStatus() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSLQSNasSwiModemStatus( void )
{
    local LTEInfo  lteinfo;

    /* Define the test cases to be executed */
    SLQSNasSwiModemStatusTestCase_t SLQSNasSwiModemStatusTestCases[] =
    {
        { eQCWWAN_ERR_NONE, 1, NULL, "Valid Test Case - Passing NULL"},

        { eQCWWAN_ERR_NONE, 2, &lteinfo, "Valid Test Case - Fetching LTE info"},

    };

    FILE *fp = tfopen( "../../nas/test/results/slqsnasswimodemstatus.txt", "w" );
    swiModemStatusResp resp;

    if ( fp )
    {
        BYTE tC     = 0;
        BYTE ncases = sizeof( SLQSNasSwiModemStatusTestCases ) /
                      sizeof( SLQSNasSwiModemStatusTestCases[0] );

        while( tC < ncases )
        {
            SLQSNasSwiModemStatusTestCase_t *pd =
                &SLQSNasSwiModemStatusTestCases[tC++];

            /* print test data */
            fprintf( fp,"\n\nTest Case %d / %d : %s\n",
                     tC,
                     ncases,
                     pd->desc );

            resp.pLTEInfo = pd->pLTEInfo;

            ULONG nRet = SLQSNasSwiModemStatus( &resp );

            /* Display result code and text */
            doprintreason( fp, nRet );

            if ( eQCWWAN_ERR_NONE == nRet )
            {
                fprintf( fp,"SLQSNasSwiModemStatus Successful \n" );
                fprintf( fp,"Modem Temperature : %x \n",resp.commonInfo.temperature );
                fprintf( fp,"Modem Mode        : %x \n",resp.commonInfo.modemMode );
                fprintf( fp,"System Mode       : %x \n",resp.commonInfo.systemMode );
                fprintf( fp,"IMS Reg State     : %x \n",resp.commonInfo.imsRegState );
                fprintf( fp,"PS Attach State   : %x \n",resp.commonInfo.psState );

                if ( resp.pLTEInfo )
                {
                    LTEInfo *t = resp.pLTEInfo;
                    fprintf( fp,"Band            : %x \n",t->band );
                    fprintf( fp,"BandWidth       : %x \n",t->bandwidth );
                    fprintf( fp,"RX Channel      : %x \n",t->RXChan );
                    fprintf( fp,"TX Channel      : %x \n",t->TXChan );
                    fprintf( fp,"EMM State       : %x \n",t->emmState );
                    fprintf( fp,"EMM Sub State   : %x \n",t->emmSubState );
                    fprintf( fp,"EMM Conn State  : %x \n",t->emmConnState );
                }
            }
        }
    }
    if (fp)
        tfclose(fp);
}

/*
 * Name:    doSLQSNasGetHDRColorCode
 *
 * Purpose: Perform the tests that call the SLQSNasGetHDRColorCode() API
 *
 * Return:  none
 *
 * Notes:   none
 *
 */
void doSLQSNasGetHDRColorCode( void )
{
    local BYTE colorcode;

    /* Test Cases */
    SLQSGetHDRColorCodeTestCase_t SLQSGetHDRColorCodeTestCases[] =
    {
        {eQCWWAN_ERR_NONE, 1, NULL, " Valid case: Passing out parameter as NULL" },
        {eQCWWAN_ERR_NONE, 2, &colorcode, "All Valid" }
    };

    nasGetHDRColorCodeResp resp;

    FILE *fp = tfopen("../../voice/test/results/SLQSNasGetHDRColorCode.txt", "w");

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSGetHDRColorCodeTestCases )/
                      sizeof( SLQSGetHDRColorCodeTestCases[0] );

        while ( tc < ncases )
        {
            SLQSGetHDRColorCodeTestCase_t *pd = &SLQSGetHDRColorCodeTestCases[tc++];
            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        tc,
                        ncases,
                        pd->desc);

            resp.pColorCode = pd->pColorCode;

            ULONG nRet = SLQSNasGetHDRColorCode( &resp );
            doprintreason( fp, nRet );

            if ( eQCWWAN_ERR_NONE == nRet )
            {
                fprintf(fp, "SLQSGetHDRColorCode Successful\n");
                IFPRINTF(fp,"Color Code is : %d \n",  pd->pColorCode);
            }
        }
    }
    if (fp)
        tfclose(fp);
}

/*
 * Name:    doSLQSNasGetTxRxInfo
 *
 * Purpose: Perform the tests that call the SLQSNasGetTxRxInfo() API
 *
 * Return:  none
 *
 * Notes:   none
 *
 */
void doSLQSNasGetTxRxInfo( void )
{
    local rxInfo RXChain0Info;
    local rxInfo RXChain1Info;
    local txInfo TXInfo;

    /* Test Cases */
    SLQSGetTxRxInfoTestCase_t SLQSGetTxRxInfoTestCases[] =
    {
        {eQCWWAN_ERR_NONE, 1, 0x01, &RXChain0Info, &RXChain1Info, NULL,
                                    "Valid Case: One optional params is NULL" },
        {eQCWWAN_ERR_NONE, 2, 0x02, &RXChain0Info, NULL, &TXInfo,
                                    "Valid Case: One optional params is NULL" },
        {eQCWWAN_ERR_NONE, 3, 0x03, NULL, &RXChain1Info, &TXInfo,
                                    "Invalid Case: Invalid Radio interface" },
        {eQCWWAN_ERR_NONE, 4, 0x04, NULL, NULL, NULL,
                                    "Valid Case: All optional params is NULL" },
        {eQCWWAN_ERR_NONE, 5, 0x05, &RXChain0Info, &RXChain1Info, &TXInfo,
                                                                  "All Valid" },
        {eQCWWAN_ERR_NONE, 6, 0x08, &RXChain0Info, &RXChain1Info, &TXInfo,
                                                                  "All Valid" }
    };

    nasGetTxRxInfoReq  req;
    nasGetTxRxInfoResp resp;

    FILE *fp = tfopen("../../voice/test/results/slqsnasgettxrxinfo.txt", "w");

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSGetTxRxInfoTestCases )/
                      sizeof( SLQSGetTxRxInfoTestCases[0] );

        while ( tc < ncases )
        {
            SLQSGetTxRxInfoTestCase_t *pd = &SLQSGetTxRxInfoTestCases[tc++];
            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        tc,
                        ncases,
                        pd->desc);

            req.radio_if = pd->radio_if;
            fprintf( fp, "Radio Interference : %x \n\n",req.radio_if );

            resp.pRXChain0Info = pd->pRXChain0Info;
            resp.pRXChain1Info = pd->pRXChain1Info;
            resp.pTXInfo       = pd->pTXInfo;

            ULONG nRet = SLQSNasGetTxRxInfo( &req, &resp );
            doprintreason( fp, nRet );

            if ( eQCWWAN_ERR_NONE != nRet )
            {
                fprintf(fp, "SLQSNasGetTxRxInfo Unsuccessful\n");
                continue;
            }

            fprintf(fp, "SLQSNasGetTxRxInfo Successful\n");
            if ( NULL != resp.pRXChain0Info )
            {
                fprintf(fp, "Rx Chain 0 Info Response \n");
                fprintf(fp, "Is radio Tuned  : 0x%X\n",
                                resp.pRXChain0Info->isRadioTuned);
                fprintf(fp, "Rx Power Value  : %d\n",
                                resp.pRXChain0Info->rxPower);
                fprintf(fp, "ECIO            : %d\n",
                                resp.pRXChain0Info->ecio);
                fprintf(fp, "RSCP            : %d\n",
                                resp.pRXChain0Info->rscp);
                fprintf(fp, "RSRP            : %d\n",
                                resp.pRXChain0Info->rsrp);
                fprintf(fp, "Phase           : 0x%X\n",
                                resp.pRXChain0Info->phase);
            }

            if ( NULL != resp.pRXChain1Info )
            {
                fprintf(fp, "Rx Chain 1 Info Response \n");
                fprintf(fp, "Is radio Tuned  : 0x%X\n",
                                resp.pRXChain1Info->isRadioTuned);
                fprintf(fp, "Rx Power Value  : %d\n",
                                resp.pRXChain1Info->rxPower);
                fprintf(fp, "ECIO            : %d\n",
                                resp.pRXChain1Info->ecio);
                fprintf(fp, "RSCP            : %d\n",
                                resp.pRXChain1Info->rscp);
                fprintf(fp, "RSRP            : %d\n",
                                resp.pRXChain1Info->rsrp);
                fprintf(fp, "Phase           : 0x%X\n",
                                resp.pRXChain1Info->phase);
            }

            if ( NULL != resp.pTXInfo )
            {
                fprintf(fp, "Tx Info Response \n");
                fprintf(fp, "Is in traffic   : 0x%X\n",
                                resp.pTXInfo->isInTraffic);
                if(resp.pTXInfo->isInTraffic)
                    fprintf(fp, "Tx Power Value  : %d\n",
                            resp.pTXInfo->txPower);
            }
        }
    }
    if (fp)
        tfclose(fp);
}

/*
 * Name:    doSLQSNasGetSigInfo
 *
 * Purpose: Perform the tests that call the SLQSNasGetSigInfo() API
 *
 * Return:  none
 *
 * Notes:   none
 *
 */
void doSLQSNasGetSigInfo( void )
{
    local CDMASSInfo  cdmaSSInfo;
    local HDRSSInfo   hdrSSInfo;
    local INT8        gsmSSInfo;
    local CDMASSInfo  wcdmaSSInfo;
    local LTESSInfo   lteSSInfo;
    local INT8        TDSCDMASigInfoRscp;
    local TDSCDMASigInfoExt tDSCDMASigInfoExt;
    /* Test Cases */
    SLQSGetSigInfoTestCase_t SLQSGetSigInfoTestCases[] =
    {
        {eQCWWAN_ERR_NONE, 1, NULL, NULL, NULL, NULL,
             NULL,NULL,NULL,"Valid case - All optional params are NULL " },
        {eQCWWAN_ERR_NONE, 2, &cdmaSSInfo, &hdrSSInfo, &gsmSSInfo, &wcdmaSSInfo,
                                   &lteSSInfo,&TDSCDMASigInfoRscp,&tDSCDMASigInfoExt,"All Valid - Fetching for All" },
    };

    nasGetSigInfoResp resp;

    FILE *fp = tfopen("../../voice/test/results/slqsnasgetsiginfo.txt", "w");

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSGetSigInfoTestCases )/
                      sizeof( SLQSGetSigInfoTestCases[0] );
        while ( tc < ncases )
        {
            SLQSGetSigInfoTestCase_t *pd = &SLQSGetSigInfoTestCases[tc++];
            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        tc,
                        ncases,
                        pd->desc);
            memset(&resp,0,sizeof(nasGetSigInfoResp));
            resp.pCDMASSInfo  = pd->pCDMASSInfo;
            resp.pHDRSSInfo   = pd->pHDRSSInfo;
            resp.pGSMSSInfo   = pd->pGSMSSInfo;
            resp.pWCDMASSInfo = pd->pWCDMASSInfo;
            resp.pLTESSInfo   = pd->pLTESSInfo;
            resp.pTDSCDMASigInfoRscp = pd->pTDSCDMASigInfoRscp;
            resp.pTDSCDMASigInfoExt = pd->pTDSCDMASigInfoExt;

            ULONG nRet = SLQSNasGetSigInfo( &resp );
            doprintreason( fp, nRet );

            if ( eQCWWAN_ERR_NONE != nRet )
            {
                fprintf(fp, "SLQSNasGetSigInfo Unsuccessful\n");
                continue;
            }

            fprintf(fp, "SLQSNasGetSigInfo Successful\n");
            if ( NULL != resp.pCDMASSInfo )
            {
                fprintf(fp, "CDMA SS Information \n");
                fprintf(fp, "RSSI  : %d\n", resp.pCDMASSInfo->rssi);
                fprintf(fp, "ECIO  : %d\n", resp.pCDMASSInfo->ecio);
            }
            if ( NULL != resp.pHDRSSInfo )
            {
                fprintf(fp, "HDR SS Information \n");
                fprintf(fp, "RSSI  : %d\n", resp.pHDRSSInfo->rssi);
                fprintf(fp, "ECIO  : %d\n", resp.pHDRSSInfo->ecio);
                fprintf(fp, "SINR  : %d\n", resp.pHDRSSInfo->sinr);
                fprintf(fp, "IO    : %d\n", resp.pHDRSSInfo->io);
            }
            IFPRINTF(fp, "GSM SIG INFO : %d\n", resp.pGSMSSInfo);
            if ( NULL != resp.pWCDMASSInfo )
            {
                fprintf(fp, "WCDMA SS Information \n");
                fprintf(fp, "RSSI  : %d\n", resp.pWCDMASSInfo->rssi);
                fprintf(fp, "ECIO  : %d\n", resp.pWCDMASSInfo->ecio);
            }
            if ( NULL != resp.pLTESSInfo )
            {
                fprintf(fp, "LTE SS Information \n");
                fprintf(fp, "RSSI  : %d\n", resp.pLTESSInfo->rssi);
                fprintf(fp, "RSRQ  : %d\n", resp.pLTESSInfo->rsrq);
                fprintf(fp, "RSRP  : %d\n", resp.pLTESSInfo->rsrp);
                fprintf(fp, "SNR   : %d\n", resp.pLTESSInfo->snr);
            }
            if( NULL != resp.pTDSCDMASigInfoRscp )
            {
                fprintf(fp, "TD-SCMDA Information  \n");
                fprintf(fp, "RSCP  : %d\n", *resp.pTDSCDMASigInfoRscp);
            }
            if( NULL != resp.pTDSCDMASigInfoExt )
            {
                fprintf(fp, "TD-SCDMA SS Information \n");
                fprintf(fp, "RSSI  : %f\n", resp.pTDSCDMASigInfoExt->rssi);
                fprintf(fp, "RSCP  : %f\n", resp.pTDSCDMASigInfoExt->rscp);
                fprintf(fp, "ECIO  : %f\n", resp.pTDSCDMASigInfoExt->ecio);
                fprintf(fp, "SINR  : %f\n", resp.pTDSCDMASigInfoExt->sinr);
            }
        }
    }
    if (fp)
        tfclose(fp);
}

/*
 * Name:    doSLQSNasIndicationRegisterExt
 *
 * Purpose: Perform the tests that call the SLQSNasIndicationRegisterExt() API
 *
 * Return:  none
 *
 * Notes:   none
 *
 */
void doSLQSNasIndicationRegisterExt( void )
{
    /* Test Cases */
    SLQSNasIndicationRegisterExtTestCase_t SLQSNasIndicationRegisterExtTestCases[] =
    {
        { eQCWWAN_ERR_NONE, 1, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
        NULL, NULL, NULL, NULL, "Invalid case - AtLeast one parameter must be set " },

        { eQCWWAN_ERR_NONE, 2, &setInd, &setInd, &setInd, &setInd, &setInd,
        &setInd, &setInd, &setInd, &setInd, &setInd, &setInd, &setInd,
        "Valid case - All params are set " },

        { eQCWWAN_ERR_NONE, 3, &unsetInd, &unsetInd, &unsetInd, &unsetInd, &unsetInd,
        &unsetInd, &unsetInd, &unsetInd, &unsetInd, &unsetInd, &unsetInd, &unsetInd,
        "Valid case - All params are unset " }
    };

    nasIndicationRegisterReq req;

    FILE *fp = tfopen("../../voice/test/results/slqsnasindicationregisterext.txt", "w");

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSNasIndicationRegisterExtTestCases )/
                      sizeof( SLQSNasIndicationRegisterExtTestCases[0] );
        while ( tc < ncases )
        {
            SLQSNasIndicationRegisterExtTestCase_t *pd =
                            &SLQSNasIndicationRegisterExtTestCases[tc++];
            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        tc,
                        ncases,
                        pd->desc);

            req.pSystemSelectionInd  = pd->pSystemSelectionInd;
            req.pDDTMInd             = pd->pDDTMInd;
            req.pServingSystemInd    = pd->pServingSystemInd;
            req.pDualStandByPrefInd  = pd->pDualStandByPrefInd;
            req.pSubscriptionInfoInd = pd->pSubscriptionInfoInd;
            req.pNetworkTimeInd      = pd->pNetworkTimeInd;
            req.pSysInfoInd          = pd->pSysInfoInd;
            req.pSignalStrengthInd   = pd->pSignalStrengthInd;
            req.pErrorRateInd        = pd->pErrorRateInd;
            req.pHDRNewUATIAssInd    = pd->pHDRNewUATIAssInd;
            req.pHDRSessionCloseInd  = pd->pHDRSessionCloseInd;
            req.pManagedRoamingInd   = pd->pManagedRoamingInd;

            IFPRINTF(fp, "System Selection Ind  : %x\n", req.pSystemSelectionInd);
            IFPRINTF(fp, "DDTM Ind              : %x\n", req.pDDTMInd);
            IFPRINTF(fp, "Serving System Ind    : %x\n", req.pServingSystemInd);
            IFPRINTF(fp, "Dual StandBy Pref Ind : %x\n", req.pDualStandByPrefInd);
            IFPRINTF(fp, "Subscription Info Ind : %x\n", req.pSubscriptionInfoInd);
            IFPRINTF(fp, "Network Time Ind      : %x\n", req.pNetworkTimeInd);
            IFPRINTF(fp, "System Info Ind       : %x\n", req.pSysInfoInd);
            IFPRINTF(fp, "Signal Strength Ind   : %x\n", req.pSignalStrengthInd);
            IFPRINTF(fp, "Error Rate Ind        : %x\n", req.pErrorRateInd);
            IFPRINTF(fp, "HDR New UATI Assn Ind : %x\n", req.pHDRNewUATIAssInd);
            IFPRINTF(fp, "HDR Session Close Ind : %x\n", req.pHDRSessionCloseInd);
            IFPRINTF(fp, "managed Roaming Ind   : %x\n", req.pManagedRoamingInd);

            ULONG nRet = SLQSNasIndicationRegisterExt( &req );
            doprintreason( fp, nRet );

            if ( eQCWWAN_ERR_NONE != nRet )
            {
                fprintf(fp, "SLQSNasIndicationRegisterExt Unsuccessful\n");
                continue;
            }

            fprintf(fp, "SLQSNasIndicationRegisterExt Successful\n");
        }
    }
    if (fp)
        tfclose(fp);
}

/*
 * Name:     doSLQSGetPLMNName
 *
 * Purpose:  Perform the tests that call the SLQSGetPLMNName() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSLQSGetPLMNName( void )
{
    ULONG           nRet;
    FILE            *fp = NULL;
    BYTE            index = 0;
    nasPLMNNameReq  PLMNNameReq;
    nasPLMNNameResp PLMNNameResp;
    BYTE mncPcsDigitF    = FALSE;

    PLMNNameReq.mcc = 310;
    PLMNNameReq.mnc = 70;    /* MNC is 070*/
    PLMNNameReq.pMncPcsStatus = &mncPcsDigitF;

    nasPLMNNameReq  PLMNNameReq1;
    BYTE mncPcsDigitT    = TRUE;

    PLMNNameReq1.mcc = 310;
    PLMNNameReq1.mnc = 70; /* MNC is 070*/
    PLMNNameReq1.pMncPcsStatus = &mncPcsDigitT;

    /* Define the test cases to be executed */
    SLQSGetPLMNNameTestCase_t SLQSGetPLMNNameTestCases[] =
    {
        { eQCWWAN_ERR_INVALID_ARG, 1, NULL, NULL,
                          "Invalid Test Case - All params NULL"},
        { eQCWWAN_ERR_NONE,        2, &PLMNNameReq, &PLMNNameResp,
                          "Valid Test Case - MNC PCS Digit False"},
        { eQCWWAN_ERR_NONE,        3, &PLMNNameReq1, &PLMNNameResp,
                          "Valid Test Case - All parameters set"},
    };

    fp = tfopen( "TestReport/slqsgetplmnname.txt", "w" );
    if ( fp )
    {
        BYTE tCaseNum     = 0;
        BYTE ncases = sizeof( SLQSGetPLMNNameTestCases ) /
                      sizeof( SLQSGetPLMNNameTestCases[0] );

        while( tCaseNum < ncases )
        {
            SLQSGetPLMNNameTestCase_t *pd =
                &SLQSGetPLMNNameTestCases[tCaseNum];

            /* print test data */
            fprintf( fp,"\n\nTest Case %d / %d : %s\n",
                     (tCaseNum + 1),
                     ncases,
                     pd->desc );

            nRet = SLQSGetPLMNName( pd->pPLMNNameReq, pd->pPLMNNameResp );

            /* Display result code and text */
            doprintreason( fp, nRet );
            if ( (!nRet) && (pd->pPLMNNameResp) )
            {
                nasPLMNNameResp *lresp = pd->pPLMNNameResp;
                fprintf( fp,"SPN Encoding :: %xH\n", lresp->spnEncoding);
                fprintf( fp,"SPN Length :: %xH\n", lresp->spnLength);
                if ((lresp->spnEncoding == 1) && (lresp->spnLength > 0))
                {
                    fprintf( fp, "SPN(unicode) :: ");
                    for (index = 0; index < lresp->spnLength; index++)
                        fprintf( fp,"%x ", lresp->spn[index]);
                    fprintf( fp,"\n");
                }
                else if ((lresp->spnEncoding == 0) && (lresp->spnLength > 0))
                {
                    fprintf( fp,"SPN(ascii) :: ");
                    for (index = 0; index < lresp->spnLength; index++)
                        fprintf( fp,"%c", lresp->spn[index]);
                    fprintf( fp,"\n");
                }

                fprintf( fp,"shortname Encoding :: %xH\n", lresp->shortNameEn);
                fprintf( fp,"shortname Country Initial :: %xH\n", lresp->shortNameCI);
                fprintf( fp,"shortname Spare Bits :: %xH\n", lresp->shortNameSB);
                fprintf( fp,"shortname Length :: %xH\n", lresp->shortNameLen);
                if ((lresp->shortNameEn == 1) && (lresp->shortNameLen > 0))
                {
                    fprintf( fp, "Short Name(unicode) :: ");
                    for (index = 0; index < lresp->shortNameLen; index++)
                        fprintf( fp,"%x ", lresp->shortName[index]);
                    fprintf( fp,"\n");
                }
                else if (lresp->shortNameLen > 0)
                {
                    fprintf( fp, "Short Name(ascii) :: ");
                    for (index = 0; index < lresp->shortNameLen; index++)
                        fprintf( fp,"%c", lresp->shortName[index]);
                    fprintf( fp,"\n");
                }

                fprintf( fp,"longname Encoding :: %xH\n", lresp->longNameEn);
                fprintf( fp,"longname Country Initial :: %xH\n", lresp->longNameCI);
                fprintf( fp,"longname Spare Bits :: %xH\n", lresp->longNameSB);
                fprintf( fp,"longname Length :: %xH\n", lresp->longNameLen);
                if ((lresp->longNameEn == 1) && (lresp->longNameLen > 0))
                {
                    fprintf( fp, "Long Name(unicode) :: ");
                    for (index = 0; index < lresp->longNameLen; index++)
                        fprintf( fp,"%x ", lresp->longName[index]);
                    fprintf( fp,"\n");
                }
                else if (lresp->longNameLen > 0)
                {
                    fprintf( fp, "Long Name(ascii) :: ");
                    for (index = 0; index < lresp->longNameLen; index++)
                        fprintf( fp,"%c", lresp->longName[index]);
                    fprintf( fp,"\n");
                }
            }
            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doSLQSGetOperatorNameData
 *
 * Purpose:  Perform the tests that call the SLQSGetOperatorNameData() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSLQSGetOperatorNameData( void )
{
    ULONG               nRet;
    FILE                *fp = NULL;
    BYTE                index = 0,index_1 = 0;
    nasOperatorNameResp OperatorNameData;
    serviceProviderName srvcProviderName;
    operatorPLMNList    operatorPLMNLists;
    PLMNNetworkName     PLMNNetworkNames;
    operatorNameString  operatorNameStrings;
    PLMNNetworkNameData NITZInformation;

    OperatorNameData.pSrvcProviderName   = &srvcProviderName;
    OperatorNameData.pOperatorPLMNList   = &operatorPLMNLists;
    OperatorNameData.pPLMNNetworkName    = &PLMNNetworkNames;
    OperatorNameData.pOperatorNameString = &operatorNameStrings;
    OperatorNameData.pNITZInformation    = &NITZInformation;

    /* Define the test cases to be executed */
    SLQSGetOperatorNameDataTestCase_t SLQSGetOperatorNameTestCases[] =
    {
        { eQCWWAN_ERR_INVALID_ARG, 1, NULL,
          "Invalid Test Case - All params NULL"},
        { eQCWWAN_ERR_NONE,        2, &OperatorNameData,
          "Valid Test Case - All parameters set"},
    };

    fp = tfopen( "TestReport/slqsgetgetoperatorname.txt", "w" );
    if ( fp )
    {
        BYTE tCaseNum     = 0;
        BYTE ncases = sizeof( SLQSGetOperatorNameTestCases ) /
                      sizeof( SLQSGetOperatorNameTestCases[0] );

        while( tCaseNum < ncases )
        {
            SLQSGetOperatorNameDataTestCase_t *pd =
                &SLQSGetOperatorNameTestCases[tCaseNum];

            /* print test data */
            fprintf( fp,"\n\nTest Case %d / %d : %s\n",
                     (tCaseNum + 1),
                     ncases,
                     pd->desc );

            nRet = SLQSGetOperatorNameData( pd->pOperatorNameData );

            /* Display result code and text */
            doprintreason( fp, nRet );

            if ( (pd->pOperatorNameData != NULL) && (!nRet) )
            {
                if( pd->pOperatorNameData->pSrvcProviderName )
                {
                    serviceProviderName *lresp =
                           pd->pOperatorNameData->pSrvcProviderName;
                    fprintf( fp, "\nService Provider Name Information\n" );
                    fprintf( fp, "Display Condition %x\n",
                                        lresp->displayCondition);
                    fprintf( fp, "SPN Length :: %d \n",
                                        lresp->spnLength);
                    fprintf( fp, "SPN :: ");
                    for(index = 0; index < lresp->spnLength; index++)
                    {
                        fprintf( fp, "%c", lresp->spn[index]);
                    }
                    fprintf( fp, "\n" );
                }

                if( pd->pOperatorNameData->pOperatorPLMNList )
                {
                    operatorPLMNList *lresp =
                                 pd->pOperatorNameData->pOperatorPLMNList;
                    fprintf( fp, "\nOperator PLMN List Information\n" );
                    fprintf( fp, "Num Instance :: %d\n", lresp->numInstance );

                    for(index = 0; index < lresp->numInstance; index++)
                    {
                        fprintf( fp, "MCC :: %c %c %c\n",
                                 lresp->PLMNData[index].mcc[0],
                                 lresp->PLMNData[index].mcc[1],
                                 lresp->PLMNData[index].mcc[2] );
                        fprintf( fp, "MNC :: %c %c %c",
                                 lresp->PLMNData[index].mnc[0],
                                 lresp->PLMNData[index].mnc[1],
                                 lresp->PLMNData[index].mnc[2] );
                        fprintf( fp, "LAC1 :: %x\n",
                                 lresp->PLMNData[index].lac1 );
                        fprintf( fp, "LAC2 :: %x\n",
                                 lresp->PLMNData[index].lac2 );
                        fprintf( fp, "PLMNRecID :: %x\n",
                                 lresp->PLMNData[index].PLMNRecID );
                    }
                }

                if( pd->pOperatorNameData->pPLMNNetworkName )
                {
                    PLMNNetworkName *qresp =
                            pd->pOperatorNameData->pPLMNNetworkName;
                    fprintf( fp, "\nPLMN Network Name\n" );
                    fprintf( fp, "Num Instance :: %d\n", qresp->numInstance );
                    for(index = 0; index < qresp->numInstance; index++)
                    {
                        PLMNNetworkNameData *lresp =
                            &pd->pOperatorNameData->pPLMNNetworkName->\
                            PLMNNetName[index];
                        fprintf( fp, "Config Scheme :: %x\n",
                                       lresp->codingScheme );
                        fprintf( fp, "CountryInitails :: %x\n",
                                      lresp->countryInitials );
                        fprintf( fp, "Long Name Length :: %x\n",
                                           lresp->longNameLen );
                        fprintf( fp, "Long Name :: ");
                        for(index_1 = 0; index_1 < lresp->longNameLen;
                                                            index_1++)
                        {
                            fprintf( fp, "%x ", lresp->longName[index_1] );
                        }
                        fprintf( fp, "\n" );
                        fprintf( fp, "Long Name Spare Bits :: %x\n",
                                         lresp->longNameSpareBits );
                        fprintf( fp, "Short Name Length :: %x\n",
                                           lresp->shortNameLen );
                        fprintf( fp, "Short Name :: ");
                        for(index_1 = 0; index_1 < lresp->shortNameLen;
                                                             index_1++)
                        {
                            fprintf( fp, "%x ", lresp->shortName[index_1] );
                        }
                        fprintf( fp, "\n" );
                        fprintf( fp, "Short Name Spare Bits :: %x\n",
                                         lresp->shortNameSpareBits );
                    }
                }

                if( pd->pOperatorNameData->pOperatorNameString )
                {
                    operatorNameString *lresp =
                                 pd->pOperatorNameData->pOperatorNameString;
                    fprintf( fp, "\nOperator Name String\n" );
                    fprintf( fp, "PLMN Name :: ");
                    for(index = 0; lresp->PLMNName[index] != EOS; index++)
                    {
                        fprintf( fp, "%c ", lresp->PLMNName[index] );
                    }
                    fprintf( fp, "\n" );
                }

                if( pd->pOperatorNameData->pNITZInformation )
                {
                    PLMNNetworkNameData *lresp =
                            pd->pOperatorNameData->pNITZInformation;
                    fprintf( fp, "\nNITZ Information\n" );
                    fprintf( fp, "Config Scheme :: %x\n",
                                   lresp->codingScheme );
                    fprintf( fp, "CountryInitails :: %x\n",
                                  lresp->countryInitials );
                    fprintf( fp, "Long Name Spare Bits :: %x\n",
                                     lresp->longNameSpareBits );
                    fprintf( fp, "Short Name Spare Bits :: %x\n",
                                     lresp->shortNameSpareBits );
                    fprintf( fp, "Long Name Length :: %d\n",
                                       lresp->longNameLen );
                    fprintf( fp, "Long Name :: ");
                    for(index = 0; index < lresp->longNameLen; index++)
                    {
                        fprintf( fp, "%c", lresp->longName[index] );
                    }
                    fprintf( fp, "\n" );
                    fprintf( fp, "Short Name Length :: %d\n",
                                        lresp->shortNameLen );
                    fprintf( fp, "Short Name :: ");
                    for(index = 0; index < lresp->shortNameLen; index++)
                    {
                        fprintf( fp, "%c", lresp->shortName[index] );
                    }
                    fprintf( fp, "\n" );
                }
           }
           tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:    doSLQSGet3GPP2Subscription
 *
 * Purpose: Perform the tests that call the SLQSNasGet3GPP2Subscription() API
 *
 * Return:  none
 *
 * Notes:   none
 *
 */
void doSLQSGet3GPP2Subscription( void )
{
    local namName      NAMNameInfo;
    local dirNum       DirNum;
    local homeSIDNID   HomeSIDNID;
    local minBasedIMSI MinBasedIMSI;
    local trueIMSI     TrueIMSI;
    local CDMAChannel  CDMAChannel;

    /* Test Cases */
    SLQSGet3GPP2SubscriptionTestCase_t SLQSGet3GPP2SubscriptionTestCases[] =
    {
        {eQCWWAN_ERR_NONE, 1, 0x01, &NAMNameInfo, &DirNum, &HomeSIDNID,
                                    &MinBasedIMSI, &TrueIMSI, NULL,
                                    "Valid Case: One optional params is NULL" },
        {eQCWWAN_ERR_NONE, 2, 0x01, &NAMNameInfo, &DirNum, &HomeSIDNID,
                                    &MinBasedIMSI, NULL, &CDMAChannel,
                                    "Valid Case: One optional params is NULL" },
        {eQCWWAN_ERR_NONE, 3, 0x01, &NAMNameInfo, &DirNum, &HomeSIDNID,
                                    NULL, &TrueIMSI, &CDMAChannel,
                                    "Valid Case: One optional params is NULL" },
        {eQCWWAN_ERR_NONE, 4, 0x01, &NAMNameInfo, &DirNum, NULL,
                                    &MinBasedIMSI, &TrueIMSI, &CDMAChannel,
                                    "Valid Case: One optional params is NULL" },
        {eQCWWAN_ERR_NONE, 5, 0x01, &NAMNameInfo, NULL, &HomeSIDNID,
                                    &MinBasedIMSI, &TrueIMSI, &CDMAChannel,
                                    "Valid Case: One optional params is NULL" },
        {eQCWWAN_ERR_NONE, 6, 0x01, NULL, &DirNum, &HomeSIDNID,
                                    &MinBasedIMSI, &TrueIMSI, &CDMAChannel,
                                    "Valid Case: One optional params is NULL" },
        {eQCWWAN_ERR_NONE, 7, 0x01, NULL, NULL, NULL,
                                    NULL, NULL, NULL,
                                    "Valid Case: All optional params is NULL" },
        {eQCWWAN_ERR_NONE, 8, 0x01, &NAMNameInfo, &DirNum, &HomeSIDNID,
                                    &MinBasedIMSI, &TrueIMSI, &CDMAChannel,
                                    "All Valid" },
        {eQCWWAN_ERR_NONE, 8, 0xFF, &NAMNameInfo, &DirNum, &HomeSIDNID,
                                    &MinBasedIMSI, &TrueIMSI, &CDMAChannel,
                                    "All Valid - For Current NAM" }
    };

    nasGet3GPP2SubscriptionInfoReq  req;
    nasGet3GPP2SubscriptionInfoResp resp;
    WORD                            lC;

    FILE *fp = tfopen("../../nas/test/results/slqsget3gpp2subscription.txt", "w");

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSGet3GPP2SubscriptionTestCases )/
                      sizeof( SLQSGet3GPP2SubscriptionTestCases[0] );

        while ( tc < ncases )
        {
            SLQSGet3GPP2SubscriptionTestCase_t *pd =
                    &SLQSGet3GPP2SubscriptionTestCases[tc++];
            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        tc,
                        ncases,
                        pd->desc);

            req.namID = pd->namID;
            fprintf( fp, "Nam Id : %x \n\n",req.namID );

            resp.pNAMNameInfo  = pd->pNAMNameInfo;
            resp.pDirNum       = pd->pDirNum;
            resp.pHomeSIDNID   = pd->pHomeSIDNID;
            resp.pMinBasedIMSI = pd->pMinBasedIMSI;
            resp.pTrueIMSI     = pd->pTrueIMSI;
            resp.pCDMAChannel  = pd->pCDMAChannel;

            ULONG nRet = SLQSNasGet3GPP2Subscription( &req, &resp );
            doprintreason( fp, nRet );

            if ( eQCWWAN_ERR_NONE != nRet )
            {
                fprintf(fp, "SLQSNasGet3GPP2Subscription Unsuccessful\n");
                continue;
            }

            fprintf(fp, "SLQSNasGet3GPP2Subscription Successful\n");
            if ( NULL != resp.pNAMNameInfo )
            {
                fprintf(fp, "Nam Name Response \n");
                fprintf(fp, "NAM name length  : %x\n",
                                                resp.pNAMNameInfo->namNameLen);
                fprintf(fp, "Nam Name Info:");
                for ( lC = 0 ; lC < resp.pNAMNameInfo->namNameLen ; lC++ )
                {
                    fprintf(fp, " %x", resp.pNAMNameInfo->namName[lC]);
                }
                fprintf(fp, "\n");
            }

            if ( NULL != resp.pDirNum )
            {
                fprintf(fp, "Directory Number Response \n");
                fprintf(fp, "Directory number length  : %x\n",
                                                resp.pDirNum->dirNumLen);
                fprintf(fp, "Directory Number Info:");
                for ( lC = 0 ; lC < resp.pDirNum->dirNumLen ; lC++ )
                {
                    fprintf(fp, " %x", resp.pDirNum->dirNum[lC]);
                }
                fprintf(fp, "\n");
            }

            if ( NULL != resp.pHomeSIDNID )
            {
                fprintf(fp, "SID/NID Response \n");
                fprintf(fp, "Number of sets : %x\n",
                                                resp.pHomeSIDNID->numInstances);
                for ( lC = 0 ; lC < resp.pHomeSIDNID->numInstances ; lC++ )
                {
                    sidNid *t = &resp.pHomeSIDNID->SidNid[lC];
                    fprintf(fp, "SID NID Info %d\n",  lC);
                    fprintf(fp, "System ID  :%x ",   t->sid);
                    fprintf(fp, "Network ID :%x\n", t->nid);
                }
            }

            if ( NULL != resp.pMinBasedIMSI )
            {
                fprintf(fp, "Min Based IMSI Response \n");
                fprintf(fp, "MCCM :");
                for ( lC = 0; lC < PLMN_LENGTH ; lC++)
                {
                    fprintf(fp, " %x ", resp.pMinBasedIMSI->mccM[lC]);
                }
                fprintf(fp, "\n");

                fprintf(fp, "IMSI_M_11_12 : %x\n", resp.pMinBasedIMSI->imsiM1112);

                fprintf(fp, "IMSI_M 1 :");
                for ( lC = 0; lC < IMSI_M_S1_LENGTH ; lC++)
                {
                    fprintf(fp, " %x ", resp.pMinBasedIMSI->imsiMS1[lC]);
                }
                fprintf(fp, "\n");

                fprintf(fp, "IMSI_M 2 :");
                for ( lC = 0; lC < IMSI_M_S2_LENGTH ; lC++)
                {
                    fprintf(fp, " %x", resp.pMinBasedIMSI->imsiMS2[lC]);
                }
                fprintf(fp, "\n");
            }

            if ( NULL != resp.pTrueIMSI )
            {
                fprintf(fp, "True IMSI Response \n");
                fprintf(fp, "MCCT :");
                for ( lC = 0; lC < PLMN_LENGTH ; lC++)
                {
                    fprintf(fp, " %x ", resp.pTrueIMSI->mccT[lC]);
                }
                fprintf(fp, "\n");

                fprintf(fp, "IMSI_T_11_12 : %x\n", resp.pTrueIMSI->imsiT1112);

                fprintf(fp, "IMSI_T 1 :");
                for ( lC = 0; lC < IMSI_M_S1_LENGTH ; lC++)
                {
                    fprintf(fp, " %x ", resp.pTrueIMSI->imsiTS1[lC]);
                }
                fprintf(fp, "\n");

                fprintf(fp, "IMSI_T 2 :");
                for ( lC = 0; lC < IMSI_M_S2_LENGTH ; lC++)
                {
                    fprintf(fp, " %x ", resp.pTrueIMSI->imsiTS2[lC]);
                }
                fprintf(fp, "\n");

                fprintf(fp, "IMSI_T_ADDR_NUM : %x\n", resp.pTrueIMSI->imsiTaddrNum);
            }
            if ( NULL != resp.pCDMAChannel )
            {
                fprintf(fp, "CDMA Channel Information \n");
                fprintf(fp, "Primary Carrier A   : %x\n", resp.pCDMAChannel->priChA);
                fprintf(fp, "Primary Carrier B   : %x\n", resp.pCDMAChannel->priChB);
                fprintf(fp, "Secondary Carrier A : %x\n", resp.pCDMAChannel->secChA);
                fprintf(fp, "Secondary Carrier A : %x\n", resp.pCDMAChannel->secChB);
            }
        }
    }
    if (fp)
        tfclose(fp);
}


/*
 * Name:     doSLQSNasGetCellLocationInfo
 *
 * Purpose:  Perform the tests that call the SLQSNasGetCellLocationInfo() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSLQSNasGetCellLocationInfo( void )
{
    local GERANInfo                geranInfo;
    local UMTSInfo                 umtsInfo;
    local CDMAInfo                 cdmaInfo;
    local LTEInfoIntrafreq         lteInfoIntrafreq;
    local LTEInfoInterfreq         lteInfoInterfreq;
    local LTEInfoNeighboringGSM    lteInfoNeighboringGSM;
    local LTEInfoNeighboringWCDMA  lteInfoNeighboringWCDMA;
    local ULONG                    UMTSCellID;
    local WCDMAInfoLTENeighborCell WCDMAInfLTENeighborCell;

    /* Define the test cases to be executed */
    SLQSNasGetCellLocationInfoTestCase_t SLQSNasGetCellLocationInfoTestCases[] =
    {
        { eQCWWAN_ERR_NONE, 1, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
          "Valid Test Case - All params NULL"},

        { eQCWWAN_ERR_NONE, 2, &geranInfo, &umtsInfo, &cdmaInfo,
          &lteInfoIntrafreq, &lteInfoInterfreq, &lteInfoNeighboringGSM,
          &lteInfoNeighboringWCDMA, &UMTSCellID, &WCDMAInfLTENeighborCell,
          "Valid Test Case - Fetching All"},

    };

    FILE *fp = tfopen( "../../nas/test/results/slqsgetcelllocationinfo.txt", "w" );
    nasCellLocationInfoResp resp;
    WORD                    lC, lC1;

    if ( fp )
    {
        BYTE tC     = 0;
        BYTE ncases = sizeof( SLQSNasGetCellLocationInfoTestCases ) /
                      sizeof( SLQSNasGetCellLocationInfoTestCases[0] );

        while( tC < ncases )
        {
            SLQSNasGetCellLocationInfoTestCase_t *pd =
                &SLQSNasGetCellLocationInfoTestCases[tC++];

            /* print test data */
            fprintf( fp,"\n\nTest Case %d / %d : %s\n",
                     tC,
                     ncases,
                     pd->desc );

            resp.pGERANInfo               = pd->pGERANInfo;
            resp.pUMTSInfo                = pd->pUMTSInfo;
            resp.pCDMAInfo                = pd->pCDMAInfo;
            resp.pLTEInfoIntrafreq        = pd->pLTEInfoIntrafreq;
            resp.pLTEInfoInterfreq        = pd->pLTEInfoInterfreq;
            resp.pLTEInfoNeighboringGSM   = pd->pLTEInfoNeighboringGSM;
            resp.pLTEInfoNeighboringWCDMA = pd->pLTEInfoNeighboringWCDMA;
            resp.pUMTSCellID              = pd->pUMTSCellID;
            resp.pWCDMAInfoLTENeighborCell = pd->pWCDMAInfoLTENeighborCell;

            ULONG nRet = SLQSNasGetCellLocationInfo( &resp );

            /* Display result code and text */
            doprintreason( fp, nRet );

            if (!nRet)
            {
                fprintf( fp,"SLQS NAS Get Cell Location Info Successful \n" );
                if ( resp.pGERANInfo )
                {
                    fprintf( fp,"\nGERAN Information \n" );
                    GERANInfo *temp = resp.pGERANInfo;
                    fprintf( fp,"Cell Id   : %x \n",temp->cellID );
                    fprintf( fp,"PLMN info : " );
                    for (lC=0;lC<PLMN_LENGTH;lC++)
                    {
                        fprintf( fp,"%x ",temp->plmn[lC] );
                    }
                    fprintf( fp,"\n" );
                    fprintf( fp,"LAC       : %x \n",temp->lac );
                    fprintf( fp,"ARFCN     : %x \n",temp->arfcn );
                    fprintf( fp,"BSIC      : %x \n",temp->bsic );
                    fprintf( fp,"Timing Adv: %x \n",temp->timingAdvance );
                    fprintf( fp,"RX-lev    : %x \n",temp->rxLev );
                    fprintf( fp,"No. of sets of following info : %x \n",temp->nmrInst );
                    for (lC=0;lC<temp->nmrInst;lC++)
                    {
                        nmrCellInfo *temp1 = &temp->insNmrCellInfo[lC];
                        fprintf( fp,"\tNMR Cell %d Information \n",lC+1 );
                        fprintf( fp,"\tNMR Cell Id   : %x \n",temp1->nmrCellID );
                        fprintf( fp,"\tNMR PLMN info : " );
                        for (lC1=0;lC1<PLMN_LENGTH;lC1++)
                        {
                            fprintf( fp,"%x ",temp1->nmrPlmn[lC1] );
                        }
                        fprintf( fp,"\n" );
                        fprintf( fp,"\tNMR LAC       : %x \n",temp1->nmrLac );
                        fprintf( fp,"\tNMR ARFCN     : %x \n",temp1->nmrArfcn );
                        fprintf( fp,"\tNMR BSIC      : %x \n",temp1->nmrBsic );
                        fprintf( fp,"\tNMR RX-lev    : %x \n",temp1->nmrRxLev );
                    }
                }
                if ( resp.pUMTSInfo )
                {
                    fprintf( fp,"\nUMTS Information \n" );
                    UMTSInfo *temp = resp.pUMTSInfo;
                    fprintf( fp,"Cell Id   : %x \n",temp->cellID );
                    fprintf( fp,"PLMN info : " );
                    for (lC=0;lC<PLMN_LENGTH;lC++)
                    {
                        fprintf( fp,"%x ",temp->plmn[lC] );
                    }
                    fprintf( fp,"\n" );
                    fprintf( fp,"LAC       : %x \n",temp->lac );
                    fprintf( fp,"UARFCN    : %x \n",temp->uarfcn );
                    fprintf( fp,"PSC       : %x \n",temp->psc );
                    fprintf( fp,"RSCP      : %d \n",temp->rscp );
                    fprintf( fp,"ECIO      : %d \n",temp->ecio );
                    fprintf( fp,"No. of sets of umts instance : %x \n",temp->umtsInst );
                    for (lC=0;lC<temp->umtsInst;lC++)
                    {
                        UMTSinstInfo *temp1 = &temp->UMTSInstInfo[lC];
                        fprintf( fp,"\tUMTS Instance %d Information \n",lC+1 );
                        fprintf( fp,"\tUMTS UARFCN  : %x \n",temp1->umtsUarfcn );
                        fprintf( fp,"\tUMTS PSC     : %x \n",temp1->umtsPsc);
                        fprintf( fp,"\tUMTS RSCP    : %d \n",temp1->umtsRscp );
                        fprintf( fp,"\tUMTS ECIO    : %d \n",temp1->umtsEcio );
                    }
                    fprintf( fp,"No. of sets of geran instance : %x \n",temp->geranInst );
                    for (lC=0;lC<temp->geranInst;lC++)
                    {
                        geranInstInfo *temp1 = &temp->GeranInstInfo[lC];
                        fprintf( fp,"\tGERAN Instance %d Information \n",lC+1 );
                        fprintf( fp,"\tGERAN ARFCN   : %x \n",temp1->geranArfcn );
                        fprintf( fp,"\tGERAN BSIC NCC: %x \n",temp1->geranBsicNcc);
                        fprintf( fp,"\tGERAN BSIC BCC: %x \n",temp1->geranBsicBcc );
                        fprintf( fp,"\tGERAN RSSI    : %d \n",temp1->geranRssi );
                    }
                }
                if ( resp.pCDMAInfo )
                 {
                     fprintf( fp,"\nCDMA Information \n" );
                     CDMAInfo *temp = resp.pCDMAInfo;
                     fprintf( fp,"System Id  : %x \n",temp->sid );
                     fprintf( fp,"Network Id : %x \n",temp->nid );
                     fprintf( fp,"Base Id    : %x \n",temp->baseId );
                     fprintf( fp,"Reference PN: %x \n",temp->refpn );
                     fprintf( fp,"Base Lat   : %x \n",temp->baseLat );
                     fprintf( fp,"Base Long  : %x \n",temp->baseLong );
                 }
                if ( resp.pLTEInfoIntrafreq )
                {
                    fprintf( fp,"\nLTE Information - INtra Frequency \n" );
                    LTEInfoIntrafreq *temp = resp.pLTEInfoIntrafreq;
                    fprintf( fp,"UE in Idle    : %x \n",temp->ueInIdle );
                    fprintf( fp,"PLMN info     : " );
                    for (lC=0;lC<PLMN_LENGTH;lC++)
                    {
                        fprintf( fp,"%x ",temp->plmn[lC] );
                    }
                    fprintf( fp,"\n" );
                    fprintf( fp,"TAC                   : %x \n",temp->tac );
                    fprintf( fp,"Global Cell Id        : %x \n",temp->globalCellId );
                    fprintf( fp,"EARFCN                : %x \n",temp->earfcn );
                    fprintf( fp,"Serving Cell Id       : %x \n",temp->servingCellId );
                    fprintf( fp,"Cell Resel Priority   : %x \n",temp->cellReselPriority );
                    fprintf( fp,"Non-Intra Search      : %x \n",temp->sNonIntraSearch );
                    fprintf( fp,"Serving Cell Threshold: %x \n",temp->threshServingLow );
                    fprintf( fp,"Intra Search          : %x \n",temp->sIntraSearch );
                    fprintf( fp,"No. of sets of following info : %x \n",temp->cellsLen );
                    for (lC=0;lC<temp->cellsLen;lC++)
                    {
                        cellParams *temp1 = &temp->CellParams[lC];
                        fprintf( fp,"\tCell Number %d Information \n",lC+1 );
                        fprintf( fp,"\tPhysical Cell Id: %x \n",temp1->pci );
                        fprintf( fp,"\tRSRQ            : %d \n",temp1->rsrq );
                        fprintf( fp,"\tRSRP            : %d \n",temp1->rsrp );
                        fprintf( fp,"\tRSSI            : %d \n",temp1->rssi );
                        fprintf( fp,"\tSRX-lev         : %d \n",temp1->srxlev );
                    }
                }
                if ( resp.pLTEInfoInterfreq )
                {
                    fprintf( fp,"\nLTE Information - Inter Frequency \n" );
                    LTEInfoInterfreq *temp = resp.pLTEInfoInterfreq;
                    fprintf( fp,"UE in Idle    : %x \n",temp->ueInIdle );
                    fprintf( fp,"Frequency Len : %d \n",temp->freqsLen );
                    for (lC=0;lC<temp->freqsLen;lC++)
                    {
                        infoInterFreq *t = &(temp->InfoInterfreq[lC]);
                        fprintf( fp,"Frequency Number %d Information \n",lC+1 );
                        fprintf( fp,"EARFCN            : %d \n",t->earfcn );
                        fprintf( fp,"Thresh X Low      : %d \n",t->threshXLow );
                        fprintf( fp,"Thresh X High     : %d \n",t->threshXHigh );
                        fprintf( fp,"Cell Resel Pty    : %d \n",t->cell_resel_priority );
                        fprintf( fp,"Cells Length      : %d \n",t->cells_len );
                        for (lC1=0;lC1<t->cells_len;lC1++)
                        {
                            cellParams *temp1 = &t->cellInterFreqParams[lC1];
                            fprintf( fp,"Cell Number %d Information \n",lC1+1 );
                            fprintf( fp,"\tPhysical Cell Id: %x \n",temp1->pci );
                            fprintf( fp,"\tRSRQ            : %d \n",temp1->rsrq );
                            fprintf( fp,"\tRSRP            : %d \n",temp1->rsrp );
                            fprintf( fp,"\tRSSI            : %d \n",temp1->rssi );
                            fprintf( fp,"\tSRX-lev         : %d \n",temp1->srxlev );
                        }
                    }
                }
                if ( resp.pLTEInfoNeighboringGSM )
                {
                    fprintf( fp,"\nLTE Information - Neighboring GSM \n" );
                    LTEInfoNeighboringGSM *temp = resp.pLTEInfoNeighboringGSM;
                    fprintf( fp,"UE in Idle    : %x \n",temp->ueInIdle );
                    fprintf( fp,"Frequency Len : %d \n",temp->freqsLen );
                    for (lC=0;lC<temp->freqsLen;lC++)
                    {
                        lteGsmCellInfo *t = &(temp->LteGsmCellInfo[lC]);
                        fprintf( fp,"\tFrequency Number %d Information \n",lC+1 );
                        fprintf( fp,"\tCell Resel Pty    : %d \n",t->cellReselPriority );
                        fprintf( fp,"\tThresh GSM High   : %d \n",t->threshGsmHigh );
                        fprintf( fp,"\tThresh GSM Low    : %d \n",t->threshGsmLow );
                        fprintf( fp,"\tNCC Permitted     : %x \n",t->nccPermitted );
                        fprintf( fp,"\tCells Length      : %d \n",t->cells_len );
                        for (lC1=0;lC1<t->cells_len;lC1++)
                        {
                            gsmCellInfo *temp1 = &t->GsmCellInfo[lC1];
                            fprintf( fp,"\tGSM Cell Number %d Information \n",lC1+1 );
                            fprintf( fp,"\t\tARFCN         : %d \n",temp1->arfcn );
                            fprintf( fp,"\t\tBand_1900     : %x \n",temp1->band1900 );
                            fprintf( fp,"\t\tCell ID Valid : %x \n",temp1->cellIdValid );
                            fprintf( fp,"\t\tBSIC ID       : %x \n",temp1->bsicId );
                            fprintf( fp,"\t\tRSSI          : %d \n",temp1->rssi );
                            fprintf( fp,"\t\tSRX-lev       : %d \n",temp1->srxlev );
                        }
                    }
                }
                if ( resp.pLTEInfoNeighboringWCDMA )
                {
                    fprintf( fp,"\nLTE Information - Neighboring WCDMA \n" );
                    LTEInfoNeighboringWCDMA *temp = resp.pLTEInfoNeighboringWCDMA;
                    fprintf( fp,"UE in Idle    : %x \n",temp->ueInIdle );
                    fprintf( fp,"Frequency Len : %d \n",temp->freqsLen );
                    for (lC=0;lC<temp->freqsLen;lC++)
                    {
                        lteWcdmaCellInfo *t = &(temp->LTEWCDMACellInfo[lC]);
                        fprintf( fp,"\tFrequency Number %d Information \n",lC+1 );
                        fprintf( fp,"\tUARFCN          : %d \n",t->uarfcn );
                        fprintf( fp,"\tCell Resel Pty  : %d \n",t->cellReselPriority );
                        fprintf( fp,"\tThresh X High   : %d \n",t->threshXhigh );
                        fprintf( fp,"\tThresh X Low    : %d \n",t->threshXlow );
                        fprintf( fp,"\tCells Length    : %d \n",t->cellsLen );
                        for (lC1=0;lC1<t->cellsLen;lC1++)
                        {
                            wcdmaCellInfo *temp1 = &t->WCDMACellInfo[lC1];
                            fprintf( fp,"\tWCDMA Cell Number %d Information \n",lC1+1 );
                            fprintf( fp,"\t\tPSC        : %d \n",temp1->psc );
                            fprintf( fp,"\t\tCPICH_RSCP : %d \n",temp1->cpich_rscp );
                            fprintf( fp,"\t\tCPICH_ECNO : %d \n",temp1->cpich_ecno );
                            fprintf( fp,"\t\tSRX-lev    : %d \n",temp1->srxlev );
                        }
                    }
                }
                if ( resp.pUMTSCellID )
                {
                     fprintf( fp,"\nUMTS Cell ID \n" );
                     IFPRINTF( fp,"UMTS Cell ID  : 0x%x \n", resp.pUMTSCellID );
                }
                if ( resp.pWCDMAInfoLTENeighborCell )
                {
                    fprintf( fp,"\nWCDMA Information - LTE Neighboring cell info \n" );
                    WCDMAInfoLTENeighborCell *temp = resp.pWCDMAInfoLTENeighborCell;
                    fprintf( fp,"WCDMA RRC State            : %x \n",temp->wcdmaRRCState );
                    fprintf( fp,"UMTS LTE neighbor cell Len : %x \n",temp->umtsLTENbrCellLen );
                    for (lC=0;lC<temp->umtsLTENbrCellLen;lC++)
                    {
                        umtsLTENbrCell *t = &(temp->UMTSLTENbrCell[lC]);
                        fprintf( fp,"\tNumber %d Information \n",lC+1 );
                        fprintf( fp,"\tEARFCN           : %d \n",t->earfcn );
                        fprintf( fp,"\tPhysical Cell ID : %d \n",t->pci );
                        fprintf( fp,"\tRSRP             : %f \n",(float)t->rsrp );
                        fprintf( fp,"\tRSRQ             : %f \n",(float)t->rsrq );
                        fprintf( fp,"\tSRX Lev          : %d \n",t->srxlev );
                        fprintf( fp,"\tCell is TDD      : %d \n",t->cellIsTDD );
                    }
                }
            }
        }
    }
    if (fp)
        tfclose(fp);
}

/*
 * Name:     doSLQSInitNetworkReg
 *
 * Purpose:  Perform the tests that call the SLQSInitiateNetworkRegistration() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSLQSInitNetworkReg( void )
{
    MNRInfo          temp;
    ULONG            changeDurationPC = 1;
    ULONG            changeDurationPer = 0;
    BOOL             mncPcsDigitT = TRUE;
    BOOL             mncPcsDigitF = FALSE;
    struct MNRegInfo MNRInfo3Digit, MNRInfo2Digit;

    MNRInfo3Digit.mcc = 302;
    MNRInfo2Digit.mcc = 302;
    MNRInfo3Digit.mnc = 220;
    MNRInfo2Digit.mnc = 0x022;
    MNRInfo3Digit.rat = 5;
    MNRInfo2Digit.rat = 5;

    SLQSNasInitNetworkRegTestCase_t SLQSNasInitNetworkRegTestCases[] =
    {
        {eQCWWAN_ERR_NONE, 1, QMI_NAS_MANUAL_REGISTRATION, &MNRInfo3Digit,
                &changeDurationPer, &mncPcsDigitT,"Valid Registration"},
        {eQCWWAN_ERR_NONE, 2, QMI_NAS_MANUAL_REGISTRATION, &MNRInfo3Digit,
                &changeDurationPC, &mncPcsDigitT, "Valid Registration"},
        {eQCWWAN_ERR_NONE, 3, QMI_NAS_MANUAL_REGISTRATION, &MNRInfo2Digit,
                &changeDurationPer, &mncPcsDigitF, "Valid Registration"},
        {eQCWWAN_ERR_NONE, 4, QMI_NAS_MANUAL_REGISTRATION, &MNRInfo2Digit,
                &changeDurationPC, &mncPcsDigitT, "Valid Registration"},
        {eQCWWAN_ERR_NONE, 5, QMI_NAS_AUTOMATIC_REGISTRATION, &MNRInfo3Digit,
                &changeDurationPer, &mncPcsDigitT, "Valid Registration"},
        {eQCWWAN_ERR_NONE, 6, QMI_NAS_AUTOMATIC_REGISTRATION, &MNRInfo3Digit,
                &changeDurationPC, &mncPcsDigitT, "Valid Registration"},
        {eQCWWAN_ERR_NONE, 7, QMI_NAS_AUTOMATIC_REGISTRATION, &MNRInfo3Digit,
                &changeDurationPer, &mncPcsDigitF, "Valid Registration"},
        {eQCWWAN_ERR_NONE, 8, QMI_NAS_AUTOMATIC_REGISTRATION, &MNRInfo3Digit,
                &changeDurationPC, &mncPcsDigitF, "Valid Registration"},
        {eQCWWAN_ERR_NONE, 9, QMI_NAS_MANUAL_REGISTRATION, &MNRInfo3Digit,
                NULL, NULL, "All Optional Parameter NULL"},
        {eQCWWAN_ERR_NONE, 10, QMI_NAS_AUTOMATIC_REGISTRATION, NULL, NULL,
                NULL, "All Optional Parameter NULL"}
    };

    ULONG nRet;
    nasInitNetworkReg req;
    FILE  *fp = NULL;

    fp = tfopen( "../../nas/test/results/slqsinitnetworkreg.txt", "w" );

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSNasInitNetworkRegTestCases )/
                      sizeof( SLQSNasInitNetworkRegTestCases[0] );
        while ( tc < ncases )
        {
            SLQSNasInitNetworkRegTestCase_t *pd =
                                     &SLQSNasInitNetworkRegTestCases[tc];
            req.regAction = pd->regAction;
            if( pd->pMNRegInfo )
            {
                req.pMNRInfo = &temp;
                req.pMNRInfo->mcc = pd->pMNRegInfo->mcc;
                req.pMNRInfo->mnc = pd->pMNRegInfo->mnc;
                req.pMNRInfo->rat = pd->pMNRegInfo->rat;
            }
            else
            {
                req.pMNRInfo = NULL;
            }
            req.pChangeDuration = pd->pChangeDuration;
            req.pMncPcsDigitStatus = pd->pMncPcsDigitStatus;

            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        (tc + 1),
                        ncases,
                        pd->desc);

            fprintf( fp, "Parameter Set:\n" );
            fprintf( fp, "regType :%u\n", req.regAction );
            if( req.pMNRInfo )
            {
                fprintf( fp, "mcc :%d\n", req.pMNRInfo->mcc );
                fprintf( fp, "mnc :%x\n", req.pMNRInfo->mnc );
                fprintf( fp, "rat :%u\n", req.pMNRInfo->rat );
            }
            if( req.pChangeDuration )
            {
                fprintf( fp, "changeDuration :%u\n", *(req.pChangeDuration) );
            }
            if( req.pMncPcsDigitStatus )
            {
                fprintf( fp, "mncpcsdigitstatus:%d\n",
                                                   *(req.pMncPcsDigitStatus) );
            }

            nRet = SLQSInitiateNetworkRegistration( &req );

            /* Display result code and text */
            doprintreason ( fp, nRet );

            if ( !nRet )
            {
                fprintf( fp, "Network Registration Initiated\n" );
            }
            tc++;
        }
    }
    if ( fp )
        tfclose( fp );
}

void doSLQSSwiGetHDRPersonality( void )
{
    WORD                   CurrentPersonality;
    BYTE                   PersonalityListLength = 0x03;
    protocolSubtypeElement protSubTypeElmnt[3];

    SLQSSwiGetHDRPersonalityTestCase_t SLQSSwiGetHDRPersonalityTestCases[] =
    {
        { eQCWWAN_ERR_NONE, 2, NULL, NULL, NULL, "No parameter fetched "},
        { eQCWWAN_ERR_NONE, 2, NULL, &PersonalityListLength, NULL, "personality List length fetched "},
        { eQCWWAN_ERR_NONE, 2, &CurrentPersonality, NULL, NULL, "Current personality fetched "},
        { eQCWWAN_ERR_NONE, 2, &CurrentPersonality, &PersonalityListLength, protSubTypeElmnt, "Valid "},
    };

    ULONG nRet;
    HDRPersonalityResp pResp;
    FILE               *fp   = NULL;
    BYTE               count = 0;

    fp = tfopen( "../../nas/test/results/slqsswigethdrpersonality.txt", "w" );

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSSwiGetHDRPersonalityTestCases )/
                      sizeof( SLQSSwiGetHDRPersonalityTestCases[0] );

        while ( tc < ncases )
        {
            SLQSSwiGetHDRPersonalityTestCase_t *pd =
                                     &SLQSSwiGetHDRPersonalityTestCases[tc];

            if( NULL != pd->pCurrentPersonality )
                pResp.pCurrentPersonality = pd->pCurrentPersonality;
            else
                pResp.pCurrentPersonality = NULL;

            if( NULL != pd->pPersonalityListLength )
                pResp.pPersonalityListLength = pd->pPersonalityListLength;
            else
                pResp.pPersonalityListLength = NULL;

            if( NULL != pd->pProtSubTypeElmnt )
                pResp.pProtocolSubtypeElement = pd->pProtSubTypeElmnt;
            else
                pResp.pProtocolSubtypeElement = NULL;

            fprintf( fp, "\nTest Case %d / %d : %s\n",
                        (tc + 1),
                        ncases,
                        pd->desc );

            nRet = SLQSSwiGetHDRPersonality( &pResp );

            /* Display result code and text */
            doprintreason ( fp, nRet );

            if ( !nRet )
            {
                fprintf( fp, "HDR Personality Retrieved successfully\n" );

                IFPRINTF( fp, "Current Personality  : 0x%x\n", pResp.pCurrentPersonality );
                IFPRINTF( fp, "PersonalityListLength:0x%x\n", pResp.pPersonalityListLength );

                if( NULL != pResp.pPersonalityListLength &&
                    NULL != pResp.pProtocolSubtypeElement )
                {
                    for( count = 0; count < *(pResp.pPersonalityListLength); count++ )
                    {
                        fprintf( fp, "Accessmac      : %d\n", pResp.pProtocolSubtypeElement[count].AccessMac );
                        fprintf( fp, "AuthPort       : %d\n", pResp.pProtocolSubtypeElement[count].AuthProt );
                        fprintf( fp, "Controlmac     : %d\n", pResp.pProtocolSubtypeElement[count].ControlMac );
                        fprintf( fp, "EncryptProt    : %d\n", pResp.pProtocolSubtypeElement[count].EncryptProt );
                        fprintf( fp, "ForwardMac     : %d\n", pResp.pProtocolSubtypeElement[count].ForwardMac );
                        fprintf( fp, "Idle State     : %d\n", pResp.pProtocolSubtypeElement[count].IdleState );
                        fprintf( fp, "KeyExchange    : %d\n", pResp.pProtocolSubtypeElement[count].KeyExchange );
                        fprintf( fp, "MultDisc       : %d\n", pResp.pProtocolSubtypeElement[count].MultDisc );
                        fprintf( fp, "Physical Layer : %d\n", pResp.pProtocolSubtypeElement[count].PhysicalLayer );
                        fprintf( fp, "Reversemac     : %d\n", pResp.pProtocolSubtypeElement[count].ReverseMac );
                        fprintf( fp, "Secprot        : %d\n", pResp.pProtocolSubtypeElement[count].SecProt );
                        fprintf( fp, "VirtStream     : %d\n", pResp.pProtocolSubtypeElement[count].VirtStream );
                    }
                }
            }
            tc++;
        }
    }
    if ( fp )
        tfclose( fp );
}

void doSLQSSwiGetHDRProtoSubtype( void )
{
    WORD                   CurrentPersonality;
    BYTE                   PersonalityListLength = 0x04;
    protocolSubtypeElement protSubTypeElmnt[4];
    ULONGLONG              appSubType;

    SLQSSwiGetHDRProtoSubtypeTestCase_t SLQSSwiGetHDRProtoSubtypeTestCase[] =
    {
        { eQCWWAN_ERR_NONE, 2, NULL, NULL, NULL, NULL, "No parameter fetched "},
        { eQCWWAN_ERR_NONE, 2, NULL, &PersonalityListLength, NULL, NULL, "personality List length fetched "},
        { eQCWWAN_ERR_NONE, 2, &CurrentPersonality, NULL, NULL, NULL, "Current personality fetched "},
        { eQCWWAN_ERR_NONE, 2, NULL, NULL, NULL, &appSubType, "AppSubtype fetched "},
        { eQCWWAN_ERR_NONE, 2, &CurrentPersonality, &PersonalityListLength, protSubTypeElmnt, &appSubType, "Valid "},
    };

    ULONG nRet;
    HDRProtSubtypResp pResp;
    FILE              *fp   = NULL;
    BYTE              count = 0;

    fp = tfopen( "../../nas/test/results/slqsswigethdrpersonality.txt", "w" );

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSSwiGetHDRProtoSubtypeTestCase )/
                      sizeof( SLQSSwiGetHDRProtoSubtypeTestCase[0] );

        while ( tc < ncases )
        {
            SLQSSwiGetHDRProtoSubtypeTestCase_t *pd =
                                     &SLQSSwiGetHDRProtoSubtypeTestCase[tc];

            if( NULL != pd->pCurrentPersonality )
                pResp.pCurrentPrsnlty = pd->pCurrentPersonality;
            else
                pResp.pCurrentPrsnlty = NULL;

            if( NULL != pd->pPersonalityListLength )
                pResp.pPersonalityListLength = pd->pPersonalityListLength;
            else
                pResp.pPersonalityListLength = NULL;

            if( NULL != pd->pProtSubTypeElmnt )
                pResp.pProtoSubTypElmnt= pd->pProtSubTypeElmnt;
            else
                pResp.pProtoSubTypElmnt= NULL;

            if( NULL != pd->pAppSubtype)
                pResp.pAppSubType = pd->pAppSubtype;
            else
                pResp.pAppSubType = NULL;

            fprintf( fp, "\nTest Case %d / %d : %s\n",
                        (tc + 1),
                        ncases,
                        pd->desc );

            nRet = SLQSSwiGetHDRProtSubtype( &pResp );

            /* Display result code and text */
            doprintreason ( fp, nRet );

            if ( !nRet )
            {
                fprintf( fp, "HDR Protosubtype Retrieved successfully\n" );

                IFPRINTF( fp, "Current Personality  : 0x%x\n", pResp.pCurrentPrsnlty );
                IFPRINTF( fp, "PersonalityListLength:0x%x\n", pResp.pPersonalityListLength );

                if( NULL != pResp.pPersonalityListLength &&
                    NULL != pResp.pProtoSubTypElmnt )
                {
                    for( count = 0; count < *(pResp.pPersonalityListLength); count++ )
                    {
                        fprintf( fp, "Accessmac      : %d\n", pResp.pProtoSubTypElmnt[count].AccessMac );
                        fprintf( fp, "AuthPort       : %d\n", pResp.pProtoSubTypElmnt[count].AuthProt );
                        fprintf( fp, "Controlmac     : %d\n", pResp.pProtoSubTypElmnt[count].ControlMac );
                        fprintf( fp, "EncryptProt    : %d\n", pResp.pProtoSubTypElmnt[count].EncryptProt );
                        fprintf( fp, "ForwardMac     : %d\n", pResp.pProtoSubTypElmnt[count].ForwardMac );
                        fprintf( fp, "Idle State     : %d\n", pResp.pProtoSubTypElmnt[count].IdleState );
                        fprintf( fp, "KeyExchange    : %d\n", pResp.pProtoSubTypElmnt[count].KeyExchange );
                        fprintf( fp, "MultDisc       : %d\n", pResp.pProtoSubTypElmnt[count].MultDisc );
                        fprintf( fp, "Physical Layer : %d\n", pResp.pProtoSubTypElmnt[count].PhysicalLayer );
                        fprintf( fp, "Reversemac     : %d\n", pResp.pProtoSubTypElmnt[count].ReverseMac );
                        fprintf( fp, "Secprot        : %d\n", pResp.pProtoSubTypElmnt[count].SecProt );
                        fprintf( fp, "VirtStream     : %d\n", pResp.pProtoSubTypElmnt[count].VirtStream );
                    }
                }

                IFPRINTF( fp, "App Subtype  : 0x%llx\n", pResp.pAppSubType );
            }
            tc++;
        }
    }
    if ( fp )
        tfclose( fp );
}

/*
 * Name:     doSLQSSwiPSDetach
 *
 * Purpose:  Perform the tests that call the SLQSSwiPSDetach() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSLQSSwiPSDetach( void )
{
    BYTE        PSDetach    = 2;
    BYTE        PSDetachInv = 3;

    SLQSSwiPSDetachTestCase_t SLQSSwiPSDetachTestCases[] =
    {
        { eQCWWAN_ERR_INVALID_ARG, 1, NULL, "NULL value" },
        { eQCWWAN_ERR_NONE, 2, &PSDetach,"Valid Detach"},
        { eQCWWAN_ERR_NONE, 3, &PSDetachInv, "Invalid Detach"},
    };

    ULONG nRet;
    PSDetachReq req;
    FILE  *fp = NULL;

    fp = tfopen( "../../nas/test/results/slqsswipsdetach.txt", "w" );

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSSwiPSDetachTestCases )/
                      sizeof( SLQSSwiPSDetachTestCases[0] );
        while ( tc < ncases )
        {
            SLQSSwiPSDetachTestCase_t *pd =
                                     &SLQSSwiPSDetachTestCases[tc];
            req.pDetachAction = pd->pDetachAction;

            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        (tc + 1),
                        ncases,
                        pd->desc);

            fprintf( fp, "Parameter Set:\n" );
            IFPRINTF( fp, "PS Detach Action : 0x%x\n", req.pDetachAction );

            nRet = SLQSSwiPSDetach( &req );

            /* Display result code and text */
            doprintreason ( fp, nRet );

            if ( !nRet )
            {
                fprintf( fp, "Immediate Packet Domain Detach Initiated\n" );
            }
            tc++;
        }
    }
    if ( fp )
        tfclose( fp );
}

void doSLQSGetErrorRate( void )
{
    WORD           CDMAFrameErrRate;
    WORD           HDRPackErrRate;
    BYTE           GSMBER;
    BYTE           WCDMABER;
    GetErrRateResp pResp;

    SLQSGetErrorRateTestCase_t SLQSGetErrorRateTestCases[] =
    {
        { eQCWWAN_ERR_NONE, 1, &pResp, "Valid Test Case" },
    };

    ULONG nRet;
    FILE  *fp   = NULL;

    fp = tfopen( "../../nas/test/results/slqsgeterrorrate.txt", "w" );

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSGetErrorRateTestCases )/
                      sizeof( SLQSGetErrorRateTestCases[0] );

        while ( tc < ncases )
        {
            SLQSGetErrorRateTestCase_t *pd =
                                     &SLQSGetErrorRateTestCases[tc];

            pResp.pCDMAFrameErrRate = &CDMAFrameErrRate;
            pResp.pHDRPackErrRate   = &HDRPackErrRate;
            pResp.pGSMBER           = &GSMBER;
            pResp.pWCDMABER         = &WCDMABER;

            fprintf( fp, "\nTest Case %d / %d : %s\n",
                        (tc + 1),
                        ncases,
                        pd->desc );

            nRet = SLQSGetErrorRate( &pResp );

            /* Display result code and text */
            doprintreason ( fp, nRet );

            if ( !nRet )
            {
                fprintf( fp, "Error Rate successfully retrieved\n" );

                IFPRINTF( fp, "CDMA Frame Error Rate : 0x%x\n", pResp.pCDMAFrameErrRate );
                IFPRINTF( fp, "HDR Packet Error Rate :0x%x\n", pResp.pHDRPackErrRate );
                IFPRINTF( fp, "GSM Bit Error Rate : 0x%x\n", pResp.pGSMBER );
                IFPRINTF( fp, "WCDMA Block Error Rate :0x%x\n", pResp.pWCDMABER );
            }
            tc++;
        }
    }
    if ( fp )
        tfclose( fp );
}

void doSLQSSwiGetHRPDStats( void )
{
    DRCParams      DRCParameters;
    BYTE           UATI[UATISIZE];
    PilotSetData   PSData;
    PilotSetParams PSParams[255];
    BYTE           numPilValid = 255;
    BYTE           numPilInValid = 0;
    GetHRPDStatsResp HRPDStats;
    BYTE           ctr = 0;

    SLQSSwiGetHRPDStatsTestCase_t SLQSSwiGetHRPDStatsTestCases[] =
    {
        { eQCWWAN_ERR_BUFFER_SZ, 1, numPilInValid,
          &HRPDStats, "InValid Test Case - Invalid number of Pilots" },

        { eQCWWAN_ERR_NONE, 2, numPilValid,
          &HRPDStats, "Valid Test Case " },
    };

    ULONG nRet;
    GetHRPDStatsResp *pResp;
    PilotSetData     *pPSDResp;
    FILE  *fp   = NULL;

    fp = tfopen( "../../nas/test/results/slqsswigethrpdstats.txt", "w" );

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSSwiGetHRPDStatsTestCases )/
                      sizeof( SLQSSwiGetHRPDStatsTestCases[0] );

        while ( tc < ncases )
        {
            SLQSSwiGetHRPDStatsTestCase_t *pd =
                                     &SLQSSwiGetHRPDStatsTestCases[tc];

            /* Assign memory to all parameters */
            pResp = pd->pGetHRPDStatsResp;
            pResp->pDRCParams = &DRCParameters;
            pResp->pUATI      = UATI;
            pResp->pPilotSetData = &PSData;
            pResp->pPilotSetData->pPilotSetInfo = PSParams;

            /* Assign input parameters */
            PSData.NumPilots = pd->numPilotSets;

            fprintf( fp, "\nTest Case %d / %d : %s\n",
                        (tc + 1),
                        ncases,
                        pd->desc );

            nRet = SLQSSwiGetHRPDStats( pResp );

            /* Display result code and text */
            doprintreason ( fp, nRet );

            if ( !nRet )
            {
                fprintf( fp, "HRPD Stats successfully retrieved\n" );
                DRCParams        *pDPResp;
                /* Display DRC*/
                pDPResp = pResp->pDRCParams;
                fprintf( fp, "DRC Parameters\n" );
                fprintf( fp, "DRC Value : 0x%x\n", pDPResp->DRCValue );
                fprintf( fp, "DRC Cover : 0x%x\n", pDPResp->DRCCover );

                /* Display UATI */
                fprintf( fp, "UATI: " );
                for( ctr = 0; ctr < UATISIZE; ctr++ )
                {
                    fprintf( fp, "%x ", pResp->pUATI[ctr] );
                }
                fprintf( fp, "\n" );

                /* Display Pilot Set */
                pPSDResp = pResp->pPilotSetData;
                fprintf( fp, "Pilot Set Data \n" );
                fprintf( fp, "Number of Pilot Sets: 0x%x \n", pPSDResp->NumPilots );

                for( ctr = 0; ctr < pPSDResp->NumPilots; ctr++ )
                {
                    PilotSetParams   *pPSPResp;
                    pPSPResp = &pPSDResp->pPilotSetInfo[ctr];
                    fprintf( fp, "Pilot Type: 0x%u \n", pPSPResp->PilotType );
                    fprintf( fp, "Pilot PN: 0x%d \n", pPSPResp->PilotPN );
                    fprintf( fp, "Pilot Strength: 0x%d \n", pPSPResp->PilotStrength );
                }
            }
            tc++;
        }
    }
    if ( fp )
        tfclose( fp );
}

void doSLQSSwiNetworkDebug( void )
{
    BYTE               ObjVer;
    NetworkStat1x      NetStat1x;
    NetworkStatEVDO    NetStatEVDO;
    DeviceConfigDetail DevCfgDetail;
    DataStatusDetail   DataStatDetail;

    ULONG              PilotSetCnt = 255;
    ActPilotPNElement  ActPilPNElmt[255];
    WORD               NeighPilSet[255];
    WORD               SectID[255];
    NetworkDebugResp   NwDbgStats;
    BYTE               ctr = 0;

    SLQSSwiNetworkDebugTestCase_t SLQSSwiNetworkDebugTestCases[] =
    {
        { eQCWWAN_ERR_NONE, 1, &NwDbgStats, "Valid Test Case" },
    };

    ULONG nRet;
    FILE  *fp   = NULL;

    fp = tfopen( "../../nas/test/results/slqsswinetworkdebug.txt", "w" );

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSSwiNetworkDebugTestCases )/
                      sizeof( SLQSSwiNetworkDebugTestCases[0] );

        while ( tc < ncases )
        {
            NetworkDebugResp *pResp;
            SLQSSwiNetworkDebugTestCase_t *pd =
                                     &SLQSSwiNetworkDebugTestCases[tc];

            /* Assign memory to all parameters */
            pResp = pd->pNetworkDebugResp;
            pResp->pObjectVer             = &ObjVer;

            NetStat1x.ActSetCnt           = PilotSetCnt;
            NetStat1x.pActPilotPNElements = ActPilPNElmt;
            NetStat1x.NeighborSetCnt      = PilotSetCnt;
            NetStat1x.pNeighborSetPilotPN = NeighPilSet;
            pResp->pNetworkStat1x         = &NetStat1x;

            NetStatEVDO.pSectorID  = SectID;
            pResp->pNetworkStatEVDO = &NetStatEVDO;

            pResp->pDeviceConfigDetail = &DevCfgDetail;
            pResp->pDataStatusDetail   = &DataStatDetail;


            fprintf( fp, "\nTest Case %d / %d : %s\n",
                        (tc + 1),
                        ncases,
                        pd->desc );

            nRet = SLQSSwiNetworkDebug( pResp );

            /* Display result code and text */
            doprintreason ( fp, nRet );

            if ( !nRet )
            {
                fprintf( fp, "Network successfully debugged\n" );

                /* Display Object version */
                IFPRINTF( fp, "Object Version : 0x%x\n", pResp->pObjectVer );

                /* Display 1x Network details*/
                if( NULL != pResp->pNetworkStat1x )
                {
                    fprintf( fp, "1x Network Status\n" );
                    fprintf( fp, "State   : 0x%x\n", NetStat1x.State );
                    fprintf( fp, "SO      : 0x%x\n", NetStat1x.SO );
                    fprintf( fp, "RX Pwr  : 0x%x\n", NetStat1x.RX_PWR );
                    fprintf( fp, "RX ECIO : 0x%x\n", NetStat1x.RX_EC_IO );
                    fprintf( fp, "TX Pwr  : 0x%x\n", NetStat1x.TX_PWR );

                    fprintf( fp, "Active Set Pilot PN Elements\n" );
                    fprintf( fp,
                             "Active Set Pilot Count : 0x%x\n",
                             NetStat1x.ActSetCnt );
                    for( ctr = 0; ctr < NetStat1x.ActSetCnt; ctr++ )
                    {
                        fprintf( fp,
                                 "Active Set Pilot PN : 0x%x\n",
                                 ActPilPNElmt[ctr].ActSetPilotPN );
                        fprintf( fp,
                                 "Active Set Pilot Strength  : 0x%x\n",
                                 ActPilPNElmt[ctr].ActSetPilotPNStrength );
                    }

                    fprintf( fp,
                             "Neighbouring Set Pilot Count : 0x%x\n",
                             NetStat1x.NeighborSetCnt );
                    for( ctr = 0; ctr < NetStat1x.NeighborSetCnt; ctr++ )
                    {
                        fprintf( fp,
                                 "Neighbouring Set Pilot PN : 0x%x\n",
                                 NetStat1x.pNeighborSetPilotPN[ctr] );
                    }
                }

                /* Display EVDO network details */
                if( NULL != pResp->pNetworkStatEVDO )
                {
                    fprintf( fp, "EVDO network status:\n" );
                    fprintf( fp, "State   : 0x%x\n", NetStatEVDO.State );
                    fprintf( fp, "SO      : 0x%x\n", NetStatEVDO.MACIndex );
                    fprintf( fp,
                             "Sector ID Length : 0x%x\n",
                             NetStatEVDO.SectorIDLen );
                    fprintf( fp, "Sector ID: " );
                    for( ctr = 0; ctr < NetStatEVDO.SectorIDLen; ctr++ )
                    {
                        fprintf( fp, "0x%x ", NetStatEVDO.pSectorID[ctr] );
                    }
                    fprintf( fp, "\n" );
                    fprintf( fp, "RX PWR       : 0x%x\n", NetStatEVDO.RX_PWR );
                    fprintf( fp, "PER          : 0x%x\n", NetStatEVDO.PER );
                    fprintf( fp, "Pilot Energy : 0x%x\n", NetStatEVDO.PilotEnergy );
                    fprintf( fp, "SNR          : 0x%x\n", NetStatEVDO.SNR );
                }
                /* Display device configuration details */
                fprintf( fp, "Device Configuration details:\n" );
                if( NULL != pResp->pDeviceConfigDetail )
                {
                    fprintf( fp, "Technology   : 0x%x\n", DevCfgDetail.Technology );
                    fprintf( fp, "QLIC         : 0x%x\n", DevCfgDetail.QLIC );
                    fprintf( fp, "Chipset      : 0x%x\n", DevCfgDetail.Chipset );
                    fprintf( fp, "Hw Version   : 0x%x\n", DevCfgDetail.HWVersion );
                }

                /* Data status details */
                if( NULL != pResp->pDataStatusDetail )
                {
                    fprintf( fp, "Data status details:\n" );
                    fprintf( fp, "IP address      : 0x%x\n", DataStatDetail.IPAddress );
                    fprintf( fp, "Last Error Code : 0x%x\n", DataStatDetail.LastErrCode );
                }
            }
            tc++;
        }
    }
    if ( fp )
        tfclose( fp );
}

void doSLQSSwiGetLteCQI( void )
{

	LteCQIParm templteCQIParm;

    SLQSSwiGetLteCQITestCase_t SLQSSwiGetLteCQITestCase[] =
    {
        { eQCWWAN_ERR_NONE, 1, NULL, "Invalid "},
        { eQCWWAN_ERR_NONE, 2, &templteCQIParm, "Valid "},
    };

    ULONG      nRet;
    FILE       *fp   = NULL;

    fp = tfopen( "../../nas/test/results/slqsswigetltecqi.txt", "w" );

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSSwiGetLteCQITestCase )/
                      sizeof( SLQSSwiGetLteCQITestCase[0] );

        while ( tc < ncases )
        {
            LteCQIParm *pResp;
            SLQSSwiGetLteCQITestCase_t *pd =
                                     &SLQSSwiGetLteCQITestCase[tc];

            fprintf( fp, "\nTest Case %d / %d : %s\n",
                        (tc + 1),
                        ncases,
                        pd->desc );

            pResp = pd->pLteCQIParm;
            nRet = SLQSSwiGetLteCQI( pResp );

            /* Display result code and text */
            doprintreason ( fp, nRet );

            if ( !nRet )
            {
                fprintf( fp, "LTE CQI Retrieved successfully\n" );
                fprintf( fp, "Validity of cw0  : 0x%x\n", pResp->ValidityCW0 );
                fprintf( fp, "CQI value of cw0 : 0x%x\n", pResp->CQIValueCW0 );
                fprintf( fp, "Validity of cw1  : 0x%x\n", pResp->ValidityCW1 );
                fprintf( fp, "CQI value of cw1 : 0x%x\n", pResp->CQIValueCW1 );
            }
            tc++;
        }
    }
    if ( fp )
        tfclose( fp );
}

/*
 * Name:     doSLQSConfigSigInfo
 *
 * Purpose:  Perform the tests that call the SLQSConfigSigInfo() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSLQSConfigSigInfo( void )
{
    INT8          rssiThresList[3] = { 0x96, 0xAA, 0xB4 };
    BYTE          rssiThresListlen = 0x03;
    RSSIThresh    rssiThresh;

    SHORT         ecioThresList[3] = { 0x02, 0x10, 0x18 }; /* -1dB, -8dB, -12dB */
    BYTE          ecioThresListLen = 0x03;
    ECIOThresh    ecioThresh;

    BYTE          hdrsinrThresList[3] = { 0x02, 0x10, 0x1A };
    BYTE          hdrsinrThresListLen = 3;
    HDRSINRThresh hdrsinrThresh;

    SHORT        ltesnrThresList[3] = { -2, 24, 26 }; /* -2dB, 24dB, 26dB */
    BYTE         ltesnrThresListLen = 3;
    LTESNRThresh ltesnrThresh;

    INT32        ioThresList[3] = { -10, 6, 14 };
    BYTE         ioThresListLen = 3;
    IOThresh     ioThresh;

    INT8         rsrqThresList[1] = { -11 };
    BYTE         rsrqThresListLen = 1;
    RSRQThresh   rsrqThresh;

    SHORT        rsrpThresList[3] = { 150, 200 };
    BYTE         rsrpThresListlen = 3;
    RSRPThresh   rsrpThresh;

    BYTE         rptRate = 2;
    BYTE         avgPeriod = 2;
    LTESigRptCfg lteSigRptCfg;


    BYTE         TDSCDMASINRCONFThreshListLen = 3;
    FLOAT       TDSCDMASINRCONFThreshList[3]={-1.1,-2.1,-5.99};
    TDSCDMASINRCONFThresh tDSCDMASINRCONFThresh;
    BYTE lCtr = 0;

    SLQSConfigSigInfoTestCase_t SLQSConfigSigInfoTestCases[] =
    {
        { eQCWWAN_ERR_NONE, 1, &rssiThresh, &ecioThresh, &hdrsinrThresh,
          &ltesnrThresh, &ioThresh, &rsrqThresh, &rsrpThresh, &lteSigRptCfg,&tDSCDMASINRCONFThresh,
          "All parameters not NULL"},
        { eQCWWAN_ERR_NONE, 2, &rssiThresh, NULL, NULL,
          NULL, NULL, NULL, NULL, NULL,NULL, "Only RSSI threshold"},
        { eQCWWAN_ERR_NONE, 3, NULL, &ecioThresh, NULL,
          NULL, NULL, NULL, NULL, NULL,NULL, "Only ECIO threshold"},
        { eQCWWAN_ERR_NONE, 4, NULL, NULL, &hdrsinrThresh,
          NULL, NULL, NULL, NULL, NULL,NULL, "Only HDR SINR threshold"},
        { eQCWWAN_ERR_NONE, 5, NULL, NULL, NULL,
          &ltesnrThresh, NULL, NULL, NULL, NULL,NULL, "Only LTE SNR threshold"},
        { eQCWWAN_ERR_NONE, 6, NULL, NULL, NULL,
          NULL, &ioThresh, NULL, NULL, NULL,NULL, "Only IO threshold"},
        { eQCWWAN_ERR_NONE, 7, NULL, NULL, NULL,
          NULL, NULL, &rsrqThresh, NULL, NULL,NULL, "Only RSRQ threshold"},
        { eQCWWAN_ERR_NONE, 8, NULL, NULL, NULL,
          NULL, NULL, NULL, &rsrpThresh, NULL,NULL, "Only RSRP threshold"},
        { eQCWWAN_ERR_NONE, 9, NULL, NULL, NULL,
          NULL, NULL, NULL, NULL, &lteSigRptCfg,NULL,
          "Only LTE Signal report configuration"},
        { eQCWWAN_ERR_NONE, 10, NULL, NULL, NULL,
          &ltesnrThresh, NULL, NULL, NULL, &lteSigRptCfg,NULL,
          "LTE SNR Threshold and LTE Signal report configuration"},
        { eQCWWAN_ERR_NONE, 11, NULL, NULL, NULL,
          NULL, NULL, NULL, NULL, NULL,&tDSCDMASINRCONFThresh,
          "Only TD-SCDMA SINR threshold"},
    };

    ULONG nRet;
    sigInfo req;
    FILE  *fp = NULL;

    fp = tfopen( "../../nas/test/results/slqsconfigsiginfo.txt", "w" );

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSConfigSigInfoTestCases )/
                      sizeof( SLQSConfigSigInfoTestCases[0] );
        while ( tc < ncases )
        {
            SLQSConfigSigInfoTestCase_t *pd =
                                     &SLQSConfigSigInfoTestCases[tc];

            /* Assign request parameters */
            rssiThresh.RSSIThresListLen = rssiThresListlen;
            rssiThresh.pRSSIThresList   = rssiThresList;
            req.pRSSIThresh             = pd->pRSSIThresh;

            ecioThresh.ECIOThresListLen = ecioThresListLen;
            ecioThresh.pECIOThresList   = ecioThresList;
            req.pECIOThresh             = pd->pECIOThresh;

            hdrsinrThresh.HDRSINRThresListLen = hdrsinrThresListLen;
            hdrsinrThresh.pHDRSINRThresList   = hdrsinrThresList;
            req.pHDRSINRThresh                = pd->pHDRSINRThresh;

            ltesnrThresh.LTESNRThresListLen = ltesnrThresListLen;
            ltesnrThresh.pLTESNRThresList   = ltesnrThresList;
            req.pLTESNRThresh               = pd->pLTESNRThresh;

            ioThresh.IOThresListLen = ioThresListLen;
            ioThresh.pIOThresList   = ioThresList;
            req.pIOThresh           = pd->pIOThresh;

            rsrqThresh.RSRQThresListLen = rsrqThresListLen;
            rsrqThresh.pRSRQThresList   = rsrqThresList;
            req.pRSRQThresh             = pd->pRSRQThresh;

            rsrpThresh.RSRPThresListLen = rsrpThresListlen;
            rsrpThresh.pRSRPThresList   = rsrpThresList;
            req.pRSRPThresh             = pd->pRSRPThresh;

            lteSigRptCfg.rptRate   = rptRate;
            lteSigRptCfg.avgPeriod = avgPeriod;
            req.pLTESigRptCfg = pd->pLTESigRptCfg;

            tDSCDMASINRCONFThresh.TDSCDMASINRCONFThreshListLen = TDSCDMASINRCONFThreshListLen;
            tDSCDMASINRCONFThresh.pTDSCDMASINRCONFThreshList = TDSCDMASINRCONFThreshList;
            req.pTDSCDMASINRCONFThresh = pd->pTDSCDMASINRCONFThresh;

            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        (tc + 1),
                        ncases,
                        pd->desc);

            fprintf( fp, "Parameter Set:\n" );
            if(pd->pRSSIThresh)
            {
                fprintf( fp, "\nRSSI Threshold list\n");
                fprintf( fp, "RSSI Threshold list length: %d\n", rssiThresh.RSSIThresListLen );
                for( lCtr=0; lCtr < rssiThresh.RSSIThresListLen; lCtr++ )
                {
                    fprintf( fp, "RSSI Threshold [0x%x]: %d\n", lCtr, rssiThresList[lCtr] );
                }
            }
            if(pd->pECIOThresh)
            {
                fprintf( fp, "\nECIO Threshold list\n");
                fprintf( fp, "ECIO Threshold list length: %d\n", ecioThresh.ECIOThresListLen );
                for( lCtr=0; lCtr < ecioThresh.ECIOThresListLen; lCtr++ )
                {
                    fprintf( fp, "ECIO Threshold [%d]: %d\n", lCtr, ecioThresList[lCtr] );
                }
            }
            if(pd->pHDRSINRThresh)
            {
                fprintf( fp, "\nHDR SINR Threshold list\n");
                fprintf( fp, "HDR SINR Threshold list length: %d\n", hdrsinrThresh.HDRSINRThresListLen );
                for( lCtr=0; lCtr < hdrsinrThresh.HDRSINRThresListLen; lCtr++ )
                {
                    fprintf( fp, "HDR SINR Threshold [0x%x]: %d\n", lCtr, hdrsinrThresList[lCtr] );
                }
            }
            if(pd->pLTESNRThresh)
            {
                fprintf( fp, "\nLTE SNR Threshold list\n");
                fprintf( fp, "LTE SNR Threshold list length: %d\n", ltesnrThresh.LTESNRThresListLen );
                for( lCtr=0; lCtr < ltesnrThresh.LTESNRThresListLen; lCtr++ )
                {
                    fprintf( fp, "LTE SNR Threshold [0x%x]: %d\n", lCtr, ltesnrThresList[lCtr] );
                }
            }
            if(pd->pIOThresh)
            {
                fprintf( fp, "\nIO Threshold list\n");
                fprintf( fp, "IO Threshold list length: %d\n", ioThresh.IOThresListLen );
                for( lCtr=0; lCtr < ioThresh.IOThresListLen; lCtr++ )
                {
                    fprintf( fp, "IO Threshold [%d]: %d\n", lCtr, ioThresList[lCtr] );
                }
            }
            if(pd->pRSRQThresh)
            {
                fprintf( fp, "\nRSRQ Threshold list\n");
                fprintf( fp, "RSRQ Threshold list length: %d\n", rsrqThresh.RSRQThresListLen );
                for( lCtr=0; lCtr < rsrqThresh.RSRQThresListLen; lCtr++ )
                {
                    fprintf( fp, "RSRQ Threshold [%d]: %d\n", lCtr, rsrqThresList[lCtr] );
                }
            }
            if(pd->pRSRPThresh)
            {
                fprintf( fp, "\nRSRP Threshold list\n");
                fprintf( fp, "RSRP Threshold list length: %d\n", rsrpThresh.RSRPThresListLen );
                for( lCtr=0; lCtr < rsrpThresh.RSRPThresListLen; lCtr++ )
                {
                    fprintf( fp, "RSRP Threshold [%d]: %d\n", lCtr, rsrpThresList[lCtr] );
                }
            }
            if(pd->pLTESigRptCfg)
            {
                fprintf( fp, "\nLTE signal report configuration\n");
                fprintf( fp, "Report Rate %d\n", lteSigRptCfg.rptRate );
                fprintf( fp, "Averaging period %d\n", lteSigRptCfg.avgPeriod );
            }

            if(pd->pTDSCDMASINRCONFThresh)
            {
                fprintf( fp, "\nTD-SCDMA SINR Threshold list\n");
                fprintf( fp, "TD-SCDMA SINR Threshold list length: %d\n", tDSCDMASINRCONFThresh.TDSCDMASINRCONFThreshListLen);
                for( lCtr=0; lCtr < tDSCDMASINRCONFThresh.TDSCDMASINRCONFThreshListLen; lCtr++ )
                {
                    fprintf( fp, "TD-SCDMA SINR Threshold [0x%x]: %f\n", lCtr, TDSCDMASINRCONFThreshList[lCtr] );
                }
            }
            nRet = SLQSConfigSigInfo( &req );

            /* Display result code and text */
            doprintreason ( fp, nRet );

            if ( !nRet )
            {
                fprintf( fp, "Sig Info configured\n" );
            }
            tc++;
        }
    }
    if ( fp )
        tfclose( fp );
}

/*
 * Name:     SLQSNasSwiIndicationRegister
 *
 * Purpose:  Perform the tests that call the SLQSNasSwiIndicationRegister() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSLQSNasSwiIndicationRegister( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    FILE  *fp      = NULL;
    NasSwiIndReg req;

    fp = tfopen("TestReport/slqsNasSwiIndicationRegister.txt", "a");

    if (fp)
    {
        while ( tCaseNum < MAX_SLQS_SWI_INDICATION_REGISTER_TESTCASE_NUM )
        {
            const SLQSSwiIndicationRegisterTestCase_t *pParams =
                                &(SLQSSwiIndicationRegisterTestCases[tCaseNum]);

            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum+1,
                       MAX_SLQS_SWI_INDICATION_REGISTER_TESTCASE_NUM,
                       SLQSSwiIndicationRegisterTestCases[tCaseNum].desc );

            req.lteEsmUl = pParams->lteEsmUl;
            req.lteEsmDl = pParams->lteEsmDl;
            req.lteEmmUl = pParams->lteEmmUl;
            req.lteEmmDl = pParams->lteEmmDl;
            req.gsmUmtsUl= pParams->gsmUmtsUl;
            req.gsmUmtsDl = pParams->gsmUmtsDl;


            nRet = SLQSNasSwiIndicationRegister(  &req );

            fprintf(fp, "Return Code: %u\n", nRet);
            doprintreason( fp, nRet );

            if (!nRet)
            {
                fprintf(fp, "NAS Swi Indication Register Success\n");
            }
            tCaseNum++;
        }
    }
    if (fp)
    {
        tfclose(fp);
    }
}

/*
 * Name:     convertOctetTo7Bit
 *
 * Purpose:  Converts a byte stream to 7-bit ASCII or 7-Bit GSM format.
 *
 * Return:   none
 *
 * Notes:    none
 *
 */

void convertOctetTo7Bit(BYTE *pNw2Name, BYTE nw2DescLen, BYTE *pNw2NameOut)
{
    BYTE i;
    BYTE mask = 0x7F;
    BYTE msbData;
    BYTE prevBits = 0;
    BYTE ascii7bit = 0;
    BYTE newLen =0;

    for(i=0; i<nw2DescLen; i++)
    {
        msbData= pNw2Name[i] >> (7-(i%7));   /*extracted MSBs to add as LSBs in next received Byte*/
        ascii7bit = pNw2Name[i] << (i%7);    /*left shift data according to number of byte (0th byte */
        ascii7bit = ascii7bit | prevBits;    /*logical OR of data with previous byte reserved bits*/
        ascii7bit = ascii7bit & mask;        /* mask MSB */
        pNw2NameOut[newLen++] = ascii7bit;
        prevBits = msbData;                /*Preserve data for the next byte in another variable*/

        /*If 7 bits are extracted from the previous byte*/
        if(i>0 && ((i+1)%7)==0)
        {
            pNw2NameOut[newLen++] = prevBits;
          prevBits = 0;
        }
    }
    pNw2NameOut[newLen] = 0;  /*null character end of string*/
}

/*
 * Name:     doGetHomeNetwork3GPP2
 *
 * Purpose:  Perform the tests that call the GetHomeNetwork3GPP2() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */

void doGetHomeNetwork3GPP2( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    WORD  pMCC;
    WORD  pMNC;
    BYTE  nameSize;
    CHAR  pName[nMaxStrLen];
    WORD  pSID;
    WORD  pNID;
    WORD  pNw2MCC;
    WORD  pNw2MNC;
    BYTE  pNw2DescDisp;
    BYTE  pNw2DescEnc;
    BYTE  nw2DescLen;
    BYTE  pNw2Name[nMaxStrLen];

    FILE  *fp = NULL;
    BYTE  nw2NameOut[nMaxStrLen];
    BYTE i;

    fp = tfopen("../../nas/test/results/gethomenetwork3GPP2.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_GET_HOME_NETWORK_3GPP2_TESTCASE_NUM)
        {
            nameSize    = GetHomeNetwork3GPP2TestCases[tCaseNum].nameSize;
            nw2DescLen  = GetHomeNetwork3GPP2TestCases[tCaseNum].nw2DescLen;


            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_GET_HOME_NETWORK_3GPP2_TESTCASE_NUM,
                       GetHomeNetwork3GPP2TestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "nameSize :%d\n", nameSize);

            nRet = GetHomeNetwork3GPP2(&pMCC,
                                       &pMNC,
                                       nameSize,
                                       &pName[0],
                                       &pSID,
                                       &pNID,
                                       &pNw2MCC,
                                       &pNw2MNC,
                                       &pNw2DescDisp,
                                       &pNw2DescEnc,
                                       &nw2DescLen,
                                       &pNw2Name[0]);

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                BYTE  *pNw2NameOut;
                fprintf(fp, "pMCC         : %d\n", pMCC);
                fprintf(fp, "pMNC         : %d\n", pMNC);
                fprintf(fp, "pName        : %s\n", pName);
                fprintf(fp, "pSID         : %d\n", pSID);
                fprintf(fp, "pNID         : %d\n", pNID);
                fprintf(fp, "pNw2MCC      : %d\n", pNw2MCC);
                fprintf(fp, "pNw2MNC      : %d\n", pNw2MNC);
                fprintf(fp, "pNw2DescDisp : %d\n", pNw2DescDisp);
                fprintf(fp, "pNw2DescEnc  : %d\n", pNw2DescEnc);
                switch(pNw2DescEnc)
                {
                    case OCTET:
                        fprintf(fp, "pNw2Name     : %s\n", pNw2Name);
                        break;

                    case ASCII7BIT:
                        convertOctetTo7Bit(pNw2Name,nw2DescLen,nw2NameOut);
                        fprintf(fp, "pNw2Name     : %s\n", nw2NameOut);
                        break;

                    case UNICODE:
                        fprintf(fp, "pNw2Name     :");
                        for(i=0; i<nw2DescLen; i=i+2)
                        {
                            WORD *pUnicode;
                            pUnicode =(WORD*)&pNw2Name[i];
                           fprintf(fp, "%lc",*pUnicode);
                        }
                        fprintf(fp, "\n");
                        break;

                    case GSM7BIT:
                        convertOctetTo7Bit(pNw2Name,nw2DescLen,nw2NameOut);
                        pNw2NameOut = &nw2NameOut[0];
                        fprintf(fp, "pNw2Name     :");
                        /* Print only alphabets & digits. Rest of the characters are
                         *  printed as '*' */
                        while(*pNw2NameOut!=0)
                        {
                            if((*pNw2NameOut > 0x2F &&  *pNw2NameOut < 0x5B) ||
                                (*pNw2NameOut > 0x60 &&  *pNw2NameOut < 0x7B))
                                fprintf(fp, "%c", *pNw2NameOut);
                            else
                                fprintf(fp, "*" );
                            pNw2NameOut++;
                        }
                        fprintf(fp, "\n");
                        break;

                    default:
                        break;
                }
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doSLQSConfigSigInfo2
 *
 * Purpose:  Perform the tests that call the SLQSNasConfigSigInfo2() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSLQSConfigSigInfo2( void )
{
    WORD              cdmaRssiThreshList[] = { -1020,-400 };/* -102dB, -40dB */
    BYTE              cdmaRssiThreshListlen = 0x02;
    CDMARSSIThresh    cdmaRssiThresh;

    WORD              cdmaEcioThreshList[] = { -400, -310 }; /* -20dB, -15.5dB */
    BYTE              cdmaEcioThreshListLen = 0x02;
    CDMAECIOThresh    cdmaEcioThresh;

    WORD              hdrRssiThreshList[2] = { -500, -150 }; /* -50dB, -15dB */
    BYTE              hdrRssiThresListLen = 2;
    HDRRSSIThresh     hdrRssiThresh;

    WORD              hdrEcioThreshList[2] = { -400, -310 }; /* -20dB, -15.5dB */
    BYTE              hdrEcioThresListLen = 2;
    HDRECIOThresh     hdrEcioThresh;

    WORD              hdrSinrThreshList[2] = { 0x01, 0x03 }; /* -6dB, -3dB */
    BYTE              hdrSinrThresListLen = 2;
    HDRSINRThreshold  hdrSinrThresh;

    WORD              hdrIoThreshList[2] = { -1110, -730 }; /* -110dB, -73dB */
    BYTE              hdrIoThresListLen = 2;
    HDRIOThresh       hdrIoThresh;

    WORD              gsmRssiThreshList[2] = { -950, -800 }; /* -95dB, -80dB */
    BYTE              gsmRssiThresListLen = 2;
    GSMRSSIThresh     gsmRssiThresh;

    WORD              wcdmaRssiThreshList[2] = { -1000, -200 }; /* -100dB, -20dB */
    BYTE              wcdmaRssiThresListLen = 2;
    WCDMARSSIThresh   wcdmaRssiThresh;

    WORD              wcdmaEcioThreshList[2] = { -400, -310 }; /* -20dB, -15.5dB */
    BYTE              wcdmaEcioThresListLen = 2;
    WCDMAECIOThresh   wcdmaEcioThresh;

    WORD              lteRssiThreshList[2] = { -1000, -400 }; /* -100dB, -40dB */
    BYTE              lteRssiThresListLen = 2;
    LTERSSIThresh     lteRssiThresh;

    WORD              lteSnrThreshList[2] = { -198, -230 }; /* -19.8dB, 23dB */
    BYTE              lteSnrThresListLen = 2;
    LTESNRThreshold   lteSnrThresh;

    WORD              lteRsrqThreshList[2] = { -110, -60 }; /* -11dB, -6dB */
    BYTE              lteRsrqThresListLen = 2;
    LTERSRQThresh     lteRsrqThresh;

    WORD              lteRsrpThreshList[2] = { -1250, -640 }; /* -125dB, -64dB */
    BYTE              lteRsrpThresListLen = 2;
    LTERSRPThresh     lteRsrpThresh;

    WORD              tdscdmaRscpThreshList[2] = { -950, -800 }; /* -95dB, -80dB */
    BYTE              tdscdmaRscpThresListLen = 2;
    TDSCDMARSCPThresh tdscdmaRscpThresh;

    float             tdscdmaRssiThreshList[2] = { -950, -800 }; /* -95dB, -80dB */
    BYTE              tdscdmaRssiThresListLen = 2;
    TDSCDMARSSIThresh tdscdmaRssiThresh;

    float             tdscdmaEcioThreshList[2] = { -400, -310 }; /* -20dB, -15.5dB */
    BYTE              tdscdmaEcioThresListLen = 2;
    TDSCDMAECIOThresh tdscdmaEcioThresh;
    float             tdscdmaSinrThreshList[2] = { 0x01, 0x03 };
    BYTE              tdscdmaSinrThresListLen = 2;
    TDSCDMASINRThresh tdscdmaSinrThresh;

    BYTE                rptRate = 2;
    BYTE                avgPeriod = 2;
    LTESigRptConfig     lteSigRptCfg;

    WORD              CDMARSSIDelta    = 100; /*10dB*/
    WORD              CDMAECIODelta    = 100;
    WORD              HDRRSSIDelta     = 100;
    WORD              HDRECIODelta     = 100;
    WORD              HDRSINRDelta     = 100;
    WORD              HDRIODelta       = 100;
    WORD              GSMRSSIDelta     = 100;
    WORD              WCDMARSSIDelta   = 100;
    WORD              WCDMAECIODelta   = 100;
    WORD              LTERSSIDelta     = 100;
    WORD              LTESNRDelta      = 100;
    WORD              LTERSRQDelta     = 100;
    WORD              LTERSRPDelta     = 100;
    WORD              TDSCDMARSCPDelta = 100;
    ULONG             TDSCDMARSSIDelta = 100;
    ULONG             TDSCDMAECIODelta = 100;
    ULONG             TDSCDMASINRDelta = 100;

    SLQSConfigSigInfo2TestCase_t SLQSConfigSigInfo2TestCases[] =
    {
        { eQCWWAN_ERR_NONE, 1, &cdmaRssiThresh,&CDMARSSIDelta, &cdmaEcioThresh, &CDMAECIODelta, &hdrRssiThresh, &HDRRSSIDelta,
          &hdrEcioThresh, &HDRECIODelta, &hdrSinrThresh,&HDRSINRDelta, &hdrIoThresh, &HDRIODelta, &gsmRssiThresh, &GSMRSSIDelta,
          &wcdmaRssiThresh, &WCDMARSSIDelta,&wcdmaEcioThresh, &WCDMAECIODelta, &lteRssiThresh, &LTERSSIDelta, &lteSnrThresh, &LTESNRDelta,
          &lteRsrqThresh, &LTERSRQDelta, &lteRsrpThresh, &LTERSRPDelta, &lteSigRptCfg, &tdscdmaRscpThresh, &TDSCDMARSCPDelta, &tdscdmaRssiThresh,
          &TDSCDMARSSIDelta, &tdscdmaEcioThresh, &TDSCDMAECIODelta, &tdscdmaSinrThresh , &TDSCDMASINRDelta,
          "All parameters not NULL"},
       { eQCWWAN_ERR_NONE, 2, &cdmaRssiThresh,NULL, NULL, NULL, NULL, NULL,
            NULL, NULL, NULL,NULL, NULL, NULL, NULL, NULL,
            NULL, NULL,NULL, NULL, NULL, NULL, NULL, NULL,
            NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
            NULL, NULL, NULL, NULL , NULL,
            "CDMA RSSI Threshold Config"},
       { eQCWWAN_ERR_NONE, 3, NULL,&CDMARSSIDelta, NULL, NULL, NULL, NULL,
            NULL, NULL, NULL,NULL, NULL, NULL, NULL, NULL,
            NULL, NULL,NULL, NULL, NULL, NULL, NULL, NULL,
            NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
            NULL, NULL, NULL, NULL , NULL,
            "CDMA RSSI Delta Config"},
       { eQCWWAN_ERR_NONE, 4, NULL,NULL, &cdmaEcioThresh, NULL, NULL, NULL,
            NULL, NULL, NULL,NULL, NULL, NULL, NULL, NULL,
            NULL, NULL,NULL, NULL, NULL, NULL, NULL, NULL,
            NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
            NULL, NULL, NULL, NULL , NULL,
            "CDMA ECIO Threshold Config"},
       { eQCWWAN_ERR_NONE, 5, NULL,NULL, NULL, NULL, &hdrRssiThresh, NULL,
            NULL, NULL, NULL,NULL, NULL, NULL, NULL, NULL,
            NULL, NULL,NULL, NULL, NULL, NULL, NULL, NULL,
            NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
            NULL, NULL, NULL, NULL , NULL,
            "HDR RSSI Threshold Config"},
       { eQCWWAN_ERR_NONE, 6, NULL,NULL, NULL, NULL, NULL, NULL,
            NULL, NULL, NULL,NULL, NULL, NULL, &gsmRssiThresh, NULL,
            NULL, NULL,NULL, NULL, NULL, NULL, NULL, NULL,
            NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
            NULL, NULL, NULL, NULL , NULL,
            "GSM RSSI Threshold Config"},
       { eQCWWAN_ERR_NONE, 7, NULL,NULL, NULL, NULL, NULL, NULL,
            NULL, NULL, NULL,NULL, NULL, NULL, NULL, NULL,
            NULL, NULL,NULL, NULL, &lteRssiThresh, NULL, NULL, NULL,
            NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
            NULL, NULL, NULL, NULL , NULL,
            "LTE RSSI Threshold Config"},
       { eQCWWAN_ERR_NONE, 8, NULL,NULL, NULL, NULL, NULL, NULL,
            NULL, NULL, NULL,NULL, NULL, NULL, NULL, NULL,
            NULL, NULL,NULL, NULL, NULL, NULL, NULL, NULL,
            NULL, NULL, NULL, NULL, NULL, NULL, NULL, &tdscdmaRssiThresh,
            NULL, NULL, NULL, NULL , NULL,
             "TD SCDMA RSSI Threshold Config"},
    };

    FILE  *fp = NULL;

    fp = tfopen( "../../nas/test/results/slqsconfigsiginfo2.txt", "w" );

    if (fp)
    {
        BYTE lCtr = 0;
        ULONG nRet;
        setSignalStrengthInfo req;
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSConfigSigInfo2TestCases )/
                      sizeof( SLQSConfigSigInfo2TestCases[0] );
        while ( tc < ncases )
        {
            SLQSConfigSigInfo2TestCase_t *pd =
                                     &SLQSConfigSigInfo2TestCases[tc];

            req.pCDMARSSIDelta  = pd->pCDMARSSIDelta;
            req.pCDMAECIODelta  = pd->pCDMAECIODelta;
            req.pHDRRSSIDelta   = pd->pHDRRSSIDelta;
            req.pHDRECIODelta   = pd->pHDRECIODelta;
            req.pHDRSINRDelta   = pd->pHDRSINRDelta;
            req.pHDRIODelta     = pd->pHDRIODelta;
            req.pGSMRSSIDelta   = pd->pGSMRSSIDelta;
            req.pWCDMARSSIDelta =  pd->pWCDMARSSIDelta;
            req.pWCDMAECIODelta = pd->pWCDMAECIODelta;
            req.pLTERSSIDelta   = pd->pLTERSSIDelta;
            req.pLTESNRDelta    = pd->pLTESNRDelta;
            req.pLTERSRQDelta   = pd->pLTERSRQDelta;
            req.pLTERSRPDelta   = pd->pLTERSRPDelta;
            req.pTDSCDMARSCPDelta = pd->pTDSCDMARSCPDelta;
            req.pTDSCDMARSSIDelta = pd->pTDSCDMARSSIDelta;
            req.pTDSCDMAECIODelta = pd->pTDSCDMAECIODelta;
            req.pTDSCDMASINRDelta = pd->pTDSCDMASINRDelta;

            /* Assign request parameters */
            cdmaRssiThresh.CDMARSSIThreshListLen = cdmaRssiThreshListlen;
            cdmaRssiThresh.pCDMARSSIThreshList   = cdmaRssiThreshList;
            req.pCDMARSSIThresh                  = pd->pCDMARSSIThresh;

            cdmaEcioThresh.CDMAECIOThreshListLen = cdmaEcioThreshListLen;
            cdmaEcioThresh.pCDMAECIOThreshList   = cdmaEcioThreshList;
            req.pCDMAECIOThresh                  = pd->pCDMAECIOThresh;

            hdrRssiThresh.HDRRSSIThreshListLen   = hdrRssiThresListLen;
            hdrRssiThresh.pHDRRSSIThreshList     = hdrRssiThreshList;
            req.pHDRRSSIThresh                 = pd->pHDRRSSIThresh;

            hdrEcioThresh.HDRECIOThreshListLen   = hdrEcioThresListLen;
            hdrEcioThresh.pHDRECIOThreshList     = hdrEcioThreshList;
            req.pHDRECIOThresh                   = pd->pHDRECIOThresh;

            hdrSinrThresh.HDRSINRThreshListLen   = hdrSinrThresListLen;
            hdrSinrThresh.pHDRSINRThreshList     = hdrSinrThreshList;
            req.pHDRSINRThresh                   = pd->pHDRSINRThresh;

            hdrIoThresh.HDRIOThreshListLen       = hdrIoThresListLen;
            hdrIoThresh.pHDRIOThreshList         = hdrIoThreshList;
            req.pHDRIOThresh                     = pd->pHDRIOThresh;

            gsmRssiThresh.GSMRSSIThreshListLen   = gsmRssiThresListLen;
            gsmRssiThresh.pGSMRSSIThreshList     = gsmRssiThreshList;
            req.pGSMRSSIThresh                   = pd->pGSMRSSIThresh;

            wcdmaRssiThresh.WCDMARSSIThreshListLen   = wcdmaRssiThresListLen;
            wcdmaRssiThresh.pWCDMARSSIThreshList     = wcdmaRssiThreshList;
            req.pWCDMARSSIThresh                     = pd->pWCDMARSSIThresh;

            wcdmaEcioThresh.WCDMAECIOThreshListLen   = wcdmaEcioThresListLen;
            wcdmaEcioThresh.pWCDMAECIOThreshList     = wcdmaEcioThreshList;
            req.pWCDMAECIOThresh                     = pd->pWCDMAECIOThresh;

            lteRssiThresh.LTERSSIThreshListLen   = lteRssiThresListLen;
            lteRssiThresh.pLTERSSIThreshList     = lteRssiThreshList;
            req.pLTERSSIThresh                   = pd->pLTERSSIThresh;

            lteSnrThresh.LTESNRThreshListLen   = lteSnrThresListLen;
            lteSnrThresh.pLTESNRThreshList     = lteSnrThreshList;
            req.pLTESNRThresh                  = pd->pLTESNRThresh;

            lteRsrqThresh.LTERSRQThreshListLen   = lteRsrqThresListLen;
            lteRsrqThresh.pLTERSRQThreshList     = lteRsrqThreshList;
            req.pLTERSRQThresh                  = pd->pLTERSRQThresh;

            lteRsrpThresh.LTERSRPThreshListLen = lteRsrpThresListLen;
            lteRsrpThresh.pLTERSRPThreshList   = lteRsrpThreshList;
            req.pLTERSRPThresh               = pd->pLTERSRPThresh;

            tdscdmaRscpThresh.TDSCDMARSCPThreshListLen = tdscdmaRscpThresListLen;
            tdscdmaRscpThresh.pTDSCDMARSCPThreshList   = tdscdmaRscpThreshList;
            req.pTDSCDMARSCPThresh               = pd->pTDSCDMARSCPThresh;

            tdscdmaRssiThresh.TDSCDMARSSIThreshListLen = tdscdmaRssiThresListLen;
            tdscdmaRssiThresh.pTDSCDMARSSIThreshList   = (ULONG*)tdscdmaRssiThreshList;
            req.pTDSCDMARSSIThresh                     = pd->pTDSCDMARSSIThresh;

            tdscdmaEcioThresh.TDSCDMAECIOThreshListLen = tdscdmaEcioThresListLen;
            tdscdmaEcioThresh.pTDSCDMAECIOThreshList   = (ULONG*)tdscdmaEcioThreshList;
            req.pTDSCDMAECIOThresh                     = pd->pTDSCDMAECIOThresh;

            tdscdmaSinrThresh.TDSCDMASINRThreshListLen = tdscdmaSinrThresListLen;
            tdscdmaSinrThresh.pTDSCDMASINRThreshList   = (ULONG*)tdscdmaSinrThreshList;
            req.pTDSCDMASINRThresh                     = pd->pTDSCDMASINRThresh;

            lteSigRptCfg.rptRate   = rptRate;
            lteSigRptCfg.avgPeriod = avgPeriod;
            req.pLTESigRptConfig   = pd->pLTESigRptConfig;

            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        (tc + 1),
                        ncases,
                        pd->desc);

            fprintf( fp, "Parameter Set:\n" );

            if(req.pCDMARSSIThresh)
            {
                fprintf( fp, "\nCDMA RSSI Threshold list\n");
                fprintf( fp, "CDMA RSSI Threshold list length: %d\n", cdmaRssiThresh.CDMARSSIThreshListLen );
                for( lCtr=0; lCtr < cdmaRssiThresh.CDMARSSIThreshListLen; lCtr++ )
                {
                    fprintf( fp, "CDMA RSSI Threshold [0x%x]: %d\n", lCtr, (short)cdmaRssiThreshList[lCtr] );
                }
            }
            if(req.pCDMAECIOThresh)
            {
                fprintf( fp, "\nCDMA ECIO Threshold list\n");
                 fprintf( fp, "CDMA ECIO Threshold list length: %d\n", cdmaEcioThresh.CDMAECIOThreshListLen );
                 for( lCtr=0; lCtr < cdmaEcioThresh.CDMAECIOThreshListLen; lCtr++ )
                 {
                     fprintf( fp, "CDMA ECIO Threshold [%d]: %d\n", lCtr, (short)cdmaEcioThreshList[lCtr] );
                 }
            }
            if(req.pHDRRSSIThresh)
            {
                fprintf( fp, "\nHDR RSSI Threshold list\n");
                fprintf( fp, "HDR RSSI Threshold list length: %d\n", hdrRssiThresh.HDRRSSIThreshListLen );
                for( lCtr=0; lCtr < hdrRssiThresh.HDRRSSIThreshListLen; lCtr++ )
                {
                    fprintf( fp, "HDR RSSI Threshold [%d]: %d\n", lCtr, (short)hdrRssiThreshList[lCtr] );
                }
            }
            if(req.pHDRECIOThresh)
            {
                fprintf( fp, "\nHDR ECIO Threshold list\n");
                fprintf( fp, "HDR ECIO Threshold list length: %d\n", hdrEcioThresh.HDRECIOThreshListLen );
                for( lCtr=0; lCtr < hdrEcioThresh.HDRECIOThreshListLen; lCtr++ )
                {
                    fprintf( fp, "HDR ECIO Threshold [%d]: %d\n", lCtr, (short)hdrEcioThreshList[lCtr] );
                }
            }
            if(req.pHDRSINRThresh)
            {
                fprintf( fp, "\nHDR SINR Threshold list\n");
                fprintf( fp, "HDR SINR Threshold list length: %d\n", hdrSinrThresh.HDRSINRThreshListLen );
                for( lCtr=0; lCtr < hdrSinrThresh.HDRSINRThreshListLen; lCtr++ )
                {
                    fprintf( fp, "HDR SINR Threshold [%d]: %d\n", lCtr, (short)hdrSinrThreshList[lCtr] );
                }
            }
            if(req.pHDRIOThresh)
            {
                fprintf( fp, "\nHDR IO Threshold list\n");
                fprintf( fp, "HDR IO Threshold list length: %d\n", hdrIoThresh.HDRIOThreshListLen );
                for( lCtr=0; lCtr < hdrIoThresh.HDRIOThreshListLen; lCtr++ )
                {
                    fprintf( fp, "HDR IO [%d]: %d\n", lCtr, (short)hdrIoThreshList[lCtr] );
                }
            }
            if(req.pGSMRSSIThresh)
            {
                fprintf( fp, "\nGSM RSSI Threshold list\n");
                fprintf( fp, "GSM RSSI Threshold list length: %d\n", gsmRssiThresh.GSMRSSIThreshListLen );
                for( lCtr=0; lCtr < gsmRssiThresh.GSMRSSIThreshListLen; lCtr++ )
                {
                    fprintf( fp, "CDMA ECIO Threshold [%d]: %d\n", lCtr, (short)gsmRssiThreshList[lCtr] );
                }
            }
            if(req.pWCDMARSSIThresh)
            {
                fprintf( fp, "\nWCDMA RSSI Threshold list\n");
                fprintf( fp, "WCDMA RSSI Threshold list length: %d\n", wcdmaRssiThresh.WCDMARSSIThreshListLen );
                for( lCtr=0; lCtr < wcdmaRssiThresh.WCDMARSSIThreshListLen; lCtr++ )
                {
                    fprintf( fp, "WCDMA RSSI Threshold [%d]: %d\n", lCtr, (short)wcdmaRssiThreshList[lCtr] );
                }
            }
            if(req.pWCDMAECIOThresh)
            {
                fprintf( fp, "\nWCDMA ECIO Threshold list\n");
                fprintf( fp, "WCDMA ECIO Threshold list length: %d\n", wcdmaEcioThresh.WCDMAECIOThreshListLen );
                for( lCtr=0; lCtr < wcdmaEcioThresh.WCDMAECIOThreshListLen; lCtr++ )
                {
                    fprintf( fp, "WCDMA ECIO Threshold [%d]: %d\n", lCtr, (short)wcdmaEcioThreshList[lCtr] );
                }
            }
            if(req.pLTERSSIThresh)
            {
                fprintf( fp, "\nLTE RSSI Threshold list\n");
                fprintf( fp, "LTE RSSI Threshold list length: %d\n", lteRssiThresh.LTERSSIThreshListLen );
                for( lCtr=0; lCtr < lteRssiThresh.LTERSSIThreshListLen; lCtr++ )
                {
                    fprintf( fp, "LTE RSSI Threshold [%d]: %d\n", lCtr, (short)lteRssiThreshList[lCtr] );
                }
            }
            if(req.pLTESNRThresh)
            {
                fprintf( fp, "\nLTE SNR Threshold list\n");
                fprintf( fp, "LTE SNR Threshold list length: %d\n", lteSnrThresh.LTESNRThreshListLen );
                for( lCtr=0; lCtr < lteSnrThresh.LTESNRThreshListLen; lCtr++ )
                {
                    fprintf( fp, "LTE SNR Threshold [%d]: %d\n", lCtr, (short)lteSnrThreshList[lCtr] );
                }
            }
            if(req.pLTERSRQThresh)
            {
                fprintf( fp, "\nLTE RSRQ Threshold list\n");
                fprintf( fp, "LTE RSRQ Threshold list length: %d\n", lteRsrqThresh.LTERSRQThreshListLen );
                for( lCtr=0; lCtr < lteRsrqThresh.LTERSRQThreshListLen; lCtr++ )
                {
                    fprintf( fp, "LTE RSRQ Threshold [%d]: %d\n", lCtr, (short)lteRsrqThreshList[lCtr] );
                }
            }
            if(req.pLTERSRPThresh)
            {
                fprintf( fp, "\nLTE RSRP Threshold list\n");
                fprintf( fp, "LTE RSRP Threshold list length: %d\n", lteRsrpThresh.LTERSRPThreshListLen );
                for( lCtr=0; lCtr < lteRsrpThresh.LTERSRPThreshListLen; lCtr++ )
                {
                    fprintf( fp, "LTE RSRP Threshold [%d]: %d\n", lCtr, (short)lteRsrpThreshList[lCtr] );
                }
            }
            if(req.pTDSCDMARSCPThresh)
            {
                fprintf( fp, "\nTDSCDMA RSCP Threshold list\n");
                fprintf( fp, "TDSCDMA RSCP Threshold list length: %d\n", tdscdmaRscpThresh.TDSCDMARSCPThreshListLen );
                for( lCtr=0; lCtr < tdscdmaRscpThresh.TDSCDMARSCPThreshListLen; lCtr++ )
                {
                    fprintf( fp, "TDSCDMA RSCP Threshold [%d]: %d\n", lCtr, (short)tdscdmaRscpThreshList[lCtr] );
                }
            }
            if(req.pTDSCDMARSSIThresh)
            {
                fprintf( fp, "\nTDSCDMA RSSI Threshold list\n");
                fprintf( fp, "TDSCDMA RSSI Threshold list length: %d\n", tdscdmaRssiThresh.TDSCDMARSSIThreshListLen );
                for( lCtr=0; lCtr < tdscdmaRssiThresh.TDSCDMARSSIThreshListLen; lCtr++ )
                {
                    fprintf( fp, "TDSCDMA RSSI Threshold [%d]: %f\n", lCtr, (float)tdscdmaRssiThreshList[lCtr] );
                }
            }
            if(req.pTDSCDMAECIOThresh)
            {
                fprintf( fp, "\nTDSCDMA ECIO Threshold list\n");
                fprintf( fp, "TDSCDMA ECIO Threshold list length: %d\n", tdscdmaEcioThresh.TDSCDMAECIOThreshListLen );
                for( lCtr=0; lCtr < tdscdmaEcioThresh.TDSCDMAECIOThreshListLen; lCtr++ )
                {
                    fprintf( fp, "TDSCDMA ECIO Threshold [%d]: %f\n", lCtr, (float)tdscdmaEcioThreshList[lCtr] );
                }
            }
            if(req.pTDSCDMASINRThresh)
            {
                fprintf( fp, "\nTDSCDMA SINR Threshold list\n");
                fprintf( fp, "TDSCDMA SINR Threshold list length: %d\n", tdscdmaSinrThresh.TDSCDMASINRThreshListLen );
                for( lCtr=0; lCtr < tdscdmaSinrThresh.TDSCDMASINRThreshListLen; lCtr++ )
                {
                    fprintf( fp, "TDSCDMA SINR Threshold [%d]: %f\n", lCtr, (float)tdscdmaSinrThreshList[lCtr] );
                }
            }

            nRet = SLQSNasConfigSigInfo2( &req );

            /* Display result code and text */
            doprintreason ( fp, nRet );

            if ( !nRet )
            {
                fprintf( fp, "Sig Info2 configured\n" );
            }
            tc++;
        }
    }
    if ( fp )
        tfclose( fp );
}

/*
 * Name:     doSLQSNASGetLTECPHYCaInfo
 *
 * Purpose:  Perform the tests that call the SLQSNASGetLTECPHYCaInfo() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSLQSNASGetLTECPHYCaInfo( void )
{
    FILE  *fp = NULL;
    nasGetLTECphyCa snasGetLTECphyCa;
    SLQSNASGetLTECPHYCaInfoTestCase_t tTestCases[] =
    {
         {eQCWWAN_ERR_INVALID_ARG, 1, NULL, "All fields are enable"},
         {eQCWWAN_ERR_NONE, 2, &snasGetLTECphyCa, "All fields are enable"}
    };

    fp = tfopen( "../../nas/test/results/slqsnasgetltecphycainfo.txt", "w" );
    int tc=0;
    if (fp)
    {
        
        BYTE ncases = sizeof( tTestCases )/
                      sizeof( tTestCases[0] );
        while ( tc < ncases )
        {
            SLQSNASGetLTECPHYCaInfoTestCase_t *pd =
                                     &tTestCases[tc++];
            nasGetLTECphyCa *pnasGetLTECphyCa;
            pnasGetLTECphyCa = pd->pNasGetLTECphyCa; 
            ULONG nRet = eQCWWAN_ERR_ENUM_BEGIN;
            nRet = SLQSNASGetLTECPHYCaInfo(pnasGetLTECphyCa);
            doprintreason ( fp, nRet );
            if ( !nRet )
            {
                if(pnasGetLTECphyCa->sPhyCaAggScellIndType.TlvPresent)
                {
                    fprintf( fp, "Scell Indicator Type:\n");
                    fprintf( fp, "PCI: %02X\n",pnasGetLTECphyCa->sPhyCaAggScellIndType.pci);
                    fprintf( fp, "Frequency: %d\n",pnasGetLTECphyCa->sPhyCaAggScellIndType.freq);
                    fprintf( fp, "Scell State: %02X\n",pnasGetLTECphyCa->sPhyCaAggScellIndType.scell_state);
                }
                
                if(pnasGetLTECphyCa->sPhyCaAggScellDlBw.TlvPresent)
                    fprintf( fp, "Downlink Bandwidth for Scell: %02X\n",pnasGetLTECphyCa->sPhyCaAggScellDlBw.dl_bw_value);

                if(pnasGetLTECphyCa->sPhyCaAggScellInfo.TlvPresent)
                {
                    fprintf( fp, "Scell Info:\n");
                    fprintf( fp, "\tDownlink Bandwidth: %02X\n",pnasGetLTECphyCa->sPhyCaAggScellInfo.dl_bw_value);
                    fprintf( fp, "\tFrequency: %d\n",pnasGetLTECphyCa->sPhyCaAggScellInfo.freq);
                    fprintf( fp, "\tLTE Band Value: %d\n",pnasGetLTECphyCa->sPhyCaAggScellInfo.iLTEbandValue);
                    fprintf( fp, "\tPCI: %d\n",pnasGetLTECphyCa->sPhyCaAggScellInfo.pci);
                    fprintf( fp, "\tCell State: %d\n",pnasGetLTECphyCa->sPhyCaAggScellInfo.scell_state);
                }

                if(pnasGetLTECphyCa->sPhyCaAggPcellInfo.TlvPresent)
                {
                    fprintf( fp,"Pcell Info\n");
                    fprintf( fp, "\tDownlink Bandwidth: %02X\n",pnasGetLTECphyCa->sPhyCaAggPcellInfo.dl_bw_value);
                    fprintf( fp, "\trequency: %d\n",pnasGetLTECphyCa->sPhyCaAggPcellInfo.freq);
                    fprintf( fp, "\tLTE Band Value: %d\n",pnasGetLTECphyCa->sPhyCaAggPcellInfo.iLTEbandValue);
                    fprintf( fp, "\tPCI: %d\n",pnasGetLTECphyCa->sPhyCaAggPcellInfo.pci);
                }

                if(pnasGetLTECphyCa->sPhyCaAggScellIndex.TlvPresent)
                {
                    fprintf( fp, "Scell Index : %d\n",(int)pnasGetLTECphyCa->sPhyCaAggScellIndex.scell_idx);
                }
            }
        }
        
    }
    
    if ( fp )
            tfclose( fp );
    
}

/*
 * Name:     doSLQSNASSwiGetChannelLock
 *
 * Purpose:  Perform the tests that call the SLQSNASSwiGetChannelLock() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSLQSNASSwiGetChannelLock( void )
{
    FILE  *fp = NULL;
    nasSwiGetChannelLockResp NasSwiGetChannelLockResp;
    wcdmaUARFCN WcdmaUARFCN;
    lteEARFCN   LteEARFCN;
    ltePCI      LtePCI;
    NasSwiGetChannelLockResp.pWcdmaUARFCN = &WcdmaUARFCN;
    NasSwiGetChannelLockResp.pLteEARFCN = &LteEARFCN;
    NasSwiGetChannelLockResp.pLtePCI = &LtePCI;

    SLQSNASSwiGetChannelLockTestCase_t tTestCases[] =
    {
         {eQCWWAN_ERR_INVALID_ARG, 1, NULL, "All output Parameters NULL"},
         {eQCWWAN_ERR_NONE, 2, &NasSwiGetChannelLockResp, "Valid test case"}
    };

    fp = tfopen( "../../nas/test/results/slqsnasswigetchannellock.txt", "w" );
    int tc=0;
    nasSwiGetChannelLockResp *pNasSwiGetChannelLockResp;
    ULONG nRet = eQCWWAN_ERR_ENUM_BEGIN;

    if (fp)
    {

        BYTE ncases = sizeof( tTestCases )/
                      sizeof( tTestCases[0] );
        while ( tc < ncases )
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tc,
                       ncases,
                       tTestCases[tc].desc);

        SLQSNASSwiGetChannelLockTestCase_t *pd =
                                     &tTestCases[tc++];
            pNasSwiGetChannelLockResp = pd->pNasSwiGetChannelLockResp;

            nRet = SLQSNASSwiGetChannelLock(pNasSwiGetChannelLockResp);
            doprintreason ( fp, nRet );
            if ( !nRet )
            {
                if(pNasSwiGetChannelLockResp->pWcdmaUARFCN)
                {
                    fprintf( fp, "\nWCDMA UARFCN:\n");
                    fprintf( fp, "Status: 0x%02X\n",pNasSwiGetChannelLockResp->pWcdmaUARFCN->status);
                    fprintf( fp, "UARFCN: %u\n",pNasSwiGetChannelLockResp->pWcdmaUARFCN->uarfcn);
                }
                if(pNasSwiGetChannelLockResp->pLteEARFCN)
                {
                    fprintf( fp, "\nLTE EARFCN:\n");
                    fprintf( fp, "Status: 0x%02X\n",pNasSwiGetChannelLockResp->pLteEARFCN->status);
                    fprintf( fp, "EARFCN Primary: %u\n",pNasSwiGetChannelLockResp->pLteEARFCN->earfcn0);
                    fprintf( fp, "EARFCN Secondary: %u\n",pNasSwiGetChannelLockResp->pLteEARFCN->earfcn1);
                }
                if(pNasSwiGetChannelLockResp->pLtePCI)
                {
                    fprintf( fp, "\nLTE PCI:\n");
                    fprintf( fp, "Status: 0x%02X\n",pNasSwiGetChannelLockResp->pLtePCI->status);
                    fprintf( fp, "EARFCN: %u\n",pNasSwiGetChannelLockResp->pLtePCI->earfcn);
                    fprintf( fp, "PCI: %u\n",pNasSwiGetChannelLockResp->pLtePCI->pci);
                }
            }
        }
    }
    if ( fp )
        tfclose( fp );
}

/*
 * Name:     doSLQSNASSwiSetChannelLock
 *
 * Purpose:  Perform the tests that call the SLQSNASSwiSetChannelLock() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSLQSNASSwiSetChannelLock( void )
{
    ULONG nRet;
    FILE  *fp = NULL;

    fp = tfopen("../../nas/test/results/slqsnasswisetchannellock.txt", "w");

    wcdmaUARFCN    WcdmaUARFCN;
    lteEARFCN      LteEARFCN;
    ltePCI         LtePCI;

    SLQSNASSwiSetChannelLockTestCase_t \
    SLQSNASSwiSetChannelLockTestCases[MAX_SET_CHANNEL_LOCK_TESTCASE_NUM] =
    {
        {eQCWWAN_ERR_NONE, 1, NULL, NULL, NULL, "All input Parameters NULL"},
        {eQCWWAN_ERR_NONE, 2, &WcdmaUARFCN, NULL, NULL, "Valid test Case"},
        {eQCWWAN_ERR_NONE, 3, &WcdmaUARFCN, &LteEARFCN, NULL, "Valid test Case"},
        {eQCWWAN_ERR_NONE, 4, NULL, &LteEARFCN, &LtePCI, "Valid test Case"}
    };

    nasSwiSetChannelLockReq  req;

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSNASSwiSetChannelLockTestCases )/
                      sizeof( SLQSNASSwiSetChannelLockTestCases[0] );
        while ( tc < ncases )
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tc,
                       ncases,
                       SLQSNASSwiSetChannelLockTestCases[tc].desc);

            SLQSNASSwiSetChannelLockTestCase_t *pd =
                                     &SLQSNASSwiSetChannelLockTestCases[tc];

            WcdmaUARFCN.status = 1;
            WcdmaUARFCN.uarfcn = 2500;

            if(tc==2)
            {
                WcdmaUARFCN.status = 0;
                WcdmaUARFCN.uarfcn = 0;
                LteEARFCN.status  = 1;
                LteEARFCN.earfcn0 = 2500;
                LteEARFCN.earfcn1 = 2700;
            }

            if(tc==3)
            {
                LteEARFCN.status  = 0;
                LteEARFCN.earfcn0 = 0;
                LteEARFCN.earfcn1 = 0;
                LtePCI.status = 1;
                LtePCI.earfcn = 2500;
                LtePCI.pci    = 100;
            }
            req.pWcdmaUARFCN = pd->pWcdmaUARFCN;
            req.pLteEARFCN    = pd->pLteEARFCN;
            req.pLtePCI    = pd->pLtePCI;

            fprintf(fp, "Parameter Set:");
            if(pd->pWcdmaUARFCN)
            {
                fprintf( fp, "\nWCDMA UARFCN\n");
                fprintf( fp, "WCDMA UARFCN Status: %x\n", WcdmaUARFCN.status);
                fprintf( fp, "WCDMA UARFCN Value: %u\n", WcdmaUARFCN.uarfcn);
            }
            if(pd->pLteEARFCN)
            {
                fprintf( fp, "\nLTE EARFCN\n");
                fprintf( fp, "LTE EARFCN Status: %x\n", LteEARFCN.status);
                fprintf( fp, "LTE Primary EARFCN Value: %u\n", LteEARFCN.earfcn0);
                fprintf( fp, "LTE Secondary EARFCN Value: %u\n", LteEARFCN.earfcn1);
            }
            if(pd->pLtePCI)
            {
                fprintf( fp, "\nLTE PCI\n");
                fprintf( fp, "LTE PCI Status: %x\n", LtePCI.status);
                fprintf( fp, "LTE EARFCN Value: %u\n", LtePCI.earfcn);
                fprintf( fp, "LTE PCI Value: %u\n", LtePCI.pci);
            }
            nRet = SLQSNASSwiSetChannelLock(&req);

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "Channel Lock Set\n");
            }

            tc++;
        }
    }

    if (fp)
        tfclose(fp);
}

