/**************
 *  Filename: qaGobiApiLocTest.c
 *
 *  Purpose:  LOC API test driver function definitions
 *
 * Copyright: � 2015 Sierra Wireless, Inc., all rights reserved
 *
 **************/

/* Linux definitions */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <unistd.h>

/* include files */
#include "SwiDataTypes.h"
#include "qmerrno.h"
#include "qmudefs.h"

#include "qaGobiApiLoc.h"
#include "qaGobiApiCbk.h"
#include "qaGobiApiLocTest.h"
#include "qatesthelper.h"
#include "qatestproto.h"

/*******************
    TEST DATA
 *******************/
/* Constants */

/* Global Variables */
ULONGLONG     qmiLocEventMaskPositionReport           = 0x00000001;
ULONGLONG     qmiLocEventMaskGnssSvInfo               = 0x00000002;
ULONGLONG     qmiLocEventMaskNmea                     = 0x00000004;
ULONGLONG     qmiLocEventMaskNiNotifyVerifyReq        = 0x00000008;
ULONGLONG     qmiLocEventMaskInjectTimeReq            = 0x00000010;
ULONGLONG     qmiLocEventMaskInjectPredictedOrbitsReq = 0x00000020;
ULONGLONG     qmiLocEventMaskInjectPositionReq        = 0x00000040;
ULONGLONG     qmiLocEventMaskEngineState              = 0x00000080;
ULONGLONG     qmiLocEventMaskFixSessionState          = 0x00000100;
ULONGLONG     qmiLocEventMaskWifiReq                  = 0x00000200;
ULONGLONG     qmiLocEventMaskSensorStreamingReadyStatus = 0x00000400;
ULONGLONG     qmiLocEventMaskTimeSyncReq              = 0x00000800;
ULONGLONG     qmiLocEventMaskSetSpiStreamingReport    = 0x00001000;
ULONGLONG     qmiLocEventMaskLocationServerConnectionReq = 0x00002000;
ULONGLONG     qmiLocEventMaskNiGeofenceNotification   = 0x00004000;
ULONGLONG     qmiLocEventMaskGeofenceGenAlert         = 0x00008000;
ULONGLONG     qmiLocEventMaskGeofenceBreachNotification = 0x00010000;
ULONGLONG     qmiLocEventMaskPedometerControl         = 0x00020000;
ULONGLONG     qmiLocEventMaskMotionDataControl        = 0x00040000;
ULONGLONG     qmiLocEventMaskBatchFullNotification    = 0x00080000;
ULONGLONG     qmiLocEventMaskLiveBatchedPositionReport = 0x00100000;
ULONGLONG     qmiLocEventMaskInjectWifiApDataReq       = 0x00200000;
ULONGLONG     qmiLocEventMaskGeofenceBatchBreachNotification = 0x00400000;
ULONGLONG     qmiLocEventMaskVehicleDataReadyStatus    = 0x00800000;
ULONGLONG     qmiLocEventMaskGnssMeasurementReport     = 0x01000000;
ULONGLONG     qmiLocEventMaskInvalidValue              = 0xFFFFFFFF;

local ULONG   qmiLocRecurrence        = 1;
local ULONG   qmiLocAccuracy          = 1;
local ULONG   intermediateReportState = 1;
local ULONG   configAltitudeAssumed   = 1;

local CHAR appProvider[] = "ABC";
local CHAR appName[]     = "XYZ";
local CHAR appVer[]      = "12";

local struct LocApplicationInfo    AppInfo = {3, appProvider,3,appName,1,2,appVer };

local ULONGLONG utcTime = 0;
local ULONG uncTime = 10;

local double latitude  = 30.7274072;
local double longitude = 76.8436267;
local float  horUnc    = 3.5;
local float  altitudeWrtEllipsoid = 16.0;
local float  vertUnc = 3.5;

/*
 * Name:    doSLQSLOCEventRegister
 *
 * Purpose: Perform the tests that call the SLQSLOCEventRegister() API
 *
 * Return:  none
 *
 * Notes:   none
 *
 */
void doSLQSLOCEventRegister( void )
{
    /* Test Cases */
    SLQSLocEventRegisterTestCase_t SLQSLocEventRegisterTestCase[]=
    {
         { eQCWWAN_ERR_NONE, 1, qmiLocEventMaskPositionReport, "Mask Position Report"},
         { eQCWWAN_ERR_NONE, 2, qmiLocEventMaskGnssSvInfo, "Mask GNSS SV Info"},
         { eQCWWAN_ERR_NONE, 3, qmiLocEventMaskNmea, "Mask NMEA"},
         { eQCWWAN_ERR_NONE, 4, qmiLocEventMaskEngineState, "Mask Engine State"},
         { eQCWWAN_ERR_NONE, 5, qmiLocEventMaskTimeSyncReq, "Mask Time Sync Req"},
         { eQCWWAN_ERR_NONE, 6, qmiLocEventMaskInvalidValue, "InValid Case"},
    };

    LOCEventRegisterReqResp req;
    FILE *fp = tfopen("../../loc/test/results/slqsloceventregister.txt", "w");
    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSLocEventRegisterTestCase )/
                      sizeof( SLQSLocEventRegisterTestCase[0] );
        while ( tc < ncases )
        {
            SLQSLocEventRegisterTestCase_t *pd =
                   &SLQSLocEventRegisterTestCase[tc++];
            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        tc,
                        ncases,
                        pd->desc);

            req.eventRegister = pd->eventRegister;
            fprintf(fp, "Event mask = %#010llx\n",req.eventRegister);
            ULONG nRet = SLQSLOCEventRegister( &req );
            doprintreason( fp, nRet );

            if ( eQCWWAN_ERR_NONE == nRet )
            {
                fprintf(fp, "SLQSLOCEventRegister Successful\n");
                fprintf(fp, "Event mask = %#010llx\n",req.eventRegister);
            }
        }
    }
    if (fp)
        tfclose(fp);
}

/*
 * Name:    doSLQSLOCSetExtPowerState
 *
 * Purpose: Perform the tests that call the SLQSLOCSetExtPowerState() API
 *
 * Return:  none
 *
 * Notes:   none
 *
 */
void doSLQSLOCSetExtPowerState(void)
{

    /* Test Cases */
    SLQSLocSetExtPowerTestCase_t SLQSLocSetExtPowerTestCase[]=
    {
         { eQCWWAN_ERR_NONE, 1, 0, "Device is not connected to an external power source"},
         { eQCWWAN_ERR_NONE, 2, 1, "Device is connected to an external power source"},
         { eQCWWAN_ERR_NONE, 3, 2, "Unknown external power state"},
         { eQCWWAN_ERR_NONE, 4, 3, "Failue Case"},
    };
LOCExtPowerStateReqResp req;
FILE *fp = tfopen("../../loc/test/results/slqslocsetextpower.txt", "w");
if (fp)
{
    BYTE tc = 0;
    BYTE ncases = sizeof( SLQSLocSetExtPowerTestCase)/
                  sizeof( SLQSLocSetExtPowerTestCase[0] );
    while ( tc < ncases )
    {
        SLQSLocSetExtPowerTestCase_t *pd =
               &SLQSLocSetExtPowerTestCase[tc++];
        fprintf(fp, "\nTest Case %d / %d : %s\n",
                    tc,
                    ncases,
                    pd->desc);

        req.extPowerState = pd->extPowerState;
        fprintf(fp, "Event mask = %d\n",req.extPowerState);
        ULONG nRet = SLQSLOCSetExtPowerState( &req );
        doprintreason( fp, nRet );

        if ( eQCWWAN_ERR_NONE == nRet )
        {
            fprintf(fp, "SLQSLOCExtPowerState Configured Successful\n");
            fprintf(fp, "External Power State = %d\n",req.extPowerState);
        }
    }
}
if (fp)
    tfclose(fp);
}
/*
 * Name:    doSLQSLOCStart
 *
 * Purpose: Perform the tests that call the SLQSLOCStart() API
 *
 * Return:  none
 *
 * Notes:   none
 *
 */

void doSLQSLOCStart(void)
{
    ULONG               nRet;
    FILE                *fp = NULL;
    LOCStartReq req;
    /* Define the test cases to be executed */
    SLQSLocStartTestCase_t SLQSLocStartTestCases[] =
    {
        { eQCWWAN_ERR_NONE, 1, 1, NULL, NULL, NULL, NULL, NULL, NULL,
           "Valid Test Case - All Optional params NULL"},

        { eQCWWAN_ERR_NONE, 2, 1, &qmiLocRecurrence, NULL, NULL, NULL, NULL, NULL,
           "Valid Test Case - LOC Recurrence Perodic with 0 session ID"},

        { eQCWWAN_ERR_NONE, 3, 1, NULL, &qmiLocAccuracy, NULL, NULL, NULL, NULL,
           "Valid Test Case - Low Accuracy"},

        { eQCWWAN_ERR_NONE, 4, 1, NULL, NULL, &intermediateReportState, NULL, NULL, NULL,
           "Valid Test Case - Intermediate Report Turned On"},

        { eQCWWAN_ERR_NONE, 5, 1, NULL, NULL, NULL, NULL, &AppInfo, NULL,
           "Valid Test Case - Set App Info"},

        { eQCWWAN_ERR_NONE, 6, 1, NULL, NULL, NULL, NULL, NULL, &configAltitudeAssumed,
           "Valid Test Case - Altitude Assumed in GNSS SV INFO enabled"},
     };

    fp = tfopen( "../../loc/test/results/slqslocstart.txt", "w" );
    if ( fp )
    {
        BYTE tCaseNum     = 0;
        BYTE ncases = sizeof( SLQSLocStartTestCases ) /
                      sizeof( SLQSLocStartTestCases[0] );

        while( tCaseNum < ncases )
        {
        SLQSLocStartTestCase_t *pd =
                &SLQSLocStartTestCases[tCaseNum++];

        fprintf(fp, "\nTest Case %d / %d : %s\n",
                        tCaseNum,
                        ncases,
                        pd->desc);
        req.SessionId                 =  pd->sessionId;
        req.pRecurrenceType           =  pd->pRecurrenceType;
        req.pHorizontalAccuracyLvl    =  pd->pHorizontalAccuracyLvl;
        req.pIntermediateReportState  =  pd->pIntermediateReportState;
        req.pMinIntervalTime          =  pd->pMinIntervalTime;
        req.pApplicationInfo          =  pd->pApplicationInfo;
        req.pConfigAltitudeAssumed    =  pd->pConfigAltitudeAssumed;

        fprintf( fp,"\nParameters set for System Selection Preference:\n" );
        fprintf(fp, "Session Id = %d\n",req.SessionId);
        IFPRINTF( fp, "Recurrence Type: %d\n", pd->pRecurrenceType );
        IFPRINTF( fp, "Horizontal Accuracy Lvl: %d\n", pd->pHorizontalAccuracyLvl );
        IFPRINTF( fp, "Intermediate Report State: %d\n", pd->pIntermediateReportState);
        IFPRINTF( fp, "Min Interval Time : %d\n", pd->pMinIntervalTime );
        if( pd->pApplicationInfo)
        {
            fprintf( fp, "\nLOC Application Info :\n");
            BYTE i = 0;
            fprintf( fp, "App Provider Length : 0x%x\n",
                      pd->pApplicationInfo->appProviderLength);

            fprintf( fp, "APP Provider:");
            for ( i = 0; i < pd->pApplicationInfo->appProviderLength; i++ )
            {
                fprintf( fp, "%c", pd->pApplicationInfo->pAppProvider[i] );
            }
            fprintf( fp, "\nApp Provider Length : 0x%x\n",
                      pd->pApplicationInfo->appNameLength);

            fprintf( fp, "APP Name:");
            for ( i = 0; i < pd->pApplicationInfo->appNameLength; i++ )
            {
               fprintf( fp, "%c", pd->pApplicationInfo->pAppName[i] );
            }

            fprintf( fp, "\nApp Version Valid : 0x%x\n",
                      pd->pApplicationInfo->appVersionValid);

            fprintf( fp, "App Version Length : 0x%x\n",
                      pd->pApplicationInfo->appVersionLength);

            fprintf( fp, "APP Version:");
            for ( i = 0; i < pd->pApplicationInfo->appVersionLength; i++ )
            {
               fprintf( fp, "%c", pd->pApplicationInfo->pAppVersion[i] );
            }
            fprintf( fp, "\n");
        }

        //        IFPRINTF( fp, "Application Info : %d\n", pd->pApplicationInfo );
        IFPRINTF( fp, "Altitude Assumed : %d\n", pd->pConfigAltitudeAssumed );

        nRet = SLQSLOCStart(&req );

        /* Display result code and text */
          doprintreason( fp, nRet );
         }
     }

      if (fp)
      tfclose(fp);
}
/*
 * Name:    doSLQSLOCStop
 *
 * Purpose: Perform the tests that call the SLQSLOCStop() API
 *
 * Return:  none
 *
 * Notes:   none
 *
 */

void doSLQSLOCStop(void)
{

    /* Test Cases */
    SLQSLocStopTestCase_t SLQSLocStopTestCases[]=
    {
         { eQCWWAN_ERR_NONE, 1, 0, "Session Id 0"},
         { eQCWWAN_ERR_NONE, 2, 1, "session Id 1"},
     };
    LOCStopReq req;

    FILE *fp = tfopen("../../loc/test/results/slqslocstop.txt", "w");
    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSLocStopTestCases)/
                  sizeof( SLQSLocStopTestCases[0] );
         while ( tc < ncases )
        {
            SLQSLocStopTestCase_t *pd =
                   &SLQSLocStopTestCases[tc++];
            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        tc,
                        ncases,
                        pd->desc);

            req.sessionId = pd->sessionId;
            fprintf(fp, "Session Id = %d\n",req.sessionId);
            ULONG nRet = SLQSLOCStop( &req );
            doprintreason( fp, nRet );

            if ( eQCWWAN_ERR_NONE == nRet )
            {
                fprintf(fp, "SLQSLOCStop  Successful\n");
                fprintf(fp, "LOC Stop Session ID = %d\n",req.sessionId);
            }
        }
    }
    if (fp)
        tfclose(fp);
}


void doSLQSLOCSetOpMode(void)
{
    ULONG               nRet;
    FILE                *fp = NULL;
    /* Define the test cases to be executed */
    SLQSLocSetOpModeTestCase_t SLQSLocSetOpModeTestCases[] =
    {
        { eQCWWAN_ERR_NONE, 1, 
            "set default operation mode (1)"}
    };

    fp = tfopen( "../../loc/test/results/slqslocsetopmode.txt", "w" );
    if ( fp )
    {
        BYTE tCaseNum     = 0;
        BYTE ncases = sizeof( SLQSLocSetOpModeTestCases ) /
            sizeof( SLQSLocSetOpModeTestCases[0] );

        while( tCaseNum < ncases )
        {
            SLQSLocSetOpModeTestCase_t *pd =
                &SLQSLocSetOpModeTestCases[tCaseNum++];

            fprintf(fp, "\nTest Case %d / %d : %s\n",
                    tCaseNum,
                    ncases,
                    pd->desc);

            fprintf( fp, "Operating mode : %d\n", pd->mode );

            nRet = SLQSLOCSetOpMode(pd->mode );

            /* Display result code and text */
            doprintreason( fp, nRet );
        }
    }

    if (fp)
        tfclose(fp);
}

void doSLQSLOCDelAssData(void)
{
    ULONG nRet;
    FILE  *fp = NULL;
    GnssData gnss;
    SVInfo svinfo;
    CellDb celldb;
    ClkInfo clk;
    BdsSVInfo bds;
    
    /* SV Info */
    SV sv[1] = {{1,1,1}};
    svinfo.len = 1;
    svinfo.pSV = sv;

    /* GNSS data */
    gnss.mask = 1;

    /* Cell data base */
    celldb.mask = 1;

    /* Clock Info */
    clk.mask = 1;

    /* BDS SV Info */
    BdsSV bsv[1] = {{201,1}};
    bds.len = 1;
    bds.pSV = bsv;

    /* Define the test cases to be executed */
    SLQSLocDelAssDataTestCase_t SLQSLocDelAssDataTestCase[] =
    {
        { eQCWWAN_ERR_NONE, 1, NULL, NULL, NULL, NULL, NULL,
            "Delete All"},
        { eQCWWAN_ERR_NONE, 2, &svinfo, NULL, NULL, NULL, NULL,
            "Delete SV Info"},
        { eQCWWAN_ERR_NONE, 3, NULL, &gnss, NULL, NULL, NULL,
            "Delete GNSS Data"},
        { eQCWWAN_ERR_NONE, 4, NULL, NULL, &celldb, NULL, NULL,
            "Delete Cell Database"},
        { eQCWWAN_ERR_NONE, 5, NULL, NULL, NULL, &clk, NULL,
            "Delete Clock Info"},
        { eQCWWAN_ERR_NONE, 6, NULL, NULL, NULL, NULL, &bds,
            "Delete BDS SV Info"},
    };

    fp = tfopen( "../../loc/test/results/slqslocsetopmode.txt", "w" );
    if ( fp )
    {
        BYTE tCaseNum     = 0;
        BYTE ncases = sizeof( SLQSLocDelAssDataTestCase ) /
            sizeof( SLQSLocDelAssDataTestCase[0] );

        while( tCaseNum < ncases )
        {
            SLQSLocDelAssDataTestCase_t *pd =
                &SLQSLocDelAssDataTestCase[tCaseNum++];

            fprintf(fp, "\nTest Case %d / %d : %s\n",
                    tCaseNum,
                    ncases,
                    pd->desc);

            LocDelAssDataReq req;
            memset(&req, 0, sizeof(req));

            req.pSVInfo = pd->pSVInfo;
            req.pGnssData = pd->pGnssData;
            req.pCellDb = pd->pCellDb;
            req.pClkInfo = pd->pClkInfo;
            req.pBdsSVInfo = pd->pBdsSVInfo;

            nRet = SLQSLOCDelAssData(req);

            /* Display result code and text */
            doprintreason( fp, nRet );
        }
    }

    if (fp)
        tfclose(fp);
}

void doSwiLocGetAutoStart(void)
{
    ULONG nRet;
    FILE  *fp = NULL;
    SwiLocGetAutoStartResp resp;

    fp = tfopen( "../../loc/test/results/swilocgetautostart.txt", "w" );
    if ( fp )
    {
        fprintf(fp, "\t null pointer\n");
        nRet = SwiLocGetAutoStart(NULL);
        doprintreason( fp, nRet );

        nRet = SwiLocGetAutoStart(&resp);

        if (nRet == eQCWWAN_ERR_NONE)
        {
            if (resp.function_reported)
                printf("\tfunction: %d\n", resp.function );
            if (resp.fix_type_reported)
                printf("\tfix_type: %d \n", resp.fix_type);
            if (resp.max_time_reported)
                printf("\tmax_time: %d\n", resp.max_time);
            if (resp.max_dist_reported)
                printf("\tmax_dist: %d\n", resp.max_dist);
            if (resp.fix_rate_reported)
                printf("\tfix_rate: %d\n", resp.fix_rate);
        }
        doprintreason( fp, nRet );
        tfclose(fp);
    }
}

void doSwiLocSetAutoStart(void)
{
    ULONG nRet;
    FILE  *fp = NULL;
    SwiLocSetAutoStartReq req;

    fp = tfopen( "../../loc/test/results/swilocgetautostart.txt", "w" );
    if ( fp )
    {
        fprintf(fp, "\t null pointer\n");
        nRet = SwiLocSetAutoStart(NULL);
        doprintreason( fp, nRet );

        //below values are extracted from EM7455
        memset(&req, 0, sizeof(req));
        fprintf(fp, "\t set nothing\n");
        nRet = SwiLocSetAutoStart(&req);
        doprintreason( fp, nRet );
        req.set_function = 1;
        req.function = 2;
        fprintf(fp, "\t set function to %d\n", req.function);
        nRet = SwiLocSetAutoStart(&req);
        doprintreason( fp, nRet );
        req.set_fix_type = 1;
        req.fix_type = 1;
        fprintf(fp, "\t and set fix_type %d\n", req.fix_type);
        nRet = SwiLocSetAutoStart(&req);
        doprintreason( fp, nRet );
        req.set_max_time = 1;
        req.max_time = 0xff;
        fprintf(fp, "\t and set max_time %d\n", req.max_time);
        nRet = SwiLocSetAutoStart(&req);
        doprintreason( fp, nRet );
        req.set_max_dist = 1;
        req.max_dist = 1000;
        fprintf(fp, "\t and set max_dist %d\n", req.max_dist);
        nRet = SwiLocSetAutoStart(&req);
        doprintreason( fp, nRet );
        req.set_fix_rate = 1;
        req.fix_rate = 1;
        fprintf(fp, "\t and set fix_rate %d\n", req.fix_rate);
        nRet = SwiLocSetAutoStart(&req);
        doprintreason( fp, nRet );

        tfclose(fp);
    }
}

void GetCurrentUTCTime(void)
{
    struct timeval tv;

    gettimeofday(&tv, NULL);

    utcTime =
    (ULONGLONG)(tv.tv_sec) * 1000 +
    (ULONGLONG)(tv.tv_usec) / 1000;

    fprintf(stderr,"\n UTC Time = %llu, Uncertainty Time = %u\n", utcTime, uncTime );
    return;
}

/*************
 *
 * Name:    cbkTestSLQSSetLocInjectUTCTimeCallback
 *
 * Purpose: SLQSSetLocInjectUTCTimeCallback API callback function
 *
 **************/
local void cbkTestSLQSSetLocInjectUTCTimeCallback(
    QmiCbkLocInjectUTCTimeInd *pInjectUTCTimeNotification  )
{
    FILE *fp;

    fp = tfopen("TestReport/SLQSSetLocInjectUTCTimeCallback.txt", "a");

    if ( fp == NULL )
        perror("cbkTestSLQSSetLocInjectUTCTimeCallback");
    else
    {
        fprintf( fp, "CB : Inject UTC Time \n" );
        fprintf( fp, "Status %u \n",
                 pInjectUTCTimeNotification->status );

        fprintf( fp, "\n\n" );
    }
    tfclose(fp);
}

void doSLQSLOCInjectUTCTime(void)
{
    ULONG               nRet;
    FILE                *fp = NULL;
    /* Define the test cases to be executed */
    SLQSLocInjectUTCTimeTestCase_t SLQSLocInjectUTCTimeTestCases[] =
    {
            { eQCWWAN_ERR_NONE, 1, "Inject UTC Time"}
    };

    fp = tfopen( "../../loc/test/results/slqslocinjectUTCtime.txt", "w" );
    if ( fp )
    {
        ULONG rc;
        BYTE tCaseNum     = 0;
        BYTE ncases = sizeof( SLQSLocInjectUTCTimeTestCases ) /
            sizeof( SLQSLocInjectUTCTimeTestCases[0] );

        rc = SLQSSetLocInjectUTCTimeCallback( &cbkTestSLQSSetLocInjectUTCTimeCallback );
        if ( rc != eQCWWAN_ERR_NONE )
            fprintf( stderr, "SLQSSetLocInjectUTCTimeCallback: Failed : %u\r\n", rc );
        else
            printf( "SLQSSetLocInjectUTCTimeCallback: Enabled callback\r\n" );

        while( tCaseNum < ncases )
        {
            SLQSLocInjectUTCTimeTestCase_t *pd =
                &SLQSLocInjectUTCTimeTestCases[tCaseNum++];

            fprintf(fp, "\nTest Case %d / %d : %s\n",
                    tCaseNum,
                    ncases,
                    pd->desc);

            GetCurrentUTCTime();
            nRet = SLQSLOCInjectUTCTime(utcTime, uncTime);

            /* Display result code and text */
            doprintreason( fp, nRet );
        }
        sleep(1);
        rc = SLQSSetLocInjectUTCTimeCallback( NULL );
        if ( rc != eQCWWAN_ERR_NONE )
            fprintf( stderr, "SLQSSetLocInjectUTCTimeCallback: Failed : %u\r\n", rc );
        else
            printf( "SLQSSetLocInjectUTCTimeCallback: Disabled\r\n" );

    }

    if (fp)
        tfclose(fp);
}

/*************
 *
 * Name:    cbkTestSLQSSetLocInjectPositionCallback
 *
 * Purpose: SLQSSetLocInjectPositionCallback API callback function
 *
 **************/
local void cbkTestSLQSSetLocInjectPositionCallback(
    QmiCbkLocInjectPositionInd *pInjectPositionNotification  )
{
    FILE *fp;

    fp = tfopen("TestReport/SLQSSetLocInjectPositionCallback.txt", "a");

    if ( fp == NULL )
        perror("cbkTestSLQSSetLocInjectPositionCallback");
    else
    {
        fprintf( fp, "CB : Inject Position \n" );
        fprintf( fp, "Status %u \n",
                pInjectPositionNotification->status );

        fprintf( fp, "\n\n" );
    }
    tfclose(fp);
}


void doSLQSLOCInjectPosition(void)
{
    ULONG               nRet;
    FILE                *fp = NULL;

    /* Define the test cases to be executed */
    SLQSLocInjectPositionTestCase_t SLQSLocInjectPositionTestCases[] =
    {
            { eQCWWAN_ERR_NONE, 1, NULL, NULL, NULL, NULL, NULL, NULL,
                     NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
                     NULL, "All parameters NULL"},
            { eQCWWAN_ERR_NONE, 2, NULL, NULL, NULL, NULL, NULL, NULL,
                     NULL, NULL, NULL, NULL, NULL, &utcTime, NULL, NULL, NULL,
                     NULL, "Input UTC Time"},
            { eQCWWAN_ERR_NONE, 3, (ULONGLONG*)&latitude, (ULONGLONG*)&longitude, (ULONG*)&horUnc, NULL, NULL, NULL,
                     NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
                     NULL, "Input Latitude , Longitude & Hor Unc Value"},
            { eQCWWAN_ERR_NONE, 4, NULL, NULL, NULL, NULL, NULL, (ULONG*)&altitudeWrtEllipsoid,
                     NULL, (ULONG*)&vertUnc, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
                     NULL, "Input Altitude wrt Ellipsoid"},
    };

    fp = tfopen( "../../loc/test/results/slqslocinjectposition.txt", "w" );
    if ( fp )
    {
        ULONG rc;
        BYTE tCaseNum     = 0;
        BYTE ncases = sizeof( SLQSLocInjectPositionTestCases ) /
            sizeof( SLQSLocInjectPositionTestCases[0] );

        rc = SLQSSetLocInjectPositionCallback( &cbkTestSLQSSetLocInjectPositionCallback );
        if ( rc != eQCWWAN_ERR_NONE )
            fprintf( stderr, "SLQSSetLocInjectPositionCallback: Failed : %u\r\n", rc );
        else
            printf( "SLQSSetLocInjectPositionCallback: Enabled callback\r\n" );

        while( tCaseNum < ncases )
        {
            SLQSLocInjectPositionTestCase_t *pd =
                &SLQSLocInjectPositionTestCases[tCaseNum++];

            fprintf(fp, "\nTest Case %d / %d : %s\n",
                    tCaseNum,
                    ncases,
                    pd->desc);

            LocInjectPositionReq locInjectPositionReq;
            memset(&locInjectPositionReq, 0, sizeof(locInjectPositionReq));

            if(tCaseNum==2)
                GetCurrentUTCTime();

            locInjectPositionReq.pAltitudeSrcInfo         = pd->pAltitudeSrcInfo;
            locInjectPositionReq.pAltitudeWrtEllipsoid    = pd->pAltitudeWrtEllipsoid;
            locInjectPositionReq.pAltitudeWrtMeanSeaLevel = pd->pAltitudeWrtMeanSeaLevel;
            locInjectPositionReq.pHorConfidence           = pd->pHorConfidence;
            locInjectPositionReq.pHorReliability          = pd->pHorReliability;
            locInjectPositionReq.pHorUncCircular          = pd->pHorUncCircular;
            locInjectPositionReq.pLatitude                = pd->pLatitude;
            locInjectPositionReq.pLongitude               = pd->pLongitude;
            locInjectPositionReq.pPositionSrc             = pd->pPositionSrc;
            locInjectPositionReq.pRawHorConfidence        = pd->pRawHorConfidence;
            locInjectPositionReq.pRawHorUncCircular       = pd->pRawHorUncCircular;
            locInjectPositionReq.pTimestampAge            = pd->pTimestampAge;
            locInjectPositionReq.pTimestampUtc            = pd->pTimestampUtc;
            locInjectPositionReq.pVertConfidence          = pd->pVertConfidence;
            locInjectPositionReq.pVertReliability         = pd->pVertReliability;
            locInjectPositionReq.pVertUnc                 = pd->pVertUnc;

            nRet = SLQSLOCInjectPosition(&locInjectPositionReq);

            /* Display result code and text */
            doprintreason( fp, nRet );
            sleep(1);
        }
        sleep(1);
        rc = SLQSSetLocInjectPositionCallback( NULL );
        if ( rc != eQCWWAN_ERR_NONE )
            fprintf( stderr, "SLQSSetLocInjectPositionCallback: Failed : %u\r\n", rc );
        else
            printf( "SLQSSetLocInjectPositionCallback: Disabled\r\n" );

    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:    doSLQSLOCSetCradleMountConfig
 *
 * Purpose: Perform the tests that call the SLQSLOCSetCradleMountConfig() API
 *
 * Return:  none
 *
 * Notes:   none
 *
 */
void doSLQSLOCSetCradleMountConfig( void )

{
    BYTE confidence  = 70;

    /* Test Cases */
    SLQSLOCSetCradleMountConfigTestCase_t SLQSLOCSetCradleMountConfigTestCase[]=
    {
         { eQCWWAN_ERR_NONE, 1, 1, NULL, "Device mounted on Cradle"},
         { eQCWWAN_ERR_NONE, 2, 1, &confidence, "Device mounted on Cradle with confidence"},
         { eQCWWAN_ERR_NONE, 3, 0, NULL, "Device not mounted on Cradle "},
         { eQCWWAN_ERR_NONE, 4, 9 ,NULL, "InValid Value"},
    };

    LocSetCradleMountReq req;
    FILE *fp = tfopen("../../loc/test/results/slqsloccradlemountconfig.txt", "w");
    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSLOCSetCradleMountConfigTestCase )/
                      sizeof( SLQSLOCSetCradleMountConfigTestCase[0] );
        while ( tc < ncases )
        {
        	SLQSLOCSetCradleMountConfigTestCase_t *pd =
                   &SLQSLOCSetCradleMountConfigTestCase[tc++];
            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        tc,
                        ncases,
                        pd->desc);

            req.state = pd->state;
            req.pConfidence = pd->pConfidence;
            fprintf(fp, "Cradle Mount State = %d\n",req.state);
            if(req.pConfidence != NULL)
            fprintf(fp, "Cradle Mount Confidence = %d\n",*req.pConfidence);
            ULONG nRet = SLQSLOCSetCradleMountConfig( &req );
            doprintreason( fp, nRet );

            if ( eQCWWAN_ERR_NONE == nRet )
            {
                fprintf(fp, "SLQSLOCSetCradleMountConfig Successful\n");
            }
        }
    }
    if (fp)
        tfclose(fp);
}


/*
 * Name:    doSLQSLOCInjectSensorData
 *
 * Purpose: Perform the tests that call the SLQSLOCInjectSensorData() API
 *
 * Return:  none
 *
 * Notes:   none
 *
 */
void doSLQSLOCInjectSensorData( void )

{
    ULONG opaqueIdentifier = 123456;

    sensorData AcceleroData;
    AcceleroData.timeOfFirstSample = 100000;
    AcceleroData.flags             = 0x02;
    AcceleroData.sensorDataLen     = 1;
    AcceleroData.timeOffset[0]     = 10000;
    float xAxis = 10.234;
    float yAxis = 10.235;
    float zAxis = 10.236;
    AcceleroData.xAxis[0]          = *(ULONG*)&xAxis;
    AcceleroData.yAxis[0]          = *(ULONG*)&yAxis;
    AcceleroData.zAxis[0]          = *(ULONG*)&zAxis;

    tempratureData   AcceleroTempData;
    AcceleroTempData.timeOfFirstSample  = 100000;
    AcceleroTempData.timeSource         = 0x01;
    AcceleroTempData.temperatureDataLen = 1;
    AcceleroTempData.timeOffset[0]      = 10000;
    float temperature    = 28.24;
    AcceleroTempData.temperature[0] = *(ULONG*)&temperature;

    /* Test Cases */
    SLQSLOCInjectSensorDataTestCase_t SLQSLOCInjectSensorDataTestCase[]=
    {
         { eQCWWAN_ERR_NONE, 1,NULL, NULL, NULL,NULL, NULL, NULL, NULL, "All Optional Null"},
         { eQCWWAN_ERR_NONE, 2, &opaqueIdentifier, NULL, NULL,NULL, NULL, NULL, NULL, "Opaque Identifier"},
         { eQCWWAN_ERR_NONE, 3, &opaqueIdentifier, &AcceleroData, NULL,NULL, NULL, NULL, NULL, "Accelerometer Data"},
         { eQCWWAN_ERR_NONE, 4, &opaqueIdentifier, &AcceleroData, NULL,NULL, NULL, &AcceleroTempData, NULL, "Accelerometer temperature"}
    };

    LocInjectSensorDataReq req;
    BYTE count;

    FILE *fp = tfopen("../../loc/test/results/slqslocinjectsensordata.txt", "w");
    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSLOCInjectSensorDataTestCase )/
                      sizeof( SLQSLOCInjectSensorDataTestCase[0] );
        while ( tc < ncases )
        {
            SLQSLOCInjectSensorDataTestCase_t *pd =
                   &SLQSLOCInjectSensorDataTestCase[tc++];
            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        tc,
                        ncases,
                        pd->desc);

            req.pOpaqueIdentifier = pd->pOpaqueIdentifier;
            req.pAcceleroData     = pd->pAcceleroData;
            req.pGyroData         = pd->pGyroData;
            req.pAcceleroTimeSrc  = pd->pAcceleroTimeSrc;
            req.pGyroTimeSrc      = pd->pGyroTimeSrc;
            req.pAcceleroTempData = pd->pAcceleroTempData;
            req.pGyroTempData     = pd->pGyroTempData;

            IFPRINTF(fp, "Opaque Identifier = %d\n",req.pOpaqueIdentifier);
            if(NULL != req.pAcceleroData)
            {
                fprintf(fp, "\nAccelerometer Data :  \n");
                fprintf(fp, "Time of First Sample = %d\n",req.pAcceleroData->timeOfFirstSample);
                fprintf(fp, "Flags = %d\n",req.pAcceleroData->flags);
                for(count=0 ; count< req.pAcceleroData->sensorDataLen ; count++)
                {
                    fprintf(fp, "Time offset = %d\n",req.pAcceleroData->timeOffset[count]);
                    fprintf(fp, "Sensor x-Axis = %f\n",*(float*)&req.pAcceleroData->xAxis[count]);
                    fprintf(fp, "Sensor y-Axis = %f\n",*(float*)&req.pAcceleroData->yAxis[count]);
                    fprintf(fp, "Sensor z-Axis = %f\n",*(float*)&req.pAcceleroData->zAxis[count]);
                }
            }
            IFPRINTF(fp, "Accelerometer Time Src = %d\n",req.pAcceleroTimeSrc);
            IFPRINTF(fp, "Gyroscope Time Src = %d\n",req.pGyroTimeSrc);
            if(NULL != req.pAcceleroTempData)
            {
                fprintf(fp, "\nAccelerometer Temperature Data :  \n");
                fprintf(fp, "Time Source = %d\n",req.pAcceleroTempData->timeSource);
                fprintf(fp, "Time of First Sample = %d\n",req.pAcceleroTempData->timeOfFirstSample);
                for(count=0 ; count< req.pAcceleroTempData->temperatureDataLen ; count++)
                {
                    fprintf(fp, "Time offset = %d\n",req.pAcceleroTempData->timeOffset[count]);
                    fprintf(fp, "Sensor Temperature = %f\n",*(float*)&req.pAcceleroTempData->temperature[count]);
                }
            }

            ULONG nRet = SLQSLOCInjectSensorData( &req );
            doprintreason( fp, nRet );

            if ( eQCWWAN_ERR_NONE == nRet )
            {
                fprintf(fp, "SLQSLOCInjectSensorData Successful\n");
            }
        }
    }
    if (fp)
        tfclose(fp);
}
