/**************
 *
 *  Filename: qaGobiApiDmsTest.c
 *
 *  Purpose:  Contains functions implementing specific tests for
 *            Device Management Service (DMS)
 *            called by the main() routine in qatest.c
 *
 * Copyright: © 2011-2013 Sierra Wireless, Inc., all rights reserved
 *
 **************/

/* Linux definitions */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* include files */
#include "SwiDataTypes.h"
#include "qmerrno.h"
#include "qaGobiApiDms.h"
#include "qaGobiApiDmsTest.h"
#include "qatesthelper.h"
#include "qatestproto.h"
#include "sludefs.h"

/*******************
    TEST DATA
 *******************/
/* Constants */
#define QMI_DMS_PM_ONLINE         0x00 /* Online */
#define QMI_DMS_PM_LOW            0x01 /* Low Power */
#define QMI_DMS_PM_FACTORY        0x02 /* Factory Test Mode */
#define QMI_DMS_PM_OFFLINE        0x03 /* Offline */
#define QMI_DMS_PM_RESET          0x04 /* Reset */
#define QMI_DMS_PM_SHUT_DOWN      0x05 /* Shut Down */
#define QMI_DMS_PM_PERSISTENT_LOW 0x06 /* Persistent Low Power */
/* Global Variables */

/* Get Custom Features parameters */

local BYTE disableIMSI;
local WORD IPFamSupport;
local BYTE RMAutoConn;
local BYTE GPSSel;
local BYTE SMSSupport;
local BYTE isVoiceEnab;
local BYTE DHCPRelayEnab;
local BYTE GPSLPMEnab;
local custFeaturesInfo CustFeatures;

/* Set Custom Features parameters*/
local BYTE  GPSSelSetting = 0x00;
local ULONG GPSEnable[3] = {0, 1, 9};
local BYTE  IsVoiceEnableFalse = 2;
local BYTE  IsVoiceEnableTrue  = 0;
local BYTE  DHCPRelayDisabled  = 0;
local BYTE  DHCPRelayEnabled   = 1;
local BYTE  GPSLPMEnabled      = 0;
local BYTE  GPSLPMDisabled     = 1;

/*******************
  TEST FUNCTIONS
 *******************/

/*
 * Name:     doGetManufacturer
 *
 * Purpose:  Perform the tests that call the GetManufacturer() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doGetManufacturer( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    BYTE  stringSize = nMaxStrLen;
    CHAR  mfrString[stringSize];
    FILE  *fp = NULL;

    fp = tfopen("../../dms/test/results/getmanufacturer.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_GET_MANUFACTURER_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_GET_MANUFACTURER_TESTCASE_NUM,
                       GetManufacturerTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "Stringsize :%d\n", stringSize);

            nRet = GetManufacturer(stringSize,
                                   &mfrString[0]);

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "Manufacturer : %s\n", mfrString);
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doGetModelID
 *
 * Purpose:  Perform the tests that call the GetModelID() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doGetModelID( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    BYTE  stringSize = nMaxStrLen;
    CHAR  modelString[stringSize];
    FILE  *fp = NULL;

    fp = tfopen("../../dms/test/results/getmodelid.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_GET_MODEL_ID_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_GET_MODEL_ID_TESTCASE_NUM,
                       GetModelIDTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "Stringsize :%d\n", stringSize);

            nRet = GetModelID(stringSize,
                              &modelString[0]);

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "Model : %s\n", modelString);
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doGetFirmwareRevision
 *
 * Purpose:  Perform the tests that call the GetFirmwareRevision() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doGetFirmwareRevision( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    BYTE stringSize = nMaxStrLen;
    CHAR fmrString[stringSize];
    FILE  *fp = NULL;

    fp = tfopen("../../dms/test/results/getfirmwarerevision.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_GET_FIRMWARE_REVISION_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_GET_FIRMWARE_REVISION_TESTCASE_NUM,
                       GetFirmwareRevisionTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "Stringsize :%d\n", stringSize);

            nRet = GetFirmwareRevision(stringSize,
                                       &fmrString[0]);

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "Firmware Revision : %s\n", fmrString);
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doGetFirmwareRevisions
 *
 * Purpose:  Perform the tests that call the GetFirmwareRevisions() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doGetFirmwareRevisions( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    BYTE  amssSize = nMaxStrLen;
    CHAR  AMSString[nMaxStrLen];
    BYTE  bootSize = nMaxStrLen;
    CHAR  bootString[nMaxStrLen];
    BYTE  priSize = nMaxStrLen;
    CHAR  PRIString[nMaxStrLen];
    FILE  *fp = NULL;

    fp = tfopen("../../dms/test/results/getfirmwarerevisions.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_GET_FIRMWARE_REVISIONS_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_GET_FIRMWARE_REVISIONS_TESTCASE_NUM,
                       GetFirmwareRevisionsTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "amssSize :%d\n", amssSize);
            fprintf(fp, "bootSize :%d\n", bootSize);
            fprintf(fp, "priSize :%d\n", priSize);

            nRet = GetFirmwareRevisions(amssSize,
                                        &AMSString[0],
                                        bootSize,
                                        &bootString[0],
                                        priSize,
                                        &PRIString[0]);

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "AMS : %s\n", AMSString);
                fprintf(fp, "Boot : %s\n", bootString);
                fprintf(fp, "PRI : %s\n", PRIString);
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doGetPRLVersion
 *
 * Purpose:  Perform the tests that call the GetPRLVersion() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doGetPRLVersion( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    FILE  *fp = NULL;
    WORD  prlVersion;

    fp = tfopen("../../dms/test/results/getprlversion.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_GET_PRL_VERSION_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_GET_PRL_VERSION_TESTCASE_NUM,
                       GetPRLVersionTestCases[tCaseNum].desc);

            nRet = GetPRLVersion(&prlVersion);

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "PRL Version : %d\n", prlVersion);
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doGetIMSI
 *
 * Purpose:  Perform the tests that call the GetIMSI() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doGetIMSI( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    BYTE  stringSize = nMaxStrLen;
    CHAR  imsiString[stringSize];
    FILE  *fp = NULL;

    fp = tfopen("../../dms/test/results/getimsi.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_GET_IMSI_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_GET_IMSI_TESTCASE_NUM,
                       GetIMSITestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "Stringsize :%d\n", stringSize);

            nRet = GetIMSI(stringSize, &imsiString[0]);

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "IMSI : %s\n", imsiString);
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doGetSerialNumbers
 *
 * Purpose:  Perform the tests that call the GetSerialNumbers() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doGetSerialNumbers( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    BYTE  esnSize = nMaxStrLen;
    CHAR  ESNString[nMaxStrLen];
    BYTE  imeiSize = nMaxStrLen;
    CHAR  IMEIString[nMaxStrLen];
    BYTE  meidSize = nMaxStrLen;
    CHAR  MEIDString[nMaxStrLen];
    FILE  *fp = NULL;

    fp = tfopen("../../dms/test/results/getserialnumbers.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_GET_SERIAL_NUMBERS_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_GET_SERIAL_NUMBERS_TESTCASE_NUM,
                       GetSerialNumbersTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "esnSize :%d\n", esnSize);
            fprintf(fp, "imeiSize :%d\n", imeiSize);
            fprintf(fp, "meidSize :%d\n", meidSize);

            nRet = GetSerialNumbers(esnSize,
                                    &ESNString[0],
                                    imeiSize,
                                    &IMEIString[0],
                                    meidSize,
                                    &MEIDString[0]);

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "ESNString : %s\n", ESNString);
                fprintf(fp, "IMEIString : %s\n", IMEIString);
                fprintf(fp, "MEIDString : %s\n", MEIDString);
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doGetHardwareRevision
 *
 * Purpose:  Perform the tests that call the GetHardwareRevision() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doGetHardwareRevision( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    BYTE  stringSize = nMaxStrLen;
    CHAR  hwString[stringSize];
    FILE  *fp = NULL;

    fp = tfopen("../../dms/test/results/gethardwarerevision.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_GET_HARDWARE_REVISION_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_GET_HARDWARE_REVISION_TESTCASE_NUM,
                       GetHardwareRevisionTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "Stringsize :%d\n", stringSize);

            nRet = GetHardwareRevision(stringSize,
                                       &hwString[0]);
            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "Hardware Revision : %s\n", hwString);
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doGetNetworkTimeTime
 *
 * Purpose:  Perform the tests that call the GetNetworkTime() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doGetNetworkTime( void )
{
    BYTE       tCaseNum = 0;
    ULONG      nRet;
    FILE       *fp = NULL;

    fp = tfopen("../../dms/test/results/getnetworktime.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_GET_NETWORK_TIME_TESTCASE_NUM)
        {
            ULONG      *timeSource;
            ULONGLONG  *timeStamp;
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       (tCaseNum + 1),
                       MAX_GET_NETWORK_TIME_TESTCASE_NUM,
                       GetNetworkTimeTestCases[tCaseNum].desc);

            timeStamp  = GetNetworkTimeTestCases[tCaseNum].timeStamp;
            timeSource = GetNetworkTimeTestCases[tCaseNum].timeSource;
            fprintf(fp, "There are no IN paramenters\n");

            nRet = GetNetworkTime( timeStamp, timeSource );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "TimeStamp is : %llx\n", *timeStamp);
                fprintf(fp, "TimeSource is : %x\n", *timeSource);
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doUIMSetPINProtection
 *
 * Purpose:  Perform the tests that call the UIMSetPINProtection() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doUIMSetPINProtection( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    FILE  *fp = NULL;

    fp = tfopen("../../dms/test/results/uimsetpinprotection.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_UIM_SET_PIN_PROTECTION_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                        (tCaseNum + 1),
                        MAX_UIM_SET_PIN_PROTECTION_TESTCASE_NUM,
                        UIMSetPINProtectionTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "Id : %u\n", UIMSetPINProtectionTestCases[tCaseNum].id);
            fprintf(fp, "bEnable : %u\n", UIMSetPINProtectionTestCases[tCaseNum].bEnable);
            fprintf(fp, "Value : %s\n", UIMSetPINProtectionTestCases[tCaseNum].value);

            nRet = UIMSetPINProtection( UIMSetPINProtectionTestCases[tCaseNum].id,
                       UIMSetPINProtectionTestCases[tCaseNum].bEnable,
                       (CHAR *)&UIMSetPINProtectionTestCases[tCaseNum].value[0],
                       UIMSetPINProtectionTestCases[tCaseNum].pVerifyRetries,
                       UIMSetPINProtectionTestCases[tCaseNum].pUnblockRetries );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if ( !nRet )
                fprintf(fp, "Set PIN Protection Successful\n");

            fprintf(fp, "Verify Retries Left: %x\n", verifyRetries);
            fprintf(fp, "Unblock Retries Left: %x\n", unblockRetries);

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doUIMUnblockPIN
 *
 * Purpose:  Perform the tests that call the UIMUnblockPIN() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doUIMUnblockPIN( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    FILE  *fp = NULL;
    ULONG verifyRetriesLeft = 0;
    ULONG unblockRetriesLeft = 0;

    fp = tfopen("../../dms/test/results/uimunblockpin.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_UIM_UNBLOCK_PIN_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                        (tCaseNum + 1),
                        MAX_UIM_UNBLOCK_PIN_TESTCASE_NUM,
                        UIMUnblockPINTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "Id : %u\n", UIMUnblockPINTestCases[tCaseNum].id);
            fprintf(fp, "PUK Value : %s\n", UIMUnblockPINTestCases[tCaseNum].pukValue);
            fprintf(fp, "New Value : %s\n", UIMUnblockPINTestCases[tCaseNum].newValue);

            nRet = UIMUnblockPIN( UIMUnblockPINTestCases[tCaseNum].id,
                                  (CHAR *)&UIMUnblockPINTestCases[tCaseNum].pukValue[0],
                                  (CHAR *)&UIMUnblockPINTestCases[tCaseNum].newValue[0],
                                  &verifyRetriesLeft,
                                  &unblockRetriesLeft );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if ( !nRet )
                fprintf(fp, "Unblock PIN Successful\n");

            fprintf(fp, "Verify Retries Left: %u\n", verifyRetriesLeft);
            fprintf(fp, "Unblock Retries Left: %u\n", unblockRetriesLeft);

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}


/*
 * Name:     doUIMVerifyPIN
 *
 * Purpose:  Perform the tests that call the UIMVerifyPIN() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doUIMVerifyPIN( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    FILE  *fp = NULL;
    ULONG verifyRetriesLeft = 0;
    ULONG unblockRetriesLeft = 0;

    fp = tfopen("../../dms/test/results/uimverifypin.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_UIM_VERIFY_PIN_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                        (tCaseNum + 1),
                        MAX_UIM_VERIFY_PIN_TESTCASE_NUM,
                        UIMVerifyPINTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "Id : %u\n", UIMVerifyPINTestCases[tCaseNum].id);
            fprintf(fp, "Value : %s\n", UIMVerifyPINTestCases[tCaseNum].value);

            nRet = UIMVerifyPIN( UIMVerifyPINTestCases[tCaseNum].id,
                                 (CHAR *)&UIMVerifyPINTestCases[tCaseNum].value[0],
                                 &verifyRetriesLeft,
                                 &unblockRetriesLeft );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if ( !nRet )
                fprintf(fp, "Verify PIN Successful\n");

            fprintf(fp, "Verify Retries Left: %u\n", verifyRetriesLeft);
            fprintf(fp, "Unblock Retries Left: %u\n", unblockRetriesLeft);

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doUIMChangePIN
 *
 * Purpose:  Perform the tests that call the UIMChangePIN() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doUIMChangePIN( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    FILE  *fp = NULL;
    ULONG verifyRetriesLeft = 0;
    ULONG unblockRetriesLeft = 0;

    fp = tfopen("../../dms/test/results/uimchangepin.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_UIM_CHANGE_PIN_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                        (tCaseNum + 1),
                        MAX_UIM_CHANGE_PIN_TESTCASE_NUM,
                        UIMChangePINTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "Id : %u\n", UIMChangePINTestCases[tCaseNum].id);
            fprintf(fp, "Old Value : %s\n", UIMChangePINTestCases[tCaseNum].oldValue);
            fprintf(fp, "New Value : %s\n", UIMChangePINTestCases[tCaseNum].newValue);

            nRet = UIMChangePIN( UIMChangePINTestCases[tCaseNum].id,
                                 (CHAR *)&UIMChangePINTestCases[tCaseNum].oldValue[0],
                                 (CHAR *)&UIMChangePINTestCases[tCaseNum].newValue[0],
                                 &verifyRetriesLeft,
                                 &unblockRetriesLeft );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if ( !nRet )
                fprintf(fp, "Change PIN Successful\n");

            fprintf(fp, "Verify Retries Left: %u\n", verifyRetriesLeft);
            fprintf(fp, "Unblock Retries Left: %u\n", unblockRetriesLeft);

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doGetVoiceNumber
 *
 * Purpose:  Perform the tests that call the GetVoiceNumber() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doGetVoiceNumber( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    FILE  *fp = NULL;

    fp = tfopen("../../dms/test/results/GetVoiceNumber.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_GET_VOICE_NUMBER_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                        (tCaseNum + 1),
                        MAX_GET_VOICE_NUMBER_TESTCASE_NUM,
                        GetVoiceNumberTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf( fp,
                     "Voice Numer Size : %d\n",
                     GetVoiceNumberTestCases[tCaseNum].voiceNumberSize );
            fprintf( fp,
                     "MIN Size : %d\n",
                     GetVoiceNumberTestCases[tCaseNum].minSize );

            nRet = GetVoiceNumber( GetVoiceNumberTestCases[tCaseNum].voiceNumberSize,
                                   GetVoiceNumberTestCases[tCaseNum].pVoiceNumber,
                                   GetVoiceNumberTestCases[tCaseNum].minSize,
                                   GetVoiceNumberTestCases[tCaseNum].pMIN );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if ( !nRet )
            {
                fprintf(fp, "Voice Number Retrieved\n");
                fprintf( fp,
                         "Voice Number : %s\n",
                         GetVoiceNumberTestCases[tCaseNum].pVoiceNumber );
                fprintf( fp,
                         "MIN: %s\n",
                         GetVoiceNumberTestCases[tCaseNum].pMIN );
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}


/*
 * Name:     doSetPower
 *
 * Purpose:  Perform the tests that call the SetPower() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSetPower( void )
{
    ULONG nRet;
    SetPowerTestCase_t SetPowerTestCases[] =
    {
        {eQCWWAN_ERR_NONE, 1, QMI_DMS_PM_ONLINE-1,
         "Invalid Power Mode : Lower Bounday Check"},
        {eQCWWAN_ERR_NONE, 1, QMI_DMS_PM_LOW,
         "Valid : QMI_DMS_PM_LOW"},
        {eQCWWAN_ERR_NONE, 1, QMI_DMS_PM_FACTORY,
         "Valid : QMI_DMS_PM_FACTORY"},
        {eQCWWAN_ERR_NONE, 1, QMI_DMS_PM_OFFLINE,
         "Valid : QMI_DMS_PM_OFFLINE"},
        {eQCWWAN_ERR_NONE, 1, QMI_DMS_PM_SHUT_DOWN,
         "Valid : QMI_DMS_PM_SHUT_DOWN"},
        {eQCWWAN_ERR_NONE, 1, QMI_DMS_PM_PERSISTENT_LOW,
         "Valid : QMI_DMS_PM_PERSISTENT_LOW"},
        {eQCWWAN_ERR_NONE, 1, QMI_DMS_PM_PERSISTENT_LOW + 1,
         "Invalid Power Mode : Upper Bounday Check"},
        {eQCWWAN_ERR_NONE, 1, QMI_DMS_PM_ONLINE,
         "Valid : QMI_DMS_PM_ONLINE"},
    };

    FILE *fp = tfopen("../../dms/test/results/setpower.txt", "w");

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SetPowerTestCases ) /
                      sizeof( SetPowerTestCases[0] );
        while( tc < ncases )
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                        (tc + 1),
                        ncases,
                        SetPowerTestCases[tc].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "Power Mode : %u\n", SetPowerTestCases[tc].powerMode);

            nRet = SetPower( SetPowerTestCases[tc].powerMode );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "Set Power Successful \n");
            }

            tc++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doGetPower
 *
 * Purpose:  Perform the tests that call the GetPower() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doGetPower( void )
{
    ULONG nRet;

    GetPowerTestCase_t GetPowerTestCases[] =
    {
        {eQCWWAN_ERR_NONE, 1, "ALL Valid"}
    };

    FILE *fp = tfopen("../../dms/test/results/getpower.txt", "w");

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( GetPowerTestCases ) /
                      sizeof( GetPowerTestCases[0] );
        while (tc < ncases)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                        (tc + 1),
                        ncases,
                        GetPowerTestCases[tc].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "There are no IN parameters\n");

            ULONG powerMode = 0;
            nRet = GetPower( &powerMode );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "Get Power Successful \n");
                fprintf(fp, "Power Mode is : %u\n", powerMode);
            }

            tc++;
        }
    }

    if(fp)
        tfclose(fp);
}

/*
 * Name:     doUIMGetControlKeyStatus
 *
 * Purpose:  Perform the tests that call the UIMGetControlKeyStatus() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doUIMGetControlKeyStatus( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    ULONG status = 0;
    ULONG verifyRetriesLeft = 0;
    ULONG unblockRetriesLeft = 0;
    FILE  *fp = NULL;

    fp = tfopen("../../dms/test/results/uimgetcontrolkeystatus.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_UIM_GET_CK_STATUS_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                        (tCaseNum + 1),
                        MAX_UIM_GET_CK_STATUS_TESTCASE_NUM,
                        UIMGetCKStatusTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "Id : %u\n", UIMGetCKStatusTestCases[tCaseNum].id);

            nRet = UIMGetControlKeyStatus( UIMGetCKStatusTestCases[tCaseNum].id,
                                           &status,
                                           &verifyRetriesLeft,
                                           &unblockRetriesLeft );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "Required Control Key Status is as follows: \n");
                fprintf(fp, "Status : %u\n",status);
                fprintf(fp, "Verify Retries Left : %u\n",verifyRetriesLeft);
                fprintf(fp, "Unblock Retries Left : %u\n",unblockRetriesLeft);
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doUIMGetICCID
 *
 * Purpose:  Perform the tests that call the UIMGetICCID() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doUIMGetICCID( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;

    FILE  *fp = NULL;

    fp = tfopen("../../dms/test/results/uimgeticcid.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_UIM_GET_ICCID_TESTCASE_NUM)
        {
            CHAR  *string;
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                        (tCaseNum + 1),
                        MAX_UIM_GET_ICCID_TESTCASE_NUM,
                        UIMGetICCIDTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "Size : %d\n", UIMGetICCIDTestCases[tCaseNum].strSize);
            string = UIMGetICCIDTestCases[tCaseNum].pString;

            nRet = UIMGetICCID( UIMGetICCIDTestCases[tCaseNum].strSize,
                                &string[0] );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "UIMGetICCID Successful \n");
                fprintf(fp, "ICCID is : %s\n", string);
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doUIMGetPINStatus
 *
 * Purpose:  Perform the tests that call the UIMGetPINStatus() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doUIMGetPINStatus( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;

    ULONG verifyRetriesLeft;
    ULONG unblockRetriesLeft;
    FILE  *fp = NULL;

    fp = tfopen("../../dms/test/results/uimgetpinstatus.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_UIM_GET_PIN_STATUS_TESTCASE_NUM)
        {
            ULONG *status;
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                        (tCaseNum + 1),
                        MAX_UIM_GET_PIN_STATUS_TESTCASE_NUM,
                        UIMGetPINStatusTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "Id : %u\n", UIMGetPINStatusTestCases[tCaseNum].id);
            status = UIMGetPINStatusTestCases[tCaseNum].pStatus;

            nRet = UIMGetPINStatus( UIMGetPINStatusTestCases[tCaseNum].id,
                                    status,
                                    &verifyRetriesLeft,
                                    &unblockRetriesLeft );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "Pin Status is as follows\n");
                fprintf(fp, "Status : %u\n", *status);
            }

            fprintf(fp, "Verify Retries Left : %u\n", verifyRetriesLeft);
            fprintf(fp, "Unblock Retries Left : %u\n", unblockRetriesLeft);

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doGetOfflineReason
 *
 * Purpose:  Perform the tests that call the GetOfflineReason() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doGetOfflineReason( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    FILE  *fp = NULL;

    fp = tfopen("../../dms/test/results/getofflinereason.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_GET_OFFLINE_REASON_TESTCASE_NUM)
        {
            ULONG *Platform;
            ULONG *ReasonMask;
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                        (tCaseNum + 1),
                        MAX_GET_OFFLINE_REASON_TESTCASE_NUM,
                        GetOfflineReasonTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "There are no IN parameters\n");
            ReasonMask = GetOfflineReasonTestCases[tCaseNum].ReasonMask;
            Platform   = GetOfflineReasonTestCases[tCaseNum].Platform;

            nRet =GetOfflineReason( ReasonMask, Platform );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "Get Offline Reason Successful \n");
                fprintf(fp, "ReasonMask is : %x\n", *ReasonMask);
                fprintf(fp, "Platform is : %x\n", *ReasonMask);
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doUIMSetCkProtection
 *
 * Purpose:  Perform the tests that call the UIMSetControlKeyProtection() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doUIMSetCkProtection( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    ULONG verifyRetriesLeft = 0;
    FILE  *fp = NULL;

    fp = tfopen("../../dms/test/results/uimsetcontrolkeyprotection.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_UIM_SET_CK_PROTECTION_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                        (tCaseNum + 1),
                        MAX_UIM_SET_CK_PROTECTION_TESTCASE_NUM,
                        UIMSetCkProtectionTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "Id : %u\n", UIMSetCkProtectionTestCases[tCaseNum].id);
            fprintf(fp, "Status : %u\n", UIMSetCkProtectionTestCases[tCaseNum].status);
            fprintf(fp, "Value : %s\n", UIMSetCkProtectionTestCases[tCaseNum].value);

            nRet = UIMSetControlKeyProtection(
                       UIMSetCkProtectionTestCases[tCaseNum].id,
                       UIMSetCkProtectionTestCases[tCaseNum].status,
                       (CHAR *)UIMSetCkProtectionTestCases[tCaseNum].value,
                       &verifyRetriesLeft );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
                fprintf(fp, "Control Key Protection Successful \n");

            fprintf(fp, "Verify Retries Left : %u\n",verifyRetriesLeft);

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doUIMUnblockCk
 *
 * Purpose:  Perform the tests that call the UIMUnblockControlKey() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doUIMUnblockCk( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    ULONG unblockRetriesLeft = 0;
    FILE  *fp = NULL;

    fp = tfopen("../../dms/test/results/uimunblockcontrolkey.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_UIM_UNBLOCK_CK_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                        (tCaseNum + 1),
                        MAX_UIM_UNBLOCK_CK_TESTCASE_NUM,
                        UIMUnblockCkTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "Id : %u\n", UIMUnblockCkTestCases[tCaseNum].id);
            fprintf(fp, "Value : %s\n", UIMUnblockCkTestCases[tCaseNum].value);

            nRet = UIMUnblockControlKey( UIMUnblockCkTestCases[tCaseNum].id,
                                         (CHAR *)UIMUnblockCkTestCases[tCaseNum].value,
                                         &unblockRetriesLeft );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
                fprintf(fp, "Control Key Unblock Successful \n");

            fprintf(fp, "Unblock Retries Left : %u\n",unblockRetriesLeft);
            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     DoGetDeviceCapabilities
 *
 * Purpose:  Perform the tests that call the GetDeviceCapabilities() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doGetDeviceCapabilities ( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    BYTE  count;
    ULONG maxTXChannelRate ;
    ULONG maxRXChannelRate ;
    ULONG dataServiceCapability ;
    ULONG simCapability ;
    ULONG radioIfacesSize;
    BYTE  radioIfaces [nMaxStrLen];
    FILE  *fp = NULL;

    fp = tfopen("../../dms/test/results/devicecapabilties.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_DEVICE_CAP_TESTCASE_NUM)
        {
            radioIfacesSize = GetDeviceCapabilitiesTestCases[tCaseNum]
                                                  .radioIfacesSize;

            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_DEVICE_CAP_TESTCASE_NUM,
                       GetDeviceCapabilitiesTestCases[tCaseNum].desc);
            fprintf(fp, "RadioIfacesSize: %u\n", radioIfacesSize);
            nRet = GetDeviceCapabilities( &maxTXChannelRate,
                                          &maxRXChannelRate,
                                          &dataServiceCapability,
                                          &simCapability,
                                          &radioIfacesSize,
                                          (BYTE *)&radioIfaces[0] );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "MaxTXChannelRate : %x\n", maxTXChannelRate );
                fprintf(fp, "MaxRXChannelRate : %x\n", maxRXChannelRate );
                fprintf(fp, "DataServiceCapability : %x\n",
                                                   dataServiceCapability );
                fprintf(fp, "SimCapability : %x\n", simCapability );
                fprintf(fp, "RadioIfacesSize: %x\n", radioIfacesSize);
                for(count = 0; count < radioIfacesSize; count++)
                {
                    fprintf(fp, "radioInterface[%d] : %x\n",
                            count, radioIfaces[count]);
                }

            }
            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doResetToFactoryDefaults
 *
 * Purpose:  Perform the tests that call the ResetToFactoryDefaults() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doResetToFactoryDefaults( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    FILE  *fp = NULL;

    fp = tfopen("../../dms/test/results/resettofactorydefaults.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_RESET_FACTORY_DEFAULTS_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_RESET_FACTORY_DEFAULTS_TESTCASE_NUM,
                       ResetFactoryDefaultsTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "pSPC :%s\n",
                    ResetFactoryDefaultsTestCases[tCaseNum].pSPC);


            nRet = ResetToFactoryDefaults(
                     (CHAR *)&ResetFactoryDefaultsTestCases[tCaseNum].pSPC[0] );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "Factory Defaults Restored\n");
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:     doValidateSPC
 *
 * Purpose:  Perform the tests that call the ValidateSPC() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doValidateSPC( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    FILE  *fp = NULL;

    fp = tfopen("../../dms/test/results/validatespc.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_VALIDATE_SPC_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_VALIDATE_SPC_TESTCASE_NUM,
                       ValidateSpcTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "pSPC :%s\n",
                    ValidateSpcTestCases[tCaseNum].pSPC);


            nRet = ValidateSPC( (CHAR *)ValidateSpcTestCases[tCaseNum].pSPC );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "SPC Validated\n");
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

/**
 * Name:     doActivateAutomatic
 *
 * Purpose:  Perform the tests that call the ActivateAutomatic() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doActivateAutomatic( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    FILE  *fp = NULL;

    fp = tfopen("../../dms/test/results/activateautomatic.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_ACTIVATE_AUTOMATIC_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                        (tCaseNum + 1),
                        MAX_ACTIVATE_AUTOMATIC_TESTCASE_NUM,
                        ActivateAutomaticTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "Length of Activation Code : %c\n",\
                ActivateAutomaticTestCases[tCaseNum].activationCode[0]);
            fprintf(fp, "Activation Code : %s\n",\
                &(ActivateAutomaticTestCases[tCaseNum].activationCode[1]));

            nRet = ActivateAutomatic( (CHAR *) (ActivateAutomaticTestCases[tCaseNum].\
                                                activationCode) );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "Activate Automatic Successful");
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

void doGetActivationState( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    ULONG pActivationState;
    FILE  *fp = NULL;

    fp = tfopen("../../dms/test/results/getactivationstate.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_GET_ACTIVATION_STATE_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_GET_ACTIVATION_STATE_TESTCASE_NUM,
                       GetActivationStateTestCases[tCaseNum].desc);

            nRet = GetActivationState( &pActivationState );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "pActivationState : %u\n", pActivationState);
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

void doUIMGetState( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    ULONG pUIMState;
    FILE  *fp = NULL;

    fp = tfopen("../../dms/test/results/uimgetstate.txt", "w");

    if (fp)
    {
        while (tCaseNum < MAX_UIM_GET_STATE_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_UIM_GET_STATE_TESTCASE_NUM,
                       UIMGetStateTestCases[tCaseNum].desc);

            nRet = SLQSUIMGetState( &pUIMState );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "pUIMState : %u\n", pUIMState);
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

void doSLQSgetBandCapability( void )
{
    BYTE  tCaseNum = 0;
    ULONG nRet;
    FILE  *fp = NULL;

    fp = tfopen("../../dms/test/results/SLQSgetBandCapability.txt", "a");

    if (fp)
    {
        while (tCaseNum < MAX_GET_BANDCAPABLTY_TESTCASE_NUM)
        {
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_GET_BANDCAPABLTY_TESTCASE_NUM,
                       GetBandCapabilityTestCases[tCaseNum].desc);

            nRet = SLQSGetBandCapability(
                    GetBandCapabilityTestCases[tCaseNum].pBandCapability );

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "pBandCapability : %llx\n",
                    *(GetBandCapabilityTestCases[tCaseNum].pBandCapability));
            }
            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

void doSLQSGetCustFeatures( void )
{
    /* Test Cases */
    SLQSGetCustFeaturesTestCase_t SLQSGetCustFeaturesTestCases[] =
    {
        { eQCWWAN_ERR_INVALID_ARG, 1, NULL,          "Invalid Argument" },
        { eQCWWAN_ERR_NONE,        2, &CustFeatures, "Valid Test Case" }
    };

    FILE *fp = tfopen("../../dms/test/results/SLQSGetCustFeatures.txt", "w");

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSGetCustFeaturesTestCases )/
                      sizeof( SLQSGetCustFeaturesTestCases[0] );
        while ( tc < ncases )
        {
            custFeaturesInfo *pResp;
            SLQSGetCustFeaturesTestCase_t *pd =
                &SLQSGetCustFeaturesTestCases[tc];

            pResp = pd->pCustFeaturesInfo;
            if( pResp )
            {
                pResp->pDisableIMSI      = &disableIMSI;
                pResp->pIPFamSupport     = &IPFamSupport;
                pResp->pRMAutoConnect    = &RMAutoConn;
                pResp->pGPSSel           = &GPSSel;
                pResp->pSMSSupport       = &SMSSupport;
                pResp->pIsVoiceEnabled   = &isVoiceEnab;
                pResp->pDHCPRelayEnabled = &DHCPRelayEnab;
                pResp->pGPSLPM           = &GPSLPMEnab;
            }

            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        (tc + 1),
                        ncases,
                        pd->desc);

            ULONG nRet = SLQSGetCustFeatures( pResp );

            doprintreason( fp, nRet );
            if ( eQCWWAN_ERR_NONE == nRet )
            {
                fprintf(fp, "SLQSGetCustFeatures Successful\n");
                fprintf(fp, "GPS Enable       : 0x%X\n", pResp->GpsEnable);
                IFPRINTF(fp,"Disable IMSI     : 0x%X\n",  pResp->pDisableIMSI);
                IFPRINTF(fp,"IP Family support: 0x%X\n",  pResp->pIPFamSupport);
                IFPRINTF(fp,"RM Auto Connect  : 0x%X\n",  pResp->pRMAutoConnect);
                IFPRINTF(fp,"GPS Select       : 0x%X\n",  pResp->pGPSSel);
                IFPRINTF(fp,"SMS Support      : 0x%X\n",  pResp->pSMSSupport);
                IFPRINTF(fp,"Is Voice Enabled : 0x%X\n",  pResp->pIsVoiceEnabled);
                IFPRINTF(fp,"DHCP Relay Enabled : 0x%X\n", pResp->pDHCPRelayEnabled);
                IFPRINTF(fp,"GPSLPM           : 0x%X\n", pResp->pGPSLPM);
            }
            tc++;
        }
    }

    if (fp)
        tfclose(fp);
}

void doSLQSSetCustFeatures( void )
{
    /* Test Cases */
    SLQSSetCustFeaturesTestCase_t SLQSSetCustFeaturesTestCases[] =
    {
        { eQCWWAN_ERR_INVALID_ARG, 1, NULL, NULL, NULL, NULL, NULL, "Invalid Argument" },

        { eQCWWAN_ERR_NONE,        2, &GPSSelSetting, &GPSEnable[0], NULL,
          NULL, NULL, "Valid Test Case, GPS Select = 0, GPS enable = 0, Rest NULL" },

        { eQCWWAN_ERR_NONE,        3, &GPSSelSetting, &GPSEnable[1], NULL,
          NULL, NULL, "Valid Test Case, GPS Select = 0, GPS enable = 1, Rest NULL" },

        { eQCWWAN_ERR_QMI_INVALID_ARG, 4, &GPSSelSetting, &GPSEnable[2], NULL,
          NULL, NULL, "Invalid Test Case, GPS Select = 0, GPS enable = 9, Rest NULL" },

        { eQCWWAN_ERR_NONE,        5, NULL, NULL, &IsVoiceEnableFalse,
          NULL, NULL, "Valid Test Case, Voice enable = 2(False), Rest NULL" },

        { eQCWWAN_ERR_NONE,        6, NULL, NULL, &IsVoiceEnableTrue,
          NULL, NULL, "Valid Test Case, Voice enable = 0(True), Rest NULL" },

        { eQCWWAN_ERR_NONE,        7, NULL, NULL, NULL, &DHCPRelayDisabled,
          NULL, "Valid Test Case, DHCP Relay disabled, Rest NULL" },

        { eQCWWAN_ERR_NONE,        8, NULL, NULL, NULL, &DHCPRelayEnabled,
          NULL, "Valid Test Case, DHCP Relay enabled, Rest NULL" },

        { eQCWWAN_ERR_NONE,        9, NULL, NULL, NULL, NULL,
          &GPSLPMEnabled, "Valid Test Case, GPSLPM enabled(=0), Rest NULL" },

        { eQCWWAN_ERR_NONE,        10, NULL, NULL, NULL, NULL,
          &GPSLPMDisabled, "Valid Test Case, GPSLPM disabled(=1), Rest NULL" },
    };

    FILE *fp = tfopen("../../dms/test/results/SLQSSetCustFeatures.txt", "w");
    custFeaturesSetting Resp;

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSSetCustFeaturesTestCases )/
                      sizeof( SLQSSetCustFeaturesTestCases[0] );
        while ( tc < ncases )
        {
            SLQSSetCustFeaturesTestCase_t *pd =
                &SLQSSetCustFeaturesTestCases[tc];

            Resp.pGPSSel           = pd->pGPSSel;
            Resp.pGPSEnable        = pd->pGPSEnable;
            Resp.pIsVoiceEnabled   = pd->pIsVoiceEnabled;
            Resp.pDHCPRelayEnabled = pd->pDHCPRelayEnabled;
            Resp.pGPSLPM           = pd->pGPSLPM;

            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        (tc + 1),
                        ncases,
                        pd->desc);

            fprintf( fp, "SLQSSetCustFeatures parameters set:\n" );

            IFPRINTF( fp, "GPS Antenna Select: 0x%x \n",
                      Resp.pGPSSel );

            IFPRINTF( fp, "GPS Enable/Disable: 0x%u \n",
                      Resp.pGPSEnable );

            IFPRINTF( fp, "Voice Enable/Disable: 0x%x \n",
                      Resp.pIsVoiceEnabled );

            IFPRINTF( fp, "DHCP Relay Enable/Disable: 0x%x \n",
                      Resp.pDHCPRelayEnabled );

            IFPRINTF( fp, "GPSLPM: 0x%x \n", Resp.pGPSLPM );

            ULONG nRet = SLQSSetCustFeatures( &Resp );
            doprintreason( fp, nRet );
            tc++;
        }
        fprintf( fp, "\nReset the module to refresh the GPS status\n");
    }

    if (fp)
        tfclose(fp);
}

/*
 * Name:    doSLQGetCurrentPRLInfo
 *
 * Purpose: Perform the tests that call the SLQGetCurrentPRLInfo() API
 *
 * Return:  none
 *
 * Notes:   none
 *
 */
void doSLQGetCurrentPRLInfo( void )
{
    WORD prlVersion;
    BYTE prlPref;

    /* Test Cases */
    SLQSGetCurrentPRLInfo_t SLQSGetCurrentPRLInformation[] =
    {
        { eQCWWAN_ERR_NONE, 1, NULL, NULL, "Fetching None" },
        { eQCWWAN_ERR_NONE, 2, &prlVersion, &prlPref, "Fetching All Values" }
    };

    FILE  *fp = tfopen("../../dms/test/results/dmsgetcurrentprlinfo.txt", "w");
    dmsCurrentPRLInfo resp;

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSGetCurrentPRLInformation )/
                      sizeof( SLQSGetCurrentPRLInformation[0] );
        while ( tc < ncases )
        {
            SLQSGetCurrentPRLInfo_t *pd = &SLQSGetCurrentPRLInformation[tc++];
            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        tc,
                        ncases,
                        pd->desc);

            resp.pPRLVersion    = pd->pPRLVersion;
            resp.pPRLPreference = pd->pPRLPreference;

            ULONG nRet = SLQSGetCurrentPRLInfo( &resp );
            doprintreason( fp, nRet );

            if ( eQCWWAN_ERR_NONE == nRet )
            {
                fprintf(fp, "SLQSDmsGetCurrentProtocol Successful\n");
                IFPRINTF(fp, "PRL Version: %x\n", resp.pPRLVersion);
                IFPRINTF(fp, "PRL Preference: %d\n", resp.pPRLPreference);
            }
        }
    }
    if (fp)
        tfclose(fp);
}

/*
 * Name:    doSLQSGetERIFile
 *
 * Purpose: Perform the tests that call the SLQSGetERIFile() API
 *
 * Return:  none
 *
 * Notes:   none
 *
 */
void doSLQSGetERIFile( void )
{
    BYTE ERIFile[1024];
    WORD ERIFileSize = 1024;
    WORD ctr = 0;

    /* Test Cases */
    SLQSGetERIFile_t SLQSGetERIFileTestCases[] =
    {
        { eQCWWAN_ERR_INVALID_ARG, 1, 0, NULL, "Invalid Test Case - ERI File Size 0" },
        { eQCWWAN_ERR_INVALID_ARG, 2, &ERIFileSize, NULL, "Invalid Test Case - ERI File NULL" },
        { eQCWWAN_ERR_NONE, 3, &ERIFileSize, ERIFile, "Valid Test Case - "}
    };

    FILE  *fp = tfopen("../../dms/test/results/dmsgeterifile.txt", "w");
    ERIFileparams resp;

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSGetERIFileTestCases )/
                      sizeof( SLQSGetERIFileTestCases[0] );
        while ( tc < ncases )
        {
            SLQSGetERIFile_t *pd = &SLQSGetERIFileTestCases[tc++];
            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        tc,
                        ncases,
                        pd->desc);

            resp.pFileSize = pd->pERIFileSize;
            resp.pFile = pd->pERIFile;

            ULONG nRet = SLQSGetERIFile( &resp );
            doprintreason( fp, nRet );

            if ( eQCWWAN_ERR_NONE == nRet )
            {
                fprintf(fp, "SLQSGetERIFile Successful\n");
                IFPRINTF(fp, "ERI File Size: 0x%x\n", resp.pFileSize);
                if( NULL != resp.pFile )
                {
                    fprintf(fp, "ERI File: ");
                    for( ctr = 0; ctr < *(resp.pFileSize); ctr++ )
                    {
                        fprintf(fp, "0x%x ", resp.pFile[ctr]);
                    }
                }
            }
        }
    }
    if (fp)
        tfclose(fp);
}

/*
 * Name:    doSLQSSwiGetFSN
 *
 * Purpose: Perform the tests that call the SLQSSwiGetFSN() API
 *
 * Return:  none
 *
 * Notes:   none
 *
 */
void doSLQSSwiGetFSN( void )
{
    FactorySequenceNumber fsnumber;

    /* Test Cases */
    SLQSSwiGetFSN_t SLQSSwiGetFSNTestCases[] =
    {
        { eQCWWAN_ERR_INVALID_ARG, 1, NULL, "Invalid Test Case " },
        { eQCWWAN_ERR_INVALID_ARG, 2, &fsnumber, "Valid Test Case" },
    };

    FILE  *fp = tfopen("../../dms/test/results/dmsgetfsn.txt", "w");

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSSwiGetFSNTestCases )/
                      sizeof( SLQSSwiGetFSNTestCases[0] );
        while ( tc < ncases )
        {
            FactorySequenceNumber *resp;
            SLQSSwiGetFSN_t *pd = &SLQSSwiGetFSNTestCases[tc];
            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        tc,
                        ncases,
                        pd->desc);

            resp = pd->pFSNumber;

            ULONG nRet = SLQSSwiGetFSN( resp );
            doprintreason( fp, nRet );

            if ( eQCWWAN_ERR_NONE == nRet )
            {
                fprintf(fp, "SLQSSwiGetFSN Successful\n");
                if( NULL != resp->FSNumber )
                {
                    fprintf( fp, "%s ", resp->FSNumber );
                    fprintf( fp, "\n" );
                }
            }
            tc++;
        }
    }
    if (fp)
        tfclose(fp);
}

/*
 * Name:    doSLQSSwiGetFirmwareCurr
 *
 * Purpose: Perform the tests that call the SLQSSwiGetFirmwareCurr() API
 *
 * Return:  none
 *
 * Notes:   none
 *
 */
void doSLQSSwiGetFirmwareCurr( void )
{
    CurrentImgList CurrImgList;
    CurrImageInfo  currImgInfo[5];
    BYTE           numEntries  = 5;
    BYTE           ctr         = 0;

    /* Test Cases */
    SLQSSwiGetFirmwareCurr_t SLQSSwiGetFirmwareCurrTestCases[] =
    {
        { eQCWWAN_ERR_INVALID_ARG, 1, NULL, "Invalid Test Case " },
        { eQCWWAN_ERR_NONE, 2, &CurrImgList, "Valid Test Case" },
    };

    FILE  *fp = tfopen("../../dms/test/results/dmsgetfirmwarecurr.txt", "w");
    CurrentImgList *resp;

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSSwiGetFirmwareCurrTestCases )/
                      sizeof( SLQSSwiGetFirmwareCurrTestCases[0] );
        while ( tc < ncases )
        {
            SLQSSwiGetFirmwareCurr_t *pd =
                    &SLQSSwiGetFirmwareCurrTestCases[tc];
            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        tc+1,
                        ncases,
                        pd->desc);

            if( NULL != pd->pCurrentImgList )
            {
                pd->pCurrentImgList->pCurrImgInfo = currImgInfo;
                pd->pCurrentImgList->numEntries   = numEntries;
            }
            resp = pd->pCurrentImgList;

            ULONG nRet = SLQSSwiGetFirmwareCurr( resp );
            doprintreason( fp, nRet );

            if ( eQCWWAN_ERR_NONE == nRet )
            {
                fprintf( fp,
                         "Details of Firmware currently running Successfully retrieved\n");
                if( NULL != resp )
                {
                    fprintf( fp, "Number of Images: 0x%x\n", resp->numEntries );
                    for( ctr = 0; ctr < resp->numEntries; ctr++ )
                    {
                        CurrImageInfo  *pCurrImPtr = NULL;
                        pCurrImPtr = &resp->pCurrImgInfo[ctr];
                        fprintf( fp, "Image Index: 0x%x\n", ctr );
                        fprintf( fp, "Image Type: 0x%x\n", pCurrImPtr->imageType );
                        fprintf( fp, "Unique ID: %s\n", (CHAR*)pCurrImPtr->uniqueID);
                        fprintf( fp, "BuildID Length: 0x%x\n", pCurrImPtr->buildIDLen );
                        fprintf( fp, "Build ID: %s\n", (CHAR*)pCurrImPtr->buildID);
                    }

                    fprintf( fp, "PRI version: %s\n", resp->priver );
                    fprintf( fp, "Pkg version: %s\n", resp->pkgver );
                    fprintf( fp, "Firmware version: %s\n", resp->fwvers );
                    fprintf( fp, "Carrier : %s\n", resp->carrier );

                    fprintf( fp, "\n" );
                }
            }
            tc++;
        }
    }
    if (fp)
        tfclose(fp);
}


/*
 * Name:    doSLQSSwiGetFwUpdatStat
 *
 * Purpose: Perform the tests that call the SLQSSwiGetFwUpdateStatus() API
 *
 * Return:  none
 *
 * Notes:   none
 *
 */
void doSLQSSwiGetFwUpdateStatus( void )
{
    
    BYTE              ImgType[2]     = {0,0};
    ULONG             RefData[2]     = {0,0};
    BYTE              RefStrLen[2]   = {0xFF,0xFF};
    BYTE              RefStr[255][2] = {"\0","\0"};
    BYTE              LogStrLen[2]   = {0xFF,0xFF};
    BYTE              LogStr[255][2] = {"\0","\0"};
    FirmwareUpdatStat FirmwareUpdatStatus[] = {
        {0,&ImgType[0],&RefData[0],&RefStrLen[0],RefStr[0],0,NULL},
        {0,&ImgType[1],&RefData[1],&RefStrLen[1],RefStr[1],&LogStrLen[1],LogStr[1]}
    };


    /* Test Cases */
    SLQSSwiGetFwUpdatStat_t SLQSSwiGetFwUpdatStatTestCases[] =
    {
        { eQCWWAN_ERR_INVALID_ARG, 1, NULL,                 "Invalid Test Case " },
        { eQCWWAN_ERR_NONE,        2, &FirmwareUpdatStatus[0], "Valid Test Case Without Log String" },
        { eQCWWAN_ERR_NONE,        3, &FirmwareUpdatStatus[1], "Valid Test Case All Parameters" }
    };

    FILE  *fp = tfopen("../../dms/test/results/dmsgetfwupdatstat.txt", "w");

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSSwiGetFwUpdatStatTestCases )/
                      sizeof( SLQSSwiGetFwUpdatStatTestCases[0] );
        while ( tc < ncases )
        {
            FirmwareUpdatStat *resp;
            SLQSSwiGetFwUpdatStat_t *pd =
                    &SLQSSwiGetFwUpdatStatTestCases[tc];
            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        tc,
                        ncases,
                        pd->desc);

            if( NULL != pd->pFirmwareUpdatStat )
            {
                
            }
            resp = pd->pFirmwareUpdatStat;

            ULONG nRet = SLQSSwiGetFwUpdateStatus( resp );
            doprintreason( fp, nRet );

            if ( eQCWWAN_ERR_NONE == nRet )
            {
                fprintf( fp, "Firmware Update Status\n" );
                if( NULL != resp )
                {
                    fprintf( fp, "Result Code: 0x%x\n", resp->ResCode );
                    IFPRINTF( fp, "Image Type: 0x%x\n", resp->pImgType );
                    IFPRINTF( fp, "Ref Data: 0x%x\n", resp->pRefData );
                    IFPRINTF( fp, "Ref String Length: 0x%x\n", resp->pRefStringLen );
                    fprintf( fp, "Ref String : %s\n", resp->pRefString );
                    if( (resp->pLogStringLen)&&
                        (resp->pLogString))
                    {
                        IFPRINTF( fp, "Log String Length: 0x%x\n", resp->pLogStringLen );
                        fprintf( fp, "Log String : %s\n", resp->pLogString );
                    }
                    fprintf( fp, "\n" );
                }
            }
            tc++;
        }
    }
    if (fp)
        tfclose(fp);
}

/*
 * Name:    doSLQSSwiGetUSBComp
 *
 * Purpose: Perform the tests that call the SLQSSwiGetUSBComp() API
 *
 * Return:  none
 *
 * Notes:   none
 *
 */
void doSLQSSwiGetUSBComp( void )
{
    USBCompParams USBCompositionParams;
    BYTE          currUSBComp = 0xFF;
    BYTE          numUSBComp  = 0xFF;
    BYTE          USBComps[255] = "\0";
    BYTE          ctr = 0;

    /* Test Cases */
    SLQSSwiGetUSBComp_t SLQSSwiGetUSBCompTestCases[] =
    {
        { eQCWWAN_ERR_INVALID_ARG, 1, NULL,                  "Invalid Test Case " },
        { eQCWWAN_ERR_NONE,        2, &USBCompositionParams, "Valid Test Case" },
    };

    FILE  *fp = tfopen("../../dms/test/results/dmsgetusbcomp.txt", "w");

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSSwiGetUSBCompTestCases )/
                      sizeof( SLQSSwiGetUSBCompTestCases[0] );
        while ( tc < ncases )
        {
            USBCompParams *resp;
            SLQSSwiGetUSBComp_t *pd =
                    &SLQSSwiGetUSBCompTestCases[tc];
            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        tc,
                        ncases,
                        pd->desc);

            if( NULL != pd->pUSBCompParams )
            {
                pd->pUSBCompParams->pUSBComp        = &currUSBComp;
                pd->pUSBCompParams->pNumSupUSBComps = &numUSBComp;
                pd->pUSBCompParams->pSupUSBComps    = USBComps;
            }
            resp = pd->pUSBCompParams;

            ULONG nRet = SLQSSwiGetUSBComp( resp );
            doprintreason( fp, nRet );

            if ( eQCWWAN_ERR_NONE == nRet )
            {
                fprintf( fp, "USB Composition\n" );
                if( NULL != resp )
                {
                    IFPRINTF( fp, "Current USB Composition: 0x%x\n", resp->pUSBComp );
                    IFPRINTF( fp, "Count of supported USB Compositions: 0x%x\n", resp->pNumSupUSBComps );
                    fprintf( fp, "Supported USB Compositions \n" );
                    for( ctr = 0; ctr < *(resp->pNumSupUSBComps); ctr++ )
                    {
                        fprintf( fp, "Composition [%d] - 0x%x\n", ctr, resp->pSupUSBComps[ctr] );
                    }
                    fprintf( fp, "\n" );
                }
            }
            tc++;
        }
    }
    if (fp)
        tfclose(fp);
}

/*
 * Name:    doSLQSSwiSetUSBComp
 *
 * Purpose: Perform the tests that call the SLQSSwiSetUSBComp() API
 *
 * Return:  none
 *
 * Notes:   none
 *
 */
void doSLQSSwiSetUSBComp( void )
{
    BYTE USBComp = 0x06;

    /* Test Cases */
    SLQSSwiSetUSBComp_t SLQSSwiSetUSBCompTestCases[] =
    {
        { eQCWWAN_ERR_INVALID_ARG, 1, NULL,     "Invalid Argument" },
        { eQCWWAN_ERR_NONE,        2, &USBComp, "Valid Test Case, USB Composition 7" },
    };

    FILE *fp = tfopen("../../dms/test/results/SLQSSwiSetUSBComp.txt", "w");
    USBCompConfig Req;

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSSwiSetUSBCompTestCases )/
                      sizeof( SLQSSwiSetUSBCompTestCases[0] );
        while ( tc < ncases )
        {
            SLQSSwiSetUSBComp_t *pd =
                &SLQSSwiSetUSBCompTestCases[tc];

            Req.pUSBComp = pd->pUSBComp;

            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        (tc + 1),
                        ncases,
                        pd->desc);

            fprintf( fp, "SLQSSwiSetUSBComp parameters set:\n" );

            IFPRINTF( fp, "USB Composition: 0x%x \n", Req.pUSBComp );

            ULONG nRet = SLQSSwiSetUSBComp( &Req );
            doprintreason( fp, nRet );
            tc++;
        }
        fprintf( fp, "\nUSB Composition set\n");
    }

    if (fp)
        tfclose(fp);
}

void doSLQSSwiGetCrashInfo( void )
{
    CrashInfoParams CrashInfoParm;
    CrashInfo       CrashInfo;
    BYTE            Clear = 0;

    CrashInfoParm.pCrashInfo = & CrashInfo;
    BYTE  devCrashStatus     = 0xFF;
    WORD  numCrashes         = 0xFFFF;
    ULONG crashId            = 0xFFFF;
    ULONG crashData          = 0xFFFF;
    WORD  crashStrLen       = 0;
    WORD  gcDumpStrLen      = 0;

    /* Test Cases */
    SLQSSwiGetCrashInfo_t SLQSSwiGetCrashInfoTestCases[] =
    {
        { eQCWWAN_ERR_INVALID_ARG, 1, NULL, NULL, 0, 0, 0, 0, 0,
                                   "Invalid Test Case " },
        { eQCWWAN_ERR_INVALID_ARG, 1, &Clear, &devCrashStatus, numCrashes, crashId, crashData,
                                   crashStrLen,  gcDumpStrLen,
                                   "Valid test case" },
        { eQCWWAN_ERR_INVALID_ARG, 1, NULL, &devCrashStatus, numCrashes, crashId, crashData,
                                   crashStrLen, gcDumpStrLen,
                                   "Valid test case" },
    };

    FILE  *fp = tfopen("../../dms/test/results/dmsgetCrashInfo.txt", "w");

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSSwiGetCrashInfoTestCases )/
                      sizeof( SLQSSwiGetCrashInfoTestCases[0] );
        while ( tc < ncases )
        {
            BYTE *req;
            CrashInfoParams *resp;
            SLQSSwiGetCrashInfo_t *pd =
                    &SLQSSwiGetCrashInfoTestCases[tc];
            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        tc+1,
                        ncases,
                        pd->desc);

            req = pd->pClear;
            resp = &CrashInfoParm;
            resp->pCrashInfo->crashData     = pd->crashData;
            resp->pCrashInfo->crashId       = pd->crashId;
            resp->pCrashInfo->crashStrLen   = pd->crashStrLen;
            resp->pCrashInfo->gcDumpStrLen  = pd->gcDumpStrLen;
            resp->pCrashInfo->numCrashes    = pd->numCrashes;
            resp->pCrashInfo->pCrashString  = NULL;
            resp->pCrashInfo->pGCDumpString = NULL;
            resp->pDevCrashStatus           = pd->pdevCrashStatus;

            ULONG nRet = SLQSSwiGetCrashInfo(req, resp );
            doprintreason( fp, nRet );

            if ( eQCWWAN_ERR_NONE == nRet )
            {
                    IFPRINTF( fp, "Clear Value Provided : 0x%x\n", req );
                    IFPRINTF( fp, "Device Crash Status  : 0x%x\n", resp->pDevCrashStatus );
                    IFPRINTF( fp, "Crash Data           : 0x%x\n", &(resp->pCrashInfo->crashData) );
                    IFPRINTF( fp, "Crash Id             : 0x%x\n", &(resp->pCrashInfo->crashId) );
                    IFPRINTF( fp, "Crash String Length  : 0x%x\n", &(resp->pCrashInfo->crashStrLen) );
                    IFPRINTF( fp, "GCDump String Length : 0x%x\n", &(resp->pCrashInfo->gcDumpStrLen) );
                    fprintf( fp, "Crash String         :");
                    if(resp->pCrashInfo->pCrashString!=NULL)
                    {
                        fprintf(fp," %s\n", resp->pCrashInfo->pCrashString );
                        resp->pCrashInfo->pCrashString = NULL;
                    }
                    else
                    {
                        fprintf(fp," NULL\n");
                    }
                    fprintf( fp, "GCDump String        :");
                    if(resp->pCrashInfo->pGCDumpString!=NULL)
                    {
                        fprintf(fp," %s\n", resp->pCrashInfo->pGCDumpString );
                        resp->pCrashInfo->pGCDumpString = NULL;
                        
                    }
                    else
                    {
                        fprintf(fp," NULL\n");
                    }
                    fprintf( fp, "\n" );
            }
            if(eQCWWAN_ERR_BUFFER_SZ == nRet)
            {
                IFPRINTF( fp, "Crash String Length should be  : 0x%x\n", &(resp->pCrashInfo->crashStrLen) );
                IFPRINTF( fp, "GCDump String Length should be : 0x%x\n", &(resp->pCrashInfo->gcDumpStrLen) );
            }
            tc++;
        }
    }
    if (fp)
        tfclose(fp);
}

/*
 * Name:    doSLQSSwiGetCrashAction
 *
 * Purpose: Perform the tests that call the SLQSSwiGetCrashAction() API
 *
 * Return:  none
 *
 * Notes:   none
 *
 */
void doSLQSSwiGetCrashAction( void )
{
    BYTE  crashAction;
    BYTE  currCrashAction = 0xFF;

    /* Test Cases */
    SLQSSwiGetCrashAction_t SLQSSwiGetCrashActionTestCases[] =
    {
        { eQCWWAN_ERR_INVALID_ARG, 1, NULL,     "Invalid Argument" },
        { eQCWWAN_ERR_NONE,        2, &crashAction, "Valid Test Case" },
    };

    FILE *fp = tfopen("../../dms/test/results/SLQSSwiGetCrashAction.txt", "w");

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSSwiGetCrashActionTestCases )/
                      sizeof( SLQSSwiGetCrashActionTestCases[0] );
        while ( tc < ncases )
        {
            BYTE *resp;
            SLQSSwiGetCrashAction_t *pd =
                &SLQSSwiGetCrashActionTestCases[tc];

            if( NULL != pd->pCrashAction )
            {
                pd->pCrashAction = &currCrashAction;
            }
            resp = pd->pCrashAction;

            ULONG nRet = SLQSSwiGetCrashAction( resp );
            doprintreason( fp, nRet );

            if ( eQCWWAN_ERR_NONE == nRet )
            {
                fprintf( fp, "Device Crash State:\n" );
                if( NULL != resp )
                {
                    IFPRINTF( fp, "Current Device Crash State: 0x%x\n", resp );
                    fprintf( fp, "\n" );
                }
            }
            tc++;
        }
    }
    if (fp)
        tfclose(fp);
}

/*
 * Name:    doSLQSSwiSetCrashAction
 *
 * Purpose: Perform the tests that call the SLQSSwiSetCrashAction() API
 *
 * Return:  none
 *
 * Notes:   none
 *
 */
void doSLQSSwiSetCrashAction( void )
{
    BYTE DevCrashAction = 0x02;

    /* Test Cases */
    SLQSSwiSetCrashAction_t SLQSSwiSetCrashActionTestCases[] =
    {
        { eQCWWAN_ERR_INVALID_ARG, 1, 4,     "Invalid Argument" },
        { eQCWWAN_ERR_NONE,        2, DevCrashAction, "Valid Test Case, "
          "Dev Crash Action 2 (No Action)" },
    };

    FILE *fp = tfopen("../../dms/test/results/SLQSSwiSetCrashAction.txt", "w");
    BYTE Req;

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSSwiSetCrashActionTestCases )/
                      sizeof( SLQSSwiSetCrashActionTestCases[0] );
        while ( tc < ncases )
        {
            SLQSSwiSetCrashAction_t *pd =
                &SLQSSwiSetCrashActionTestCases[tc];

            Req = pd->DevCrashAction;

            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        (tc + 1),
                        ncases,
                        pd->desc);

            fprintf( fp, "SLQSSwiSetCrashAction parameters set:\n" );
            fprintf( fp, "Crash Action: 0x%x \n", Req );

            ULONG nRet = SLQSSwiSetCrashAction( Req );
            doprintreason( fp, nRet );
            tc++;
        }
        fprintf( fp, "\nCrash Action set\n");
    }

    if (fp)
    tfclose(fp);
}

void doSLQSSwiGetHostDevInfo( void )
{
    FILE *fp = tfopen("../../dms/test/results/SLQSSwiGetHostDevInfo.txt", "w");
	SLQSSwiGetHostDevInfoParams hostDevInfoParams;
    SLQSSwiGetHostDevInfoParams *Req;
	Req = &hostDevInfoParams;

    if (fp)
    {
        BYTE tc = 0;
        while ( tc < MAX_GET_HOST_DEV_INFO_NUM )
        {

			Req->bManSize = GetHostDevInfoTestCases[tc].bManSize;
			Req->pManString = GetHostDevInfoTestCases[tc].szManString;
			Req->bModelSize = GetHostDevInfoTestCases[tc].bModelSize;
			Req->pModelString = GetHostDevInfoTestCases[tc].szModelString;
			Req->bSWVerSize = GetHostDevInfoTestCases[tc].bSWVerSize;
			Req->pSWVerString = GetHostDevInfoTestCases[tc].szSWVerString;
			Req->bPlasmaIDSize = GetHostDevInfoTestCases[tc].bPlasmaIDSize;
			Req->pPlasmaIDString = GetHostDevInfoTestCases[tc].szPlasmaIDString;
			
            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        (tc + 1),
                        MAX_GET_HOST_DEV_INFO_NUM,
                        GetHostDevInfoTestCases[tc].desc);


            ULONG nRet = SLQSSwiGetHostDevInfo( Req );
			 if ( eQCWWAN_ERR_NONE == nRet )
            {

                 fprintf( fp, "Manufacturer\t: %s\n", Req->pManString );
                 fprintf( fp, "Model String\t: %s\n", Req->pModelString);
                 fprintf( fp, "Software Version\t: %s\n", Req->pSWVerString);
                 fprintf( fp, "Plasma ID String\t: %s\n", Req->pPlasmaIDString);
                 fprintf( fp, "\n" );
            }
            doprintreason( fp, nRet );
            tc++;
        }
        fprintf( fp, "\n Get Host Dev Info finish\n");
    }

    if (fp)
    tfclose(fp);
}


void doSLQSSwiSetHostDevInfo( void )
{

    FILE *fp = tfopen("../../dms/test/results/SLQSSwiSetHostDevInfo.txt", "w");
	SLQSSwiSetHostDevInfoParams hostDevInfoParams;
    SLQSSwiSetHostDevInfoParams *Req;
	Req = &hostDevInfoParams;

    if (fp)
    {
        BYTE tc = 0;

        while ( tc < MAX_SET_HOST_DEV_INFO_NUM )
        {
            Req->bManSize = SetHostDevInfoTestCases[tc].bManSize;
			Req->pManString = SetHostDevInfoTestCases[tc].szManString;
			Req->bModelSize = SetHostDevInfoTestCases[tc].bModelSize;
			Req->pModelString = SetHostDevInfoTestCases[tc].szModelString;
			Req->bSWVerSize = SetHostDevInfoTestCases[tc].bSWVerSize;
			Req->pSWVerString = SetHostDevInfoTestCases[tc].szSWVerString;
			Req->bPlasmaIDSize = SetHostDevInfoTestCases[tc].bPlasmaIDSize;
			Req->pPlasmaIDString = SetHostDevInfoTestCases[tc].szPlasmaIDString;

            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        (tc + 1),
                        MAX_SET_HOST_DEV_INFO_NUM,
                        GetHostDevInfoTestCases[tc].desc);
			
            ULONG nRet = SLQSSwiSetHostDevInfo( Req );
			if ( eQCWWAN_ERR_NONE == nRet )
            {
                 fprintf( fp, "Manufacturer\t\t: %s\n", Req->pManString );
                 fprintf( fp, "Model String\t\t: %s\n", Req->pModelString);
                 fprintf( fp, "Software Version\t: %s\n", Req->pSWVerString);
                 fprintf( fp, "Plasma ID String\t: %s\n", Req->pPlasmaIDString);
                 fprintf( fp, "\n" );
            }
            doprintreason( fp, nRet );
            tc++;
        }
        fprintf( fp, "\n Set Host Dev Info finish\n");
    }

    if (fp)
    tfclose(fp);
}

void doSLQSSwiGetOSInfo( void )
{
    
    FILE *fp = tfopen("../../dms/test/results/SLQSSwiGetOSInfo.txt", "w");
	SLQSSwiGetOSInfoParams sLQSSwiGetOSInfoParams;
    SLQSSwiGetOSInfoParams *Req;
	Req = &sLQSSwiGetOSInfoParams;

    if (fp)
    {
        BYTE tc = 0;

        while ( tc < MAX_GET_OS_INFO_NUM )
        {
            Req->bNameSize = GetOSInfoTestCases[tc].bNameSize;
			Req->pNameString = GetOSInfoTestCases[tc].szNameString;
			Req->bVersionSize = GetOSInfoTestCases[tc].bVersionSize;
			Req->pVersionString = GetOSInfoTestCases[tc].szVersionString;

            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        (tc + 1),
                        MAX_GET_OS_INFO_NUM,
                        GetOSInfoTestCases[tc].desc);
			
            ULONG nRet = SLQSSwiGetOSInfo( Req );
			if ( eQCWWAN_ERR_NONE == nRet )
            {
                 fprintf( fp, "Name\t: %s\n", Req->pNameString );
                 fprintf( fp, "Version\t: %s\n", Req->pVersionString);
                 fprintf( fp, "\n" );
            }
            doprintreason( fp, nRet );
            tc++;
        }
        fprintf( fp, "\n Set Host Dev Info finish\n");
    }

    if (fp)
    tfclose(fp);

}

void doSLQSSwiSetOSInfo( void )
{

	FILE *fp = tfopen("../../dms/test/results/SLQSSwiSetOSInfo.txt", "w");
	SLQSSwiSetOSInfoParams OSInfoParams;
    SLQSSwiSetOSInfoParams *Req;
	Req = &OSInfoParams;

    if (fp)
    {
        BYTE tc = 0;

        while ( tc < MAX_SET_OS_INFO_NUM )
        {
            Req->bNameSize = SetOSInfoTestCases[tc].bNameSize;
			Req->pNameString = SetOSInfoTestCases[tc].szNameString;
			Req->bVersionSize = SetOSInfoTestCases[tc].bVersionSize;
			Req->pVersionString = SetOSInfoTestCases[tc].szVersionString;

            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        (tc + 1),
                        MAX_SET_OS_INFO_NUM,
                        SetOSInfoTestCases[tc].desc);
			
            ULONG nRet = SLQSSwiSetOSInfo( Req );
			if ( eQCWWAN_ERR_NONE == nRet )
            {
                 fprintf( fp, "Name\t: %s\n", Req->pNameString );
                 fprintf( fp, "Version\t: %s\n", Req->pVersionString);
                 fprintf( fp, "\n" );
            }
            doprintreason( fp, nRet );
            tc++;
        }
        fprintf( fp, "\n Set OS Info finish\n");
    }

    if (fp)
    tfclose(fp);


}

void doSLQSSwiGetSerialNoExt( void )
{

    FILE *fp = tfopen("../../dms/test/results/SLQSSwiGetSerialNoExt.txt", "w");
    SLQSSwiGetSerialNoExtParams sLQSSwiGetSerialNoExtParams;
    SLQSSwiGetSerialNoExtParams *Req;
    Req = &sLQSSwiGetSerialNoExtParams;

    if (fp)
    {
        BYTE tc = 0;

        while ( tc < MAX_GET_SR_NUM_EXT )
        {
            Req->meidLength = GetSerialNoExtTestCase[tc].meidLength;
            Req->pMeidString = GetSerialNoExtTestCase[tc].meidString;

            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        (tc + 1),
                        MAX_GET_SR_NUM_EXT,
                        GetSerialNoExtTestCase[tc].desc);

            ULONG nRet = SLQSSwiGetSerialNoExt( Req );
            if ( eQCWWAN_ERR_NONE == nRet )
            {
                 fprintf( fp, "MEID\t: %s\n", Req->pMeidString );
                 fprintf( fp, "\n" );
            }
            doprintreason( fp, nRet );
            tc++;
        }
        fprintf( fp, "\n Get Serial No. Extended finish\n");
    }

    if (fp)
    tfclose(fp);

}

/*
 * Name:     doSLQSGetSerialNumbers
 *
 * Purpose:  Perform the tests that call the SLQSGetSerialNumbers() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSLQSGetSerialNumbers( void )
{
    BYTE  esnSize = nMaxStrLen;
    CHAR  ESNString[nMaxStrLen];
    BYTE  imeiSize = nMaxStrLen;
    CHAR  IMEIString[nMaxStrLen];
    BYTE  meidSize = nMaxStrLen;
    CHAR  MEIDString[nMaxStrLen];
    BYTE  imeiSvnSize = nMaxStrLen;
    CHAR  ImeiSvnString[nMaxStrLen];
    serialNumbersInfo serialNumbersInfo;

    FILE  *fp = NULL;

    serialNumbersInfo.esnSize        = esnSize;
    serialNumbersInfo.pESNString     = ESNString;
    serialNumbersInfo.imeiSize       = imeiSize;
    serialNumbersInfo.pIMEIString    = IMEIString;
    serialNumbersInfo.meidSize       = meidSize;
    serialNumbersInfo.pMEIDString    = MEIDString;
    serialNumbersInfo.imeiSvnSize    = imeiSvnSize;
    serialNumbersInfo.pImeiSvnString = ImeiSvnString;

    fp = tfopen("../../dms/test/results/slqsgetserialnumbers.txt", "w");

    if (fp)
    {
        BYTE  tCaseNum = 0;
        while (tCaseNum < MAX_GET_SERIAL_NUMBERS_TESTCASE_NUM)
        {
            ULONG nRet;
            fprintf(fp, "\n\nTest Case %d / %d : %s\n",
                       tCaseNum,
                       MAX_GET_SERIAL_NUMBERS_TESTCASE_NUM,
                       GetSerialNumbersTestCases[tCaseNum].desc);

            fprintf(fp, "Parameter Set:\n");
            fprintf(fp, "esnSize :%d\n", esnSize);
            fprintf(fp, "imeiSize :%d\n", imeiSize);
            fprintf(fp, "meidSize :%d\n", meidSize);
            fprintf(fp, "imeiSvnSize :%d\n", imeiSvnSize);

            nRet = SLQSGetSerialNumbers(&serialNumbersInfo);

            /* Display result code and text */
            doprintreason (fp, nRet);

            if (!nRet)
            {
                fprintf(fp, "ESNString : %s\n", serialNumbersInfo.pESNString);
                fprintf(fp, "IMEIString : %s\n",serialNumbersInfo.pIMEIString);
                fprintf(fp, "MEIDString : %s\n", serialNumbersInfo.pMEIDString);
                fprintf(fp, "IMEI SVN String : %s\n", serialNumbersInfo.pImeiSvnString);
            }

            tCaseNum++;
        }
    }

    if (fp)
        tfclose(fp);
}

void doSLQSGetCustFeaturesV2( void )
{
    getCustomFeatureV2 getCustFeaturesV2;
    custSettingInfo custSettInfo;
    custSettingList custSettList;
    getCustomInput custSettInput;
    BYTE i, j, temp_value;
    ULONGLONG value, value2;
    CHAR customId[] = "WAKEHOSTEN";

    strcpy(&custSettInput.cust_id[0], &customId[0]);
    custSettInput.list_type = 0;

    /* Test Cases */
    SLQSGetCustFeaturesV2TestCase_t SLQSGetCustFeaturesV2TestCases[] =
    {
        { eQCWWAN_ERR_INVALID_ARG, 1, NULL,          "Invalid Argument" },
        { eQCWWAN_ERR_NONE,        2, &getCustFeaturesV2, "Valid Test Case" }
    };

    FILE *fp = tfopen("../../dms/test/results/SLQSGetCustFeaturesV2.txt", "w");

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSGetCustFeaturesV2TestCases )/
                      sizeof( SLQSGetCustFeaturesV2TestCases[0] );
        while ( tc < ncases )
        {
            getCustomFeatureV2 *pResp;
            SLQSGetCustFeaturesV2TestCase_t *pd =
                &SLQSGetCustFeaturesV2TestCases[tc];
            value = 0;
            value2 = 0;
            temp_value = 0;

            pResp = pd->pGetCustFeaturesInfoV2;
            if( pResp )
            {
                pResp->pGetCustomInput = &custSettInput;
                pResp->pCustSettingInfo= &custSettInfo;
                pResp->pCustSettingList= &custSettList;
            }

            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        (tc + 1),
                        ncases,
                        pd->desc);

            ULONG nRet = SLQSGetCustFeaturesV2( pResp );

            doprintreason( fp, nRet );
            if ( eQCWWAN_ERR_NONE == nRet )
            {
                fprintf(fp, "SLQSGetCustFeaturesV2 Successful\n");
                if ( pResp->pCustSettingInfo != NULL)
                {
                    fprintf( fp, "Customization Setting:\n"\
                                 "id length: %d\n", pResp->pCustSettingInfo->id_length );
                    
                    fprintf( fp, "custom id: %s \n", pResp->pCustSettingInfo->cust_id );
                    fprintf( fp, "value length: %d\n", pResp->pCustSettingInfo->value_length );
                    for ( i = 0; i < pResp->pCustSettingInfo->value_length; i++)
                    {
                        temp_value = pResp->pCustSettingInfo->cust_value[i];
                        value = value | temp_value << (8*i);
                    }
                    fprintf( fp, "value id: %llu\n",  value);
                    fprintf( fp, "custom attribute: %d\n", pResp->pCustSettingInfo->cust_attr);
                }
                if ( pResp->pCustSettingList != NULL)
                {
                    fprintf( fp, "\nCustomization List:\n"\
                                 "list type: %d\n",  pResp->pCustSettingList->list_type);
                    fprintf( fp, "number instances: %d\n",  pResp->pCustSettingList->num_instances);
                    for ( i = 0; i < pResp->pCustSettingList->num_instances; i++)
                    {
                        /* reset custom value for every single custom filed */
                        value2 = 0;
                        fprintf( fp, "Instance[%i]:\n", i);
                        fprintf( fp, "id length: %d\n",  pResp->pCustSettingList->custSetting[i].id_length);
                        fprintf( fp, "custtom id: %s\n",  pResp->pCustSettingList->custSetting[i].cust_id);

                        for ( j = 0; j < pResp->pCustSettingList->custSetting[i].value_length; j++ )
                        {
                            temp_value = pResp->pCustSettingList->custSetting[i].cust_value[j];
                            value2 = value2 | temp_value << (8*j);
                        }
                        fprintf( fp, "custom value : %llu\n",  value2);
                        fprintf( fp, "custom attribute: %d\n", pResp->pCustSettingList->custSetting[i].cust_attr);
                    }
                }
            }
            tc++;
        }
    }

    if (fp)
        tfclose(fp);
}

void doSLQSSetCustFeaturesV2( void )
{
    setCustomSettingV2 cust_setting;
    BYTE i;
    CHAR customId[] = "WAKEHOSTEN";

    strcpy(&cust_setting.cust_id[0], &customId[0]);
    cust_setting.value_length = 1;
    cust_setting.cust_value[0] = 1;
    cust_setting.cust_value[1] = '\0';

    /* Test Cases */
    SLQSSetCustFeaturesV2TestCase_t SLQSSetCustFeaturesV2TestCases[] =
    {
        { eQCWWAN_ERR_INVALID_ARG, 1, NULL, "Invalid Argument" },
        { eQCWWAN_ERR_NONE,        2, &cust_setting, "Valid Test Case, custom_setting (WAKEHOSTEN=1)" }
    };

    FILE *fp = tfopen("../../dms/test/results/SLQSSetCustFeaturesV2.txt", "w");

    if (fp)
    {
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSSetCustFeaturesV2TestCases )/
                      sizeof( SLQSSetCustFeaturesV2TestCases[0] );
        while ( tc < ncases )
        {
            SLQSSetCustFeaturesV2TestCase_t *pd =
                &SLQSSetCustFeaturesV2TestCases[tc];

            fprintf(fp, "\nTest Case %d / %d : %s\n",
                        (tc + 1),
                        ncases,
                        pd->desc);

            fprintf( fp, "SLQSSetCustFeaturesV2 parameters set:\n" );
            if ( pd->pSetCustomSetting != NULL )
            {
                fprintf( fp, "Custom Setting:\n");
                fprintf( fp, "custom id: %s \n", pd->pSetCustomSetting->cust_id);
                fprintf( fp, "value length: %d\n", pd->pSetCustomSetting->value_length);
                for ( i = 0; i < pd->pSetCustomSetting->value_length; i++)
                {
                    fprintf( fp, "value id[%d]: %d\n", i, pd->pSetCustomSetting->cust_value[i] );
                }
            }
            ULONG nRet = SLQSSetCustFeaturesV2( pd->pSetCustomSetting );
            doprintreason( fp, nRet );
            tc++;
        }
        fprintf( fp, "\nReset the module to refresh the custom setting status\n");
    }

    if (fp)
        tfclose(fp);
}

void doSLQSSwiGetDyingGaspCfg( void )
{
    getDyingGaspCfg config[4];
    BYTE SMSContent[MAX_DYING_GASP_CFG_SMS_CONTENT_LENGTH];
    BYTE SMSNumber[MAX_DYING_GASP_CFG_SMS_NUMBER_LENGTH];
    config[0].pDestSMSContent = NULL;
    config[0].pDestSMSNum = NULL;
    config[1].pDestSMSContent = NULL;
    config[1].pDestSMSNum = &SMSNumber[0];
    config[2].pDestSMSContent = &SMSContent[0];
    config[2].pDestSMSNum = NULL;
    config[3].pDestSMSNum = &SMSNumber[0];
    config[3].pDestSMSContent = &SMSContent[0];
    FILE *fp = tfopen("../../dms/test/results/SLQSSwiGetDyingGaspCfg.txt", "w");
    ULONG nRet = 0;
    if (fp)
    {
        /* Test Cases */
        SLQSSwiGetDyingGaspCfgTestCase_t SLQSSwiGetDyingGaspCfgTestCases[] =
        {
            { eQCWWAN_ERR_INVALID_ARG, 1, NULL, "Invalid Argument" },
            { eQCWWAN_ERR_NONE,        2, &config[0], "Valid Test Case, No parameters" },
            { eQCWWAN_ERR_NONE,        3, &config[1], "Valid Test Case, Only SMS No." },
            { eQCWWAN_ERR_NONE,        4, &config[2], "Valid Test Case, Only SMS Content" },
            { eQCWWAN_ERR_NONE,        5, &config[3], "Valid Test Case, Both SMS No. and Content" },
        };
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSSwiGetDyingGaspCfgTestCases )/
                      sizeof( SLQSSwiGetDyingGaspCfgTestCases[0] );
        while ( tc < ncases )
        {
            SLQSSwiGetDyingGaspCfgTestCase_t *pd =
                &SLQSSwiGetDyingGaspCfgTestCases[tc];
            
            memset(&SMSNumber,0,MAX_DYING_GASP_CFG_SMS_NUMBER_LENGTH);
            memset(&SMSContent,0,MAX_DYING_GASP_CFG_SMS_CONTENT_LENGTH);
            fprintf(fp, "\nTest Case %d / %d : %s\n",
                    (tc + 1),
                    ncases,
                    pd->desc);
            nRet = SLQSSwiGetDyingGaspCfg( pd->pgetDyingGaspCfg );
            doprintreason( fp, nRet );
            fprintf( fp, "Custom Setting:\n");
            if(nRet==eQCWWAN_ERR_NONE)
            {
                if(pd->pgetDyingGaspCfg)
                {
                    if(pd->pgetDyingGaspCfg->pDestSMSContent!=NULL)
                        fprintf( fp, "SMS Content: %s \n", pd->pgetDyingGaspCfg->pDestSMSContent);
                    if(pd->pgetDyingGaspCfg->pDestSMSNum!=NULL)
                        fprintf( fp, "SMS Number: %s\n", pd->pgetDyingGaspCfg->pDestSMSNum);
                }
            }
            tc++;
        }
    }
     if (fp)
        tfclose(fp);
}

void doSLQSSwiSetDyingGaspCfg( void )
{
    setDyingGaspCfg config[4];
    BYTE SMSContent[MAX_DYING_GASP_CFG_SMS_CONTENT_LENGTH]="hello world123456!";
    BYTE SMSNumber[MAX_DYING_GASP_CFG_SMS_NUMBER_LENGTH]="+85298455141";
    config[0].pDestSMSContent = NULL;
    config[0].pDestSMSNum = NULL;
    config[1].pDestSMSContent = NULL;
    config[1].pDestSMSNum = &SMSNumber[0];
    config[2].pDestSMSContent = &SMSContent[0];
    config[2].pDestSMSNum = NULL;
    config[3].pDestSMSNum = &SMSNumber[0];
    config[3].pDestSMSContent = &SMSContent[0];
    
    FILE *fp = tfopen("../../dms/test/results/SLQSSwiSetDyingGaspCfg.txt", "w");
    ULONG nRet;
    if (fp)
    {
             /* Test Cases */
        SLQSSwiSetDyingGaspCfgTestCase_t SLQSSwiSetDyingGaspCfgTestCases[] =
        {
            { eQCWWAN_ERR_INVALID_ARG, 1, NULL, "Invalid Argument" },
            { eQCWWAN_ERR_INVALID_ARG, 2, &config[0], "Invalid Test Case, No parameters" },
            { eQCWWAN_ERR_NONE,        3, &config[1], "Valid Test Case, Only SMS No." },
            { eQCWWAN_ERR_NONE,        4, &config[2], "Valid Test Case, Only SMS Content" },
            { eQCWWAN_ERR_NONE,        5, &config[3], "Valid Test Case, Both SMS No. and Content" },
        };
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSSwiSetDyingGaspCfgTestCases )/
                      sizeof( SLQSSwiSetDyingGaspCfgTestCases[0] );
        while ( tc < ncases )
        {
            SLQSSwiSetDyingGaspCfgTestCase_t *pd =
                &SLQSSwiSetDyingGaspCfgTestCases[tc];
            fprintf(fp, "\nTest Case %d / %d : %s\n",
                    (tc + 1),
                    ncases,
                    pd->desc);
            
            if ( pd->psetDyingGaspCfg !=NULL )
            {
                fprintf( fp, "New Settings:\n");
                if(pd->psetDyingGaspCfg->pDestSMSContent!=NULL)
                    fprintf( fp, "SMS Content: %s \n", pd->psetDyingGaspCfg->pDestSMSContent);
                 if(pd->psetDyingGaspCfg->pDestSMSNum!=NULL)
                    fprintf( fp, "SMS Number: %s\n", pd->psetDyingGaspCfg->pDestSMSNum);
            }
            nRet = SLQSSwiSetDyingGaspCfg( pd->psetDyingGaspCfg);
            doprintreason( fp, nRet );
            tc++;
        }
            
    }
     if (fp)
        tfclose(fp);
}


void doSLQSSwiClearDyingGaspStatistics( void )
{
    FILE *fp = tfopen("../../dms/test/results/SLQSSwiClearDyingGaspStatistics.txt", "w");
    ULONG nRet;
    if (fp)
    {
        nRet = SLQSSwiClearDyingGaspStatistics();
        doprintreason( fp, nRet );
    }
     if (fp)
        tfclose(fp);
}

void doSLQSSwiGetDyingGaspStatistics(void)
{
    getDyingGaspStatistics Statistics1,Statistics2,Statistics3,Statistics4;
    ULONG TimeStamp;
    BYTE  SMSAttemptedFlag;
    Statistics1.pTimeStamp= NULL;
    Statistics1.pSMSAttemptedFlag= NULL;
    
    Statistics2.pTimeStamp = &TimeStamp;
    Statistics2.pSMSAttemptedFlag = NULL;
    
    Statistics3.pTimeStamp = NULL;
    Statistics3.pSMSAttemptedFlag = &SMSAttemptedFlag;
    
    Statistics4.pTimeStamp = &TimeStamp;
    Statistics4.pSMSAttemptedFlag = &SMSAttemptedFlag;

    FILE *fp = tfopen("../../dms/test/results/SLQSSLQSSwiGetDyingGaspStatistics.txt", "w");
    ULONG nRet = 0;
    if (fp)
    {
        /* Test Cases */
        SLQSGetDyingGaspStatisticsTestCase_t SLQSGetDyingGaspStatisticsTestCases[] =
        {
            { eQCWWAN_ERR_INVALID_ARG, 1, NULL, "Invalid Argument" },
            { eQCWWAN_ERR_NONE,        2, &Statistics1, "Valid Test Case, No parameters" },
            { eQCWWAN_ERR_NONE,        3, &Statistics2, "Valid Test Case, Only Time Stamp." },
            { eQCWWAN_ERR_NONE,        4, &Statistics3, "Valid Test Case, Only Attempted Flag" },
            { eQCWWAN_ERR_NONE,        5, &Statistics4, "Valid Test Case, Both Time Stamp. and Attempted Flag" },
        };
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSGetDyingGaspStatisticsTestCases )/
                      sizeof( SLQSGetDyingGaspStatisticsTestCases[0] );
        while ( tc < ncases )
        {
            SLQSGetDyingGaspStatisticsTestCase_t *pd =
                &SLQSGetDyingGaspStatisticsTestCases[tc];
            
            TimeStamp=-1;
            SMSAttemptedFlag=-1;
            fprintf(fp, "\nTest Case %d / %d : %s\n",
                    (tc + 1),
                    ncases,
                    pd->desc);
            nRet = SLQSSwiGetDyingGaspStatistics( pd->pgetDyingGaspStatistics);
            doprintreason( fp, nRet );
            fprintf( fp, "Dying GASP Statistics:\n");
            if(nRet==eQCWWAN_ERR_NONE)
            {
                if(pd->pgetDyingGaspStatistics)
                {
                    if(pd->pgetDyingGaspStatistics->pSMSAttemptedFlag!=NULL)
                        fprintf( fp, "SMS Attempted Flag: %d \n", *pd->pgetDyingGaspStatistics->pSMSAttemptedFlag);
                    if(pd->pgetDyingGaspStatistics->pTimeStamp!=NULL)
                        fprintf( fp, "TimeStamp: %u\n", *pd->pgetDyingGaspStatistics->pTimeStamp);
                }
            }
            tc++;
        }
    }
     if (fp)
        tfclose(fp);
}

/*
 * Name:     SLQSDmsSwiIndicationRegister
 *
 * Purpose:  Perform the tests that call the SLQSDmsSwiIndicationRegister() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSLQSDmsSwiIndicationRegister( void )
{
    ULONG nRet = 0;
    FILE  *fp      = NULL;
    BYTE enabled = 1;
    BYTE disabled = 0;

    fp = tfopen("TestReport/slqsDmsIndicationRegister.txt", "a");

    dmsIndicationRegisterReq req;

    if (fp)
    {
        /* Test Cases */
        SLQSDmsSwiIndicationRegisterTestCase_t SLQSDmsSwiIndicationRegisterTestCases[] =
        {
            { eQCWWAN_ERR_INVALID_ARG, 1, NULL, "Invalid Argument" },
            { eQCWWAN_ERR_NONE,        2, &enabled, "Valid Test Case, enable get reset info indication" },
            { eQCWWAN_ERR_NONE,        3, &disabled, "Valid Test Case, disable get reset info indication" },
            { eQCWWAN_ERR_NONE,        4, &enabled, "Valid Test Case, enable get reset info indication" },
        };
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSDmsSwiIndicationRegisterTestCases )/
                      sizeof( SLQSDmsSwiIndicationRegisterTestCases[0] );
        while ( tc < ncases )
        {
            SLQSDmsSwiIndicationRegisterTestCase_t *pd =
                &SLQSDmsSwiIndicationRegisterTestCases[tc];

            req.pSwiGetResetInd = pd->pGetResetInfoInd;

            fprintf(fp, "\nTest Case %d / %d : %s\n",
                    (tc + 1),
                    ncases,
                    pd->desc);
            nRet = SLQSDmsSwiIndicationRegister( &req);
            doprintreason( fp, nRet );

            if(nRet==eQCWWAN_ERR_NONE)
                fprintf( fp, "API return: success\n");
            else
                fprintf( fp, "API return: fail\n");
            
            tc++;
        }
    }
    if (fp)
        tfclose(fp);
}

/*
 * Name:     SLQSDmsSwiGetResetInfo
 *
 * Purpose:  Perform the tests that call the SLQSDmsSwiGetResetInfo() API
 *
 * Return:   none
 *
 * Notes:    none
 *
 */
void doSLQSDmsSwiGetResetInfo( void )
{
    ULONG nRet = 0;
    FILE  *fp      = NULL;
    BYTE type, source;

    fp = tfopen("TestReport/slqsDmsSwiGetResetInfo.txt", "a");

    dmsSwiGetResetInfo resp;

    if (fp)
    {
        /* Test Cases */
        SLQSDmsSwiGetResetInfoTestCase_t SLQSDmsSwiGetResetInfoTestCases[] =
        {
            { eQCWWAN_ERR_INVALID_ARG, 1, NULL, NULL, "Invalid Argument" },
            { eQCWWAN_ERR_NONE,        2, &type, &source, "Valid Test Case" },
        };
        BYTE tc = 0;
        BYTE ncases = sizeof( SLQSDmsSwiGetResetInfoTestCases )/
                      sizeof( SLQSDmsSwiGetResetInfoTestCases[0] );
        while ( tc < ncases )
        {
            SLQSDmsSwiGetResetInfoTestCase_t *pd =
                &SLQSDmsSwiGetResetInfoTestCases[tc];

            resp.pType   = pd->pType;
            resp.pSource = pd->pSource;

            fprintf(fp, "\nTest Case %d / %d : %s\n",
                    (tc + 1),
                    ncases,
                    pd->desc);
            nRet = SLQSDmsSwiGetResetInfo( &resp);
            doprintreason( fp, nRet );

            if(nRet==eQCWWAN_ERR_NONE)
            {
                fprintf( fp, "API return: success\n");
                if (resp.pType)
                    fprintf( fp, "type: %d\n", *resp.pType);
                if (resp.pSource)
                    fprintf( fp, "source: %d\n", *resp.pSource);
            }
            else
                fprintf( fp, "API return: fail\n");
            
            tc++;
        }
    }
    if (fp)
        tfclose(fp);
}

