/*
 * \ingroup sms
 *
 * \file    qaSmsSwiSetSMSStorage.c
 *
 * \brief   Contains Packing and UnPacking routines for the
 *          QMI_WMS_SWI_SET_SMS_STORAGE message.
 *
 * Copyright: © 2014 Sierra Wireless, Inc. all rights reserved
 */

/* include files */
#include "SwiDataTypes.h"
#include "sludefs.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaSmsSwiSetSMSStorage.h"

/******************************************************************************
 * Request handling
 ******************************************************************************/

/*
 * This function packs the SetSmsStorage Address field
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                 data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE   - Increment successful
 * \return eQCWWAN_ERR_MEMORY - Access beyond allowed size attempted
 *
 */
enum eQCWWANError BuildTlvSmsStorage( BYTE *pBuf, BYTE *pParam )
{
    struct QmiSmsSwiSetSMSStorageReq *pReq =
                     (struct QmiSmsSwiSetSMSStorageReq *)pParam;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Add TLV data */
    eRCode = PutByte( pBuf, pReq->smsStorage );
    return eRCode;
}

/*
 * This function packs the SetSmsStorage parameters to the
 * QMI message SDU
 *
 * \param  pParamField [OUT] - Pointer to storage into which the packed
 *                                    data will be placed by this function.
 *
 * \param  pMlength    [OUT] - Total length of built message.
 *
 * \return eQCWWAN_ERR_NONE   - Increment successful
 * \return eQCWWAN_ERR_MEMORY - Access beyond allowed size attempted
 *
 * \sa     qaGobiApiSms.h for remaining parameter descriptions.
 *
 */
enum eQCWWANError PkQmiSmsSwiSetSMSStorage(
    WORD *pMlength,
    BYTE *pParamField,
    BYTE smsStorage )
{
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_SET_SMS_STORAGE,  &BuildTlvSmsStorage },
        { eTLV_TYPE_INVALID, NULL } /* Important. Sentinel.
                                     * Signifies last item in map.
                                     */
    };

    enum   eQCWWANError            eRCode;
    struct QmiSmsSwiSetSMSStorageReq req;

    slmemset((CHAR *)&req, 0, sizeof(struct QmiSmsSwiSetSMSStorageReq));
    req.smsStorage = smsStorage;

    eRCode = qmbuild ( pParamField,
                       (BYTE *)&req,
                       map,
                       eQMI_WMS_SWI_SET_SMS_STORAGE,
                       pMlength );
    return eRCode;
}

/******************************************************************************
 * Response handling
 ******************************************************************************/
/*
 * This function unpacks the SetSmsStorage response
 * message to a user-provided response structure.
 *
 * \param  pMdmResp  [IN]  - Pointer to packed response from the modem.
 *
 * \param  pApiResp  [OUT] - Pointer to storage to unpack into.
 *
 * \return eQCWWAN_ERR_NONE   - Increment successful
 * \return eQCWWAN_ERR_MEMORY - Access beyond allowed size attempted
 *
 */
enum eQCWWANError UpkQmiSmsSwiSetSMSStorage(
    BYTE                              *pMdmResp,
    struct QmiSmsSwiSetSMSStorageResp *pApiResp )
{
    enum eQCWWANError eRCode;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,  &qmUnpackTlvResultCode },
        { eTLV_TYPE_INVALID, NULL } /* Important. Sentinel.
                                     * Signifies last item in map.
                                     */
    };
    eRCode = qmunpackresp( pMdmResp,
                           (BYTE *)pApiResp,
                           map,
                           eQMI_WMS_SWI_SET_SMS_STORAGE );
    return eRCode;
}

