/*
 * \ingroup sms
 *
 * \file    qaSmsAsyncRawSend.c
 *
 * \brief   Contains Packing and UnPacking routines for the
 *          QMI_WMS_RAW_SEND message.
 *
 * Copyright: © 2014 Sierra Wireless, Inc. all rights reserved
 */

/* include files */
#include "SwiDataTypes.h"
#include "sludefs.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaSmsAsyncRawSend.h"

/******************************************************************************
 * Request handling
 ******************************************************************************/

/*
 * This function packs the AsyncSendSMS Message Data field
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                 data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE   - Increment successful
 * \return eQCWWAN_ERR_MEMORY - Access beyond allowed size attempted
 *
 */
enum eQCWWANError BuildTlvAsyncMessageData( BYTE *pBuf, BYTE *pParam )
{
    struct QmiSmsAsyncRawSendReq *pReq =
                     (struct QmiSmsAsyncRawSendReq *)pParam;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Add TLV data */
    eRCode = PutByte( pBuf, pReq->messageFormat );
    if ( eRCode == eQCWWAN_ERR_NONE )
        eRCode = PutWord( pBuf, pReq->messageSize );

    while ( pReq->messageSize && (eRCode == eQCWWAN_ERR_NONE) )
    {
        eRCode = PutByte( pBuf, *(pReq->pMessage)++ );
        pReq->messageSize--;
    }

    return eRCode;
}

/*
 * This function packs the Force On DC field
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to link timer which the packed by this function.
 *
 * \param pParam - Pointer to structure containing link timer for this TLV.
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 */
enum eQCWWANError BuildTlvAsyncForceOnDC( BYTE *pBuf, BYTE *pParam )
{
    struct QmiSmsAsyncRawSendReq *pReq =
                     (struct QmiSmsAsyncRawSendReq *)pParam;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    if ( NULL == pReq->pForceOnDC ||
         NULL == pReq->pServiceOption)
        return eRCode;

    /* Add TLV data */
    eRCode = PutByte( pBuf, *pReq->pForceOnDC );

    if(eRCode == eQCWWAN_ERR_NONE)
        /* Add TLV data */
        eRCode = PutByte( pBuf, *pReq->pServiceOption );
    return eRCode;
}

/*
 * This function packs the Follow On DC field
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to link timer which the packed by this function.
 *
 * \param pParam - Pointer to structure containing link timer for this TLV.
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 */
enum eQCWWANError BuildTlvAsyncFollowOnDC( BYTE *pBuf, BYTE *pParam )
{
    struct QmiSmsAsyncRawSendReq *pReq =
                     (struct QmiSmsAsyncRawSendReq *)pParam;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    if(NULL == pReq->pFollowOnDC)
        return eRCode;

    /* Add TLV data */
    return PutByte( pBuf, *pReq->pFollowOnDC );
}

/*
 * This function packs the Link timer field
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to link timer which the packed by this function.
 *
 * \param pParam - Pointer to structure containing link timer for this TLV.
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 */
enum eQCWWANError BuildTlvAsyncLinkTimer( BYTE *pBuf, BYTE *pParam )
{
    struct QmiSmsAsyncRawSendReq *pReq =
                     (struct QmiSmsAsyncRawSendReq *)pParam;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    /* Add TLV data */
    if((pReq->pLinktimer) && (*pReq->pLinktimer))
        eRCode = PutByte( pBuf, *pReq->pLinktimer );
    return eRCode;
}

/*
 * This function packs the SMS On IMS field
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to link timer which the packed by this function.
 *
 * \param pParam - Pointer to structure containing link timer for this TLV.
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 */
enum eQCWWANError BuildTlvAsyncSMSOnIMS( BYTE *pBuf, BYTE *pParam )
{
    struct QmiSmsAsyncRawSendReq *pReq =
                     (struct QmiSmsAsyncRawSendReq *)pParam;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    if(NULL == pReq->pSmsOnIms)
        return eRCode;

    /* Add TLV data */
    return PutByte( pBuf, *pReq->pSmsOnIms );
}

/*
 * This function packs the Retry Message field
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to link timer which the packed by this function.
 *
 * \param pParam - Pointer to structure containing link timer for this TLV.
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 */
enum eQCWWANError BuildTlvAsyncRetryMsg( BYTE *pBuf, BYTE *pParam )
{
    struct QmiSmsAsyncRawSendReq *pReq =
                     (struct QmiSmsAsyncRawSendReq *)pParam;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    if(NULL == pReq->pRetryMessage)
        return eRCode;

    /* Add TLV data */
    return PutByte( pBuf, *pReq->pRetryMessage );

}

/*
 * This function packs the Retry Message ID field
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to link timer which the packed by this function.
 *
 * \param pParam - Pointer to structure containing link timer for this TLV.
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 */
enum eQCWWANError BuildTlvAsyncRetryMsgID( BYTE *pBuf, BYTE *pParam )
{
    struct QmiSmsAsyncRawSendReq *pReq =
                     (struct QmiSmsAsyncRawSendReq *)pParam;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    if(NULL == pReq->pRetryMessage)
        return eRCode;

    if(0x01 == *(pReq->pRetryMessage))
        return eRCode;

    if(NULL == pReq->pRetryMessageId)
        return eRCode;

    /* Add TLV data */
    return PutLong( pBuf, *pReq->pRetryMessageId );

}

/*
 * This function packs the User Data field
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to link timer which the packed by this function.
 *
 * \param pParam - Pointer to structure containing link timer for this TLV.
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 */
enum eQCWWANError BuildTlvAsyncUserData( BYTE *pBuf, BYTE *pParam )
{
    struct QmiSmsAsyncRawSendReq *pReq =
                     (struct QmiSmsAsyncRawSendReq *)pParam;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    if(NULL == pReq->pUserData)
        return eRCode;

    /* Add TLV data */
    return PutLong( pBuf, *pReq->pUserData );
}

/*
 * This function packs the AsyncSendSMS parameters to the QMI message SDU
 *
 * \param  pParamField [OUT] - Pointer to storage into which the packed
 *                             data will be placed by this function.
 *
 * \param  pMlength    [OUT] - Total length of built message.
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * \sa     qaGobiApiSms.h for remaining parameter descriptions.
 */
enum eQCWWANError PkQmiSmsAsyncRawSend(
    WORD  *pMlength,
    BYTE  *pParamField,
    slqssendasyncsmsparams_s *pSendSMSParam )
{
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_ASYNC_MESSAGE_DATA, &BuildTlvAsyncMessageData },
        { eTLV_ASYNC_FORCE_ON_DC,  &BuildTlvAsyncForceOnDC },
        { eTLV_ASYNC_FOLLOW_ON_DC, &BuildTlvAsyncFollowOnDC },
        { eTLV_ASYNC_LINK_TIMER,   &BuildTlvAsyncLinkTimer },
        { eTLV_ASYNC_SMS_ON_IMS,   &BuildTlvAsyncSMSOnIMS },
        { eTLV_ASYNC_RETRY_MSG,    &BuildTlvAsyncRetryMsg },
        { eTLV_ASYNC_RETRY_MSG_ID, &BuildTlvAsyncRetryMsgID },
        { eTLV_ASYNC_USER_DATA,    &BuildTlvAsyncUserData },
        { eTLV_TYPE_INVALID, NULL } /* Important. Sentinel.
                                     * Signifies last item in map.
                                     */
    };

    struct QmiSmsAsyncRawSendReq req;

    enum eQCWWANError eRCode;

    slmemset((CHAR *)&req, 0, sizeof(struct QmiSmsAsyncRawSendReq));
    req.messageFormat   = pSendSMSParam->messageFormat;
    req.messageSize     = pSendSMSParam->messageSize;
    req.pLinktimer      = pSendSMSParam->pLinktimer;
    req.pMessage        = pSendSMSParam->pMessage;
    req.pFollowOnDC     = pSendSMSParam->pFollowOnDC;
    req.pForceOnDC      = pSendSMSParam->pForceOnDC;
    req.pRetryMessage   = pSendSMSParam->pRetryMessage;
    req.pRetryMessageId = pSendSMSParam->pRetryMessageId;
    req.pServiceOption  = pSendSMSParam->pServiceOption;
    req.pSmsOnIms       = pSendSMSParam->pSmsOnIms;
    req.pUserData       = pSendSMSParam->pUserData;

    eRCode = qmbuild ( pParamField,
                       (BYTE *)&req,
                       map,
                       eQMI_WMS_ASYNC_RAW_SEND,
                       pMlength );

    return eRCode;
}

/******************************************************************************
 * Response handling
 ******************************************************************************/

/*
 * This function unpacks the ASYNC_RAW_SEND response
 * message to a user-provided response structure.
 *
 * \param  pMdmResp  [IN]  - Pointer to packed response from the modem.
 *
 * \param  pApiResp  [OUT] - Pointer to storage to unpack into.
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 */
enum eQCWWANError UpkQmiSmsAsyncRawSend(
    BYTE                     *pMdmResp,
    struct QmiSmsAsyncRawSendResp *pApiResp )
{
    enum eQCWWANError eRCode;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,  &qmUnpackTlvResultCode },
        { eTLV_TYPE_INVALID, NULL } /* Important. Sentinel.
                                     * Signifies last item in map.
                                     */
    };
    eRCode = qmunpackresp( pMdmResp,
                           (BYTE *)pApiResp,
                           map,
                           eQMI_WMS_ASYNC_RAW_SEND );
    return eRCode;
}
