/*
 * \ingroup pds
 *
 * \file qaPdsSLQSGetGPSStateInfo.c
 *
 * \brief Contains Packing and UnPacking routines for the
 *        QMI_PDS_GET_GPS_STATE message.
 *
 * Copyright: © 2014 Sierra Wireless, Inc. all rights reserved
 */

/* include files */

#include "SwiDataTypes.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaPdsSLQSGetGPSStateInfo.h"

/******************************************************************************
 * Request handling
 ******************************************************************************/

/*
 * This function packs the SLQSGetGPSStateInfo parameters to the QMI message SDU
 *
 * \param  pParamField [OUT] - Pointer to storage into which the packed
 *                             data will be placed by this function.
 *
 * \param  pMlength    [OUT] - Total length of built message.
 *
 * \return eQCWWAN_ERR_NONE on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError PkQmiPdsSlqsGetGPSStateInfo(
    WORD *pMlength,
    BYTE *pParamField )
{
    /* There is no mandatory TLV and hence the map is empty */
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_TYPE_INVALID, NULL } /* Important. Sentinel.
                                     * Signifies last item in map.
                                     */
    };
    enum eQCWWANError eRCode;
    eRCode = qmbuild( pParamField,
                      NULL,
                      map,
                      eQMI_PDS_GET_GPS_STATE,
                      pMlength );
    return eRCode;
}

/******************************************************************************
 * Response handling
 ******************************************************************************/

/*
 * This function unpacks the SLQSGetGPSStateInfo GPS state Info from the
 * QMI response message to a user provided response structure
 *
 * \param  pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param  pResp    [OUT]     - Pointer to structure containing storage
 *                              to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UnpackTlvGPSStateInfo( BYTE* pTlvData, BYTE *pResp )
{
    struct QmiPdsSLQSGetGPSStateInfoResp *lResp =
        (struct QmiPdsSLQSGetGPSStateInfoResp *)pResp;

    GPSStateInfo *tResp = lResp->pGPSStateInfo;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE              lTemp;
    ULONG             uTemp;
    ULONGLONG         ulTemp;
    WORD              wTemp;

    if ( NULL == tResp )
         return eRCode;

    /* Get Engine State */
    eRCode = GetByte( pTlvData, &lTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->EngineState = lTemp;

    /* Get Validity Mask */
    eRCode = GetLong( pTlvData, &uTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->ValidMask = uTemp;

    /* Get Latitude */
    eRCode = GetLongLong( pTlvData, &ulTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->Latitude = ulTemp;

    /* Get Longitude */
    eRCode = GetLongLong( pTlvData, &ulTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->Longitude = ulTemp;

    /* Get Horizontal Uncertainty */
    eRCode = GetLong( pTlvData, &uTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->HorizontalUncertainty = uTemp;

    /* Get Altitude */
    eRCode = GetLong( pTlvData, &uTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->Altitude = uTemp;

    /* Get Vertical Uncertainty */
    eRCode = GetLong( pTlvData, &uTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->VerticalUncertainty = uTemp;

    /* Get Timestamp tow ms */
    eRCode = GetLong( pTlvData, &uTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->TimeStmp_tow_ms = uTemp;

    /* Get Timestamp gps week */
    eRCode = GetWord( pTlvData, &wTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->TimeStmp_gps_week = wTemp;

    /* Get Time uncertainty ms */
    eRCode = GetLong( pTlvData, &uTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->Time_uncert_ms = uTemp;

    /* Get Iono validity */
    eRCode = GetByte( pTlvData, &lTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->Iono_valid = uTemp;

    /* Get GPS ephemeris SV mask */
    eRCode = GetLong( pTlvData, &uTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->gps_ephemeris_sv_msk = uTemp;

    /* Get GPS almanac SV mask */
    eRCode = GetLong( pTlvData, &uTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->gps_almanac_sv_msk = uTemp;

    /* Get GPS health SV mask */
    eRCode = GetLong( pTlvData, &uTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->gps_health_sv_msk = uTemp;

    /* Get GPS visible SV mask */
    eRCode = GetLong( pTlvData, &uTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->gps_visible_sv_msk = uTemp;

    /* Get GLO ephemeris SV mask */
    eRCode = GetLong( pTlvData, &uTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->glo_ephemeris_sv_msk = uTemp;

    /* Get GLO almanac SV mask */
    eRCode = GetLong( pTlvData, &uTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->glo_almanac_sv_msk = uTemp;

    /* Get GLO health SV mask */
    eRCode = GetLong( pTlvData, &uTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->glo_health_sv_msk = uTemp;

    /* Get GLO visible SV mask */
    eRCode = GetLong( pTlvData, &uTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->glo_visible_sv_msk = uTemp;

    /* Get SBAS ephemeris SV mask */
    eRCode = GetLong( pTlvData, &uTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->sbas_ephemeris_sv_msk = uTemp;

    /* Get SBAS almanac SV mask */
    eRCode = GetLong( pTlvData, &uTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->sbas_almanac_sv_msk = uTemp;

    /* Get SBAS health SV mask */
    eRCode = GetLong( pTlvData, &uTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->sbas_health_sv_msk = uTemp;

    /* Get SBAS visible SV mask */
    eRCode = GetLong( pTlvData, &uTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->sbas_visible_sv_msk = uTemp;

    /* Get Xtra Start GPS Week */
    eRCode = GetWord( pTlvData, &wTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->xtra_start_gps_week = wTemp;

    /* Get Xtra Start GPS Minutes */
    eRCode = GetWord( pTlvData, &wTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->xtra_start_gps_minutes = wTemp;

    /* Get Xtra valid duration hours */
    eRCode = GetWord( pTlvData, &wTemp );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    tResp->xtra_valid_duration_hours = wTemp;

    return eRCode;
}

/*
 * This function unpacks the SLQSGetGPSStateInfo response message to a
 * user-provided response structure.
 *
 * \param  pMdmResp   [IN]  - Pointer to packed response from the modem.
 *
 * \param  pApiResp   [OUT] - Pointer to storage to unpack into.
 *
 * \return eQCWWAN_ERR_NONE on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UpkQmiSlqsGetGPSStateInfo(
    BYTE                                 *pMdmResp,
    struct QmiPdsSLQSGetGPSStateInfoResp *pApiResp )
{
    ULONG eRCode;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,    &qmUnpackTlvResultCode },
        { eTLV_GPS_STATE_INFO, &UnpackTlvGPSStateInfo },
        { eTLV_TYPE_INVALID,   NULL } /* Important. Sentinel.
                                       * Signifies last item in map.
                                       */
    };
    eRCode = qmunpackresp( pMdmResp,
                           (BYTE *)pApiResp,
                           map,
                           eQMI_PDS_GET_GPS_STATE );
    return eRCode;
}
