/*
 * \ingroup : nas
 *
 * \file    : qaNasSwiIndicationRegister.c
 *
 * \brief   : Contains Packing and UnPacking routines for the
 *            eQMI_NAS_INDICATION_REGISTER message.
 *
 * Copyright: © 2014 Sierra Wireless, Inc. all rights reserved
 */

/* include files */

#include "SwiDataTypes.h"
#include "sludefs.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaNasSwiIndicationRegister.h"
#include "qaGobiApiNas.h"

/******************************************************************************
 * Request handling
 ******************************************************************************/
/*
 * This function packs the qaNasSwiIndicationRegister field to the
 * QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                  data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE  on success, eQCWWAN_ERR_XXX on error
 *
 */
enum eQCWWANError BuildTlvSwiIndicationReg( BYTE *pBuf, BYTE *pParam )
{
    NasSwiIndReg *pReq = (NasSwiIndReg *)pParam;
    enum eQCWWANError eRCode;

    if( NULL == pReq )
    {
        return eQCWWAN_ERR_NONE;
    }

    /* Insert lte_esm_ul field value */
    eRCode = PutByte( pBuf, pReq->lteEsmUl );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    /* Insert lte_esm_dl field value */
    eRCode = PutByte( pBuf, pReq->lteEsmDl);
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    /* Insert lte_emm_ul field value */
    eRCode = PutByte( pBuf, pReq->lteEmmUl );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    /* Insert lte_emm_dl field value */
    eRCode = PutByte( pBuf, pReq->lteEmmDl);
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    /* Insert lte_umts_ul field value */
    eRCode = PutByte( pBuf, pReq->gsmUmtsUl );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    /* Insert lte_umts_dl field value */
    eRCode = PutByte( pBuf, pReq->gsmUmtsDl);
    return eRCode;
}


enum eQCWWANError BuildTlvSwiRankIndicatorIndicationReg( BYTE *pBuf, BYTE *pParam )
{
    NasSwiIndReg *pReq = (NasSwiIndReg *)pParam;
    enum eQCWWANError eRCode;

    if( NULL == pReq )
    {
        return eQCWWAN_ERR_NONE;
    }

    if(pReq->pRankIndicatorInd == NULL)
    {
        return eQCWWAN_ERR_NONE;
    }
    /* Insert Rank Indicator field value */
    eRCode = PutByte( pBuf, *(pReq->pRankIndicatorInd) );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    return eRCode;
}

/*
 * This function packs the SwiIndicationRegister parameters to the QMI message SDU
 *
 * \param  pParamField [OUT] - Pointer to storage into which the packed
 *                                    data will be placed by this function.
 *
 * \param  pMlength    [OUT] - Total length of built message.
 *
 * \return eQCWWAN_ERR_NONE  on success, eQCWWAN_ERR_XXX on error
 *
 * \sa     qaGobiApiNas.h for remaining parameter descriptions.
 */
enum eQCWWANError PkQmiNasSwiIndicationRegister(
    WORD             *pMlength,
    BYTE             *pParamField,
    NasSwiIndReg *pSwiIndiReg
    )
{
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_OTA_MSG_IND, &BuildTlvSwiIndicationReg },
        {eTLV_NAS_SWI_INDICATION_RANK_INDICATOR_IND,
             &BuildTlvSwiRankIndicatorIndicationReg },
        { eTLV_TYPE_INVALID,NULL } /* Important. Sentinel.
                                    * Signifies last item in map.
                                    */
    };

    enum   eQCWWANError             eRCode;

    eRCode = qmbuild( pParamField,
                      (BYTE *)pSwiIndiReg,
                      map,
                      eQMI_NAS_SWI_INDICATION_REGISTER,
                      pMlength );
    return eRCode;
}

/******************************************************************************
 * Response handling
 *****************************************************************************/

/*
 * This function unpacks the eQMI_NAS_INDICATION_REGISTER response
 * message to a user-provided response structure.
 *
 * \param  pMdmResp   [IN]  - Pointer to packed response from the modem.
 *
 * \param  pApiResp  [OUT] - Pointer to storage to unpack into.
 *
 * \return eQCWWAN_ERR_NONE  on success, eQCWWAN_ERR_XXX on error
 *
 */
enum eQCWWANError UpkQmiNasSwiIndicationRegister(
    BYTE                                   *pMdmResp,
    struct QmiNasSwiIndicationRegisterResp *pApiResp)
{
    enum eQCWWANError eRCode;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,  &qmUnpackTlvResultCode },
        { eTLV_TYPE_INVALID, NULL } /* Important. Sentinel.
                                     * Signifies last item in map.
                                     */
    };
    eRCode = qmunpackresp( pMdmResp,
                           (BYTE *)pApiResp,
                           map,
                           eQMI_NAS_SWI_INDICATION_REGISTER );
    return eRCode;
}

