/*
 * \ingroup nas
 *
 * \file qaNasSLQSSwiNetworkDebug.c
 *
 * \brief  Contains Packing and UnPacking routines for the
 *         QMI_NAS_SWI_NETWORK_DEBUG message.
 *
 * Copyright: © 2014 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */

#include "SwiDataTypes.h"
#include "sludefs.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaGobiApiNas.h"
#include "qaNasSLQSSwiNetworkDebug.h"

/*****************************************************************************
 * Request handling
 ******************************************************************************/

/*
 * This function packs the SLQSSwiNetworkDebug parameters
 * to the QMI message SDU
 *
 * \param  pParamField [OUT] - Pointer to storage into which the packed
 *                             data will be placed by this function.
 *
 * \param  pMlength    [OUT] - Total length of built message.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError PkQmiNasSlqsSwiNetworkDebug(
    WORD *pMlength,
    BYTE *pParamField )
{
    /* There is no mandatory TLV and hence the map is empty */
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_TYPE_INVALID, NULL }  /* Important. Sentinel.
                                      * Signifies last item in map.
                                      */
    };

    enum eQCWWANError eRCode;
    eRCode = qmbuild( pParamField,
                      NULL,
                      map,
                      eQMI_NAS_SWI_NETWORK_DEBUG,
                      pMlength);
    return eRCode;
}

/*****************************************************************************
 * Response handling
 ******************************************************************************/

/*
 * This function unpacks the version number from the
 * QMI response message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError UnpackTlvVerNumber(
    BYTE *pTlvData,
    BYTE *pResp )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    struct QmiNasSlqsSwiNetworkDebugResp *lResp =
            ( struct QmiNasSlqsSwiNetworkDebugResp * )pResp;
    NetworkDebugResp *tResp = lResp->pNetworkDebugResp;

    /*Check for parameter for NULL*/
    if( NULL == tResp->pObjectVer )
    {
        return eQCWWAN_ERR_NONE;
    }

    /* Get Version Number from the response */
    eRCode = GetByte( pTlvData, tResp->pObjectVer );
    return eRCode;
}

/*
 * This function unpacks the 1x Network Status Details from the
 * QMI response message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError UnpackTlv1xNWStat(
    BYTE *pTlvData,
    BYTE *pResp )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    struct QmiNasSlqsSwiNetworkDebugResp *lResp =
            ( struct QmiNasSlqsSwiNetworkDebugResp * )pResp;
    NetworkDebugResp *tResp = lResp->pNetworkDebugResp;
    NetworkStat1x    *NSResp = tResp->pNetworkStat1x;

    BYTE  bTemp, ctr = 0;
    WORD  wTemp;
    ULONG uTemp;

    /*Check for parameter for NULL*/
    if( NULL == NSResp )
    {
        return eQCWWAN_ERR_NONE;
    }

    if( NULL == NSResp->pActPilotPNElements ||
        NULL == NSResp->pNeighborSetPilotPN )
    {
       return eQCWWAN_ERR_BUFFER_SZ;
    }

    /* Get state from the response */
    eRCode = GetByte( pTlvData, &bTemp );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    NSResp->State = bTemp;

    /* Get SO from the response */
    eRCode = GetWord( pTlvData, &wTemp );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    NSResp->SO = wTemp;

    /* Get RX Pwr from the response */
    eRCode = GetLong( pTlvData, &uTemp );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    NSResp->RX_PWR = uTemp;

    /* Get RX EC/IO from the response */
    eRCode = GetWord( pTlvData, &wTemp );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    NSResp->RX_EC_IO = wTemp;

    /* Get TX Pwr from the response */
    eRCode = GetLong( pTlvData, &uTemp );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    NSResp->TX_PWR = uTemp;

    /* Get Active Set count from the response */
    eRCode = GetByte( pTlvData, &bTemp );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    NSResp->ActSetCnt = bTemp;

    for( ctr = 0; ctr < NSResp->ActSetCnt; ctr++ )
    {
        ActPilotPNElement *APResp;
        APResp = &NSResp->pActPilotPNElements[ctr];

        /* Get Active Pilot PN from the response */
        eRCode = GetWord( pTlvData, &wTemp );
        if( eQCWWAN_ERR_NONE != eRCode )
        {
            return eRCode;
        }
        APResp->ActSetPilotPN = wTemp;

        /* Get Active Pilot Strength from the response */
        eRCode = GetByte( pTlvData, &bTemp );
        if( eQCWWAN_ERR_NONE != eRCode )
        {
            return eRCode;
        }
        APResp->ActSetPilotPNStrength = bTemp;
    }

    /* Get Neighbor set count from the response */
    eRCode = GetByte( pTlvData, &bTemp );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    NSResp->NeighborSetCnt = bTemp;

    for( ctr = 0; ctr < NSResp->NeighborSetCnt; ctr++ )
    {
        /* Get Neighbor set pilot PN from the response */
        eRCode = GetWord( pTlvData, &wTemp );
        if( eQCWWAN_ERR_NONE != eRCode )
        {
            return eRCode;
        }
        NSResp->pNeighborSetPilotPN[ctr] = wTemp;
    }
    return eRCode;
}

/*
 * This function unpacks the EVDO Network Status Details from the
 * QMI response message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError UnpackTlvEVDONWStat(
    BYTE *pTlvData,
    BYTE *pResp )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    struct QmiNasSlqsSwiNetworkDebugResp *lResp =
            ( struct QmiNasSlqsSwiNetworkDebugResp * )pResp;
    NetworkDebugResp *tResp = lResp->pNetworkDebugResp;
    NetworkStatEVDO  *NSResp = tResp->pNetworkStatEVDO;

    BYTE bTemp, ctr = 0;
    WORD wTemp;

    /*Check for parameter for NULL*/
    if( NULL == NSResp )
    {
        return eQCWWAN_ERR_NONE;
    }

    if( NULL == NSResp->pSectorID )
    {
       return eQCWWAN_ERR_BUFFER_SZ;
    }

    /* Get state from the response */
    eRCode = GetByte( pTlvData, &bTemp );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    NSResp->State = bTemp;

    /* Get MAC index from the response */
    eRCode = GetByte( pTlvData, &bTemp );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    NSResp->MACIndex = bTemp;

    /* Get Sector ID length from the response */
    eRCode = GetByte( pTlvData, &bTemp );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    NSResp->SectorIDLen = bTemp;

    for( ctr = 0; ctr < NSResp->SectorIDLen; ctr++ )
    {
        /* Get Sector ID from the response */
        eRCode = GetWord( pTlvData, &wTemp );
        if( eQCWWAN_ERR_NONE != eRCode )
        {
            return eRCode;
        }
        NSResp->pSectorID[ctr] = wTemp;
    }

    /* Get RX PWR from the response */
    eRCode = GetWord( pTlvData, &wTemp );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    NSResp->RX_PWR = wTemp;

    /* Get PER from the response */
    eRCode = GetWord( pTlvData, &wTemp );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    NSResp->PER = wTemp;

    /* Get Pilot Energy count from the response */
    eRCode = GetWord( pTlvData, &wTemp );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    NSResp->PilotEnergy = wTemp;

    /* Get SNR from the response */
    eRCode = GetByte( pTlvData, &bTemp );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    NSResp->SNR = bTemp;

    return eRCode;
}

/*
 * This function unpacks the Device configuration details from the
 * QMI response message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError UnpackTlvDevCfgDetails(
    BYTE *pTlvData,
    BYTE *pResp )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    struct QmiNasSlqsSwiNetworkDebugResp *lResp =
            ( struct QmiNasSlqsSwiNetworkDebugResp * )pResp;
    NetworkDebugResp   *tResp = lResp->pNetworkDebugResp;
    DeviceConfigDetail *DCResp = tResp->pDeviceConfigDetail;

    BYTE bTemp;

    /*Check for parameter for NULL*/
    if( NULL == DCResp )
    {
        return eQCWWAN_ERR_NONE;
    }

    /* Get technology from the response */
    eRCode = GetByte( pTlvData, &bTemp );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    DCResp->Technology = bTemp;

    /* Get QLIC from the response */
    eRCode = GetByte( pTlvData, &bTemp );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    DCResp->QLIC = bTemp;

    /* Get chipset from the response */
    eRCode = GetByte( pTlvData, &bTemp );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    DCResp->Chipset = bTemp;

    /* Get Hw Version from the response */
    eRCode = GetByte( pTlvData, &bTemp );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    DCResp->HWVersion = bTemp;

    return eRCode;
}

/*
 * This function unpacks the Data status details from the
 * QMI response message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError UnpackTlvDataStatDetails(
    BYTE *pTlvData,
    BYTE *pResp )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    struct QmiNasSlqsSwiNetworkDebugResp *lResp =
            ( struct QmiNasSlqsSwiNetworkDebugResp * )pResp;
    NetworkDebugResp *tResp = lResp->pNetworkDebugResp;
    DataStatusDetail *DSResp = tResp->pDataStatusDetail;

    BYTE  bTemp;
    ULONG uTemp;

    /*Check for parameter for NULL*/
    if( NULL == DSResp )
    {
        return eQCWWAN_ERR_NONE;
    }

    /* Get technology from the response */
    eRCode = GetLong( pTlvData, &uTemp );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    DSResp->IPAddress = uTemp;

    /* Get Last Error code from the response */
    eRCode = GetByte( pTlvData, &bTemp );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    DSResp->LastErrCode = bTemp;

    return eRCode;
}
/*
 * This function unpacks the SLQSSwiNetworkDebug response message to to a
 * user-provided response structure.
 *
 * \param     pMdmResp   [IN]  - Pointer to packed response from the modem.
 *
 * \param     pApiResp  [OUT] - Pointer to storage to unpack into.
 *
 * \return    eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError UpkQmiNasSlqsSwiNetworkDebug(
    BYTE                                 *pMdmResp,
    struct QmiNasSlqsSwiNetworkDebugResp *pApiResp )
{
    enum eQCWWANError eRCode;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,           &qmUnpackTlvResultCode },
        { eTLV_VERSION_NUMBER,        &UnpackTlvVerNumber },
        { eTLV_1X_NW_STATUS,          &UnpackTlv1xNWStat },
        { eTLV_EVDO_NW_STATUS,        &UnpackTlvEVDONWStat },
        { eTLV_DEVICE_CONFIG_DETAILS, &UnpackTlvDevCfgDetails },
        { eTLV_DATA_STATUS_DETAILS,   &UnpackTlvDataStatDetails },
        { eTLV_TYPE_INVALID,          NULL }  /* Important. Sentinel.
                                               * Signifies last item in map.
                                               */
    };
    eRCode = qmunpackresp( pMdmResp,
                          (BYTE*)pApiResp,
                          map,
                          eQMI_NAS_SWI_NETWORK_DEBUG );
    return eRCode;
}
