/*
 * \ingroup nas
 *
 * \file qaNasSLQSSwiGetHRPDStats.c
 *
 * \brief  Contains Packing and UnPacking routines for the
 *         QMI_NAS_SWI_GET_HRPD_STATS message.
 *
 * Copyright: © 2014 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */

#include "SwiDataTypes.h"
#include "sludefs.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaGobiApiNas.h"
#include "qaNasSLQSSwiGetHRPDStats.h"

/*****************************************************************************
 * Request handling
 ******************************************************************************/

/*
 * This function packs the SLQSSwiGetHRPDStats parameters
 * to the QMI message SDU
 *
 * \param  pParamField [OUT] - Pointer to storage into which the packed
 *                             data will be placed by this function.
 *
 * \param  pMlength    [OUT] - Total length of built message.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError PkQmiNasSlqsSwiGetHRPDStats(
    WORD *pMlength,
    BYTE *pParamField )
{
    /* There is no mandatory TLV and hence the map is empty */
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_TYPE_INVALID, NULL }  /* Important. Sentinel.
                                      * Signifies last item in map.
                                      */
    };

    enum eQCWWANError eRCode;
    eRCode = qmbuild( pParamField,
                      NULL,
                      map,
                      eQMI_NAS_SWI_GET_HRPD_STATS,
                      pMlength);
    return eRCode;
}

/*****************************************************************************
 * Response handling
 ******************************************************************************/

/*
 * This function unpacks the Current Data Rate Channel from the
 * QMI response message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError UnpackTlvDRC(
    BYTE *pTlvData,
    BYTE *pResp )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    struct QmiNasSlqsSwiGetHRPDStatsResp *lResp =
            ( struct QmiNasSlqsSwiGetHRPDStatsResp * )pResp;
    GetHRPDStatsResp *tResp = lResp->pGetHRPDStatsResp;
    DRCParams        *pTemp = tResp->pDRCParams;
    BYTE             bTemp;

    /*Check for parameter for NULL*/
    if( NULL == pTemp )
    {
        return eQCWWAN_ERR_NONE;
    }

    /* Get DRC Value from the response */
    eRCode = GetByte( pTlvData, &bTemp );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    pTemp->DRCValue = bTemp;

    eRCode = GetByte( pTlvData, &bTemp );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    pTemp->DRCCover = bTemp;

    return eRCode;
}

/*
 * This function unpacks the UATI from the
 * QMI response message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError UnpackTlvUATI(
    BYTE *pTlvData,
    BYTE *pResp )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    struct QmiNasSlqsSwiGetHRPDStatsResp *lResp =
            ( struct QmiNasSlqsSwiGetHRPDStatsResp * )pResp;

    GetHRPDStatsResp *tResp = lResp->pGetHRPDStatsResp;
    BYTE             bTemp;
    BYTE             ctr = 0;

    /*Check for parameter for NULL*/
    if( NULL == tResp->pUATI )
    {
        return eQCWWAN_ERR_NONE;
    }

    /* Get UATI Value from the response */
    for( ctr = 0; ctr < UATISIZE ; ctr++ )
    {
        eRCode = GetByte( pTlvData, &bTemp );
        if( eQCWWAN_ERR_NONE != eRCode )
        {
            return eRCode;
        }
        tResp->pUATI[ctr] = bTemp;
    }

    return eRCode;
}

/*
 * This function unpacks the Pilot Set from the
 * QMI response message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError UnpackTlvPilotSet(
    BYTE *pTlvData,
    BYTE *pResp )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    struct QmiNasSlqsSwiGetHRPDStatsResp *lResp =
            ( struct QmiNasSlqsSwiGetHRPDStatsResp * )pResp;

    GetHRPDStatsResp *tResp = lResp->pGetHRPDStatsResp;
    PilotSetData     *pPSDResp = tResp->pPilotSetData;
    BYTE             bTemp = 0;
    BYTE             lCtr  = 0;
    ULONG            uTemp = 0;
    WORD             wTemp = 0;

    /*Check for parameter for NULL*/
    if( NULL == pPSDResp )
    {
        return eQCWWAN_ERR_NONE;
    }

    /* Get number of pilot sets */
    eRCode = GetByte( pTlvData, &bTemp );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    /* Check buffer size error */
    if( bTemp > pPSDResp->NumPilots )
    {
        return eQCWWAN_ERR_BUFFER_SZ;
    }
    pPSDResp->NumPilots = bTemp;

    /* Retrieve the Pilot Sets */
    for( lCtr = 0; lCtr < pPSDResp->NumPilots; lCtr++ )
    {
        PilotSetParams   *pPSPResp;
        pPSPResp = &pPSDResp->pPilotSetInfo[lCtr];

        /* Get Pilot Type */
        eRCode = GetLong( pTlvData, &uTemp );
        if( eQCWWAN_ERR_NONE != eRCode )
        {
            return eRCode;
        }
        pPSPResp->PilotType = uTemp;

        /* Get Pilot PN sequence offset */
        eRCode = GetWord( pTlvData, &wTemp );
        if( eQCWWAN_ERR_NONE != eRCode )
        {
            return eRCode;
        }
        pPSPResp->PilotPN = wTemp;

        /* Get Pilot Strength */
        eRCode = GetWord( pTlvData, &wTemp );
        if( eQCWWAN_ERR_NONE != eRCode )
        {
            return eRCode;
        }
        pPSPResp->PilotStrength = wTemp;
    }
    return eRCode;
}

/*
 * This function unpacks the SLQSSwiGetHRPDStats response message to to a
 * user-provided response structure.
 *
 * \param     pMdmResp   [IN]  - Pointer to packed response from the modem.
 *
 * \param     pApiResp  [OUT] - Pointer to storage to unpack into.
 *
 * \return    eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError UpkQmiNasSlqsSwiGetHRPDStats(
    BYTE                                 *pMdmResp,
    struct QmiNasSlqsSwiGetHRPDStatsResp *pApiResp )
{
    enum eQCWWANError eRCode;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,  &qmUnpackTlvResultCode },
        { eTLV_DRC,          &UnpackTlvDRC },
        { eTLV_UATI,         &UnpackTlvUATI },
        { eTLV_PILOT_SET,    &UnpackTlvPilotSet },
        { eTLV_TYPE_INVALID, NULL }  /* Important. Sentinel.
                                      * Signifies last item in map.
                                      */
    };
    eRCode = qmunpackresp( pMdmResp,
                          (BYTE*)pApiResp,
                          map,
                          eQMI_NAS_SWI_GET_HRPD_STATS );
    return eRCode;
}
