/*
 * \ingroup nas
 *
 * \file qaNasSLQSSwiGetChannelLock.c
 *
 * \brief  Contains Packing and UnPacking routines for the
 *         QMI_NAS_SWI_GET_CHANNEL_LOCK message.
 *
 * Copyright: © 2015 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */

#include "SwiDataTypes.h"
#include "sludefs.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaGobiApiNas.h"
#include "qaNasSLQSSwiGetChannelLock.h"

/*****************************************************************************
 * Request handling
 ******************************************************************************/
/*
 * This function packs the SLQSNASSwiGetChannelLock parameters
 * to the QMI message SDU
 *
 * \param  pParamField [OUT] - Pointer to storage into which the packed
 *                             data will be placed by this function.
 *
 * \param  pMlength    [OUT] - Total length of built message.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError PkQmiNasSlqsSwiGetChannelLock(
    WORD        *pMlength,
    BYTE        *pParamField )
{
    /* There is no mandatory TLV and hence the map is empty */
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_TYPE_INVALID, NULL }  /* Important. Sentinel.
                                      * Signifies last item in map.
                                      */
    };

    enum eQCWWANError eRCode;
    eRCode = qmbuild( pParamField,
                      NULL,
                      map,
                      eQMI_NAS_SWI_GET_CHANNEL_LOCK,
                      pMlength);
    return eRCode;
}

/*****************************************************************************
 * Response handling
 ******************************************************************************/
/*
 * This function unpacks the WCDMA UARCFN from the
 * QMI response message to a user provided response structure
 *
 * \param  pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param  pResp    [OUT]     - Pointer to structure containing storage
 *                              to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success
 */
enum eQCWWANError UnpackTlvWcdmaUARFCN( BYTE *pTlvData, BYTE *pResp )
{
    wcdmaUARFCN *lResp =
              ((struct QmiNasSlqsSwiGetChannelLockResp *)pResp)->pNasSwiGetChannelLockResp->pWcdmaUARFCN;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Extract the parameters into the structure */

    /* get status Byte */
    eRCode = GetByte( pTlvData, &lResp->status);
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    /* Get UARFCN */
    eRCode = GetLong( pTlvData, &lResp->uarfcn);
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    return eRCode;
}

/*
 * This function unpacks the LTE UARCFN from the
 * QMI response message to a user provided response structure
 *
 * \param  pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param  pResp    [OUT]     - Pointer to structure containing storage
 *                              to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success
 */
enum eQCWWANError UnpackTlvLteUARFCN( BYTE *pTlvData, BYTE *pResp )
{
    lteEARFCN *lResp =
              ((struct QmiNasSlqsSwiGetChannelLockResp *)pResp)->pNasSwiGetChannelLockResp->pLteEARFCN;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Extract the parameters into the structure */

    /* get status Byte */
    eRCode = GetByte( pTlvData, &lResp->status);
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    /* Get EARFCN primary */
    eRCode = GetLong( pTlvData, &lResp->earfcn0);
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    /* Get EARFCN Secondary*/
    eRCode = GetLong( pTlvData, &lResp->earfcn1);
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    return eRCode;
}

/*
 * This function unpacks the LTE PCI from the
 * QMI response message to a user provided response structure
 *
 * \param  pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param  pResp    [OUT]     - Pointer to structure containing storage
 *                              to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success
 */
enum eQCWWANError UnpackTlvLtePCI( BYTE *pTlvData, BYTE *pResp )
{
    ltePCI *lResp =
              ((struct QmiNasSlqsSwiGetChannelLockResp *)pResp)->pNasSwiGetChannelLockResp->pLtePCI;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Extract the parameters into the structure */

    /* get status Byte */
    eRCode = GetByte( pTlvData, &lResp->status);
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    /* Get EARFCN */
    eRCode = GetLong( pTlvData, &lResp->earfcn);
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    /* Get PCI */
    eRCode = GetLong( pTlvData, &lResp->pci);
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    return eRCode;
}

/*
 * This function unpacks the SLQSNASSwiGetChannelLock response message to to a
 * user-provided response structure.
 *
 * \param     pMdmResp   [IN]  - Pointer to packed response from the modem.
 *
 * \param     pApiResp  [OUT] - Pointer to storage to unpack into.
 *
 * \return    eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError UpkQmiNasSlqsSwiGetChannelLock(
    BYTE                                   *pMdmResp,
    struct QmiNasSlqsSwiGetChannelLockResp *pApiResp )
{
    enum eQCWWANError eRCode;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,  &qmUnpackTlvResultCode },
        { eTLV_WCDMA_UARFCN    ,  &UnpackTlvWcdmaUARFCN},
        { eTLV_LTE_EARFCN      ,  &UnpackTlvLteUARFCN},
        { eTLV_LTE_PCI         ,  &UnpackTlvLtePCI},
        { eTLV_TYPE_INVALID, NULL }  /* Important. Sentinel.
                                      * Signifies last item in map.
                                      */
    };
    eRCode = qmunpackresp( pMdmResp,
                          (BYTE*)pApiResp,
                          map,
                          eQMI_NAS_SWI_GET_CHANNEL_LOCK );
    return eRCode;
}
