/*
 * \ingroup nas
 *
 * \file qaNasSLQSSwiGetCQI.c
 *
 * \brief  Contains Packing and UnPacking routines for the
 *         QMI_NAS_SWI_GET_CQI message.
 *
 * Copyright: © 2014 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */

#include "SwiDataTypes.h"
#include "sludefs.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaGobiApiNas.h"
#include "qaNasSLQSSwiGetCQI.h"

/*****************************************************************************
 * Request handling
 ******************************************************************************/

/*
 * This function packs the SlqsSwiGetCQI parameters
 * to the QMI message SDU
 *
 * \param  pParamField [OUT] - Pointer to storage into which the packed
 *                             data will be placed by this function.
 *
 * \param  pMlength    [OUT] - Total length of built message.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError PkQmiNasSlqsSwiGetCQI(
    WORD        *pMlength,
    BYTE        *pParamField )
{
    /* There is no mandatory TLV and hence the map is empty */
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_TYPE_INVALID, NULL }  /* Important. Sentinel.
                                      * Signifies last item in map.
                                      */
    };

    enum eQCWWANError eRCode;
    eRCode = qmbuild( pParamField,
                      NULL,
                      map,
                      eQMI_NAS_SWI_GET_CQI,
                      pMlength);
    return eRCode;
}

/*****************************************************************************
 * Response handling
 ******************************************************************************/

/*
 * This function unpacks the LTE CQI from the
 * QMI response message to a user provided response structure
 *
 * \param  pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param  pResp    [OUT]     - Pointer to structure containing storage
 *                              to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success
 */
enum eQCWWANError UnpackTlvLteCQI( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiNasSlqsSwiGetCQIResp *lResp =
              (struct QmiNasSlqsSwiGetCQIResp *)pResp;
    LteCQIParm *temp = lResp->pLteCQI;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Extract the parameters into the structure */

    /* Validity CW0: Byte */
    eRCode = GetByte( pTlvData, &(temp->ValidityCW0) );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    /* CQI Value CW0: Byte */
    eRCode = GetByte( pTlvData, &(temp->CQIValueCW0) );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    /* Validity CW1: Byte */
    eRCode = GetByte( pTlvData, &(temp->ValidityCW1) );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    /* CQI Value CW1: Byte */
    eRCode = GetByte( pTlvData, &(temp->CQIValueCW1) );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    return eRCode;
}

/*
 * This function unpacks the SlqsSwiGetCQI response message to to a
 * user-provided response structure.
 *
 * \param     pMdmResp   [IN]  - Pointer to packed response from the modem.
 *
 * \param     pApiResp  [OUT] - Pointer to storage to unpack into.
 *
 * \return    eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError UpkQmiNasSlqsSwiGetCQI(
    BYTE                           *pMdmResp,
    struct QmiNasSlqsSwiGetCQIResp *pApiResp )
{
    enum eQCWWANError eRCode;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,  &qmUnpackTlvResultCode },
        { eTLV_LTE_CQI    ,  &UnpackTlvLteCQI},
        { eTLV_TYPE_INVALID, NULL }  /* Important. Sentinel.
                                      * Signifies last item in map.
                                      */
    };
    eRCode = qmunpackresp( pMdmResp,
                          (BYTE*)pApiResp,
                          map,
                          eQMI_NAS_SWI_GET_CQI );
    return eRCode;
}
