/*
 * \ingroup nas
 *
 * \file qaNasSLQSGetErrorRate.c
 *
 * \brief  Contains Packing and UnPacking routines for the
 *         QMI_NAS_GET_ERR_RATE message.
 *
 * Copyright: © 2014 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */

#include "SwiDataTypes.h"
#include "sludefs.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaGobiApiNas.h"
#include "qaNasSLQSGetErrorRate.h"

/*****************************************************************************
 * Request handling
 ******************************************************************************/

/*
 * This function packs the SLQSGetErrorRate parameters
 * to the QMI message SDU
 *
 * \param  pParamField [OUT] - Pointer to storage into which the packed
 *                             data will be placed by this function.
 *
 * \param  pMlength    [OUT] - Total length of built message.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError PkQmiNasSlqsGetErrorRate(
    WORD *pMlength,
    BYTE *pParamField )
{
    /* There is no mandatory TLV and hence the map is empty */
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_TYPE_INVALID, NULL }  /* Important. Sentinel.
                                      * Signifies last item in map.
                                      */
    };

    enum eQCWWANError eRCode;
    eRCode = qmbuild( pParamField,
                      NULL,
                      map,
                      eQMI_NAS_GET_ERR_RATE,
                      pMlength);
    return eRCode;
}

/*****************************************************************************
 * Response handling
 ******************************************************************************/
/*
 * This function unpacks the CDMA Frame Error Rate from the
 * QMI response message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError UnpackTlvCDMAFameErrRate(
    BYTE *pTlvData,
    BYTE *pResp )
{
    struct QmiNasSlqsGetErrorRateResp *lResp =
            ( struct QmiNasSlqsGetErrorRateResp * )pResp;
    GetErrRateResp *tResp = lResp->pGetErrRateResp;

    /*Check for parameter for NULL*/
    if( NULL == tResp->pCDMAFrameErrRate )
    {
        return eQCWWAN_ERR_NONE;
    }

    /* Get CDMA Frame Error Rate from the response */
    return GetWord( pTlvData, tResp->pCDMAFrameErrRate );

}

/*
 * This function unpacks the HDR Packet Error Rate from the
 * QMI response message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError UnpackTlvHDRPktErrRate(
    BYTE *pTlvData,
    BYTE *pResp )
{
    struct QmiNasSlqsGetErrorRateResp *lResp =
            ( struct QmiNasSlqsGetErrorRateResp * )pResp;
    GetErrRateResp *tResp = lResp->pGetErrRateResp;

    /*Check for parameter for NULL*/
    if( NULL == tResp->pHDRPackErrRate )
    {
        return eQCWWAN_ERR_NONE;
    }

    /* Get HDR Packet Error Rate from the response */
    return GetWord( pTlvData, tResp->pHDRPackErrRate );

}

/*
 * This function unpacks the GSM Bit Error Rate from the
 * QMI response message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError UnpackTlvGSMBitErrRate(
    BYTE *pTlvData,
    BYTE *pResp )
{
    struct QmiNasSlqsGetErrorRateResp *lResp =
            ( struct QmiNasSlqsGetErrorRateResp * )pResp;
    GetErrRateResp *tResp = lResp->pGetErrRateResp;

    /*Check for parameter for NULL*/
    if( NULL == tResp->pGSMBER )
    {
        return eQCWWAN_ERR_NONE;
    }

    /* Get GSM Bit Error Rate from the response */
    return GetByte( pTlvData, tResp->pGSMBER );

}

/*
 * This function unpacks the WCDMA Block Error Rate from the
 * QMI response message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError UnpackTlvWCDMABlkErrRate(
    BYTE *pTlvData,
    BYTE *pResp )
{
    struct QmiNasSlqsGetErrorRateResp *lResp =
            ( struct QmiNasSlqsGetErrorRateResp * )pResp;
    GetErrRateResp *tResp = lResp->pGetErrRateResp;

    /*Check for parameter for NULL*/
    if( NULL == tResp->pWCDMABER )
    {
        return eQCWWAN_ERR_NONE;
    }

    /* Get WCDMA Block Error Rate from the response */
    return GetByte( pTlvData, tResp->pWCDMABER );

}
/*
 * This function unpacks the SlqsgetErrorRate response message to to a
 * user-provided response structure.
 *
 * \param     pMdmResp   [IN]  - Pointer to packed response from the modem.
 *
 * \param     pApiResp  [OUT] - Pointer to storage to unpack into.
 *
 * \return    eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError UpkQmiNasSlqsGetErrorRate(
    BYTE                              *pMdmResp,
    struct QmiNasSlqsGetErrorRateResp *pApiResp )
{
    enum eQCWWANError eRCode;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,          &qmUnpackTlvResultCode },
        { eTLV_CDMA_FRAME_ERR_RATE,  &UnpackTlvCDMAFameErrRate },
        { eTLV_HDR_PACKET_ERR_RATE,  &UnpackTlvHDRPktErrRate },
        { eTLV_GSM_BIT_ERR_RATE,     &UnpackTlvGSMBitErrRate },
        { eTLV_WCDMA_BLOCK_ERR_RATE, &UnpackTlvWCDMABlkErrRate },
        { eTLV_TYPE_INVALID,         NULL }  /* Important. Sentinel.
                                              * Signifies last item in map.
                                              */
    };
    eRCode = qmunpackresp( pMdmResp,
                          (BYTE*)pApiResp,
                          map,
                          eQMI_NAS_GET_ERR_RATE );
    return eRCode;
}
