/*
 * \ingroup nas
 *
 * \file    qaNasGetLteCphycaInfo.c
 *
 * \brief   Contains Packing and UnPacking routines for the
 *          eQMI_NAS_GET_LTE_CPHY_CA_INFO message.
 *
 * Copyright: © 2015 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */
#include "SwiDataTypes.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaNasGetLteCphycaInfo.h"

/******************************************************************************
 * Request handling
 ******************************************************************************/

/*
 * This function packs the SLQS Get Tx/Rx information request parameters
 * to the QMI message SDU
 *
 * \param  pMlength     [OUT] - Total length of built message.
 *
 * \param  pParamField  [OUT] - Pointer to storage into which the packed
 *                              data will be placed by this function.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError PkQmiNasSlqsGetLTECphyCAInfo(
    WORD              *pMlength,
    BYTE              *pParamField)
{
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_TYPE_INVALID,    NULL } /* Important. Sentinel.
                                        * Signifies last item in map
                                        */
    };

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    eRCode = qmbuild( pParamField,
                      NULL,
                      map,
                      eQMI_NAS_GET_LTE_CPHY_CA_INFO,
                      pMlength );
    return eRCode;
}

/* Functions */
/*
 * This function unpacks the Physical Carrier Aggregation of Scell Indicator from the QMI
 * indication message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UnpackTlvGetPhyCaScellInd(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    QmiNasSLQSGetLTECphyCAResp    *pReq = (QmiNasSLQSGetLTECphyCAResp *)pResp;
    nasGetLTECphyCa *lResp = NULL;

    enum eQCWWANError eRCode = eQCWWAN_ERR_ENUM_BEGIN;
    swi_uint16 iTemp = -1;
    swi_uint8 bTemp = 0;

    if(!pReq)
        return eQCWWAN_ERR_MEMORY;
    if(!pReq->pnasGetLTECphyCaResp)
        return eQCWWAN_ERR_MEMORY;
    lResp = pReq->pnasGetLTECphyCaResp;
    lResp->sPhyCaAggScellIndType.TlvPresent = FALSE;
    /* Extract the Physical cell ID of the Scell Range*/
    eRCode = GetWord( pTlvData, &iTemp);
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggScellIndType.pci = iTemp;
    /* Extract the Frequency of the absolute cell Range*/
    eRCode = GetWord( pTlvData,  &iTemp);
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggScellIndType.freq = iTemp;
    /* Extract the Scell state values*/
    eRCode = GetByte( pTlvData, &bTemp);
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggScellIndType.scell_state = bTemp;
    lResp->sPhyCaAggScellIndType.TlvPresent = TRUE;
    return eRCode;
}

/* Functions */
/*
 * This function unpacks the Physical Carrier Aggregation Dwonlonk Bandwidth Type from the QMI
 * indication message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UnpackTlvGetPhyCaScellDlBw(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    QmiNasSLQSGetLTECphyCAResp    *pReq = (QmiNasSLQSGetLTECphyCAResp *)pResp;
    nasGetLTECphyCa *lResp = NULL;
    
    
    enum eQCWWANError eRCode = eQCWWAN_ERR_ENUM_BEGIN;
    ULONG uTemp = -1;

    if(!pReq)
        return eQCWWAN_ERR_MEMORY;
    if(!pReq->pnasGetLTECphyCaResp)
        return eQCWWAN_ERR_MEMORY;
    lResp = pReq->pnasGetLTECphyCaResp;
    lResp->sPhyCaAggScellDlBw.TlvPresent = FALSE;
    /* Extract the Donwlink Bandwidth Values*/
    eRCode = GetLong( pTlvData, &uTemp);
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggScellDlBw.dl_bw_value = uTemp;
    lResp->sPhyCaAggScellDlBw.TlvPresent = TRUE;
    return eRCode;
}

/* Functions */
/*
 * This function unpacks the Scell Information from the QMI
 * indication message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UnpackTlvGetPhyCaScellInfo(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    QmiNasSLQSGetLTECphyCAResp    *pReq = (QmiNasSLQSGetLTECphyCAResp *)pResp;
    nasGetLTECphyCa *lResp = NULL;
    enum eQCWWANError eRCode = eQCWWAN_ERR_ENUM_BEGIN;
    WORD iTemp = -1;
    ULONG uTemp = -1;

    if(!pReq)
        return eQCWWAN_ERR_MEMORY;
    if(!pReq->pnasGetLTECphyCaResp)
        return eQCWWAN_ERR_MEMORY;
    lResp = pReq->pnasGetLTECphyCaResp;
    
    
    lResp->sPhyCaAggScellInfo.TlvPresent = FALSE;
    
    /* Extract the Physical cell ID of the Scell Range*/
    eRCode = GetWord( pTlvData, &iTemp);
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    
    lResp->sPhyCaAggScellInfo.pci = iTemp;
    /* Extract the Frequency of the absolute Scell Range*/
    eRCode = GetWord( pTlvData, &iTemp);
    
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggScellInfo.freq = iTemp;

    /* Extract the Scell Dwonlink Bandwidth values*/
    eRCode = GetLong( pTlvData, &uTemp);
    
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggScellInfo.dl_bw_value = uTemp;
    
    /* Extract the Scell Dwonlink Bandwidth values*/
    eRCode = GetWord( pTlvData, &iTemp);
    
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggScellInfo.iLTEbandValue = iTemp;

    eRCode = GetLong( pTlvData, &uTemp);
    
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggScellInfo.scell_state = uTemp;
    lResp->sPhyCaAggScellInfo.TlvPresent = TRUE;
    return eRCode;
}

/* Functions */
/*
 * This function unpacks the Pcell Information from the QMI
 * indication message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UnpackTlvGetPhyCaPcellInfo(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    QmiNasSLQSGetLTECphyCAResp    *pReq = (QmiNasSLQSGetLTECphyCAResp *)pResp;
    nasGetLTECphyCa *lResp = NULL;

    enum eQCWWANError eRCode = eQCWWAN_ERR_ENUM_BEGIN;
    WORD iTemp = -1;
    ULONG uTemp = -1;

    if(!pReq)
        return eQCWWAN_ERR_MEMORY;
    if(!pReq->pnasGetLTECphyCaResp)
        return eQCWWAN_ERR_MEMORY;
    lResp = pReq->pnasGetLTECphyCaResp;
    lResp->sPhyCaAggPcellInfo.TlvPresent = FALSE;
    
    /* Extract the Physical cell ID of the Pcell Range*/
    eRCode = GetWord( pTlvData, &iTemp);
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggPcellInfo.pci = iTemp;
    /* Extract the Frequency of the absolute Pcell Range*/
    eRCode = GetWord( pTlvData, &iTemp);
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggPcellInfo.freq = iTemp;

    /* Extract the Pcell Dwonlink Bandwidth values*/
    eRCode = GetLong( pTlvData, &uTemp);
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggPcellInfo.dl_bw_value = uTemp;

    /* Extract the Pcell Dwonlink Bandwidth values*/
    eRCode = GetWord( pTlvData, &iTemp);
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggPcellInfo.iLTEbandValue = iTemp;
    lResp->sPhyCaAggPcellInfo.TlvPresent = TRUE;
    return eRCode;
}

/* Functions */
/*
 * This function unpacks the Scell Index from the QMI
 * indication message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UnpackTlvGetPhyCaScellIndex(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    QmiNasSLQSGetLTECphyCAResp    *pReq = (QmiNasSLQSGetLTECphyCAResp *)pResp;
    nasGetLTECphyCa *lResp = NULL;

    enum eQCWWANError eRCode = eQCWWAN_ERR_ENUM_BEGIN;
    swi_uint8 bTemp = -1;

    if(!pReq)
        return eQCWWAN_ERR_MEMORY;
    if(!pReq->pnasGetLTECphyCaResp)
        return eQCWWAN_ERR_MEMORY;
    lResp = pReq->pnasGetLTECphyCaResp;
    lResp->sPhyCaAggScellIndex.TlvPresent = FALSE;
    /* Extract the Scell index*/
    eRCode = GetByte( pTlvData, &bTemp);
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggScellIndex.scell_idx = bTemp;
    lResp->sPhyCaAggScellIndex.TlvPresent = TRUE;
    return eRCode;
}

/*
 * This function unpacks the SLQS Get LTE Physical Carrier Aggregation response message to a
 * user-provided response structure.
 *
 * \param  pMdmResp  - Pointer to packed response from the modem.
 *
 * \param  pApiResp  - Pointer to storage to unpack into.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UpkQmiNasSlqsGetLTECphyCAInfo(
    BYTE                             *pMdmResp,
    QmiNasSLQSGetLTECphyCAResp *pApiResp )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,     &qmUnpackTlvResultCode },
        { eTLV_GET_LTE_CPHY_CA_SCELL_IND_TYPE, &UnpackTlvGetPhyCaScellInd },
        { eTLV_GET_LTE_CPHY_CA_SCELL_DL_BW, &UnpackTlvGetPhyCaScellDlBw },
        { eTLV_GET_LTE_CPHY_CA_SCELL_INFO,         &UnpackTlvGetPhyCaScellInfo},
        { eTLV_GET_LTE_CPHY_CA_PCELL_INFO,         &UnpackTlvGetPhyCaPcellInfo },
        { eTLV_GET_LTE_CPHY_CA_SCELL_INDEX,         &UnpackTlvGetPhyCaScellIndex },
        { eTLV_TYPE_INVALID,    NULL } /* Important. Sentinel.
                                        * Signifies last item in map.
                                        */
    };
    eRCode = qmunpackresp( pMdmResp,
                           (BYTE *)pApiResp,
                           map,
                           eQMI_NAS_GET_LTE_CPHY_CA_INFO );
    return eRCode;
}

