/*
 * \ingroup nas
 *
 * \file qaNasGetHomeNetwork.c
 *
 * \brief  Contains Packing and UnPacking routines for the
 *         QMI_NAS_GET_HOME_NETWORK message.
 *
 * Copyright: © 2011 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */

#include "SwiDataTypes.h"
#include "sludefs.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaQmiBasic.h"

#include "qaNasGetHomeNetwork.h"

#define ASCII7BIT         0x02
#define nMaxStrLen        0xFF

/*****************************************************************************
 * Request handling
 ******************************************************************************/

/*
 * This function packs the GetHomeNetwork parameters
 * to the QMI message SDU
 *
 * \param       pParamField [OUT] - Pointer to storage into which the packed
 *                                    data will be placed by this function.
 *
 * \param       pMlength    [OUT] - Total length of built message.
 *
 * \param       nameSize    [IN]  - Size of the Network Descriptor Array.
 *
 * \return      eQCWWAN_ERR_NONE, on success
 * \return      eQCWWAN_ERR_MEMORY, on error
 *
 */
enum eQCWWANError PkQmiNasGetHomeNetwork(
    WORD     *pMlength,
    BYTE     *pParamField )
{
    /* There is no mandatory TLV and hence the map is empty */
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_TYPE_INVALID, NULL }  /* Important. Sentinel.
                                      * Signifies last item in map.
                                      */
    };

    enum eQCWWANError eRCode;
    eRCode = qmbuild( pParamField,
                      NULL,
                      map,
                      eQMI_NAS_GET_HOME_INFO,
                      pMlength );
    return eRCode;
}


/*****************************************************************************
 * Response handling
 ******************************************************************************/

/*
 * This function unpacks the GetHomeNetwork from the
 * QMI response message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                     to return data for this TLV.
 *
 * \return      eQCWWAN_ERR_NONE, on success
 * \return      eQCWWAN_ERR_MEMORY, on error
 */
enum eQCWWANError UnpackTlvGetHomeNetwork(
    BYTE    *pTlvData,
    BYTE    *pResp )
{
    struct QmiNasGetHomeNetworkResp *lResp =
        (struct QmiNasGetHomeNetworkResp *)pResp;
    BYTE lnameSize = 0, lNameIndex = 0;
    enum eQCWWANError eRCode;

    /* Extract the parameters into the structure */
    eRCode  = GetWord (pTlvData, lResp->pMCC);
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }
    eRCode  = GetWord (pTlvData, lResp->pMNC);
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }
    eRCode  = GetByte (pTlvData, &lnameSize);
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }

    /* Check for the Buffer Undersize Error */
    if ( lnameSize > lResp->nameSize )
        return eQCWWAN_ERR_BUFFER_SZ;

    /* Fill the Network Description */
    while ( lnameSize-- )
    {
        eRCode = GetByte ( pTlvData,
                           (BYTE *)&(lResp->pName[lNameIndex++]) );
    }
    /* Terminate the string with a NULL */
    lResp->pName[lNameIndex] = 0;
    return eRCode;
}

/*
 * This function unpacks the HomeSystemID from the
 * QMI response message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]   - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]      - Pointer to structure containing storage
 *                                     to return data for this TLV.
 *
 * \return      eQCWWAN_ERR_NONE, on success
 * \return      eQCWWAN_ERR_MEMORY, on error
 */
enum eQCWWANError UnpackTlvHomeSystemID(
    BYTE    *pTlvData,
    BYTE    *pResp )
{
    struct QmiNasGetHomeNetworkResp *lResp =
        (struct QmiNasGetHomeNetworkResp *)pResp;
    enum eQCWWANError eRCode;

    if ( !lResp->pSID ||
         !lResp->pNID )
        return eQCWWAN_ERR_NONE;

    /* Extract the parameters into the structure */
    /* Two Byte System ID */
    eRCode  = GetWord (pTlvData, lResp->pSID);
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }
    /* Two Byte Network ID */
    eRCode  = GetWord (pTlvData, lResp->pNID);
    return eRCode;
}

/*
 * This function unpacks the 3GPP2 Home Network extension from the
 * QMI response message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]   - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]      - Pointer to structure containing storage
 *                                     to return data for this TLV.
 *
 * \return      eQCWWAN_ERR_NONE, on success
 * \return      eQCWWAN_ERR_MEMORY, on error
 */
enum eQCWWANError UnpackTlv3GPP2HomeNetExt(
    BYTE    *pTlvData,
    BYTE    *pResp )
{
    struct QmiNasGetHomeNetworkResp *lResp =
        (struct QmiNasGetHomeNetworkResp *)pResp;

    struct HomeNetwork3GPP2Ext *pHomeNetExt = &(lResp->homeNetworkExt);

    BYTE lnameSize = 0, lNameIndex = 0;
    enum eQCWWANError eRCode;

    /* Extract the parameters into the structure */
    eRCode  = GetWord (pTlvData, pHomeNetExt->pNw2MCC);
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }
    eRCode  = GetWord (pTlvData, pHomeNetExt->pNw2MNC);
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }
    eRCode  = GetByte (pTlvData,pHomeNetExt->pNw2DescDisp);
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }
    eRCode  = GetByte (pTlvData,pHomeNetExt->pNw2DescEnc);
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }
    eRCode  = GetByte (pTlvData,&lnameSize);
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }
    /* Check for the Buffer Undersize Error */
    if ( lnameSize > pHomeNetExt->nw2DescLen )
        return eQCWWAN_ERR_BUFFER_SZ;

    pHomeNetExt->nw2DescLen = lnameSize;

    /* Fill the Network Description */
    while ( lnameSize-- )
    {
        eRCode = GetByte ( pTlvData,
                           (BYTE *)&(pHomeNetExt->pNw2Name[lNameIndex++]) );
    }
    /* Terminate the string with a NULL */
    pHomeNetExt->pNw2Name[lNameIndex] = 0;
    return eRCode;
}

void Ascii7BitDecoding(BYTE *pInput, BYTE *pOutput, BYTE length)
{
    BYTE pbuffer = 0, nbuffer = 0, mask = 0;
    BYTE i = 0, j = 0;

    for ( i = 0; i < length; i++)
    {
        /* store next buffer */
        pbuffer = nbuffer;

        mask = mask >> 1;
        mask = mask | 0x80;

        /* mask the higher bits which should be dropped for current buffer */
        nbuffer = (pInput[i] & mask);
        /* mask the lower bits which should be retrieved for current buffer*/
        pOutput[i+j] = (pInput[i] & (~mask));

        /* construct the upper buffer */
        pOutput[i+j] = (pOutput[i+j] << (i % 7));
        /* construct the lower buffer */
        pbuffer = pbuffer >> (8 - (i % 7));

        /* final output = upper buffer | lower buffer */
        pOutput[i+j] = pOutput[i+j] | pbuffer;

        /* special handling for every 7th byte value since an extra 7 bit value will be appeared
           (7 bytes * 8 bit = 56 bits, 56/7 = 8 bytes) */
        if ( i % 7 == 6)
        {
           j++;
           pOutput[i+j] = pInput[i] >> 1;
           mask = 0;
        }
    }
}

/*
 * This function unpacks the GetHomeNetwork response message to to a
 * user-provided response structure.
 *
 * \param     MdmResp   [IN]  - Pointer to packed response from the modem.
 *
 * \param     pApiResp  [OUT] - Pointer to storage to unpack into.
 *
 * \return    eQCWWAN_ERR_NONE, on success
 * \return    eQCWWAN_ERR_MEMORY, on error
 */
enum eQCWWANError UpkQmiNasGetHomeNetwork(
    BYTE   *pMdmResp,
    struct QmiNasGetHomeNetworkResp *pApiResp)
{
    enum eQCWWANError eRCode;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,             &qmUnpackTlvResultCode },
        { eTLV_HOME_NETWORK,            &UnpackTlvGetHomeNetwork },
        { eTLV_HOME_SYSTEM_ID,          &UnpackTlvHomeSystemID },
        { eTLV_3GPP2_HOME_NETWORK_EXT,  &UnpackTlv3GPP2HomeNetExt },
        { eTLV_TYPE_INVALID,            NULL }  /* Important. Sentinel.
                                                 * Signifies last item in map.
                                                 */
    };
    eRCode = qmunpackresp( pMdmResp,
                          (BYTE*)pApiResp,
                          map,
                          eQMI_NAS_GET_HOME_INFO );

    /* if  network description is not present in mandatory TLV, extract it form optional TLV */
    if ( *(pApiResp->pName) == 0)
    {
        if (   (pApiResp->homeNetworkExt.pNw2DescDisp  != NULL) &&
               (pApiResp->homeNetworkExt.pNw2DescEnc   != NULL) &&
             (*(pApiResp->homeNetworkExt.pNw2DescDisp) == 1) &&
             (*(pApiResp->homeNetworkExt.pNw2DescEnc) == ASCII7BIT))
        {
            BYTE *pInput;
            CHAR *pOutput;
            BYTE len = 0;

            pInput = pApiResp->homeNetworkExt.pNw2Name;
            pOutput = pApiResp->pName;
            len = pApiResp->homeNetworkExt.nw2DescLen;

            Ascii7BitDecoding(pInput, (BYTE*)pOutput, len);
        }
    }

    return eRCode;
}
