/*
 * \ingroup : nas
 *
 * \file    : qaNasConfigSigInfo2.c
 *
 * \brief   : Contains Packing and UnPacking routines for the
 *            eQMI_NAS_CONFIG_SIG_INFO2 message.
 *
 * Copyright: © 2015 Sierra Wireless, Inc. all rights reserved
 */

/* include files */

#include "SwiDataTypes.h"
#include "sludefs.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaQmiBasic.h"
#include "qaNasConfigSigInfo2.h"

/******************************************************************************
 * Request handling
 ******************************************************************************/
/*
 * This function packs the SLQSNasConfigSigInfo2 CDMA RSSI threshold
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvCDMARSSIThresh( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;
    CDMARSSIThresh *pRSReq = pReq->pCDMARSSIThresh;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE              lCtr   = 0;

    if( NULL == pRSReq )
    {
        return eRCode;
    }

    /* Add TLV data */
    eRCode = PutByte( pBuf, pRSReq->CDMARSSIThreshListLen );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    for( lCtr = 0; lCtr < pRSReq->CDMARSSIThreshListLen; lCtr++ )
    {
        eRCode = PutWord( pBuf, pRSReq->pCDMARSSIThreshList[lCtr] );
        if( eQCWWAN_ERR_NONE != eRCode )
            return eRCode;
    }

    return eRCode;
}
/*
 * This function packs the SLQSNasConfigSigInfo2 CDMA RSSI Delta
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvCDMARSSIDelta( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;

    if ( pReq->pCDMARSSIDelta == NULL )
        return eQCWWAN_ERR_NONE;

    return PutWord( pBuf, *(pReq->pCDMARSSIDelta) );
}
/*
 * This function packs the SLQSNasConfigSigInfo2 CDMA ECIO threshold
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvCDMAECIOThresh( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;
    CDMAECIOThresh *pRSReq = pReq->pCDMAECIOThresh;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE              lCtr   = 0;

    if( NULL == pRSReq )
    {
        return eRCode;
    }

    /* Add TLV data */
    eRCode = PutByte( pBuf, pRSReq->CDMAECIOThreshListLen );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    for( lCtr = 0; lCtr < pRSReq->CDMAECIOThreshListLen; lCtr++ )
    {
        eRCode = PutWord( pBuf, pRSReq->pCDMAECIOThreshList[lCtr] );
        if( eQCWWAN_ERR_NONE != eRCode )
            return eRCode;
    }

    return eRCode;
}
/*
 * This function packs the SLQSNasConfigSigInfo2 CDMA ECIO Delta
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvCDMAECIODelta( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;

    if ( pReq->pCDMAECIODelta == NULL )
        return eQCWWAN_ERR_NONE;

    return PutWord( pBuf, *(pReq->pCDMAECIODelta) );
}

/*
 * This function packs the SLQSNasConfigSigInfo2 HDR RSSI threshold
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvHDRRSSIThresh( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;
    HDRRSSIThresh *pRSReq = pReq->pHDRRSSIThresh;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE              lCtr   = 0;

    if( NULL == pRSReq )
    {
        return eRCode;
    }

    /* Add TLV data */
    eRCode = PutByte( pBuf, pRSReq->HDRRSSIThreshListLen );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    for( lCtr = 0; lCtr < pRSReq->HDRRSSIThreshListLen; lCtr++ )
    {
        eRCode = PutWord( pBuf, pRSReq->pHDRRSSIThreshList[lCtr] );
        if( eQCWWAN_ERR_NONE != eRCode )
            return eRCode;
    }

    return eRCode;
}
/*
 * This function packs the SLQSNasConfigSigInfo2 HDR RSSI Delta
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvHDRRSSIDelta( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;

    if ( pReq->pHDRRSSIDelta == NULL )
        return eQCWWAN_ERR_NONE;

    return PutWord( pBuf, *(pReq->pHDRRSSIDelta) );
}

/*
 * This function packs the SLQSNasConfigSigInfo2 HDR ECIO threshold
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvHDRECIOThresh( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;
    HDRECIOThresh *pRSReq = pReq->pHDRECIOThresh;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE              lCtr   = 0;

    if( NULL == pRSReq )
    {
        return eRCode;
    }

    /* Add TLV data */
    eRCode = PutByte( pBuf, pRSReq->HDRECIOThreshListLen );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    for( lCtr = 0; lCtr < pRSReq->HDRECIOThreshListLen; lCtr++ )
    {
        eRCode = PutWord( pBuf, pRSReq->pHDRECIOThreshList[lCtr] );
        if( eQCWWAN_ERR_NONE != eRCode )
            return eRCode;
    }

    return eRCode;
}
/*
 * This function packs the SLQSNasConfigSigInfo2 HDR ECIO Delta
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvHDRECIODelta( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;

    if ( pReq->pHDRECIODelta == NULL )
        return eQCWWAN_ERR_NONE;

    return PutWord( pBuf, *(pReq->pHDRECIODelta) );
}

/*
 * This function packs the SLQSNasConfigSigInfo2 HDR SINR threshold
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvHDRSINRThresh( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;
    HDRSINRThreshold *pRSReq = pReq->pHDRSINRThresh;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE              lCtr   = 0;

    if( NULL == pRSReq )
    {
        return eRCode;
    }

    /* Add TLV data */
    eRCode = PutByte( pBuf, pRSReq->HDRSINRThreshListLen );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    for( lCtr = 0; lCtr < pRSReq->HDRSINRThreshListLen; lCtr++ )
    {
        eRCode = PutWord( pBuf, pRSReq->pHDRSINRThreshList[lCtr] );
        if( eQCWWAN_ERR_NONE != eRCode )
            return eRCode;
    }

    return eRCode;
}
/*
 * This function packs the SLQSNasConfigSigInfo2 HDR SINR Delta
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvHDRSINRDelta( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;

    if ( pReq->pHDRSINRDelta == NULL )
        return eQCWWAN_ERR_NONE;

    return PutWord( pBuf, *(pReq->pHDRSINRDelta) );
}

/*
 * This function packs the SLQSNasConfigSigInfo2 HDR IO threshold
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvHDRIOThresh( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;
    HDRIOThresh *pRSReq = pReq->pHDRIOThresh;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE              lCtr   = 0;

    if( NULL == pRSReq )
    {
        return eRCode;
    }

    /* Add TLV data */
    eRCode = PutByte( pBuf, pRSReq->HDRIOThreshListLen );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    for( lCtr = 0; lCtr < pRSReq->HDRIOThreshListLen; lCtr++ )
    {
        eRCode = PutWord( pBuf, pRSReq->pHDRIOThreshList[lCtr] );
        if( eQCWWAN_ERR_NONE != eRCode )
            return eRCode;
    }

    return eRCode;
}
/*
 * This function packs the SLQSNasConfigSigInfo2 HDR IO Delta
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvHDRIODelta( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;

    if ( pReq->pHDRIODelta == NULL )
        return eQCWWAN_ERR_NONE;

    return PutWord( pBuf, *(pReq->pHDRIODelta) );
}

/*
 * This function packs the SLQSNasConfigSigInfo2 GSM RSSI threshold
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvGSMRSSIThresh( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;
    GSMRSSIThresh *pRSReq = pReq->pGSMRSSIThresh;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE              lCtr   = 0;

    if( NULL == pRSReq )
    {
        return eRCode;
    }

    /* Add TLV data */
    eRCode = PutByte( pBuf, pRSReq->GSMRSSIThreshListLen );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    for( lCtr = 0; lCtr < pRSReq->GSMRSSIThreshListLen; lCtr++ )
    {
        eRCode = PutWord( pBuf, pRSReq->pGSMRSSIThreshList[lCtr] );
        if( eQCWWAN_ERR_NONE != eRCode )
            return eRCode;
    }

    return eRCode;
}
/*
 * This function packs the SLQSNasConfigSigInfo2 GSM RSSI Delta
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvGSMRSSIDelta( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;

    if ( pReq->pGSMRSSIDelta == NULL )
        return eQCWWAN_ERR_NONE;

    return PutWord( pBuf, *(pReq->pGSMRSSIDelta) );
}

/*
 * This function packs the SLQSNasConfigSigInfo2 WCDMA RSSI threshold
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvWCDMARSSIThresh( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;
    WCDMARSSIThresh *pRSReq = pReq->pWCDMARSSIThresh;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE              lCtr   = 0;

    if( NULL == pRSReq )
    {
        return eRCode;
    }

    /* Add TLV data */
    eRCode = PutByte( pBuf, pRSReq->WCDMARSSIThreshListLen );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    for( lCtr = 0; lCtr < pRSReq->WCDMARSSIThreshListLen; lCtr++ )
    {
        eRCode = PutWord( pBuf, pRSReq->pWCDMARSSIThreshList[lCtr] );
        if( eQCWWAN_ERR_NONE != eRCode )
            return eRCode;
    }

    return eRCode;
}
/*
 * This function packs the SLQSNasConfigSigInfo2 WCDMA RSSI Delta
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvWCDMARSSIDelta( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;

    if ( pReq->pWCDMARSSIDelta == NULL )
        return eQCWWAN_ERR_NONE;

    return PutWord( pBuf, *(pReq->pWCDMARSSIDelta) );
}

/*
 * This function packs the SLQSNasConfigSigInfo2 WCDMA ECIO threshold
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvWCDMAECIOThresh( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;
    WCDMAECIOThresh *pRSReq = pReq->pWCDMAECIOThresh;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE              lCtr   = 0;

    if( NULL == pRSReq )
    {
        return eRCode;
    }

    /* Add TLV data */
    eRCode = PutByte( pBuf, pRSReq->WCDMAECIOThreshListLen );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    for( lCtr = 0; lCtr < pRSReq->WCDMAECIOThreshListLen; lCtr++ )
    {
        eRCode = PutWord( pBuf, pRSReq->pWCDMAECIOThreshList[lCtr] );
        if( eQCWWAN_ERR_NONE != eRCode )
            return eRCode;
    }

    return eRCode;
}
/*
 * This function packs the SLQSNasConfigSigInfo2 WCDMA ECIO Delta
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvWCDMAECIODelta( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;

    if ( pReq->pWCDMAECIODelta == NULL )
        return eQCWWAN_ERR_NONE;

    return PutWord( pBuf, *(pReq->pWCDMAECIODelta) );
}

/*
 * This function packs the SLQSNasConfigSigInfo2 LTE RSSI threshold
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvLTERSSIThresh( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;
    LTERSSIThresh *pRSReq = pReq->pLTERSSIThresh;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE              lCtr   = 0;

    if( NULL == pRSReq )
    {
        return eRCode;
    }

    /* Add TLV data */
    eRCode = PutByte( pBuf, pRSReq->LTERSSIThreshListLen );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    for( lCtr = 0; lCtr < pRSReq->LTERSSIThreshListLen; lCtr++ )
    {
        eRCode = PutWord( pBuf, pRSReq->pLTERSSIThreshList[lCtr] );
        if( eQCWWAN_ERR_NONE != eRCode )
            return eRCode;
    }

    return eRCode;
}
/*
 * This function packs the SLQSNasConfigSigInfo2 LTE RSSI Delta
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvLTERSSIDelta( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;

    if ( pReq->pLTERSSIDelta == NULL )
        return eQCWWAN_ERR_NONE;

    return PutWord( pBuf, *(pReq->pLTERSSIDelta) );
}

/*
 * This function packs the SLQSNasConfigSigInfo2 LTE SNR threshold
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvLTESNRThreshold( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;
    LTESNRThreshold *pRSReq = pReq->pLTESNRThresh;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE              lCtr   = 0;

    if( NULL == pRSReq )
    {
        return eRCode;
    }

    /* Add TLV data */
    eRCode = PutByte( pBuf, pRSReq->LTESNRThreshListLen );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    for( lCtr = 0; lCtr < pRSReq->LTESNRThreshListLen; lCtr++ )
    {
        eRCode = PutWord( pBuf, pRSReq->pLTESNRThreshList[lCtr] );
        if( eQCWWAN_ERR_NONE != eRCode )
            return eRCode;
    }

    return eRCode;
}
/*
 * This function packs the SLQSNasConfigSigInfo2 LTE SNR Delta
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvLTESNRDelta( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;

    if ( pReq->pLTESNRDelta == NULL )
        return eQCWWAN_ERR_NONE;

    return PutWord( pBuf, *(pReq->pLTESNRDelta) );
}

/*
 * This function packs the SLQSNasConfigSigInfo2 LTE RSRQ threshold
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvLTERSRQThresh( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;
    LTERSRQThresh *pRSReq = pReq->pLTERSRQThresh;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE              lCtr   = 0;

    if( NULL == pRSReq )
    {
        return eRCode;
    }

    /* Add TLV data */
    eRCode = PutByte( pBuf, pRSReq->LTERSRQThreshListLen );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    for( lCtr = 0; lCtr < pRSReq->LTERSRQThreshListLen; lCtr++ )
    {
        eRCode = PutWord( pBuf, pRSReq->pLTERSRQThreshList[lCtr] );
        if( eQCWWAN_ERR_NONE != eRCode )
            return eRCode;
    }

    return eRCode;
}
/*
 * This function packs the SLQSNasConfigSigInfo2 LTE RSRQ Delta
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvLTERSRQDelta( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;

    if ( pReq->pLTERSRQDelta == NULL )
        return eQCWWAN_ERR_NONE;

    return PutWord( pBuf, *(pReq->pLTERSRQDelta) );
}

/*
 * This function packs the SLQSNasConfigSigInfo2 LTE RSRP threshold
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvLTERSRPThresh( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;
    LTERSRPThresh *pRSReq = pReq->pLTERSRPThresh;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE              lCtr   = 0;

    if( NULL == pRSReq )
    {
        return eRCode;
    }

    /* Add TLV data */
    eRCode = PutByte( pBuf, pRSReq->LTERSRPThreshListLen );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    for( lCtr = 0; lCtr < pRSReq->LTERSRPThreshListLen; lCtr++ )
    {
        eRCode = PutWord( pBuf, pRSReq->pLTERSRPThreshList[lCtr] );
        if( eQCWWAN_ERR_NONE != eRCode )
            return eRCode;
    }

    return eRCode;
}
/*
 * This function packs the SLQSNasConfigSigInfo2 LTE RSRP Delta
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvLTERSRPDelta( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;

    if ( pReq->pLTERSRPDelta == NULL )
        return eQCWWAN_ERR_NONE;

    return PutWord( pBuf, *(pReq->pLTERSRPDelta) );
}
/*
 * This function packs the SLQSNasConfigSigInfo2 LTE RSRP Delta
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvLTESigRptConfig( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;
    LTESigRptConfig *pRSReq = pReq->pLTESigRptConfig;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    if( NULL == pRSReq )
    {
        return eRCode;
    }

    /* Add TLV data */
    eRCode = PutByte( pBuf, pRSReq->rptRate );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    /* Add TLV data */
    eRCode = PutByte( pBuf, pRSReq->avgPeriod );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;
    return eRCode;
}

/*
 * This function packs the SLQSNasConfigSigInfo2 TDSCDMA RSCP threshold
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvTDSCDMARSCPThresh( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;
    TDSCDMARSCPThresh *pRSReq = pReq->pTDSCDMARSCPThresh;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE              lCtr   = 0;

    if( NULL == pRSReq )
    {
        return eRCode;
    }

    /* Add TLV data */
    eRCode = PutByte( pBuf, pRSReq->TDSCDMARSCPThreshListLen );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    for( lCtr = 0; lCtr < pRSReq->TDSCDMARSCPThreshListLen; lCtr++ )
    {
        eRCode = PutWord( pBuf, pRSReq->pTDSCDMARSCPThreshList[lCtr] );
        if( eQCWWAN_ERR_NONE != eRCode )
            return eRCode;
    }

    return eRCode;
}
/*
 * This function packs the SLQSNasConfigSigInfo2 TDSCDMA RSCP Delta
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvTDSCDMARSCPDelta( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;

    if ( pReq->pTDSCDMARSCPDelta == NULL )
        return eQCWWAN_ERR_NONE;

    return PutWord( pBuf, *(pReq->pTDSCDMARSCPDelta) );
}

/*
 * This function packs the SLQSNasConfigSigInfo2 TDSCDMA RSSI threshold
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvTDSCDMARSSIThresh( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;
    TDSCDMARSSIThresh *pRSReq = pReq->pTDSCDMARSSIThresh;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE              lCtr   = 0;

    if( NULL == pRSReq )
    {
        return eRCode;
    }

    /* Add TLV data */
    eRCode = PutByte( pBuf, pRSReq->TDSCDMARSSIThreshListLen );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    for( lCtr = 0; lCtr < pRSReq->TDSCDMARSSIThreshListLen; lCtr++ )
    {
        eRCode = PutLong( pBuf, pRSReq->pTDSCDMARSSIThreshList[lCtr] );
        if( eQCWWAN_ERR_NONE != eRCode )
            return eRCode;
    }

    return eRCode;
}
/*
 * This function packs the SLQSNasConfigSigInfo2 TDSCDMA RSSI Delta
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvTDSCDMARSSIDelta( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;

    if ( pReq->pTDSCDMARSSIDelta == NULL )
        return eQCWWAN_ERR_NONE;

    return PutWord( pBuf, *(pReq->pTDSCDMARSSIDelta) );
}

/*
 * This function packs the SLQSNasConfigSigInfo2 TDSCDMA ECIO threshold
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvTDSCDMAECIOThresh( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;
    TDSCDMAECIOThresh *pRSReq = pReq->pTDSCDMAECIOThresh;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE              lCtr   = 0;

    if( NULL == pRSReq )
    {
        return eRCode;
    }

    /* Add TLV data */
    eRCode = PutByte( pBuf, pRSReq->TDSCDMAECIOThreshListLen );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    for( lCtr = 0; lCtr < pRSReq->TDSCDMAECIOThreshListLen; lCtr++ )
    {
        eRCode = PutLong( pBuf, pRSReq->pTDSCDMAECIOThreshList[lCtr] );
        if( eQCWWAN_ERR_NONE != eRCode )
            return eRCode;
    }

    return eRCode;
}
/*
 * This function packs the SLQSNasConfigSigInfo2 TDSCDMA ECIO Delta
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvTDSCDMAECIODelta( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;

    if ( pReq->pTDSCDMAECIODelta == NULL )
        return eQCWWAN_ERR_NONE;

    return PutWord( pBuf, *(pReq->pTDSCDMAECIODelta) );
}

/*
 * This function packs the SLQSNasConfigSigInfo2 TDSCDMA SINR threshold
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvTDSCDMASINRThresh( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;
    TDSCDMASINRThresh *pRSReq = pReq->pTDSCDMASINRThresh;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE              lCtr   = 0;

    if( NULL == pRSReq )
    {
        return eRCode;
    }

    /* Add TLV data */
    eRCode = PutByte( pBuf, pRSReq->TDSCDMASINRThreshListLen );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    for( lCtr = 0; lCtr < pRSReq->TDSCDMASINRThreshListLen; lCtr++ )
    {
        eRCode = PutLong( pBuf, pRSReq->pTDSCDMASINRThreshList[lCtr] );
        if( eQCWWAN_ERR_NONE != eRCode )
            return eRCode;
    }

    return eRCode;
}
/*
 * This function packs the SLQSNasConfigSigInfo2 TDSCDMA SINR Delta
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvTDSCDMASINRDelta( BYTE *pBuf, BYTE *pParam )
{
    setSignalStrengthInfo    *pReq = (setSignalStrengthInfo *)pParam;

    if ( pReq->pTDSCDMASINRDelta == NULL )
        return eQCWWAN_ERR_NONE;

    return PutWord( pBuf, *(pReq->pTDSCDMASINRDelta) );
}

/*
 * This function packs the SLQSNasConfigSigInfo2 parameters to the QMI message
 * SDU
 *
 * \param  pParamField [OUT] - Pointer to storage into which the packed
 *                                    data will be placed by this function.
 *
 * \param  pMlength    [OUT] - Total length of built message.
 *
 * \param  pSigInfo    [IN] - Data to be packed
 *
 * \return eQCWWAN_ERR_NONE      - Increment successful
 * \return eQCWWAN_ERR_NO_MEMORY - Access beyond allowed size attempted
 *
 * \sa     qaGobiApiNas.h for remaining parameter descriptions.
 */
enum eQCWWANError PkQmiNasConfigSigInfo2(
    WORD    *pMlength,
    BYTE    *pParamField,
    setSignalStrengthInfo *pSetSignalStrengthInfo)
{
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_CDMA_RSSI_THRESHOLD,   &BuildTlvCDMARSSIThresh },
        { eTLV_CDMA_RSSI_DELTA,       &BuildTlvCDMARSSIDelta },
        { eTLV_CDMA_ECIO_THRESHOLD,   &BuildTlvCDMAECIOThresh },
        { eTLV_CDMA_ECIO_DELTA,       &BuildTlvCDMAECIODelta },
        { eTLV_HDR_RSSI_THRESHOLD,    &BuildTlvHDRRSSIThresh },
        { eTLV_HDR_RSSI_DELTA,        &BuildTlvHDRRSSIDelta },
        { eTLV_HDR_ECIO_THRESHOLD,    &BuildTlvHDRECIOThresh },
        { eTLV_HDR_ECIO_DELTA,        &BuildTlvHDRECIODelta },
        { eTLV_HDR_SINR_THRESH,       &BuildTlvHDRSINRThresh },
        { eTLV_HDR_SINR_DELTA,        &BuildTlvHDRSINRDelta },
        { eTLV_HDR_IO_THRESHOLD,      &BuildTlvHDRIOThresh },
        { eTLV_HDR_IO_DELTA,          &BuildTlvHDRIODelta },
        { eTLV_GSM_RSSI_THRESHOLD,    &BuildTlvGSMRSSIThresh },
        { eTLV_GSM_RSSI_DELTA,        &BuildTlvGSMRSSIDelta },
        { eTLV_WCDMA_RSSI_THRESHOLD,  &BuildTlvWCDMARSSIThresh },
        { eTLV_WCDMA_RSSI_DELTA,      &BuildTlvWCDMARSSIDelta },
        { eTLV_WCDMA_ECIO_THRESHOLD,  &BuildTlvWCDMAECIOThresh },
        { eTLV_WCDMA_ECIO_DELTA,      &BuildTlvWCDMAECIODelta },
        { eTLV_LTE_RSSI_THRESHOLD,    &BuildTlvLTERSSIThresh },
        { eTLV_LTE_RSSI_DELTA,        &BuildTlvLTERSSIDelta },
        { eTLV_LTE_SNR_THRESH,        &BuildTlvLTESNRThreshold },
        { eTLV_LTE_SNR_DELTA,         &BuildTlvLTESNRDelta },
        { eTLV_LTE_RSRQ_THRESHOLD,    &BuildTlvLTERSRQThresh },
        { eTLV_LTE_RSRQ_DELTA,        &BuildTlvLTERSRQDelta },
        { eTLV_LTE_RSRP_THRESHOLD,    &BuildTlvLTERSRPThresh },
        { eTLV_LTE_RSRP_DELTA,        &BuildTlvLTERSRPDelta },
        { eTLV_LTE_SIG_RPT_CONFIG,    &BuildTlvLTESigRptConfig },
        { eTLV_TDSCDMA_RSCP_THRESHOLD,&BuildTlvTDSCDMARSCPThresh },
        { eTLV_TDSCDMA_RSCP_DELTA,    &BuildTlvTDSCDMARSCPDelta },
        { eTLV_TDSCDMA_RSSI_THRESHOLD,&BuildTlvTDSCDMARSSIThresh },
        { eTLV_TDSCDMA_RSSI_DELTA,    &BuildTlvTDSCDMARSSIDelta },
        { eTLV_TDSCDMA_ECIO_THRESHOLD,&BuildTlvTDSCDMAECIOThresh },
        { eTLV_TDSCDMA_ECIO_DELTA,    &BuildTlvTDSCDMAECIODelta },
        { eTLV_TDSCDMA_SINR_THRESHOLD,&BuildTlvTDSCDMASINRThresh },
        { eTLV_TDSCDMA_SINR_DELTA,    &BuildTlvTDSCDMASINRDelta },
        { eTLV_TYPE_INVALID,          NULL } /* Important. Sentinel.
                                              * Signifies last item in map.
                                              */
    };

    enum eQCWWANError           eRCode;

    eRCode = qmbuild( pParamField,
                      (BYTE *)pSetSignalStrengthInfo,
                      map,
                      eQMI_NAS_CONFIG_SIG_INFO2,
                      pMlength );

    return eRCode;
}

/******************************************************************************
 * Response handling
 ******************************************************************************/

/* This function unpacks the SLQSNasConfigSigInfo2 response
 * message to a user-provided response structure.
 *
 * \param  pMdmResp   [IN]  - Pointer to packed response from the modem.
 *
 * \param  pApiResp  [OUT] - Pointer to storage to unpack into.
 *
 * \return eQCWWAN_ERR_NONE      - Increment successful
 * \return eQCWWAN_ERR_NO_MEMORY - Access beyond allowed size attempted
 *
 */
enum eQCWWANError UpkQmiNasConfigSigInfo2(
    BYTE                               *pMdmResp,
    struct QmiNasConfigSigInfo2Resp    *pApiResp)
{
    enum eQCWWANError eRCode;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,  &qmUnpackTlvResultCode },
        { eTLV_TYPE_INVALID, NULL } /* Important. Sentinel.
                                     * Signifies last item in map.
                                     */
    };

    eRCode = qmunpackresp( pMdmResp,
                           (BYTE *)pApiResp,
                           map,
                           eQMI_NAS_CONFIG_SIG_INFO2 );
    return eRCode;
}
