/*
 * \ingroup : loc
 *
 * \file    : qaSlqsLocInjectUTCTime.c
 *
 * \brief   : Contains Packing and UnPacking routines for the
 *            eQMI_LOC_INJECT_UTC_TIME message.
 *
 * Copyright: © 2011-2015 Sierra Wireless, Inc. all rights reserved
 */

#include "SwiDataTypes.h"
#include "sludefs.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaQmiBasic.h"

#include "qaSlqsLocInjectUTCTime.h"

/******************************************************************************
 * Request handling
 ******************************************************************************/
/*
 * This function packs the UTC time field to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                 data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvUTCTime( BYTE *pBuf, BYTE *pParam )
{
    struct QmiLocInjectUTCTimeReq *pReq =
        (struct QmiLocInjectUTCTimeReq *)pParam;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    eRCode = Put64( pBuf, pReq->timeMsec);
    return eRCode;
}

/*
 * This function packs the Uncertainty time field to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                 data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvUncertaintyTime( BYTE *pBuf, BYTE *pParam )
{
    struct QmiLocInjectUTCTimeReq *pReq =
        (struct QmiLocInjectUTCTimeReq *)pParam;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    eRCode = PutLong( pBuf, pReq->timeUncMsec);
    return eRCode;
}

/*
 * Packs the PkQmiLocInjectUTCTime parameters to the QMI message SDU
 *
 * \param  pParamField[OUT]         - Pointer to storage into which the packed
 *                                    data will be placed by this function.
 *
 * \param  pMlength[OUT]            - Total length of built message.
 *
 * \param  timeMsec[IN]             - UTC time in msec
 *
 * \param  timeUncMsec[IN]          - Time Uncertainty in msec
 *
 * \return  eQCWWAN_ERR_NONE on success, eQCWWAN_ERR_XXX on error
 *
 */
enum eQCWWANError PkQmiLocInjectUTCTime(
    WORD                *pMlength,
    BYTE                *pParamField,
    ULONGLONG           timeMsec,
    ULONG               timeUncMsec)
{
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_UTC_TIME,                    &BuildTlvUTCTime },
        { eTLV_UNCERTAINTY_TIME,            &BuildTlvUncertaintyTime },
        { eTLV_TYPE_INVALID,               NULL }  /* Important. Sentinel.
                                                 * Signifies last item in map.
                                                 */
    };
    struct QmiLocInjectUTCTimeReq req;

    enum eQCWWANError eRCode;

    slmemset((CHAR *)&req, 0, sizeof(struct QmiLocInjectUTCTimeReq));
    req.timeMsec    = timeMsec;
    req.timeUncMsec = timeUncMsec;

    eRCode = qmbuild( pParamField,
                      (BYTE *)&req,
                      map,
                      eQMI_LOC_INJECT_UTC_TIME,
                      pMlength );

    return eRCode;
}

/******************************************************************************
 * Response handling
 ******************************************************************************

 * This function unpacks the eQMI_LOC_INJECT_UTC_TIME response message
 * to a user-provided response structure.
 *
 * \param  pMdmResp  [IN]  - Pointer to packed response from the modem.
 *
 * \param  pApiResp  [OUT] - Pointer to storage to unpack into.
 *
 * \return eQCWWAN_ERR_NONE  - on success;  eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UpkQmiLocInjectUTCTime(
                  BYTE                              *pMdmResp,
                  struct QmiLocInjectUTCTimeResp    *pApiResp)
{
    enum eQCWWANError eRCode;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,  &qmUnpackTlvResultCode },
        { eTLV_TYPE_INVALID, NULL } /* Important. Sentinel.
                                     * Signifies last item in map.
                                     */
    };

    eRCode = qmunpackresp( pMdmResp,
                           (BYTE *)pApiResp,
                           map,
                           eQMI_LOC_INJECT_UTC_TIME);
    return eRCode;
}
