/*
 * \ingroup imsa
 *
 * \file qaImsaSLQSGetSupportedFields.c
 *
 * \brief  Contains Packing and UnPacking routines for the
 *         eQMI_IMSA_GET_SUPPORTED_FIELDS message.
 *
 * Copyright: © 2015 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */

#include "SwiDataTypes.h"
#include "sludefs.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaQmiBasic.h"
#include "qaGobiApiImsa.h"
#include "qaImsaSLQSGetSupportedFields.h"

/******************************************************************************
 * Request handling
 ******************************************************************************/

/*
 * This function packs the IMSA Service Message Id  field
 * to the QMI message SDU
 *
 * \param pBuf   - Pointer to storage into which the packed
 *                 data will be placed by this function.
 *
 * \param pParam - Pointer to structure containing data for this TLV.
 *
 */
enum eQCWWANError BuildTlvServiceMsgId( BYTE *pBuf, BYTE *pParam )
{
    WORD *lReq =  (WORD*)pParam;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Insert Session Id */
    eRCode = PutWord( pBuf, *lReq);
    return eRCode;
}
/*
 * This function packs the SLQSGetIMSASupportedFields parameters
 * to the QMI message SDU
 *
 * \param  pParamField [OUT] - Pointer to storage into which the packed
 *                             data will be placed by this function.
 *
 * \param  pMlength    [OUT] - Total length of built message.
 *
 * \param  messageID    [In] - Service Message ID.
 *
 * \return eQCWWAN_ERR_NONE, on success
 * \return eQCWWAN_ERR_NO_MEMORY, on error
 *
 */
enum eQCWWANError PkQmiImsaSLQSGetSupportFields(
    WORD *pMlength,
    BYTE *pParamField,
    WORD messageID)
{
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_IMSA_SERVICE_MSG_ID,  &BuildTlvServiceMsgId },
        { eTLV_TYPE_INVALID, NULL } /* Important. Sentinel.
                                     * Signifies last item in map.
                                     */
    };
    enum eQCWWANError eRCode;

    eRCode = qmbuild( pParamField,
                      (BYTE*)&messageID,
                      map,
                      eQMI_IMSA_GET_SUPPORTED_FIELDS,
                      pMlength );
    return eRCode;
}

/*
 * This function unpacks the SLQSGetIMSASupportedFields
 * Supported Request fields List from the QMI response
 * message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return      eQCWWAN_ERR_NONE, on success
 * \return      eQCWWAN_ERR_NO_MEMORY, on error
 */
enum eQCWWANError UnpackTlvSupportedReqFieldsList( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiImsaGetSupportFieldsResp *lResp =
                (struct QmiImsaGetSupportFieldsResp *)pResp;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE lTemp;
    if ( lResp->pIMSASupportedFieldsResp->pReqFieldsList == NULL )
         return eRCode;

    eRCode = GetByte( pTlvData, &lTemp);
    lResp->pIMSASupportedFieldsResp->pReqFieldsList->requestFieldsLen = lTemp;
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }
    WORD count = 0;

    while ( lTemp-- )
    {
         eRCode = GetByte( pTlvData,
         (BYTE *)&lResp->pIMSASupportedFieldsResp->pReqFieldsList->requestFields[count++]);

        if (eQCWWAN_ERR_NONE != eRCode)
        {
            return eRCode;
        }
    }
    return eRCode;
}
/*
 * This function unpacks the SLQSGetIMSASupportedFields
 * Supported Response fields List from the QMI response
 * message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return      eQCWWAN_ERR_NONE, on success
 * \return      eQCWWAN_ERR_NO_MEMORY, on error
 */
enum eQCWWANError UnpackTlvSupportedRespFieldsList( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiImsaGetSupportFieldsResp *lResp =
                (struct QmiImsaGetSupportFieldsResp *)pResp;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE lTemp;
    if ( lResp->pIMSASupportedFieldsResp->pRespFieldsList == NULL )
         return eRCode;

    eRCode = GetByte( pTlvData, &lTemp);
    lResp->pIMSASupportedFieldsResp->pRespFieldsList->responseFieldsLen = lTemp;
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }
    WORD count = 0;

    while ( lTemp-- )
    {
         eRCode = GetByte( pTlvData,
         (BYTE *)&lResp->pIMSASupportedFieldsResp->pRespFieldsList->responseFields[count++]);

        if (eQCWWAN_ERR_NONE != eRCode)
        {
            return eRCode;
        }
    }
    return eRCode;
}
/*
 * This function unpacks the SLQSGetIMSASupportedFields
 * Supported Indication fields List from the QMI response
 * message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return      eQCWWAN_ERR_NONE, on success
 * \return      eQCWWAN_ERR_NO_MEMORY, on error
 */
enum eQCWWANError UnpackTlvSupportedIndFieldsList( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiImsaGetSupportFieldsResp *lResp =
                (struct QmiImsaGetSupportFieldsResp *)pResp;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE lTemp;
    if ( lResp->pIMSASupportedFieldsResp->pRespFieldsList == NULL )
         return eRCode;

    eRCode = GetByte( pTlvData, &lTemp);
    lResp->pIMSASupportedFieldsResp->pIndFieldsList->indicationFieldsLen = lTemp;
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }
    WORD count = 0;

    while ( lTemp-- )
    {
         eRCode = GetByte( pTlvData,
         (BYTE *)&lResp->pIMSASupportedFieldsResp->pIndFieldsList->indicationFields[count++]);

        if (eQCWWAN_ERR_NONE != eRCode)
        {
            return eRCode;
        }
    }
    return eRCode;
}
/*
 * This function unpacks the SLQSGetIMSASupportedFields response message to a
 * user-provided response structure.
 *
 * \param     MdmResp   [IN]  - Pointer to packed response from the modem.
 *
 * \param     pApiResp  [OUT] - Pointer to storage to unpack into.
 *
 * \return    eQCWWAN_ERR_NONE, on success
 * \return    eQCWWAN_ERR_NO_MEMORY, on error
 */
enum eQCWWANError UpkQmiImsaSLQSGetSupportFields(
    BYTE                               *pMdmResp,
    struct QmiImsaGetSupportFieldsResp *pApiResp)
{
    enum eQCWWANError rCode;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,                   &qmUnpackTlvResultCode },
        { eTLV_IMSA_SUPPORT_REQ_FIELDS_LIST,  &UnpackTlvSupportedReqFieldsList },
        { eTLV_IMSA_SUPPORT_RESP_FIELDS_LIST, &UnpackTlvSupportedRespFieldsList },
        { eTLV_IMSA_SUPPORT_IND_FIELDS_LIST,  &UnpackTlvSupportedIndFieldsList },
        { eTLV_TYPE_INVALID,                  NULL } /* Important. Sentinel.
                                                      * Signifies last item in map.
                                                      */
    };
    rCode = qmunpackresp( pMdmResp,
                          (BYTE *)pApiResp,
                          map,
                          eQMI_IMSA_GET_SUPPORTED_FIELDS );
    return rCode;
}

