/*
 * \ingroup imsa
 *
 * \file qaImsaSLQSGetRegistrationStatus.c
 *
 * \brief  Contains Packing and UnPacking routines for the
 *         eQMI_IMSA_GET_REGISTRATION_STATUS message.
 *
 * Copyright: © 2015 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */

#include "SwiDataTypes.h"
#include "sludefs.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaQmiBasic.h"
#include "qaGobiApiImsa.h"
#include "qaImsaSLQSGetServiceStatus.h"

/******************************************************************************
 * Request handling
 ******************************************************************************/

/*
 * This function packs the SLQSGetIMSAServiceStatus parameters
 * to the QMI message SDU
 *
 * \param  pParamField [OUT] - Pointer to storage into which the packed
 *                             data will be placed by this function.
 *
 * \param  pMlength    [OUT] - Total length of built message.
 *
 * \return eQCWWAN_ERR_NONE, on success
 * \return eQCWWAN_ERR_NO_MEMORY, on error
 *
 */
enum eQCWWANError PkQmiImsaSLQSGetServiceStatus(
    WORD *pMlength,
    BYTE *pParamField)
{
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_TYPE_INVALID, NULL } /* Important. Sentinel.
                                     * Signifies last item in map.
                                     */
    };
    enum eQCWWANError eRCode;

    eRCode = qmbuild( pParamField,
                      NULL,
                      map,
                      eQMI_IMSA_GET_SERVICE_STATUS,
                      pMlength );
    return eRCode;
}
/*
 * This function unpacks the SLQSGetIMSAServiceStatus
 * SMS Service Status from the QMI response
 * message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return      eQCWWAN_ERR_NONE, on success
 * \return      eQCWWAN_ERR_NO_MEMORY, on error
 */
enum eQCWWANError UnpackTlvSmsServiceStatus( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiImsaGetServiceStatusResp *lResp =
                (struct QmiImsaGetServiceStatusResp *)pResp;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    if ( lResp->pIMSAServiceStatus->pSmsServiceStatus == NULL )
         return eRCode;

    eRCode = GetLong( pTlvData, lResp->pIMSAServiceStatus->pSmsServiceStatus);

    return eRCode;
}
/*
 * This function unpacks the SLQSGetIMSAServiceStatus
 * VOIP Service Status from the QMI response
 * message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return      eQCWWAN_ERR_NONE, on success
 * \return      eQCWWAN_ERR_NO_MEMORY, on error
 */
enum eQCWWANError UnpackTlvVoipServiceStatus( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiImsaGetServiceStatusResp *lResp =
                (struct QmiImsaGetServiceStatusResp *)pResp;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    if ( lResp->pIMSAServiceStatus->pVoipServiceStatus == NULL )
         return eRCode;

    eRCode = GetLong( pTlvData, lResp->pIMSAServiceStatus->pVoipServiceStatus);

    return eRCode;
}
/*
 * This function unpacks the SLQSGetIMSAServiceStatus
 * VT Service Status from the QMI response
 * message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return      eQCWWAN_ERR_NONE, on success
 * \return      eQCWWAN_ERR_NO_MEMORY, on error
 */
enum eQCWWANError UnpackTlvVtServiceStatus( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiImsaGetServiceStatusResp *lResp =
                (struct QmiImsaGetServiceStatusResp *)pResp;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    if ( lResp->pIMSAServiceStatus->pVtServiceStatus == NULL )
         return eRCode;

    eRCode = GetLong( pTlvData, lResp->pIMSAServiceStatus->pVtServiceStatus);

    return eRCode;
}

/*
 * This function unpacks the SLQSGetIMSAServiceStatus
 * SMS Service RAT from the QMI response
 * message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return      eQCWWAN_ERR_NONE, on success
 * \return      eQCWWAN_ERR_NO_MEMORY, on error
 */
enum eQCWWANError UnpackTlvSmsRat( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiImsaGetServiceStatusResp *lResp =
                (struct QmiImsaGetServiceStatusResp *)pResp;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    if ( lResp->pIMSAServiceStatus->pSmsServiceRat == NULL )
         return eRCode;

    eRCode = GetLong( pTlvData, lResp->pIMSAServiceStatus->pSmsServiceRat);

    return eRCode;
}
/*
 * This function unpacks the SLQSGetIMSAServiceStatus
 * VOIP Service RAT from the QMI response
 * message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return      eQCWWAN_ERR_NONE, on success
 * \return      eQCWWAN_ERR_NO_MEMORY, on error
 */
enum eQCWWANError UnpackTlvVoipRat( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiImsaGetServiceStatusResp *lResp =
                (struct QmiImsaGetServiceStatusResp *)pResp;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    if ( lResp->pIMSAServiceStatus->pVoipServiceRat == NULL )
         return eRCode;

    eRCode = GetLong( pTlvData, lResp->pIMSAServiceStatus->pVoipServiceRat);

    return eRCode;
}
/*
 * This function unpacks the SLQSGetIMSAServiceStatus
 * VT Service RAT from the QMI response
 * message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return      eQCWWAN_ERR_NONE, on success
 * \return      eQCWWAN_ERR_NO_MEMORY, on error
 */
enum eQCWWANError UnpackTlvVtRat( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiImsaGetServiceStatusResp *lResp =
                (struct QmiImsaGetServiceStatusResp *)pResp;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    if ( lResp->pIMSAServiceStatus->pVtServiceRat == NULL )
         return eRCode;

    eRCode = GetLong( pTlvData, lResp->pIMSAServiceStatus->pVtServiceRat);

    return eRCode;
}
/*
 * This function unpacks the SLQSGetIMSAServiceStatus
 * UT Service Status from the QMI response
 * message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return      eQCWWAN_ERR_NONE, on success
 * \return      eQCWWAN_ERR_NO_MEMORY, on error
 */
enum eQCWWANError UnpackTlvUtServiceStatus( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiImsaGetServiceStatusResp *lResp =
                (struct QmiImsaGetServiceStatusResp *)pResp;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    if ( lResp->pIMSAServiceStatus->pUtServiceStatus == NULL )
         return eRCode;

    eRCode = GetLong( pTlvData, lResp->pIMSAServiceStatus->pUtServiceStatus);

    return eRCode;
}
/*
 * This function unpacks the SLQSGetIMSAServiceStatus
 * UT Service RAT from the QMI response
 * message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return      eQCWWAN_ERR_NONE, on success
 * \return      eQCWWAN_ERR_NO_MEMORY, on error
 */
enum eQCWWANError UnpackTlvUtRat( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiImsaGetServiceStatusResp *lResp =
                (struct QmiImsaGetServiceStatusResp *)pResp;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    if ( lResp->pIMSAServiceStatus->pUtServiceRat == NULL )
         return eRCode;

    eRCode = GetLong( pTlvData, lResp->pIMSAServiceStatus->pUtServiceRat);

    return eRCode;
}
/*
 * This function unpacks the SLQSGetIMSAServiceStatus
 * VS Service Status from the QMI response
 * message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return      eQCWWAN_ERR_NONE, on success
 * \return      eQCWWAN_ERR_NO_MEMORY, on error
 */
enum eQCWWANError UnpackTlvVsServiceStatus( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiImsaGetServiceStatusResp *lResp =
                (struct QmiImsaGetServiceStatusResp *)pResp;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    if ( lResp->pIMSAServiceStatus->pVsServiceStatus == NULL )
         return eRCode;

    eRCode = GetLong( pTlvData, lResp->pIMSAServiceStatus->pVsServiceStatus);

    return eRCode;
}
/*
 * This function unpacks the SLQSGetIMSAServiceStatus
 * VS Service RAT from the QMI response
 * message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return      eQCWWAN_ERR_NONE, on success
 * \return      eQCWWAN_ERR_NO_MEMORY, on error
 */
enum eQCWWANError UnpackTlvVsRat( BYTE *pTlvData, BYTE *pResp )
{
    struct QmiImsaGetServiceStatusResp *lResp =
                (struct QmiImsaGetServiceStatusResp *)pResp;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    if ( lResp->pIMSAServiceStatus->pVsServiceRat == NULL )
         return eRCode;

    eRCode = GetLong( pTlvData, lResp->pIMSAServiceStatus->pVsServiceRat);

    return eRCode;
}
/*
 * This function unpacks the SLQSGetIMSAServiceStatus response message to a
 * user-provided response structure.
 *
 * \param     MdmResp   [IN]  - Pointer to packed response from the modem.
 *
 * \param     pApiResp  [OUT] - Pointer to storage to unpack into.
 *
 * \return    eQCWWAN_ERR_NONE, on success
 * \return    eQCWWAN_ERR_NO_MEMORY, on error
 */
enum eQCWWANError UpkQmiImsaSLQSGetServiceStatus(
    BYTE                               *pMdmResp,
    struct QmiImsaGetServiceStatusResp  *pApiResp)
{
    enum eQCWWANError rCode;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,                &qmUnpackTlvResultCode },
        { eTLV_IMSA_SMS_SERVICE_STATUS,    &UnpackTlvSmsServiceStatus },
        { eTLV_IMSA_VOIP_SERVICE_STATUS,   &UnpackTlvVoipServiceStatus },
        { eTLV_IMSA_VT_SERVICE_STATUS,     &UnpackTlvVtServiceStatus },
        { eTLV_IMSA_SMS_SERVICE_RAT,       &UnpackTlvSmsRat },
        { eTLV_IMSA_VOIP_SERVICE_RAT,      &UnpackTlvVoipRat },
        { eTLV_IMSA_VT_SERVICE_RAT,        &UnpackTlvVtRat },
        { eTLV_IMSA_UT_SERVICE_STATUS,     &UnpackTlvUtServiceStatus },
        { eTLV_IMSA_UT_SERVICE_RAT,        &UnpackTlvUtRat },
        { eTLV_IMSA_VS_SERVICE_STATUS,     &UnpackTlvVsServiceStatus },
        { eTLV_IMSA_VS_SERVICE_RAT,        &UnpackTlvVsRat },
        { eTLV_TYPE_INVALID,            NULL } /* Important. Sentinel.
                                                * Signifies last item in map.
                                                */
    };
    rCode = qmunpackresp( pMdmResp,
                          (BYTE *)pApiResp,
                          map,
                          eQMI_IMSA_GET_SERVICE_STATUS );
    return rCode;
}
