/*
 * \ingroup dms
 *
 * \file qaGobiApiDms.c
 *
 * \brief  Entry points for Gobi APIs for the Device Management Service ( DMS )
 *
 * Copyright: © 2011-2014 Sierra Wireless, Inc. all rights reserved
 *
 */

#include "SwiDataTypes.h"
#include "qaGobiApiDms.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "amudefs.h"
#include "sludefs.h"
#include "qaQmiBasic.h"
#include "qaDmsGetDeviceMfr.h"
#include "qaDmsGetDeviceModelID.h"
#include "qaDmsGetDeviceRevisionID.h"
#include "qaDmsGetPRLVersion.h"
#include "qaDmsGetMSISDN.h"
#include "qaDmsUIMGetIMSI.h"
#include "qaDmsGetDeviceSerialNumbers.h"
#include "qaDmsGetDeviceHardwareRev.h"
#include "qaDmsGetTime.h"
#include "qaDmsUIMSetPINProtection.h"
#include "qaDmsUIMVerifyPIN.h"
#include "qaDmsUIMUnblockPIN.h"
#include "qaDmsUIMChangePIN.h"
#include "qaDmsSetOperatingMode.h"
#include "qaDmsGetOperatingMode.h"
#include "qaDmsUIMGetCkStatus.h"
#include "qaDmsUIMGetICCID.h"
#include "qaDmsUIMGetPINStatus.h"
#include "qaDmsUIMSetCkProtection.h"
#include "qaDmsUIMUnblockCk.h"
#include "qaDmsGetDeviceCap.h"
#include "qaDmsRestoreFactoryDefaults.h"
#include "qaDmsValSerProgramCode.h"
#include "qaDmsActivateAutomatic.h"
#include "qaDmsGetActivationState.h"
#include "qaDmsSlqsUIMGetState.h"
#include "qaDmsGetBandCapability.h"
#include "qaDmsSLQSGetCustFeatures.h"
#include "qaDmsSLQSSetCustFeatures.h"
#include "qaDmsSLQSGetCurrentPRLInfo.h"
#include "qaDmsSLQSGetERIFile.h"
#include "qaDmsSLQSSwiGetFSN.h"
#include "qaDmsSLQSSwiGetFirmwareCurr.h"
#include "qaDmsSLQSSwiGetFwUpdatStat.h"
#include "qaDmsSLQSSwiGetUSBComp.h"
#include "qaDmsSLQSSwiSetUSBComp.h"
#include "qaDmsSLQSSwiGetCrashInfo.h"
#include "qaDmsSLQSSwiGetCrashAction.h"
#include "qaDmsSLQSSwiSetCrashAction.h"
#include "qaDmsSLQSSwiGetHostDevInfo.h"
#include "qaDmsSLQSSwiSetHostDevInfo.h"
#include "qaDmsSLQSSwiGetOSInfo.h"
#include "qaDmsSLQSSwiSetOSInfo.h"
#include "qaDmsSLQSSwiGetSerialNoExt.h"
#include "qaDmsSLQSGetCustFeaturesV2.h"
#include "qaDmsSLQSSetCustFeaturesV2.h"
#include "qaDmsSLQSSwiGetDyingGaspCfg.h"
#include "qaDmsSLQSSwiSetDyingGaspCfg.h"
#include "qaDmsSLQSSwiGetDyingGaspStatistics.h"
#include "qaDmsSLQSSwiClearDyingGaspStatistics.h"
#include "qaDmsIndicationRegister.h"
#include "qaDmsSwiGetResetInfo.h"

/* To be used by UIM PIN related APIs */
#define QMI_DMS_PIN_ONE 1
#define QMI_DMS_PIN_TWO 2

ULONG GetManufacturer(
    BYTE    stringSize,
    CHAR    *pString )
{
    ULONG              resultCode;     /* Result of SwiQmiSendnWait() */
    BYTE               *pInParam;      /* ptr to param field rx'd from modem */
    BYTE               *pOutParam;     /* ptr to outbound param field */
    BYTE               *pReqBuf;       /* Pointer to outgoing request buffer */
    USHORT             ParamLength;    /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsGetDeviceMfrResp response;

    /* Device string is OUT parameter and hence should be NULL */
    if ( !pString  )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }
    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs QMI message */
    resultCode = PkQmiDmsGetDeviceMfr( &ParamLength,
                                   pOutParam );
    if (resultCode != eQCWWAN_ERR_NONE)
    {
        qmrelreqbkp();
        return resultCode;
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  ParamLength,
                                  eQMI_TIMEOUT_2_S,
                                  &pInParam,
                                  &ParamLength );
    /* Only parse out the response data if we got a positive return */
    if (resultCode == eQCWWAN_ERR_NONE)
    {
        /* Initialisation of Buffer */
        slmemset((char *)pString, 0, stringSize);
        /* Pass the pointer for the OUT parameters */
        response.pString     = pString;
        response.stringSize  = stringSize;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsGetDeviceMfr(pInParam, &response);
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG GetModelID(
    BYTE    stringSize,
    CHAR    *pString )
{
    ULONG               resultCode;     /* Result of SwiQmiSendnWait() */
    BYTE                *pInParam;      /* ptr to param field rx'd from modem */
    BYTE                *pOutParam;     /* ptr to outbound param field */
    BYTE                *pReqBuf;       /* Pointer to outgoing request buffer */
    USHORT              ParamLength;    /* Ret'd length of the  QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsGetDeviceModelIDResp response;

    /* Device string is OUT parameter and hence should be NULL */
    if ( !pString  )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }
    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs QMI message */
    resultCode = PkQmiDmsGetDeviceModelID( &ParamLength,
                                       pOutParam );
    if (resultCode != eQCWWAN_ERR_NONE)
    {
        qmrelreqbkp();
        return resultCode;
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  ParamLength,
                                  eQMI_TIMEOUT_2_S,
                                  &pInParam,
                                  &ParamLength );
    /* Only parse out the response data if we got a positive return */
    if (resultCode == eQCWWAN_ERR_NONE)
    {
        /* Initialisation of Buffer */
        slmemset((char *)pString, 0, stringSize);
        /* Pass the pointer for the OUT parameters */
        response.pString        = pString;
        response.stringSize     = stringSize;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsGetDeviceModelID(pInParam, &response);
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG GetFirmwareRevision(
    BYTE     stringSize,
    CHAR    *pString )
{
    CHAR lBootString, lPRIString;
    return GetFirmwareRevisions ( stringSize,
                                  pString,
                                  0,
                                  &lBootString,
                                  0,
                                  &lPRIString);
}

ULONG GetFirmwareRevisions(
    BYTE    amssSize,
    CHAR    *pAMSSString,
    BYTE    bootSize,
    CHAR    *pBootString,
    BYTE    priSize,
    CHAR    *pPRIString )
{
    ULONG             resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE              *pInParam;    /* ptr to param field rx'd from modem */
    BYTE              *pOutParam;   /* ptr to outbound param field */
    BYTE              *pReqBuf;     /* Pointer to outgoing request buffer */
    USHORT            ParamLength; /* Ret'd length of the  QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsGetDeviceRevIDResp response;

    /* Device string is OUT parameter and hence should be NULL */
    if ( !pAMSSString || !pBootString || !pPRIString )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }
    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs QMI message */
    resultCode = PkQmiDmsGetDeviceRevID( &ParamLength,
                                     pOutParam  );
    if (resultCode != eQCWWAN_ERR_NONE)
    {
        qmrelreqbkp();
        return resultCode;
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  ParamLength,
                                  eQMI_TIMEOUT_2_S,
                                  &pInParam,
                                  &ParamLength );
    /* Only parse out the response data if we got a positive return */
    if (resultCode == eQCWWAN_ERR_NONE)
    {
        /* Initialisation of Buffer */
        slmemset((char *)pAMSSString, 0, amssSize);
        slmemset((char *)pBootString, 0, bootSize);
        slmemset((char *)pPRIString, 0, priSize);
        /* Pass the pointer for the OUT parameters */
        memset(&response, 0, sizeof(response));
        response.pAMSSString = pAMSSString;
        response.pBootString = pBootString;
        response.pPRIString  = pPRIString;
        response.amssSize    = amssSize;
        response.bootSize    = bootSize;
        response.priSize     = priSize;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsGetDeviceRevID(pInParam, &response);
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG GetPRLVersion(
    WORD    *pPRLVersion )
{
    ULONG               resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE                *pInParam;   /* ptr to param field rx'd from modem */
    BYTE                *pOutParam;  /* ptr to outbound param field */
    BYTE                *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT              ParamLength; /* Ret'd length of the QMI Parm field */

    /* Storage for results and response variable */
    struct QmiDmsGetPRLVersionResp response;

    /* Device string is OUT parameter and hence should be NULL */
    if ( !pPRLVersion )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Index points to first QMI header. */
    ParamLength = 0;

    /* Invoke the function which packs QMI message */
    resultCode = PkQmiDmsGetPRLVersion( &ParamLength, pOutParam );
    if (resultCode != eQCWWAN_ERR_NONE)
    {
        qmrelreqbkp();
        return resultCode;
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  ParamLength,
                                  eQMI_TIMEOUT_2_S, /* 2 seconds */
                                  &pInParam,
                                  &ParamLength );


    /* Only parse out the response data if we got a positive return */
    if (resultCode == eQCWWAN_ERR_NONE)
    {
        /* Copy the obained values to the function OUT parameters */
        response.pPRLVersion = pPRLVersion;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsGetPRLVersion( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG GetIMSI(
    BYTE   stringSize,
    CHAR   *pString )
{
    ULONG               resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE                *pInParam;   /* ptr to param field rx'd from modem */
    BYTE                *pOutParam;  /* ptr to outbound param field */
    BYTE                *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT              ParamLength; /* Ret'd length of the QMI Parm field */

    /* Storage for results and response variable */
    struct QmiDmsUIMGetIMSIResp response;

    /* Device string is OUT parameter and hence should be NULL */
    if ( !pString )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Index points to first QMI header. */
    ParamLength = 0;

    /* Invoke the function which packs QMI message */
    resultCode = PkQmiDmsUIMGetIMSI( &ParamLength, pOutParam );
    if (resultCode != eQCWWAN_ERR_NONE)
    {
        qmrelreqbkp();
        return resultCode;
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  ParamLength,
                                  eQMI_TIMEOUT_2_S, /* 2 seconds */
                                  &pInParam,
                                  &ParamLength );


    /* Only parse out the response data if we got a positive return */
    if (resultCode == eQCWWAN_ERR_NONE)
    {
        /* Initialising the Buffer */
        slmemset((char *)pString, 0, stringSize);
        /* Copy the obained values to the function OUT parameters */
        response.IMSISize            = stringSize;
        response.pIMSI               = pString;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsUIMGetIMSI( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG GetSerialNumbers(
   BYTE     esnSize,
   CHAR    *pESNString,
   BYTE     imeiSize,
   CHAR    *pIMEIString,
   BYTE     meidSize,
   CHAR    *pMEIDString )
{
    ULONG               resultCode;  /* Result of SwiQmiSendnWait() */
    serialNumbersInfo   serialNumbersInfo;
    CHAR imeiSvnString  = 0xFF;

    serialNumbersInfo.esnSize     = esnSize;
    serialNumbersInfo.pESNString  = pESNString;
    serialNumbersInfo.imeiSize    = imeiSize;
    serialNumbersInfo.pIMEIString = pIMEIString;
    serialNumbersInfo.meidSize    = meidSize;
    serialNumbersInfo.pMEIDString = pMEIDString;
    serialNumbersInfo.imeiSvnSize = 0;
    serialNumbersInfo.pImeiSvnString = &imeiSvnString;

    resultCode = SLQSGetSerialNumbers(&serialNumbersInfo);

    return resultCode;
}

ULONG SLQSGetSerialNumbers(
        serialNumbersInfo *pSerialNumbersInfo)
{
    ULONG               resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE                *pInParam;   /* ptr to param field rx'd from modem */
    BYTE                *pOutParam;  /* ptr to outbound param field */
    BYTE                *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT              ParamLength; /* Ret'd length of the QMI Parm field */

    /* Storage for results and response variable */
    struct QmiDmsGetDeviceSerialNumbersResp response;

    /* Device string is OUT parameter and hence should be NULL */
    if ( !pSerialNumbersInfo->pESNString || !pSerialNumbersInfo->pIMEIString ||
            !pSerialNumbersInfo->pMEIDString || !pSerialNumbersInfo->pImeiSvnString)
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Index points to first QMI header. */
    ParamLength = 0;

    /* Invoke the function which packs QMI message */
    resultCode = PkQmiDmsGetDeviceSerialNumbers( &ParamLength, pOutParam );
    if (resultCode != eQCWWAN_ERR_NONE)
    {
        qmrelreqbkp();
        return resultCode;
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  ParamLength,
                                  eQMI_TIMEOUT_2_S, /* 2 seconds */
                                  &pInParam,
                                  &ParamLength );


    /* Only parse out the response data if we got a positive return */
    if (resultCode == eQCWWAN_ERR_NONE)
    {
        /* Initialising the Buffer */
        slmemset((char *)pSerialNumbersInfo->pESNString,     0, pSerialNumbersInfo->esnSize);
        slmemset((char *)pSerialNumbersInfo->pIMEIString,    0, pSerialNumbersInfo->imeiSize);
        slmemset((char *)pSerialNumbersInfo->pMEIDString,    0, pSerialNumbersInfo->meidSize);
        slmemset((char *)pSerialNumbersInfo->pImeiSvnString, 0, pSerialNumbersInfo->imeiSvnSize);

        /* Copy the obtained values to the function OUT parameters */
        response.esnSize        = pSerialNumbersInfo->esnSize;
        response.imeiSize       = pSerialNumbersInfo->imeiSize;
        response.meidSize       = pSerialNumbersInfo->meidSize;
        response.imeiSvnSize    = pSerialNumbersInfo->imeiSvnSize;
        response.pESNString     = pSerialNumbersInfo->pESNString;
        response.pIMEIString    = pSerialNumbersInfo->pIMEIString;
        response.pMEIDString    = pSerialNumbersInfo->pMEIDString;
        response.pImeiSvnString = pSerialNumbersInfo->pImeiSvnString;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsGetDeviceSerialNumbers( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG GetHardwareRevision(
    BYTE   stringSize,
    CHAR * pString )
{
    ULONG               resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE                *pInParam;   /* ptr to param field rx'd from modem */
    BYTE                *pOutParam;  /* ptr to outbound param field */
    BYTE                *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT              ParamLength; /* Ret'd length of the QMI Parm field */

    /* Storage for results and response variable */
    struct QmiDmsGetDeviceHardwareRevResp response;

    /* Device string is OUT parameter and hence should be NULL */
    if ( !pString )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Index points to first QMI header. */
    ParamLength = 0;

    /* Invoke the function which packs QMI message */
    resultCode = PkQmiDmsGetDeviceHardwareRev( &ParamLength, pOutParam );
    if (resultCode != eQCWWAN_ERR_NONE)
    {
        qmrelreqbkp();
        return resultCode;
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  ParamLength,
                                  eQMI_TIMEOUT_2_S, /* 2 seconds */
                                  &pInParam,
                                  &ParamLength );


    /* Only parse out the response data if we got a positive return */
    if (resultCode == eQCWWAN_ERR_NONE)
    {
        /* Initialising the Buffer */
        slmemset((char *)pString, 0, stringSize);
        /* Copy the obained values to the function OUT parameters */
        response.pString    = pString;
        response.stringSize = stringSize;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsGetDeviceHardwareRev( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG GetNetworkTime(
    ULONGLONG *pTimeStamp,
    ULONG     *pTimeSource )
{
    ULONG  resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */
    WORD   first2bytes;  /* To extract Timesource from Device Time */

    /* Storage for results and response variable */
    struct QmiDmsGetTimeResp response;

    /* pTimeStamp & pTimeSource are OUT parameters & hence should not be NULL */
    if ( !pTimeStamp || !pTimeSource )
        return eQCWWAN_ERR_INVALID_ARG;

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsGetTime( &paramLength, pOutParam );

    if ( resultCode != eQCWWAN_ERR_NONE )
    {
        qmrelreqbkp();
        return resultCode; /* No memory */
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_2_S, /* 2 seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( resultCode == eQCWWAN_ERR_NONE )
    {
        /* Copy the obtained values to the function OUT parameters */
        response.pDeviceTime = pTimeStamp;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsGetTime( pInParam, &response );

        if ( resultCode == eQCWWAN_ERR_NONE )
        {
            /* Length of pTimeStamp is 6 bytes
             * And length of pTimeSource is 2 bytes
             * These two need to be extracted from Device Time
             */
            first2bytes  = *(response.pDeviceTime) >> QMI_DMS_SIX_BYTES_SHIFT;
            *pTimeSource = (ULONG)first2bytes;
            *pTimeStamp  = *(response.pDeviceTime) << QMI_DMS_TWO_BYTES_SHIFT;
            *pTimeStamp  = *(pTimeStamp) >> QMI_DMS_TWO_BYTES_SHIFT;
        }
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG UIMSetPINProtection(
    ULONG id,
    ULONG bEnable,
    CHAR  *pValue,
    ULONG *pVerifyRetriesLeft,
    ULONG *pUnblockRetriesLeft )
{
    ULONG  resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsUIMSetPINProtectionResp response;

    /* pVerifyRetriesLeft & pUnblockRetriesLeft are OUT parameters
     * and hence should not be NULL
     */
    if ( !pVerifyRetriesLeft || !pUnblockRetriesLeft )
        return eQCWWAN_ERR_INVALID_ARG;

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsUIMSetPINProtection( &paramLength,
                                              pOutParam,
                                              id,
                                              bEnable,
                                              pValue );

    if ( resultCode != eQCWWAN_ERR_NONE )
    {
        qmrelreqbkp();
        return resultCode; /* No memory */
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_5_S, /* 5 seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( resultCode == eQCWWAN_ERR_NONE )
    {
        /* Initialization to default value */
        *pVerifyRetriesLeft  = 0xFFFFFFFF;
        *pUnblockRetriesLeft = 0xFFFFFFFF;

        /* Copy the obtained values to the function OUT parameters */
        response.pVerifyRetriesLeft  = pVerifyRetriesLeft;
        response.pUnblockRetriesLeft = pUnblockRetriesLeft;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsUIMSetPINProtection( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG UIMUnblockPIN(
    ULONG id,
    CHAR  *pPUKValue,
    CHAR  *pNewValue,
    ULONG *pVerifyRetriesLeft,
    ULONG *pUnblockRetriesLeft )
{
    ULONG  resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsUIMUnblockPINResp response;

    /* pVerifyRetriesLeft & pUnblockRetriesLeft are OUT parameters
     * and hence should not be NULL
     */
    if ( !pVerifyRetriesLeft || !pUnblockRetriesLeft )
        return eQCWWAN_ERR_INVALID_ARG;

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsUIMUnblockPIN( &paramLength,
                                        pOutParam,
                                        id,
                                        pPUKValue,
                                        pNewValue );

    if ( resultCode != eQCWWAN_ERR_NONE )
    {
        qmrelreqbkp();
        return resultCode; /* No memory */
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_5_S, /* 5 seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( resultCode == eQCWWAN_ERR_NONE )
    {
        /* Initialization to default value */
        *pVerifyRetriesLeft  = 0xFFFFFFFF;
        *pUnblockRetriesLeft = 0xFFFFFFFF;

        /* Copy the obtained values to the function OUT parameters */
        response.pVerifyRetriesLeft  = pVerifyRetriesLeft;
        response.pUnblockRetriesLeft = pUnblockRetriesLeft;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsUIMUnblockPIN( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG UIMVerifyPIN(
    ULONG id,
    CHAR  *pValue,
    ULONG *pVerifyRetriesLeft,
    ULONG *pUnblockRetriesLeft )
{
    ULONG  resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsUIMVerifyPINResp response;

    /* pVerifyRetriesLeft & pUnblockRetriesLeft are OUT parameters
     * and hence should not be NULL
     */
    if ( !pVerifyRetriesLeft || !pUnblockRetriesLeft )
        return eQCWWAN_ERR_INVALID_ARG;

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsUIMVerifyPIN( &paramLength,
                                       pOutParam,
                                       id,
                                       pValue );

    if ( resultCode != eQCWWAN_ERR_NONE )
    {
        qmrelreqbkp();
        return resultCode; /* No memory */
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_5_S, /* 5 seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( resultCode == eQCWWAN_ERR_NONE )
    {
        /* Initialization to default value */
        *pVerifyRetriesLeft  = 0xFFFFFFFF;
        *pUnblockRetriesLeft = 0xFFFFFFFF;

        /* Copy the obtained values to the function OUT parameters */
        response.pVerifyRetriesLeft  = pVerifyRetriesLeft;
        response.pUnblockRetriesLeft = pUnblockRetriesLeft;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsUIMVerifyPIN( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG UIMChangePIN(
    ULONG id,
    CHAR  *pOldValue,
    CHAR  *pNewValue,
    ULONG *pVerifyRetriesLeft,
    ULONG *pUnblockRetriesLeft )
{
    ULONG  resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsUIMChangePINResp response;

    /* pVerifyRetriesLeft & pUnblockRetriesLeft are OUT parameters
     * and hence should not be NULL
     */
    if ( !pVerifyRetriesLeft || !pUnblockRetriesLeft )
        return eQCWWAN_ERR_INVALID_ARG;

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsUIMChangePIN( &paramLength,
                                       pOutParam,
                                       id,
                                       pOldValue,
                                       pNewValue );

    if ( resultCode != eQCWWAN_ERR_NONE )
    {
        qmrelreqbkp();
        return resultCode; /* No memory */
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_5_S, /* 5 seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( resultCode == eQCWWAN_ERR_NONE )
    {
        /* Initialization to default value */
        *pVerifyRetriesLeft  = 0xFFFFFFFF;
        *pUnblockRetriesLeft = 0xFFFFFFFF;

        /* Copy the obtained values to the function OUT parameters */
        response.pVerifyRetriesLeft  = pVerifyRetriesLeft;
        response.pUnblockRetriesLeft = pUnblockRetriesLeft;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsUIMChangePIN( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG GetVoiceNumber(
    BYTE voiceNumberSize,
    CHAR *pVoiceNumber,
    BYTE minSize,
    CHAR *pMIN )
{
    ULONG  resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsGetMSISDNResp response;

    /* pVoiceNumber & pVoiceNumber are OUT parameters
     * and hence should not be NULL
     */
    if ( !pVoiceNumber || !pMIN )
        return eQCWWAN_ERR_INVALID_ARG;

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsGetMSISDN( &paramLength,
                                              pOutParam );

    if ( resultCode != eQCWWAN_ERR_NONE )
    {
        qmrelreqbkp();
        return resultCode; /* No memory */
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_2_S, /* 2 seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( resultCode == eQCWWAN_ERR_NONE )
    {
        /* Initialising the Buffer */
        slmemset((char *)pVoiceNumber, 0, voiceNumberSize);
        slmemset((char *)pMIN,         0, minSize);

        /* Copy the obtained values to the function OUT parameters */
        response.voiceNumberSize  = voiceNumberSize;
        response.pVoiceNumber     = pVoiceNumber;
        response.minSize          = minSize;
        response.pMIN             = pMIN;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsGetMSISDN( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG SetPower(
    ULONG powerMode )
{
    ULONG  resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsSetOperatingModeResp response;

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsSetOperatingMode( &paramLength,
                                           pOutParam,
                                           powerMode );

    if ( resultCode != eQCWWAN_ERR_NONE )
    {
        qmrelreqbkp();
        return resultCode; /* No memory */
    }

    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_2_S, /* 2 Seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( resultCode == eQCWWAN_ERR_NONE )
    {
        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsSetOperatingMode( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG GetPower(
    ULONG *pPowerMode )
{
    ULONG  resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsGetOperatingModeResp response;

    /* pPowerMode is an OUT parameter and hence should not be NULL */
    if ( !pPowerMode )
        return eQCWWAN_ERR_INVALID_ARG;

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs OMI message and sends the message. */
    resultCode = PkQmiDmsGetOperatingMode( &paramLength, pOutParam );

    if ( resultCode != eQCWWAN_ERR_NONE )
    {
        qmrelreqbkp();
        return resultCode; /* No memory */
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_2_S, /* 2 Seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( resultCode == eQCWWAN_ERR_NONE )
    {
        /* Copy the obtained values to the function OUT parameters */
        response.pOperatingMode = pPowerMode;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsGetOperatingMode( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG UIMGetControlKeyStatus(
    ULONG id,
    ULONG *pStatus,
    ULONG *pVerifyRetriesLeft,
    ULONG *pUnblockRetriesLeft )
{
    ULONG  resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsUIMGetCkStatusResp response;

    /* pStatus, pVerifyRetriesLeft & pUnblockRetriesLeft are OUT parameters
     * and hence should not be NULL
     */
    if ( !pStatus || !pVerifyRetriesLeft || !pUnblockRetriesLeft )
        return eQCWWAN_ERR_INVALID_ARG;

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsUIMGetCkStatus( &paramLength,
                                         pOutParam,
                                         id );

    if ( resultCode != eQCWWAN_ERR_NONE )
    {
        qmrelreqbkp();
        return resultCode; /* No memory */
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_5_S, /* 5 seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( resultCode == eQCWWAN_ERR_NONE )
    {
        /* Initialization to default value */
        *pVerifyRetriesLeft  = 0xFFFFFFFF;
        *pUnblockRetriesLeft = 0xFFFFFFFF;

        /* Copy the obtained values to the function OUT parameters */
        response.pStatus             = pStatus;
        response.pVerifyRetriesLeft  = pVerifyRetriesLeft;
        response.pUnblockRetriesLeft = pUnblockRetriesLeft;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsUIMGetCkStatus( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG UIMGetICCID(
    BYTE stringSize,
    CHAR *pString )
{
    ULONG  resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsUIMGetICCIDResp response;

    /* pSting is an OUT parameters and hence should not be NULL */
    if ( !pString )
        return eQCWWAN_ERR_INVALID_ARG;

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsUIMGetICCID( &paramLength, pOutParam );

    if ( resultCode != eQCWWAN_ERR_NONE )
    {
        qmrelreqbkp();
        return resultCode; /* No memory */
    }
    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_2_S, /* 2 seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( resultCode == eQCWWAN_ERR_NONE )
    {
        /* Initialization of Buffer */
        slmemset((CHAR *)pString, 0, stringSize);

        /* Pass the pointers for the OUT parameters */
        response.pString    = pString;
        response.stringSize = stringSize;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsUIMGetICCID( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG UIMGetPINStatus(
    ULONG id,
    ULONG *pStatus,
    ULONG *pVerifyRetriesLeft,
    ULONG *pUnblockRetriesLeft )
{
    ULONG  resultCode;  /* Result of SwiIntSendnWait() */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsUIMGetPINStatusResp response;

    /* Input Parameter Validation */
    if ( id != QMI_DMS_PIN_ONE && id != QMI_DMS_PIN_TWO )
        return eQCWWAN_ERR_INVALID_ARG;
    else
    {
        /* This shall help to identify which PIN status needs to be fetched */
        response.id = id;
    }

    /* pStatus, pVerifyRetriesLeft & pUnblockRetriesLeft are OUT parameters
     * and hence should not be NULL
     */
    if ( !pStatus || !pVerifyRetriesLeft || !pUnblockRetriesLeft )
        return eQCWWAN_ERR_INVALID_ARG;

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsUIMGetPINStatus( &paramLength,
                                          pOutParam );

    if ( resultCode != eQCWWAN_ERR_NONE )
    {
        qmrelreqbkp();
        return resultCode; /* No memory */
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_5_S, /* 5 seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( resultCode == eQCWWAN_ERR_NONE )
    {
        /* Initialization to default value */
        *pVerifyRetriesLeft  = 0xFFFFFFFF;
        *pUnblockRetriesLeft = 0xFFFFFFFF;

        /* Copy the obtained values to the function OUT parameters */
        response.pStatus             = pStatus;
        response.pVerifyRetriesLeft  = pVerifyRetriesLeft;
        response.pUnblockRetriesLeft = pUnblockRetriesLeft;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsUIMGetPINStatus( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG GetOfflineReason(
    ULONG *pReasonMask,
    ULONG *pbPlatform )
{
    ULONG  resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */


    /* Storage for results and response variable */
    struct QmiDmsGetOperatingModeResp response;

    /*
     * OfflineReason and HardwareRestriction are OUT parameters and hence
     * should not be NULL
     */
    if ( !pReasonMask || !pbPlatform )
        return eQCWWAN_ERR_INVALID_ARG;

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs OMI message */
    resultCode = PkQmiDmsGetOperatingMode( &paramLength, pOutParam );

    if ( resultCode != eQCWWAN_ERR_NONE )
    {
        qmrelreqbkp();
        return resultCode; /* No memory */
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_2_S, /* 2 Seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( resultCode == eQCWWAN_ERR_NONE )
    {
        /* Copy the obtained values to the function OUT parameters */
        response.pOperatingMode          = NULL;
        response.pOfflineReason          = pReasonMask;
        response.pHardwareControlledMode = pbPlatform;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsGetOperatingMode( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG UIMSetControlKeyProtection(
    ULONG id,
    ULONG status,
    CHAR  *pValue,
    ULONG *pVerifyRetriesLeft )
{
    ULONG  resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsUIMSetCkProtectionResp response;

    /* pVerifyRetriesLeft is an OUT parameter and hence should not be NULL */
    if ( !pVerifyRetriesLeft )
        return eQCWWAN_ERR_INVALID_ARG;

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsUIMSetCkProtection( &paramLength,
                                             pOutParam,
                                             id,
                                             status,
                                             pValue );

    if ( resultCode != eQCWWAN_ERR_NONE )
    {
        qmrelreqbkp();
        return resultCode; /* No memory */
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_5_S, /* 5 seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( resultCode == eQCWWAN_ERR_NONE )
    {
        /* Initialization to default value */
        *pVerifyRetriesLeft  = 0xFFFFFFFF;

        /* Copy the obtained values to the function OUT parameters */
        response.pVerifyRetriesLeft = pVerifyRetriesLeft;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsUIMSetCkProtection( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG UIMUnblockControlKey(
    ULONG id,
    CHAR  *pValue,
    ULONG *pUnblockRetriesLeft )
{
    ULONG  resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsUIMUnblockCkResp response;

    /* pUnblockRetriesLeft is an OUT parameter and hence should not be NULL */
    if ( !pUnblockRetriesLeft )
        return eQCWWAN_ERR_INVALID_ARG;

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsUIMUnblockCk( &paramLength,
                                       pOutParam,
                                       id,
                                       pValue );

    if ( resultCode != eQCWWAN_ERR_NONE )
    {
        qmrelreqbkp();
        return resultCode; /* No memory */
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_5_S, /* 5 seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( resultCode == eQCWWAN_ERR_NONE )
    {
        /* Initialization to default value */
        *pUnblockRetriesLeft  = 0xFFFFFFFF;

        /* Copy the obtained values to the function OUT parameters */
        response.pUnblockRetriesLeft = pUnblockRetriesLeft;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsUIMUnblockCk( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG GetDeviceCapabilities(
    ULONG *pMaxTXChannelRate,
    ULONG *pMaxRXChannelRate,
    ULONG *pDataServiceCapability,
    ULONG *pSimCapability,
    ULONG *pRadioIfacesSize,
    BYTE  *pRadioIfaces )
{
    ULONG  resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT ParamLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsGetDeviceCapResp response;

    /* OUT parameter should not be NULL  */
    if ( !pMaxTXChannelRate      ||
         !pMaxRXChannelRate      ||
         !pDataServiceCapability ||
         !pSimCapability         ||
         !pRadioIfacesSize       ||
         !pRadioIfaces )
    {
         return eQCWWAN_ERR_INVALID_ARG;
    }
    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs QMI message */
    resultCode = PkQmiDmsGetDeviceCap( &ParamLength, pOutParam );

    if (resultCode != eQCWWAN_ERR_NONE)
    {
        qmrelreqbkp();
        return resultCode;
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  ParamLength,
                                  eQMI_TIMEOUT_2_S, /* 2 Seconds */
                                  &pInParam,
                                  &ParamLength );

    /* Only parse out the response data if we got a positive return */
    if ( resultCode == eQCWWAN_ERR_NONE )
    {
        /* Initialize the Array */
        slmemset( (char *) pRadioIfaces,
                  0,
                  *pRadioIfacesSize );

        /* Set the pointers for the function OUT parameters */
        response.pMaxTXChannelRate      = pMaxTXChannelRate;
        response.pMaxRXChannelRate      = pMaxRXChannelRate;
        response.pDataServiceCapability = pDataServiceCapability;
        response.pSimCapability         = pSimCapability;
        response.pRadioIfacesSize       = pRadioIfacesSize;
        response.pRadioIfaces           = pRadioIfaces;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsGetDeviceCap( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG ResetToFactoryDefaults(
    CHAR *pSPC )
{
    ULONG  resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsRestoreFactoryDefaultsResp response;

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs QMI message */
    resultCode = PkQmiDmsRestoreFactoryDefaults( &paramLength,
                                                 pOutParam,
                                                 pSPC );

    if ( resultCode != eQCWWAN_ERR_NONE )
    {
        qmrelreqbkp();
        return resultCode; /* No memory */
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_300_S, /* 5 minutes */
                                  &pInParam,
                                  &paramLength );


    /* Only parse out the response data if we got a positive return */
    if ( resultCode == eQCWWAN_ERR_NONE )
    {
        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsRestoreFactoryDefaults( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG ValidateSPC(
    CHAR *pSPC )
{
    ULONG  resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsValSerProgramCodeResp response;

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs QMI message */
    resultCode = PkQmiDmsValSerProgramCode( &paramLength,
                                            pOutParam,
                                            pSPC );
    /* No memory */
    if ( resultCode != eQCWWAN_ERR_NONE )
    {
        qmrelreqbkp();
        return resultCode;
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_2_S, /* 2 seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    /* Copy to the caller's buffer */
    if ( resultCode == eQCWWAN_ERR_NONE )
        resultCode = UpkQmiDmsValSerProgramCode( pInParam, &response );

    qmrelreqbkp();
    return resultCode;
}

ULONG SLQSGetERIFile(
    ERIFileparams *pERIFileparams )
{
    ULONG  resultCode;  /* Result code to be returned by this function */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsSLQSGetERIFileResp response;

    /* Checking for Invalid Parameter */
    if ( NULL == pERIFileparams )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsSLQSGetERIFile( &paramLength, pOutParam );

    if ( eQCWWAN_ERR_NONE != resultCode )
    {
        qmrelreqbkp();
        return resultCode; /* No Memory */
    }

    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_5_S, /* 5 Seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( eQCWWAN_ERR_NONE == resultCode )
    {
        /* Copy the obtained values to the function OUT parameters */
        response.pERIFileparams = pERIFileparams;
        if( pERIFileparams->pFile )
        {
            slmemset( (char *)pERIFileparams->pFile,
                      0,
                      *pERIFileparams->pFileSize );
        }
        resultCode = UpkQmiDmsSLQSGetERIFile( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG ActivateAutomatic(
    CHAR *pActivationCode )
{
    ULONG  resultCode;  /* Result of SwiIntSendnWait() */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsActivateAutomaticResp response;

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message. */
    resultCode = PkQmiDmsActivateAutomatic( &paramLength,
                                            pOutParam,
                                            pActivationCode );

    if ( eQCWWAN_ERR_NONE != resultCode )
    {
        qmrelreqbkp();
        return resultCode; /* No memory */
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_300_S, /* 5 Minutes */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( eQCWWAN_ERR_NONE == resultCode )
    {
        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsActivateAutomatic( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG GetActivationState(
    ULONG *pActivationState)
{
    ULONG  resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsGetActivationStateResp response;

    /* pActivationState is an OUT parameter and hence should not be NULL.*/
    if ( !pActivationState )
        return eQCWWAN_ERR_INVALID_ARG;

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs QMI message */
    resultCode = PkQmiDmsGetActivationState( &paramLength, pOutParam );

    if ( resultCode != eQCWWAN_ERR_NONE )
    {
        qmrelreqbkp();
        return resultCode; /* No memory */
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_2_S, /* 2 seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( resultCode == eQCWWAN_ERR_NONE )
    {
        /* Copy the obtained values to the function OUT parameters */
        response.pActivationState = pActivationState;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsGetActivationState( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG SLQSUIMGetState(
    ULONG *pUIMState)
{
    ULONG  resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsUIMGetStateResp response;

    /* pUIMState is an OUT parameter and hence should not be NULL.*/
    if ( !pUIMState )
        return eQCWWAN_ERR_INVALID_ARG;

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs QMI message */
    resultCode = PkQmiDmsUIMGetState( &paramLength, pOutParam );

    if ( resultCode != eQCWWAN_ERR_NONE )
    {
        qmrelreqbkp();
        return resultCode; /* No memory */
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_2_S, /* 2 seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( resultCode == eQCWWAN_ERR_NONE )
    {
        /* Copy the obtained values to the function OUT parameters */
        response.pUIMState = pUIMState;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsUIMGetState( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG SLQSGetBandCapability( ULONGLONG *pBandCapability )
{
    ULONG  resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsGetBandCapabilityResp response;

    /* pBandCapability is an OUT parameter and hence should not be NULL.*/
    if ( !pBandCapability )
        return eQCWWAN_ERR_INVALID_ARG;

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs QMI message */
    resultCode = PkQmiDmsGetBandCapability( &paramLength, pOutParam );

    if ( resultCode != eQCWWAN_ERR_NONE )
    {
        qmrelreqbkp();
        return resultCode; /* No memory */
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_2_S, /* 2 seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( resultCode == eQCWWAN_ERR_NONE )
    {
        /* Copy the obtained values to the function OUT parameters */
        response.pBandCapability = pBandCapability;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsGetBandCapability( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG SLQSGetCustFeatures(
    custFeaturesInfo *pCustFeaturesInfo )
{
    ULONG  resultCode;  /* Result code to be returned by this function */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsGetCustFeaturesResp response;

    /* Checking for Invalid Parameter */
    if ( NULL == pCustFeaturesInfo )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsGetCustFeatures( &paramLength,
                                          pOutParam );

    if ( eQCWWAN_ERR_NONE != resultCode )
    {
        qmrelreqbkp();
        return resultCode; /* No Memory */
    }

    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_2_S, /* 2 seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( eQCWWAN_ERR_NONE == resultCode )
    {
        /* Set the pointers for the function OUT parameters */
        response.pCustFeaturesInfo = pCustFeaturesInfo;

        pCustFeaturesInfo->GpsEnable             = -1;
        if( pCustFeaturesInfo->pDisableIMSI )
            *(pCustFeaturesInfo->pDisableIMSI)   = 0xFF;
        if( pCustFeaturesInfo->pIPFamSupport )
            *(pCustFeaturesInfo->pIPFamSupport)  = 0xFFFF;
        if( pCustFeaturesInfo->pRMAutoConnect )
            *(pCustFeaturesInfo->pRMAutoConnect) = 0xFF;
        if( pCustFeaturesInfo->pGPSSel )
            *(pCustFeaturesInfo->pGPSSel)        = 0xFF;
        if( pCustFeaturesInfo->pSMSSupport )
            *(pCustFeaturesInfo->pSMSSupport)    = 0xFF;
        if( pCustFeaturesInfo->pIsVoiceEnabled )
            *(pCustFeaturesInfo->pIsVoiceEnabled) = 0xFF;
        if( pCustFeaturesInfo->pDHCPRelayEnabled )
            *(pCustFeaturesInfo->pDHCPRelayEnabled) = 0xFF;
        if( pCustFeaturesInfo->pGPSLPM )
            *(pCustFeaturesInfo->pGPSLPM) = 0xFF;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsGetCustFeatures( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG SLQSSetCustFeatures(
    custFeaturesSetting *pCustFeaturesSetting )
{
    ULONG  resultCode;  /* Result code to be returned by this function */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsSetCustFeaturesResp response;

    /* Checking for Invalid Parameter */
    if ( NULL == pCustFeaturesSetting )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsSetCustFeatures( &paramLength,
                                          pOutParam,
                                          pCustFeaturesSetting );

    if ( eQCWWAN_ERR_NONE != resultCode )
    {
        qmrelreqbkp();
        return resultCode; /* No Memory */
    }

    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_2_S, /* 2 seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( eQCWWAN_ERR_NONE == resultCode )
    {
        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsSetCustFeatures( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG SLQSGetCurrentPRLInfo( dmsCurrentPRLInfo *pCurrentPRLInfo )
{
    ULONG  resultCode;  /* Result code to be returned by this function */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDMSSLQSGetCurrentPRLInfoResp response;

    /* Checking for Invalid Parameter */
    if ( !pCurrentPRLInfo )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsSLQSGetCurrentPRLInfo( &paramLength, pOutParam );

    if ( eQCWWAN_ERR_NONE != resultCode )
    {
        qmrelreqbkp();
        return resultCode; /* No Memory */
    }

    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_2_S, /* 2 Seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( eQCWWAN_ERR_NONE == resultCode )
    {
        /* Copy the obtained values to the function OUT parameters */
        response.pCurrentPRLInfo = pCurrentPRLInfo;
        if( pCurrentPRLInfo->pPRLVersion )
        {
            *(pCurrentPRLInfo->pPRLVersion) = 0xFFFF;
        }
        if( pCurrentPRLInfo->pPRLPreference )
        {
            *(pCurrentPRLInfo->pPRLPreference) = 0xFF;
        }
        resultCode = UpkQmiDmsSLQSGetCurrentPRLInfo( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG SLQSSwiGetFSN( FactorySequenceNumber *pFSNumber)
{
    ULONG  resultCode;  /* Result code to be returned by this function */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsSLQSSwiGetFSNResp response;

    /* Checking for Invalid Parameter */
    if ( NULL == pFSNumber ||
         NULL == pFSNumber->FSNumber )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsSLQSSwiGetFSN( &paramLength, pOutParam );

    if ( eQCWWAN_ERR_NONE != resultCode )
    {
        qmrelreqbkp();
        return resultCode; /* No Memory */
    }

    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_5_S, /* 5 Seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( eQCWWAN_ERR_NONE == resultCode )
    {
        /* Copy the obtained values to the function OUT parameters */
        response.pFSNumber = pFSNumber;
        resultCode = UpkQmiDmsSLQSSwiGetFSN( pInParam, &response );
        if ( eQCWWAN_ERR_NONE == resultCode )
        {
            //skip last two digital as it is for internal use only
            size_t len = strlen((const char*)pFSNumber->FSNumber);
            if(len>2)
            {
                pFSNumber->FSNumber[len-1] = 0;
                pFSNumber->FSNumber[len-2] = 0;
            }
        }
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG SLQSSwiGetFirmwareCurr( CurrentImgList *pCurrentImgList )
{
    ULONG  resultCode;  /* Result code to be returned by this function */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */
    CurrImageInfo *pCurrImg = NULL; /* Pointer to CurrImageInfo */

    WORD          count     = 0; /* Counter for generated tokens */

    /* Storage for results and response variable */
    struct QmiDmsGetFirmwareCurrResp response;

    /* Checking for Invalid Parameter */
    if ( NULL == pCurrentImgList )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsSLQSSwiGetCurrFw( &paramLength, pOutParam );

    if ( eQCWWAN_ERR_NONE != resultCode )
    {
        qmrelreqbkp();
        return resultCode; /* No Memory */
    }

    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_5_S, /* 5 Seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( eQCWWAN_ERR_NONE == resultCode )
    {
        BYTE ctr = 0;
        BYTE ImgListLen = 0;

        /* Copy the obtained values to the function OUT parameters */
        response.pCurrentImgList = pCurrentImgList;

        /* Initialize out paramaters */
        ImgListLen = pCurrentImgList->numEntries;
        for( ctr = 0 ; ctr < ImgListLen; ctr++ )
        {
            pCurrImg = &pCurrentImgList->pCurrImgInfo[ctr];

            /* Initialize the structure */
            slmemset( (char*)pCurrImg, 0, sizeof(CurrImageInfo) );

            /* Initialize ImageType to FF as 0 is a valid value */
            pCurrImg->imageType = 0xFF;
        }
        resultCode = UpkQmiDmsSLQSSwiGetCurrFw( pInParam, &response );
    }

    /* Parse the build ID and unique ID for details */
    if( eQCWWAN_ERR_NONE == resultCode )
    {
        /* Store unique ID,Build ID in separate variables
         * as strtok replaces token with a 0 */
        BYTE uniqueIDTmp[UNIQUE_ID_LEN];
        BYTE buildIDTmp[MAX_BUILD_ID_LEN];
        CHAR          *strPtr   = NULL; /* Pointer for generating tokens */

        slmemset( (char *)uniqueIDTmp, 0, sizeof( uniqueIDTmp) );
        slmemset( (char *)buildIDTmp, 0, sizeof( buildIDTmp) );

        pCurrImg = &pCurrentImgList->pCurrImgInfo[0];
        slmemset( (char *) pCurrentImgList->priver,
                   0,
                   sizeof(pCurrentImgList->priver) );
        slmemset( (char *) pCurrentImgList->pkgver,
                  0,
                  sizeof(pCurrentImgList->pkgver) );
        slmemset( (char *) pCurrentImgList->fwvers,
                  0,
                  sizeof(pCurrentImgList->fwvers) );
        slmemset( (char *) pCurrentImgList->carrier,
                  0 ,
                  sizeof(pCurrentImgList->carrier) );

        /* Parse PRI version and pkg version from unique ID */
        slstrncpy( (char *) uniqueIDTmp,
                   (char*)pCurrImg->uniqueID,
                   slstrlen((const char *)pCurrImg->uniqueID) );
        strPtr = strtok ((char*)uniqueIDTmp, "_");

        /* Device can return a string of zero's for unique ID.
         * In this case no details are available on the device. */
        if( NULL != strPtr )
        {
            count  = slstrlen(strPtr);
            if(IMGDETAILS_LEN <= count+1)
                return eQCWWAN_ERR_MEMORY;
            slstrncpy( pCurrentImgList->priver,
                       strPtr,
                       count + 1 );

            strPtr = strtok(NULL, "_");

            /* Account for the rare case where there is no pkgver retrieved
             * from the device */
            if( NULL != strPtr )
            {
                count  = slstrlen(strPtr);
                if(IMGDETAILS_LEN <= count+1)
                    return eQCWWAN_ERR_MEMORY;
                slstrncpy( pCurrentImgList->pkgver,
                           strPtr,
                           count + 1 );
            }
        }
        /* Parse firmware version and carrier */
        slstrncpy( (char *) buildIDTmp,
                   (char*)pCurrImg->buildID,
                   slstrlen((const char *)pCurrImg->buildID) );
        strPtr = strtok ((char*)buildIDTmp, "_");

        /* Device can return a string without an _ for Build ID.
         * Usually this is given as "unknown"
         * In this case no details are available on the device. */
        if( NULL != strPtr )
        {
            count  = slstrlen(strPtr);
            if(IMGDETAILS_LEN <= count+1)
                return eQCWWAN_ERR_MEMORY;
            slstrncpy( pCurrentImgList->fwvers,
                       strPtr,
                       count + 1 );

            strPtr = strtok(NULL, "_");
            /* Account for the rare case where there is no carrier retrieved
             * from the device */
            if( NULL != strPtr )
            {
                count  = slstrlen(strPtr);
                if(IMGDETAILS_LEN <= count+1)
                    return eQCWWAN_ERR_MEMORY;
                slstrncpy( pCurrentImgList->carrier,
                           strPtr,
                           count + 1 );
            }
        }
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG SLQSSwiGetFwUpdateStatus( FirmwareUpdatStat *pFirmwareUpdatStat )
{
    ULONG  resultCode;  /* Result code to be returned by this function */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsSLQSSwiGetFwUpdStatResp response;

    /* Checking for Invalid Parameter */
    if ( NULL == pFirmwareUpdatStat )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsSLQSSwiGetFwUpdStat( &paramLength, pOutParam );

    if ( eQCWWAN_ERR_NONE != resultCode )
    {
        qmrelreqbkp();
        return resultCode; /* No Memory */
    }

    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_5_S, /* 5 Seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( eQCWWAN_ERR_NONE == resultCode )
    {
        /* Copy the obtained values to the function OUT parameters */
        response.pFirmwareUpdatStat = pFirmwareUpdatStat;
        resultCode = UpkQmiDmsSLQSSwiGetFwUpdStat( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG SLQSSwiGetUSBComp( USBCompParams *pUSBCompParams )
{
    ULONG  resultCode;  /* Result code to be returned by this function */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsSLQSSwiGetUSBCompResp response;

    /* Checking for Invalid Parameter */
    if ( NULL == pUSBCompParams )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsSLQSSwiGetUSBComp( &paramLength, pOutParam );

    if ( eQCWWAN_ERR_NONE != resultCode )
    {
        qmrelreqbkp();
        return resultCode; /* No Memory */
    }

    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_5_S, /* 5 Seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( eQCWWAN_ERR_NONE == resultCode )
    {
        /* Initialize out parameters */
        if( NULL != pUSBCompParams->pUSBComp )
        {
            *(pUSBCompParams->pUSBComp) = 0xFF;
        }

        if( NULL != pUSBCompParams->pNumSupUSBComps )
        {
            *(pUSBCompParams->pNumSupUSBComps) = 0x00;
        }

        /* Copy the obtained values to the function OUT parameters */
        response.pUSBCompParams = pUSBCompParams;
        resultCode = UpkQmiDmsSLQSSwiGetUSBComp( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG SLQSSwiSetUSBComp( USBCompConfig *pUSBCompConfig )
{
    ULONG  resultCode;  /* Result code to be returned by this function */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsSLQSSwiSetUSBCompResp response;

    /* Checking for Invalid Parameter */
    if ( NULL == pUSBCompConfig )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsSLQSSwiSetUSBComp( &paramLength,
                                            pOutParam,
                                            pUSBCompConfig );

    if ( eQCWWAN_ERR_NONE != resultCode )
    {
        qmrelreqbkp();
        return resultCode; /* No Memory */
    }

    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_5_S, /* 5 Seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( eQCWWAN_ERR_NONE == resultCode )
    {
        resultCode = UpkQmiDmsSLQSSwiSetUSBComp( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG SLQSSwiGetCrashInfo( BYTE *pClear,CrashInfoParams *pCrashInfoParams )
{
    ULONG  resultCode;  /* Result code to be returned by this function */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsSLQSSwiGetCrashInfoResp response;

    /* Checking for Invalid Parameter */
    if ( NULL == pCrashInfoParams )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsSLQSSwiGetCrashInfo( &paramLength,
                                              pOutParam,
                                              pClear );

    if ( eQCWWAN_ERR_NONE != resultCode )
    {
        qmrelreqbkp();
        return resultCode; /* No Memory */
    }

    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_5_S, /* 5 Seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( eQCWWAN_ERR_NONE == resultCode )
    {
        /* Initialize out parameters */
        if( NULL != pCrashInfoParams->pDevCrashStatus )
        {
            *(pCrashInfoParams->pDevCrashStatus) = 0xFF;
        }

        if(NULL != pCrashInfoParams->pCrashInfo)
        {
            CrashInfo *plCrashInfo = pCrashInfoParams->pCrashInfo;

                plCrashInfo->crashData    = 0xFFFF;
                plCrashInfo->crashId      = 0xFFFF;
                plCrashInfo->numCrashes   = 0xFFFF;
        }
        /* Copy the obtained values to the function OUT parameters */
        response.pCrashInfoParams = pCrashInfoParams;
        resultCode = UpkQmiDmsSLQSSwiGetCrashInfo( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG SLQSSwiGetCrashAction( BYTE *pDevCrashState )
{
    ULONG  resultCode;  /* Result code to be returned by this function */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsSLQSSwiGetCrashActionResp response;

    /* Checking for Invalid Parameter */
    if ( NULL == pDevCrashState )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsSLQSSwiGetCrashAction( &paramLength, pOutParam );

    if ( eQCWWAN_ERR_NONE != resultCode )
    {
        qmrelreqbkp();
        return resultCode; /* No Memory */
    }

    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_5_S, /* 5 Seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( eQCWWAN_ERR_NONE == resultCode )
    {
        /* Initialize out parameters */
        if( NULL != pDevCrashState )
        {
            *(pDevCrashState) = 0xFF;
        }

        /* Copy the obtained values to the function OUT parameters */
        response.pDevCrashState = pDevCrashState;
        resultCode = UpkQmiDmsSLQSSwiGetCrashAction( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG SLQSSwiSetCrashAction( BYTE crashActionParams)
{
    ULONG  resultCode;  /* Result code to be returned by this function */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsSLQSSwiSetCrashActionResp response;

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsSLQSSwiSetCrashAction( &paramLength,
                                                pOutParam,
                                                crashActionParams);

    if ( eQCWWAN_ERR_NONE != resultCode )
    {
        qmrelreqbkp();
        return resultCode; /* No Memory */
    }

    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_5_S, /* 5 Seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( eQCWWAN_ERR_NONE == resultCode )
    {
        resultCode = UpkQmiDmsSLQSSwiSetCrashAction( pInParam,
                                                     &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG SLQSSwiGetHostDevInfo(SLQSSwiGetHostDevInfoParams *pParams)
{
    ULONG               resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE                *pInParam;   /* ptr to param field rx'd from modem */
    BYTE                *pOutParam;  /* ptr to outbound param field */
    BYTE                *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT              ParamLength; /* Ret'd length of the QMI Parm field */

    /* Storage for results and response variable */
    struct QmiDmsSLQSSwiGetHostDevInfoResp response;

    /* Device string is OUT parameter and hence should be NULL */
    if ( !pParams->pManString || !pParams->pModelString 
      || !pParams->pSWVerString || !pParams->pPlasmaIDString )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Index points to first QMI header. */
    ParamLength = 0;

    /* Invoke the function which packs QMI message */
    resultCode = PkQmiDmsSLQSSwiGetHostDevInfo( &ParamLength, pOutParam );
    if (resultCode != eQCWWAN_ERR_NONE)
    {
        qmrelreqbkp();
        return resultCode;
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  ParamLength,
                                  eQMI_TIMEOUT_2_S, /* 2 seconds */
                                  &pInParam,
                                  &ParamLength );


    /* Only parse out the response data if we got a positive return */
    if (resultCode == eQCWWAN_ERR_NONE)
    {

        /* Initialising the Buffer */
        slmemset((char *)pParams->pManString, 0, pParams->bManSize);
        slmemset((char *)pParams->pModelString, 0, pParams->bModelSize);
        slmemset((char *)pParams->pSWVerString, 0, pParams->bSWVerSize);
        slmemset((char *)pParams->pPlasmaIDString, 0, pParams->bPlasmaIDSize);

        /* Copy the obained values to the function OUT parameters */
        response.bManSize      = pParams->bManSize;
        response.bModelSize    = pParams->bModelSize;
        response.bSWVerSize    = pParams->bSWVerSize;
        response.bPlasmaIDSize = pParams->bPlasmaIDSize;

        response.pManString   = pParams->pManString;
        response.pModelString = pParams->pModelString;
        response.pSWVerString = pParams->pSWVerString;
        response.pPlasmaIDString = pParams->pPlasmaIDString;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsSLQSSwiGetHostDevInfo( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}


ULONG SLQSSwiSetHostDevInfo(SLQSSwiSetHostDevInfoParams *pParams)
{
    ULONG               resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE                *pInParam;   /* ptr to param field rx'd from modem */
    BYTE                *pOutParam;  /* ptr to outbound param field */
    BYTE                *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT              ParamLength; /* Ret'd length of the QMI Parm field */

    /* Storage for results and response variable */
    struct QmiDmsSLQSSwiSetHostDevInfoResp response;
    struct QmiDmsSLQSSwiSetHostDevInfoReq req;

    /* Device string is OUT parameter and hence should be NULL */
    if ( !pParams->pManString || !pParams->pModelString 
      || !pParams->pSWVerString || !pParams->pPlasmaIDString )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

   req.bManSize        = pParams->bManSize;
   req.bModelSize      = pParams->bModelSize;
   req.bSWVerSize      = pParams->bSWVerSize;
   req.bPlasmaIDSize   = pParams->bPlasmaIDSize;

   req.pManString      = pParams->pManString;
   req.pModelString    = pParams->pModelString;
   req.pSWVerString    = pParams->pSWVerString;
   req.pPlasmaIDString = pParams->pPlasmaIDString;
         

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Index points to first QMI header. */
    ParamLength = 0;

    /* Invoke the function which packs QMI message */
    resultCode = PkQmiDmsSLQSSwiSetHostDevInfo( &ParamLength, pOutParam ,&req);
    if (resultCode != eQCWWAN_ERR_NONE)
    {
        qmrelreqbkp();
        return resultCode;
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  ParamLength,
                                  eQMI_TIMEOUT_2_S, /* 2 seconds */
                                  &pInParam,
                                  &ParamLength );


    /* Only parse out the response data if we got a positive return */
    if (resultCode == eQCWWAN_ERR_NONE)
    {
        resultCode = UpkQmiDmsSLQSSwiSetHostDevInfo( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}


ULONG SLQSSwiGetOSInfo(SLQSSwiGetOSInfoParams *pParams)
{
    ULONG               resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE                *pInParam;   /* ptr to param field rx'd from modem */
    BYTE                *pOutParam;  /* ptr to outbound param field */
    BYTE                *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT              ParamLength; /* Ret'd length of the QMI Parm field */

    /* Storage for results and response variable */

   struct QmiDmsSLQSSwiGetOSInfoResp response;

    /* Device string is OUT parameter and hence should be NULL */
    if ( !pParams->pNameString || !pParams->pVersionString )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Index points to first QMI header. */
    ParamLength = 0;

    /* Invoke the function which packs QMI message */
    resultCode = PkQmiDmsSLQSSwiGetOSInfo( &ParamLength, pOutParam );
    if (resultCode != eQCWWAN_ERR_NONE)
    {
        qmrelreqbkp();
        return resultCode;
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  ParamLength,
                                  eQMI_TIMEOUT_2_S, /* 2 seconds */
                                  &pInParam,
                                  &ParamLength );


    /* Only parse out the response data if we got a positive return */
    if (resultCode == eQCWWAN_ERR_NONE)
    {
      /* Initialising the Buffer */
        slmemset((char *)pParams->pNameString, 0, pParams->bNameSize);
        slmemset((char *)pParams->pVersionString, 0, pParams->bVersionSize);
        response.bNameSize     = pParams->bNameSize;
        response.bVersionSize  = pParams->bVersionSize;
   
        response.pNameString    = pParams->pNameString;
        response.pVersionString = pParams->pVersionString;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsSLQSSwiGetOSInfo( pInParam, &response );

    }

    qmrelreqbkp();
    return resultCode;
}


ULONG SLQSSwiSetOSInfo(SLQSSwiSetOSInfoParams *pParams)
{
   ULONG            resultCode;  /* Result of SwiQmiSendnWait() */
   BYTE            *pInParam;    /* ptr to param field rx'd from modem */
   BYTE            *pOutParam;  /* ptr to outbound param field */
   BYTE            *pReqBuf;    /* Pointer to outgoing request buffer */
   USHORT            ParamLength; /* Ret'd length of the QMI Parm field */

   /* Storage for results and response variable */
   struct QmiDmsSLQSSwiSetOSInfoResp response;
   struct QmiDmsSLQSSwiSetOSInfoReq req;

   /* Device string is OUT parameter and hence should be NULL */
    if ( !pParams->pNameString || !pParams->pVersionString )
   {
      return eQCWWAN_ERR_INVALID_ARG;
   }

   req.bNameSize      = pParams->bNameSize;
   req.bVersionSize   = pParams->bVersionSize;

   req.pNameString    = pParams->pNameString;
   req.pVersionString = pParams->pVersionString;
         

   /* Initialize the pointer to the outgoing request buffer pointer */
   resultCode = qmgetreqbkp(&pReqBuf);
    if (resultCode != eQCWWAN_ERR_NONE)
       return resultCode;

   /* Get a pointer to the start of the outbound QMI Parameter field */
   pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

   /* Index points to first QMI header. */
   ParamLength = 0;

   /* Invoke the function which packs QMI message */
   resultCode = PkQmiDmsSLQSSwiSetOSInfo( &ParamLength, pOutParam ,&req);
   if (resultCode != eQCWWAN_ERR_NONE)
   {
      qmrelreqbkp();
      return resultCode;
   }

   /* Prepare and send the blocking call */
   resultCode = SwiQmiSendnWait( pReqBuf,
                          eQMI_SVC_DMS,
                          ParamLength,
                          eQMI_TIMEOUT_2_S, /* 2 seconds */
                          &pInParam,
                          &ParamLength );


   /* Only parse out the response data if we got a positive return */
   if (resultCode == eQCWWAN_ERR_NONE)
   {
      resultCode = UpkQmiDmsSLQSSwiSetOSInfo( pInParam, &response );
   }

   qmrelreqbkp();
   return resultCode;
}

ULONG SLQSSwiGetSerialNoExt(SLQSSwiGetSerialNoExtParams *pParams)
{
    ULONG               resultCode;  /* Result of SwiQmiSendnWait() */
    BYTE                *pInParam;   /* ptr to param field rx'd from modem */
    BYTE                *pOutParam;  /* ptr to outbound param field */
    BYTE                *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT              ParamLength; /* Ret'd length of the QMI Parm field */

    /* Storage for results and response variable */

    struct QmiDmsSLQSSwiGetSerialNoExtResp response;

    /* Device string is OUT parameter and hence should be NULL */
    if ( !pParams->pMeidString )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Index points to first QMI header. */
    ParamLength = 0;

    /* Invoke the function which packs QMI message */
    resultCode = PkQmiDmsSLQSSwiGetSerialNoExt( &ParamLength, pOutParam );
    if (resultCode != eQCWWAN_ERR_NONE)
    {
        qmrelreqbkp();
        return resultCode;
    }

    /* Prepare and send the blocking call */
    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  ParamLength,
                                  eQMI_TIMEOUT_5_S, /* 5 seconds */
                                  &pInParam,
                                  &ParamLength );


    /* Only parse out the response data if we got a positive return */
    if (resultCode == eQCWWAN_ERR_NONE)
    {
        /* Initialising the Buffer */
        slmemset((char *)pParams->pMeidString, 0, pParams->meidLength);

        response.meidLength  = pParams->meidLength;
        response.pMeidString = pParams->pMeidString;

        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsSLQSSwiGetSerialNoExt( pInParam, &response );

    }

    qmrelreqbkp();
    return resultCode;
}

ULONG SLQSGetCustFeaturesV2(
    getCustomFeatureV2 *pGetCustomFeatureV2 )
{
    ULONG  resultCode;  /* Result code to be returned by this function */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsGetCustFeaturesV2Resp response;

    /* Checking for Invalid Parameter */
    if ( NULL == pGetCustomFeatureV2 )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }
    if ( (NULL == pGetCustomFeatureV2->pCustSettingInfo) ||
         (NULL == pGetCustomFeatureV2->pCustSettingList) ||
         (NULL == pGetCustomFeatureV2->pGetCustomInput) )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsGetCustFeaturesV2( &paramLength,
                                          pOutParam,
                                          pGetCustomFeatureV2->pGetCustomInput );

    if ( eQCWWAN_ERR_NONE != resultCode )
    {
        qmrelreqbkp();
        return resultCode; /* No Memory */
    }

    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_2_S, /* 2 seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( eQCWWAN_ERR_NONE == resultCode )
    {
        /* Set the pointers for the function OUT parameters */
        response.pGetCustomFeatureV2 = pGetCustomFeatureV2;
        /* initialize the output structure */
        slmemset((char*)pGetCustomFeatureV2->pCustSettingInfo, 0, sizeof(custSettingInfo));
        slmemset((char*)pGetCustomFeatureV2->pCustSettingList, 0, sizeof(custSettingList));
        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsGetCustFeaturesV2( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG SLQSSetCustFeaturesV2(
    setCustomSettingV2 *pSetCustSetting )
{
    ULONG  resultCode;  /* Result code to be returned by this function */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    struct QmiDmsSetCustFeaturesV2Resp response;

    /* Checking for Invalid Parameter */
    if ( NULL == pSetCustSetting )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsSetCustFeaturesV2( &paramLength,
                                          pOutParam,
                                          pSetCustSetting );

    if ( eQCWWAN_ERR_NONE != resultCode )
    {
        qmrelreqbkp();
        return resultCode; /* No Memory */
    }

    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_2_S, /* 2 seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( eQCWWAN_ERR_NONE == resultCode )
    {
        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsSetCustFeaturesV2( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG SLQSSwiGetDyingGaspCfg(
    getDyingGaspCfg *pConfig )
{
    ULONG  resultCode;  /* Result code to be returned by this function */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    QmiDmsSwiGetDyingGaspCfgResp response;

    /* Checking for Invalid Parameter */
    if ( NULL == pConfig )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsSwiGetDyingGaspCfg( &paramLength,
                                          pOutParam );

    if ( eQCWWAN_ERR_NONE != resultCode )
    {
        qmrelreqbkp();
        return resultCode; /* No Memory */
    }

    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_2_S, /* 2 seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( eQCWWAN_ERR_NONE == resultCode )
    {
        /* Set the pointers for the function OUT parameters */
        response.pDyingGaspCfg= pConfig;
        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsSwiGetDyingGaspCfg( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG SLQSSwiSetDyingGaspCfg(
    setDyingGaspCfg *pConfig )
{
    ULONG  resultCode;  /* Result code to be returned by this function */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    QmiDmsSwiSetDyingGaspCfgResp response;

    
    /* Checking for Invalid Parameter */
    if ( NULL == pConfig )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }
    else if (( NULL == pConfig->pDestSMSContent) &&
                ( NULL == pConfig->pDestSMSNum))
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsSwiSetDyingGaspCfg( &paramLength,
                                          pOutParam,
                                          pConfig);

    if ( eQCWWAN_ERR_NONE != resultCode )
    {
        qmrelreqbkp();
        return resultCode; /* No Memory */
    }

    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_2_S, /* 2 seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( eQCWWAN_ERR_NONE == resultCode )
    {
        /* Copy to the caller's buffer */
        memset(&response,0,sizeof(QmiDmsSwiSetDyingGaspCfgResp));
        resultCode = UpkQmiDmsSwiSetDyingGaspCfg( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG SLQSSwiGetDyingGaspStatistics(
    getDyingGaspStatistics *pStatistics )
{
    ULONG  resultCode;  /* Result code to be returned by this function */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    QmiDmsSwiGetDyingGaspStatisticsResp response;

    
    /* Checking for Invalid Parameter */
    if ( NULL == pStatistics )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsSwiGetDyingGaspStatistics( &paramLength,
                                          pOutParam);

    if ( eQCWWAN_ERR_NONE != resultCode )
    {
        qmrelreqbkp();
        return resultCode; /* No Memory */
    }

    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_2_S, /* 2 seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( eQCWWAN_ERR_NONE == resultCode )
    {
        response.pDyingGaspstatistics = pStatistics; 
        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsSwiGetDyingGaspStatistics( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG SLQSSwiClearDyingGaspStatistics()
{
    ULONG  resultCode;  /* Result code to be returned by this function */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to outgoing request buffer */
    USHORT paramLength; /* Ret'd length of the QMI Param field */

    /* Storage for results and response variable */
    QmiDmsSwiClearDyingGaspStatisticsResp response;

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultCode = qmgetreqbkp(&pReqBuf);
     if (resultCode != eQCWWAN_ERR_NONE)
         return resultCode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs the QMI message */
    resultCode = PkQmiDmsSwiClearDyingGaspStatistics( &paramLength,
                                          pOutParam);

    if ( eQCWWAN_ERR_NONE != resultCode )
    {
        qmrelreqbkp();
        return resultCode; /* No Memory */
    }

    resultCode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_2_S, /* 2 seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we got a positive return */
    if ( eQCWWAN_ERR_NONE == resultCode )
    {
        memset(&response,0,sizeof(QmiDmsSwiClearDyingGaspStatisticsResp));
        /* Copy to the caller's buffer */
        resultCode = UpkQmiDmsSwiClearDyingGaspStatistics( pInParam, &response );
    }

    qmrelreqbkp();
    return resultCode;
}

ULONG SLQSDmsSwiIndicationRegister(
    dmsIndicationRegisterReq *pIndicationRegisterReq )
{
    ULONG  resultcode;  /* Result of SwiQmiSendnWait() */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to ougoing request buffer */
    USHORT paramLength; /* Ret'd length of the Param field */

    /* Storage for results and response variable */
    struct QmiDmsSwiIndicationRegisterResp response;

    /* Checking for Invalid Parameter */
    if ( NULL == pIndicationRegisterReq->pSwiGetResetInd )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultcode = qmgetreqbkp(&pReqBuf);
     if (resultcode != eQCWWAN_ERR_NONE)
         return resultcode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs QMI message */
    resultcode = PkQmiDmsIndicationRegister( &paramLength,
                                             pOutParam ,
                                             pIndicationRegisterReq );

    if ( resultcode != eQCWWAN_ERR_NONE )
    {
        qmrelreqbkp();
        return resultcode; /* No memory */
    }

    /* Prepare and send the blocking call */
    resultcode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_2_S, /* 2 Seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we get a positive return */
    if ( eQCWWAN_ERR_NONE == resultcode )
    {
        resultcode = UpkQmiDmsIndicationRegister( pInParam, &response );
    }

    qmrelreqbkp();
    return resultcode;
}

ULONG SLQSDmsSwiGetResetInfo(
    dmsSwiGetResetInfo *pGetResetInfoResp )
{
    ULONG  resultcode;  /* Result of SwiQmiSendnWait() */
    BYTE   *pInParam;   /* ptr to param field rx'd from modem */
    BYTE   *pOutParam;  /* ptr to outbound param field */
    BYTE   *pReqBuf;    /* Pointer to ougoing request buffer */
    USHORT paramLength; /* Ret'd length of the Param field */

    /* Storage for results and response variable */
    struct QmiDmsSwiGetResetInfoResp response;

    if ( (NULL == pGetResetInfoResp->pType) ||
         (NULL == pGetResetInfoResp->pSource) )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Initialize the pointer to the outgoing request buffer pointer */
    resultcode = qmgetreqbkp(&pReqBuf);
     if (resultcode != eQCWWAN_ERR_NONE)
         return resultcode;

    /* Get a pointer to the start of the outbound QMI Parameter field */
    pOutParam = amgetparamp( AMTYPEQMIREQUEST, pReqBuf );

    /* Invoke the function which packs QMI message */
    resultcode = PkQmiDmsSwiGetResetInfo( &paramLength,
                                             pOutParam );

    if ( resultcode != eQCWWAN_ERR_NONE )
    {
        qmrelreqbkp();
        return resultcode; /* No memory */
    }

    /* Prepare and send the blocking call */
    resultcode = SwiQmiSendnWait( pReqBuf,
                                  eQMI_SVC_DMS,
                                  paramLength,
                                  eQMI_TIMEOUT_2_S, /* 2 Seconds */
                                  &pInParam,
                                  &paramLength );

    /* Only parse out the response data if we get a positive return */
    if ( eQCWWAN_ERR_NONE == resultcode )
    {
        /* initialize the type and source */
        *pGetResetInfoResp->pType = 0xff;
        *pGetResetInfoResp->pSource = 0xff;
        response.pResetInfo = pGetResetInfoResp;
        resultcode = UpkQmiDmsSwiGetResetInfo( pInParam, &response );
    }

    qmrelreqbkp();
    return resultcode;
}


