/*
 * \ingroup dms
 *
 * \file    qaDmsSLQSSwiGetUSBComp.c
 *
 * \brief   Contains Packing and UnPacking routines for the
 *          eQMI_DMS_SWI_GET_USB_COMP message.
 *
 * Copyright: © 2014 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */
#include "SwiDataTypes.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaGobiApiDms.h"
#include "qaDmsSLQSSwiGetUSBComp.h"

/*****************************************************************************
 * Request handling
 *****************************************************************************/
/*
 * This function packs the SLQSSwiGetUSBComp request parameters to
 * the QMI message SDU
 *
 * \param  pMlength     [OUT] - Total length of built message.
 *
 * \param  pParamField  [OUT] - Pointer to storage into which the packed data
 *                              will be placed by this function.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError PkQmiDmsSLQSSwiGetUSBComp(
    WORD *pMlength,
    BYTE *pParamField )
{
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_TYPE_INVALID, NULL }/* Important. Sentinel.
                                    * Signifies last item in map
                                    */
    };

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    eRCode =  qmbuild( pParamField,
                       NULL,
                       map,
                       eQMI_DMS_SWI_GET_USB_COMP,
                       pMlength );
    return eRCode;
}

/*****************************************************************************
 * Response handling
 *****************************************************************************/
/*
 * This function unpacks the current USB Composition from the QMI response
 * message to a user provided response structure
 *
 * \param pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param pResp    - Pointer to structure containing storage
 *                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE
 */
enum eQCWWANError UnpackTlvGetUSBComp(BYTE *pTlvData, BYTE *pResp)
{
    USBCompParams *lResp =
        ((struct QmiDmsSLQSSwiGetUSBCompResp *)pResp)->pUSBCompParams;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Check For Null Parameter */
    if ( NULL == lResp->pUSBComp )
    {
        return eRCode;
    }

    /* Unpack the current USB composition parameter */
    eRCode = GetByte( pTlvData, lResp->pUSBComp );
    return eRCode;
}

/*
 * This function unpacks the Ref Data parameter from the QMI response
 * message to a user provided response structure
 *
 * \param pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param pResp    - Pointer to structure containing storage
 *                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE
 */
enum eQCWWANError UnpackTlvgetSuppUSBComp(BYTE *pTlvData, BYTE *pResp)
{
    USBCompParams *lResp =
        ((struct QmiDmsSLQSSwiGetUSBCompResp *)pResp)->pUSBCompParams;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE              lCtr   = 0;

    /* Check For Null Parameter */
    if ( NULL == lResp->pNumSupUSBComps ||
         NULL == lResp->pSupUSBComps )
    {
        return eRCode;
    }

    /* Unpack the number of supported USB compositions */
    eRCode = GetByte( pTlvData, lResp->pNumSupUSBComps );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    /* Unpack the supported USB compositions */
    for( lCtr = 0; lCtr < *(lResp->pNumSupUSBComps); lCtr++ )
    {
        eRCode = GetByte( pTlvData, &lResp->pSupUSBComps[lCtr] );
        if( eQCWWAN_ERR_NONE != eRCode )
        {
            return eRCode;
        }
    }

    return eRCode;
}

/*
 * This function unpacks the SLQSSwiGetUSBComp response to a
 * user-provided response structure.
 *
 * \param  pMdmResp           - Pointer to packed response from the modem.
 *
 * \param  pApiResp           - Pointer to storage to unpack into.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UpkQmiDmsSLQSSwiGetUSBComp(
    BYTE                               *pMdmResp,
    struct QmiDmsSLQSSwiGetUSBCompResp *pApiResp )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,           &qmUnpackTlvResultCode },
        { eTLV_DMS_GET_USB_COMP,      &UnpackTlvGetUSBComp },
        { eTLV_DMS_GET_SUPP_USB_COMP, &UnpackTlvgetSuppUSBComp },
        { eTLV_TYPE_INVALID,          NULL } /* Important. Sentinel.
                                             * Signifies last item in map.
                                             */
    };

    eRCode = qmunpackresp( pMdmResp,
                           (BYTE *)pApiResp,
                           map,
                           eQMI_DMS_SWI_GET_USB_COMP );
    return eRCode;
}
