/*
 * \ingroup dms
 *
 * \file    qaDmsSLQSSwiGetHostDevInfo.c
 *
 * \brief   Contains Packing and UnPacking routines for the
 *          eQMI_DMS_SWI_GET_HOST_DEV_INFO message.
 *
 * Copyright: Â© 2014 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */
#include "SwiDataTypes.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaGobiApiDms.h"
#include "qaDmsSLQSSwiGetHostDevInfo.h"

/*****************************************************************************
 * Request handling
 *****************************************************************************/
/*
 * This function packs the GetHostDevInfo request parameters to
 * the QMI message SDU
 *
 * \param  pMlength     [OUT] - Total length of built message.
 *
 * \param  pParamField  [OUT] - Pointer to storage into which the packed data
 *                              will be placed by this function.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError PkQmiDmsSLQSSwiGetHostDevInfo(
    WORD *pMlength,
    BYTE *pParamField )
{
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_TYPE_INVALID, NULL }/* Important. Sentinel.
                                    * Signifies last item in map
                                    */
    };

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    eRCode =  qmbuild( pParamField,
                       NULL,
                       map,
                       eQMI_DMS_SWI_GET_HOST_DEV_INFO,
                       pMlength );
    return eRCode;
}

/*****************************************************************************
 * Response handling
 *****************************************************************************/
/*
 * This function unpacks the current Host Device Manufactureer from the QMI response
 * message to a user provided response structure
 *
 * \param pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param pResp    - Pointer to structure containing storage
 *                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE
 */
enum eQCWWANError UnpackTlvGetHostDevInfoManufacturer(BYTE *pTlvData, BYTE *pResp)
{
    struct QmiDmsSLQSSwiGetHostDevInfoResp *lResp =
        (struct QmiDmsSLQSSwiGetHostDevInfoResp *)pResp;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Check For Null Parameter */
    if ( 0 == lResp->bManSize )
    {
        return eRCode;
    }

    /* Extract the String Parameter */
    /* The size of the string is stored to check for buffer size in API */
    eRCode = qmQmiExtractString( pTlvData,
                                 lResp->pManString,
                                 lResp->bManSize );
	return eRCode;
}

/*****************************************************************************
 * Response handling
 *****************************************************************************/
/*
 * This function unpacks the current Host Device Model from the QMI response
 * message to a user provided response structure
 *
 * \param pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param pResp    - Pointer to structure containing storage
 *                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE
 */
enum eQCWWANError UnpackTlvGetHostDevInfoModel(BYTE *pTlvData, BYTE *pResp)
{
    struct QmiDmsSLQSSwiGetHostDevInfoResp *lResp =
        (struct QmiDmsSLQSSwiGetHostDevInfoResp *)pResp;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Check For Null Parameter */
    if ( 0 == lResp->bModelSize )
    {
        return eRCode;
    }

    /* Extract the String Parameter */
    /* The size of the string is stored to check for buffer size in API */
    eRCode = qmQmiExtractString( pTlvData,
                                 lResp->pModelString,
                                 lResp->bModelSize );
	 return eRCode;
}

/*****************************************************************************
 * Response handling
 *****************************************************************************/
/*
 * This function unpacks the current Host Device Software Version from the QMI response
 * message to a user provided response structure
 *
 * \param pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param pResp    - Pointer to structure containing storage
 *                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE
 */
enum eQCWWANError UnpackTlvGetHostDevInfoSoftwareVersion(BYTE *pTlvData, BYTE *pResp)
{
    struct QmiDmsSLQSSwiGetHostDevInfoResp *lResp =
        (struct QmiDmsSLQSSwiGetHostDevInfoResp *)pResp;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Check For Null Parameter */
    if ( 0 == lResp->bSWVerSize )
    {
        return eRCode;
    }

    /* Extract the String Parameter */
    /* The size of the string is stored to check for buffer size in API */
    eRCode = qmQmiExtractString( pTlvData,
                                 lResp->pSWVerString,
                                 lResp->bSWVerSize );
	 return eRCode;
}


/*****************************************************************************
 * Response handling
 *****************************************************************************/
/*
 * This function unpacks the current Host Device Plasma ID from the QMI response
 * message to a user provided response structure
 *
 * \param pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param pResp    - Pointer to structure containing storage
 *                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE
 */
enum eQCWWANError UnpackTlvGetHostDevInfoPLASMAID(BYTE *pTlvData, BYTE *pResp)
{
    struct QmiDmsSLQSSwiGetHostDevInfoResp *lResp =
        (struct QmiDmsSLQSSwiGetHostDevInfoResp *)pResp;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Check For Null Parameter */
    if ( 0 == lResp->bPlasmaIDSize )
    {
        return eRCode;
    }

    /* Extract the String Parameter */
    /* The size of the string is stored to check for buffer size in API */
    eRCode = qmQmiExtractString( pTlvData,
                                 lResp->pPlasmaIDString,
                                 lResp->bPlasmaIDSize );
	 return eRCode;
}
/*
 * This function unpacks the SLQSSwiGetHostDevInfo response to a
 * user-provided response structure.
 *
 * \param  pMdmResp           - Pointer to packed response from the modem.
 *
 * \param  pApiResp           - Pointer to storage to unpack into.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UpkQmiDmsSLQSSwiGetHostDevInfo(
    BYTE                               *pMdmResp,
    struct QmiDmsSLQSSwiGetHostDevInfoResp *pApiResp )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,           					&qmUnpackTlvResultCode },
        { eTLV_DMS_GET_HOST_DEV_INFO_MANUFACTURER,		&UnpackTlvGetHostDevInfoManufacturer },
        { eTLV_DMS_GET_HOST_DEV_INFO_MODEL,				&UnpackTlvGetHostDevInfoModel },
        { eTLV_DMS_GET_HOST_DEV_INFO_SOFTWARE_VERSION,	&UnpackTlvGetHostDevInfoSoftwareVersion },
        { eTLV_DMS_GET_HOST_DEV_INFO_PLASMA_ID,			&UnpackTlvGetHostDevInfoPLASMAID },
        { eTLV_TYPE_INVALID,          		NULL } /* Important. Sentinel.
                                             * Signifies last item in map.
                                             */
    };

    eRCode = qmunpackresp( pMdmResp,
                           (BYTE *)pApiResp,
                           map,
                           eQMI_DMS_SWI_GET_HOST_DEV_INFO );
    return eRCode;
}
