/*
 * \ingroup dms
 *
 * \file    qaDmsSLQSSwiGetFwUpdatStat.c
 *
 * \brief   Contains Packing and UnPacking routines for the
 *          eQMI_DMS_SWI_GET_FW_UPDATE_STAT message.
 *
 * Copyright: © 2014 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */
#include "SwiDataTypes.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaGobiApiDms.h"
#include "qaDmsSLQSSwiGetFwUpdatStat.h"

/*****************************************************************************
 * Request handling
 *****************************************************************************/
/*
 * This function packs the SLQSSwiGetFwUpdateStatus request parameters to
 * the QMI message SDU
 *
 * \param  pMlength     [OUT] - Total length of built message.
 *
 * \param  pParamField  [OUT] - Pointer to storage into which the packed data
 *                              will be placed by this function.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError PkQmiDmsSLQSSwiGetFwUpdStat(
    WORD *pMlength,
    BYTE *pParamField )
{
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_TYPE_INVALID, NULL }/* Important. Sentinel.
                                    * Signifies last item in map
                                    */
    };

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    eRCode =  qmbuild( pParamField,
                       NULL,
                       map,
                       eQMI_DMS_SWI_GET_FW_UPDATE_STAT,
                       pMlength );
    return eRCode;
}

/*****************************************************************************
 * Response handling
 *****************************************************************************/
/*
 * This function unpacks the result code parameter from the QMI response
 * message to a user provided response structure
 *
 * \param pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param pResp    - Pointer to structure containing storage
 *                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE
 */
enum eQCWWANError UnpackTlvResCode(BYTE *pTlvData, BYTE *pResp)
{
    FirmwareUpdatStat *lResp =
        ((struct QmiDmsSLQSSwiGetFwUpdStatResp *)pResp)->pFirmwareUpdatStat;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Unpack the Result Code parameter */
    eRCode = GetLong( pTlvData, &(lResp->ResCode) );
    return eRCode;
}

/*
 * This function unpacks the Image Type parameter from the QMI response
 * message to a user provided response structure
 *
 * \param pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param pResp    - Pointer to structure containing storage
 *                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE
 */
enum eQCWWANError UnpackTlvImgType(BYTE *pTlvData, BYTE *pResp)
{
    FirmwareUpdatStat *lResp =
        ((struct QmiDmsSLQSSwiGetFwUpdStatResp *)pResp)->pFirmwareUpdatStat;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Check For Null Parameter */
    if ( NULL == lResp->pImgType )
    {
        return eRCode;
    }

    /* Unpack the image type parameter */
    eRCode = GetByte( pTlvData, lResp->pImgType );
    return eRCode;
}

/*
 * This function unpacks the Ref Data parameter from the QMI response
 * message to a user provided response structure
 *
 * \param pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param pResp    - Pointer to structure containing storage
 *                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE
 */
enum eQCWWANError UnpackTlvRefData(BYTE *pTlvData, BYTE *pResp)
{
    FirmwareUpdatStat *lResp =
        ((struct QmiDmsSLQSSwiGetFwUpdStatResp *)pResp)->pFirmwareUpdatStat;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Check For Null Parameter */
    if ( NULL == lResp->pRefData )
    {
        return eRCode;
    }

    /* Unpack the Ref. Data parameter */
    eRCode = GetLong( pTlvData, lResp->pRefData );
    return eRCode;
}

/*
 * Unpacks the Reference String from the QMI response message to a user
 * provided response structure
 *
 * \param       pTlvData [IN/OUT] - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]    - Pointer to structure containing storage
 *                                  to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError UnpackTlvRefString( BYTE *pTlvData, BYTE *pResp )
{
    FirmwareUpdatStat *lResp =
        ( ( struct QmiDmsSLQSSwiGetFwUpdStatResp * )pResp )->pFirmwareUpdatStat;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    swi_uint16        strLength = 0;

    if ( NULL == lResp->pRefStringLen ||
         NULL == lResp->pRefString )
        return eRCode;

    /* Get the Length of string as the string might not be NULL terminated */
    eRCode = GetStringLen( (swi_uint8 *)pTlvData, &strLength );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    if( strLength > *(lResp->pRefStringLen) )
    {
        return eQCWWAN_ERR_BUFFER_SZ;
    }

    *(lResp->pRefStringLen) = strLength;
    /* Extract the String Parameter */
    eRCode = qmQmiExtractString( pTlvData,
                                 (CHAR *)lResp->pRefString,
                                 strLength + 1 );
    return eRCode;
}

/*
 * Unpacks the Reference String from the QMI response message to a user
 * provided response structure
 *
 * \param       pTlvData [IN/OUT] - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]    - Pointer to structure containing storage
 *                                  to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE on success, eQCWWAN_ERR_XXX otherwise
 */
enum eQCWWANError UnpackTlvLogString( BYTE *pTlvData, BYTE *pResp )
{
    FirmwareUpdatStat *lResp =
        ( ( struct QmiDmsSLQSSwiGetFwUpdStatResp * )pResp )->pFirmwareUpdatStat;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    swi_uint16        strLength = 0;

    if ( NULL == lResp->pLogStringLen ||
         NULL == lResp->pLogString )
        return eRCode;

    /* Get the Length of string as the string might not be NULL terminated */
    eRCode = GetStringLen( (swi_uint8 *)pTlvData, &strLength );
    if( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    if( strLength > *(lResp->pLogStringLen) )
    {
        return eQCWWAN_ERR_BUFFER_SZ;
    }

    if(DMS_SLQS_SWI_MAX_LOG_STRING_SIZE < strLength)
    {
        return eQCWWAN_ERR_BUFFER_SZ;
    }

    *(lResp->pLogStringLen) = strLength;
    /* Extract the String Parameter */
    eRCode = qmQmiExtractString( pTlvData,
                                 (CHAR *)lResp->pLogString,
                                 strLength + 1 );
    return eRCode;
}

/*
 * This function unpacks the SLQSSwiGetFwUpdateStatus response to a
 * user-provided response structure.
 *
 * \param  pMdmResp           - Pointer to packed response from the modem.
 *
 * \param  pApiResp           - Pointer to storage to unpack into.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UpkQmiDmsSLQSSwiGetFwUpdStat(
    BYTE                                 *pMdmResp,
    struct QmiDmsSLQSSwiGetFwUpdStatResp *pApiResp )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,          &qmUnpackTlvResultCode },
        { eTLV_DMS_FWUPD_RES_CODE,   &UnpackTlvResCode },
        { eTLV_DMS_FWUPD_IMG_TYPE,   &UnpackTlvImgType },
        { eTLV_DMS_FWUPD_REF_DATA,   &UnpackTlvRefData },
        { eTLV_DMS_FWUPD_REF_STRING, &UnpackTlvRefString },
        { eTLV_DMS_FWUPD_LOG_STRING, &UnpackTlvLogString },
        { eTLV_TYPE_INVALID,         NULL } /* Important. Sentinel.
                                             * Signifies last item in map.
                                             */
    };

    eRCode = qmunpackresp( pMdmResp,
                           (BYTE *)pApiResp,
                           map,
                           eQMI_DMS_SWI_GET_FW_UPDATE_STAT );
    return eRCode;
}
