/*
 * \ingroup dms
 *
 * \file    qaDmsSLQSSwiGetDyingGaspStatistics.c
 *
 * \brief   Contains Packing and UnPacking routines for the
 *          eQMI_DMS_SWI_GET_DYING_GASP_STAT message.
 *
 * Copyright: © 2016 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */

#include "SwiDataTypes.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "sludefs.h"
#include "qaDmsSLQSSwiGetDyingGaspStatistics.h"

/******************************************************************************
 * Request handling
 ******************************************************************************/
enum eQCWWANError qmUnpackTlvTimeStamp(BYTE *pTlvData, BYTE *pResp)
{
    getDyingGaspStatistics *lResp =
        ((QmiDmsSwiGetDyingGaspStatisticsResp *)pResp)->pDyingGaspstatistics;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    if(NULL == lResp)
        return eQCWWAN_ERR_INVALID_ARG;
    /* Check For Null Parameter */
    if ( (NULL == lResp->pTimeStamp) )
    {
        return eRCode;
    }
    /* Extract the TimeStamp Parameter */
    eRCode = GetLong( pTlvData,
                lResp->pTimeStamp );
    return eRCode;
}

enum eQCWWANError qmUnpackTlvAttemptedFlag(BYTE *pTlvData, BYTE *pResp)
{
    getDyingGaspStatistics *lResp =
        ((QmiDmsSwiGetDyingGaspStatisticsResp *)pResp)->pDyingGaspstatistics;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    if(NULL == lResp)
        return eQCWWAN_ERR_INVALID_ARG;
    
    /* Check For Null Parameter */
    if ( (NULL == lResp->pSMSAttemptedFlag) )
    {
        return eRCode;
    }

    eRCode = GetByte( pTlvData,
                lResp->pSMSAttemptedFlag);
    return eRCode;
}

/*
 * This function packs the Get the current Statistics for the 
 * Dying Gasp feature request parameters to the QMI message SDU
 *
 * \param  pMlength            [OUT] - Total length of built message.
 *
 * \param  pParamField         [OUT] - Pointer to storage into which the packed
 *                                     data will be placed by this function.
 *
 * \param  pCustfeatureSetting [IN]  - structure containing data to be packed.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 * \sa qaGobiApiDms.h for remaining parameter descriptions.
 */
enum eQCWWANError PkQmiDmsSwiGetDyingGaspStatistics(
    WORD                *pMlength,
    BYTE                *pParamField )
{
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_TYPE_INVALID,   NULL } /* Important. Sentinel.
                                       * Signifies last item in map
                                       */
    };
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    eRCode = qmbuild( pParamField,
                      NULL,
                      map,
                      eQMI_DMS_SWI_GET_DYING_GASP_STAT,
                      pMlength );
    return eRCode;
}

/******************************************************************************
 * Response handling
 ******************************************************************************/
/*
 * This function unpacks the Get the current Statistics for the 
 * Dying Gasp feature request parameters to the QMI message SDU
 *
 * \param  pMdmResp - Pointer to packed response from the modem.
 *
 * \param  pApiResp - Pointer to storage to unpack into.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UpkQmiDmsSwiGetDyingGaspStatistics(
    BYTE                             *pMdmResp,
    QmiDmsSwiGetDyingGaspStatisticsResp *pApiResp )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,  &qmUnpackTlvResultCode },
        { eQMI_DMS_SWI_GET_DYING_GASP_STATATICS_TIMESTAMP,
            &qmUnpackTlvTimeStamp},
        {eQMI_DMS_SWI_GET_DYING_GASP_STATATICS_SMS_ATTEMPTED_COUNT,
            &qmUnpackTlvAttemptedFlag},
        { eTLV_TYPE_INVALID, NULL } /* Important. Sentinel.
                                     * Signifies last item in map.
                                     */
    };

    eRCode = qmunpackresp( pMdmResp,
                           (BYTE *)pApiResp,
                           map,
                           eQMI_DMS_SWI_GET_DYING_GASP_STAT );
    return eRCode;
}

