/*
 * \ingroup dms
 *
 * \file    qaDmsSLQSSwiGetCrashInfo.c
 *
 * \brief   Contains Packing and UnPacking routines for the
 *          eQMI_DMS_SWI_GET_CRASH_INFO message.
 *
 * Copyright: © 2014 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */
#include "SwiDataTypes.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaGobiApiDms.h"
#include "qaDmsSLQSSwiGetCrashInfo.h"
#include "sludefs.h"
#include "amudefs.h"

/*****************************************************************************
 * Request handling
 *****************************************************************************/
#define MAX_CRASH_INFO_CRASH_STRING 64
local CHAR szCrashString[MAX_CRASH_INFO_CRASH_STRING];
local CHAR szGCDumpString[AMMAXRESPBKLEN];
/*
 * This function packs the clear request parameters to
 * the QMI message SDU *
 *
 * \param  pBuf   [OUT] - Pointer to storage into which the packed
 *                        data will be placed by this function.
 *
 * \param  pParam [IN]  - Pointer to structure containing data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError BuildTlvClear( BYTE *pBuf, BYTE *pParam )
{
    struct QmiDmsSLQSSwiGetCrashInfoReq *lReq =
                     ( struct QmiDmsSLQSSwiGetCrashInfoReq *) pParam;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    if ( NULL == lReq->pClear )
    {
        return eQCWWAN_ERR_NONE;
    }

    /* Insert clear TLV */
    eRCode = PutByte( pBuf, *(lReq->pClear));

    return eRCode;
}

/*
 * This function packs the SLQSSwiGetCrashInfo request parameters to
 * the QMI message SDU
 *
 * \param  pMlength     [OUT] - Total length of built message.
 *
 * \param  pParamField  [OUT] - Pointer to storage into which the packed data
 *                              will be placed by this function.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError PkQmiDmsSLQSSwiGetCrashInfo(
    WORD *pMlength,
    BYTE *pParamField,
    BYTE *pClear)
{
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_DMS_CLEAR, &BuildTlvClear },
        { eTLV_TYPE_INVALID, NULL }/* Important. Sentinel.
                                    * Signifies last item in map
                                    */
    };

    struct QmiDmsSLQSSwiGetCrashInfoReq req;
    slmemset((CHAR *)&req, 0, sizeof(struct QmiDmsSLQSSwiGetCrashInfoReq));
    req.pClear = pClear;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    eRCode =  qmbuild( pParamField,
                       (BYTE *)&req,
                       map,
                       eQMI_DMS_SWI_GET_CRASH_INFO,
                       pMlength );
    return eRCode;
}

/*****************************************************************************
 * Response handling
 *****************************************************************************/
/*
 * This function unpacks the device crash status from the QMI response
 * message to a user provided response structure
 *
 * \param pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param pResp    - Pointer to structure containing storage
 *                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE
 */
enum eQCWWANError UnpackTlvGetDevCrashStatus(BYTE *pTlvData, BYTE *pResp)
{
    CrashInfoParams *lResp =
        ((struct QmiDmsSLQSSwiGetCrashInfoResp *)pResp)->pCrashInfoParams;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Check For Null Parameter */
    if ( NULL == lResp->pDevCrashStatus )
    {
        return eQCWWAN_ERR_INVALID_ARG;
    }

    /* Unpack the device crash status parameter */
    eRCode = GetByte( pTlvData, lResp->pDevCrashStatus );
    return eRCode;
}

/*
 * This function unpacks the crash Information from the QMI response
 * message to a user provided response structure
 *
 * \param pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param pResp    - Pointer to structure containing storage
 *                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE
 */
enum eQCWWANError UnpackTlvGetCrashInfo(BYTE *pTlvData, BYTE *pResp)
{
    CrashInfoParams *lResp =
        ((struct QmiDmsSLQSSwiGetCrashInfoResp *)pResp)->pCrashInfoParams;

    /*Local varibale to store input length*/

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Check For Null Parameter */
    if ( NULL == lResp->pCrashInfo )
    {
        return eRCode;
    }

    CrashInfo *plCrashInfo = lResp->pCrashInfo;

    /* Unpack the numCrashes parameter */
    eRCode = GetWord( pTlvData, &(plCrashInfo->numCrashes) );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    /* Unpack the Crash Id parameter */
    eRCode = GetLong( pTlvData, &(plCrashInfo->crashId) );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    /* Unpack the Crash Data parameter */
    eRCode = GetLong( pTlvData, &(plCrashInfo->crashData) );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    /* Unpack the Crash String length parameter */
    eRCode = GetWord( pTlvData, &(plCrashInfo->crashStrLen) );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }
    if( MAX_CRASH_INFO_CRASH_STRING < plCrashInfo->crashStrLen )
    {
        return eQCWWAN_ERR_BUFFER_SZ;
    }
    
    memset(&szCrashString,0,MAX_CRASH_INFO_CRASH_STRING);
    plCrashInfo->pCrashString = (CHAR*)&szCrashString;
    if(plCrashInfo->pCrashString==NULL)
    {
        return eQCWWAN_ERR_MEMORY;
    }
        
    eRCode = qmQmiExtractString ( pTlvData,
                                      plCrashInfo->pCrashString,
                                      AMMAXRESPBKLEN );
    
    /* Unpack the GCDump String Length parameter */
    eRCode = GetWord( pTlvData, &(lResp->pCrashInfo->gcDumpStrLen) );
    if ( eQCWWAN_ERR_NONE != eRCode )
    {
        return eRCode;
    }

    if(AMMAXRESPBKLEN < plCrashInfo->gcDumpStrLen)
    {
        return eQCWWAN_ERR_BUFFER_SZ;
    }

    memset(&szGCDumpString,0,AMMAXRESPBKLEN);
    plCrashInfo->pGCDumpString = (CHAR*)&szGCDumpString;
    if(plCrashInfo->pGCDumpString==NULL)
    {
        return eQCWWAN_ERR_MEMORY;
    }
    /* Unpack the GCDump String parameter */
    eRCode = qmQmiExtractString ( pTlvData,
                                      plCrashInfo->pGCDumpString,
                                      AMMAXRESPBKLEN );
    
    return eRCode;
}

/*
 * This function unpacks the SLQSSwiGetCrashInfo response to a
 * user-provided response structure.
 *
 * \param  pMdmResp           - Pointer to packed response from the modem.
 *
 * \param  pApiResp           - Pointer to storage to unpack into.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UpkQmiDmsSLQSSwiGetCrashInfo(
    BYTE                                 *pMdmResp,
    struct QmiDmsSLQSSwiGetCrashInfoResp *pApiResp )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,              &qmUnpackTlvResultCode },
        { eTLV_DMS_GET_DEV_CRASH_STATUS, &UnpackTlvGetDevCrashStatus },
        { eTLV_DMS_GET_CRASH_INFO,       &UnpackTlvGetCrashInfo },
        { eTLV_TYPE_INVALID,             NULL } /* Important. Sentinel.
                                                 * Signifies last item in map.
                                                 */
    };

    eRCode = qmunpackresp( pMdmResp,
                           (BYTE *)pApiResp,
                           map,
                           eQMI_DMS_SWI_GET_CRASH_INFO );
    return eRCode;
}
