/*
 * \ingroup dms
 *
 * \file    qaDmsSLQSGetCustFeaturesV2.c
 *
 * \brief   Contains Packing and UnPacking routines for the
 *          QMI_DMS_SWI_GET_CUST_FEATURES message.
 *
 * Copyright: © 2015 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */

#include "SwiDataTypes.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "sludefs.h"
#include "qaDmsSLQSGetCustFeaturesV2.h"

/******************************************************************************
 * Request handling
 ******************************************************************************/
enum eQCWWANError BuildTlvCustId( BYTE *pBuf, BYTE *pParam )
{
    getCustomInput *pReq =
    ((struct QmiDmsGetCustFeaturesV2Req *)pParam)->pCustSettInput;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE i;

    /* Check For Null Parameter */
    if ( NULL == pReq )
    {
        return eRCode;
    }
    /* insert custom id */
    for ( i = 0; i < strlen(pReq->cust_id); i++ )
    {
        eRCode = PutByte( pBuf, (BYTE) pReq->cust_id[i]);
        CHECK();
    }

    return eRCode;
}


enum eQCWWANError BuildTlvCustListType( BYTE *pBuf, BYTE *pParam )
{
    getCustomInput *pReq =
    ((struct QmiDmsGetCustFeaturesV2Req *)pParam)->pCustSettInput;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Check For Null Parameter */
    if ( NULL == pReq )
    {
        return eRCode;
    }

    /* insert list type */
    eRCode = PutByte( pBuf, pReq->list_type );

    return eRCode;
}

/*
 * This function packs the Get Custom Features request parameters to the QMI
 * message SDU
 *
 * \param  pMlength     [OUT] - Total length of built message.
 *
 * \param  pParamField  [OUT] - Pointer to storage into which the packed
 *                              data will be placed by this function.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 * \sa qaGobiApiDms.h for remaining parameter descriptions.
 */
enum eQCWWANError PkQmiDmsGetCustFeaturesV2(
    WORD *pMlength,
    BYTE *pParamField,
    getCustomInput *pCustSettInput)
{
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_CUST_ID_REQ,   BuildTlvCustId },
        { eTLV_CUST_LIST_TYPE_REQ,   BuildTlvCustListType },
        { eTLV_TYPE_INVALID,   NULL }  /* Important. Sentinel.
                                       * Signifies last item in map
                                       */
    };

    struct QmiDmsGetCustFeaturesV2Req req;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    slmemset((CHAR *)&req, 0, sizeof(struct QmiDmsGetCustFeaturesV2Req));
    req.pCustSettInput = pCustSettInput;

    eRCode = qmbuild( pParamField,
                      (BYTE *)&req,
                      map,
                      eQMI_DMS_SWI_GET_CUST_FEATURES,
                      pMlength );
    return eRCode;
}

/******************************************************************************
 * Response handling
 ******************************************************************************/
enum eQCWWANError UnpackTlvCustSetting(BYTE *pTlvData, BYTE *pResp)
{
    getCustomFeatureV2 *lResp =
        ((struct QmiDmsGetCustFeaturesV2Resp *)pResp)->pGetCustomFeatureV2;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE i;

    /* Check For Null Parameter */
    if ( (NULL == lResp->pCustSettingInfo) )
    {
        return eRCode;
    }

    eRCode = GetWord( pTlvData, &lResp->pCustSettingInfo->id_length );
    CHECK();
    for ( i = 0; i < lResp->pCustSettingInfo->id_length; i++ )
    {
        eRCode = GetByte(pTlvData, (BYTE*)&lResp->pCustSettingInfo->cust_id[i]);
        CHECK();
    }
    eRCode = GetWord( pTlvData, &lResp->pCustSettingInfo->value_length);
    CHECK();

    for ( i = 0; i < lResp->pCustSettingInfo->value_length; i++ )
    {
        eRCode = GetByte(pTlvData, &lResp->pCustSettingInfo->cust_value[i]);
        CHECK();
    }
   
    eRCode = GetWord( pTlvData, &lResp->pCustSettingInfo->cust_attr);
    CHECK();

    return eRCode;
}

enum eQCWWANError UnpackTlvCustList(BYTE *pTlvData, BYTE *pResp)
{
    getCustomFeatureV2 *lResp =
        ((struct QmiDmsGetCustFeaturesV2Resp *)pResp)->pGetCustomFeatureV2;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE i, j;

    /* Check For Null Parameter */
    if ( NULL == lResp->pCustSettingList)
    {
        return eRCode;
    }

    eRCode = GetByte( pTlvData, &lResp->pCustSettingList->list_type );
    CHECK();

    eRCode = GetWord( pTlvData, &lResp->pCustSettingList->num_instances);
    CHECK();
    for ( i = 0; i < lResp->pCustSettingList->num_instances; i++ )
    {
        eRCode = GetWord( pTlvData, &lResp->pCustSettingList->custSetting[i].id_length);
        CHECK();
        for ( j = 0; j < lResp->pCustSettingList->custSetting[i].id_length; j++ )
        {
            eRCode = GetByte(pTlvData, (BYTE*)&lResp->pCustSettingList->custSetting[i].cust_id[j]);
            CHECK();
        }

        eRCode = GetWord( pTlvData, &lResp->pCustSettingList->custSetting[i].value_length);
        CHECK();
        for ( j = 0; j < lResp->pCustSettingList->custSetting[i].value_length; j++ )
        {
            eRCode = GetByte(pTlvData, &lResp->pCustSettingList->custSetting[i].cust_value[j]);
            CHECK();
        }

        eRCode = GetWord( pTlvData, &lResp->pCustSettingList->custSetting[i].cust_attr);
        CHECK();
    }

    return eRCode;
}

/*
 * This function unpacks the get custom features response message to a user
 * provided response structure.
 *
 * \param  pMdmResp - Pointer to packed response from the modem.
 *
 * \param  pApiResp - Pointer to storage to unpack into.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UpkQmiDmsGetCustFeaturesV2(
    BYTE                             *pMdmResp,
    struct QmiDmsGetCustFeaturesV2Resp *pApiResp )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,       &qmUnpackTlvResultCode },
        { eTLV_CUST_SETTING_INFO, &UnpackTlvCustSetting },
        { eTLV_CUST_LIST,         &UnpackTlvCustList },
        { eTLV_TYPE_INVALID,      NULL } /* Important. Sentinel.
                                          * Signifies last item in map.
                                          */
    };

    eRCode = qmunpackresp( pMdmResp,
                           (BYTE *)pApiResp,
                           map,
                           eQMI_DMS_SWI_GET_CUST_FEATURES );
    return eRCode;
}
