/*
 * \ingroup dms
 *
 * \file qaDmsIndicationRegister.c
 *
 * \brief  Contains Packing and UnPacking routines for the
 *   QMI_DMS_SWI_INDICATION_REGISTER message.
 *
 * Copyright: © 2016 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */

#include "SwiDataTypes.h"
#include "sludefs.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaQmiBasic.h"

#include "qaDmsIndicationRegister.h"

/*****************************************************************************
 * Local Variables
 ******************************************************************************/

/*****************************************************************************
 * Request handling
 ******************************************************************************/
/*
 * Packs the Swi Get Reset Info Indication field to the QMI message
 *
 * \param  pBuf   - Pointer to storage into which the packed
 *                  data will be placed by this function.
 *
 * \param  pParam - Pointer to structure containing data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE on success, eQCWWAN_ERR_XXX on error
 */
enum eQCWWANError BuildTlvSwiGetResetInfoInd( BYTE *pBuf, BYTE *pParam )
{
    dmsIndicationRegisterReq *pReq = ( dmsIndicationRegisterReq *)pParam;

    if ( NULL == pReq->pSwiGetResetInd )
    {
        return eQCWWAN_ERR_NONE;
    }

    /* Insert Get Reset Info Indication event */
    return PutByte( pBuf, *(pReq->pSwiGetResetInd) );
}


/*
 * This function packs the SWI Indication Regster parameters
 * to the QMI message SDU
 *
 * \param    pParamField [OUT] - Pointer to storage into which the packed
 *                                    data will be placed by this function.
 *
 * \param    pMlength    [OUT] - Total length of built message.
 *
 * \return   eQCWWAN_ERR_NONE     - Increment successful
 * \return   eQCWWAN_ERR_MEMORY    - Increment beyond allowed size attempted
 *
 */
enum eQCWWANError PkQmiDmsIndicationRegister (
    WORD    *pMlength,
    BYTE    *pParamField,
    dmsIndicationRegisterReq *pIndicationRegisterReq  )
{
    /* There is no mandatory TLV and hence the map is empty */
    static struct qmTlvBuilderItem map[] =
    {
        { eTLV_SWI_GET_RESET_INFO_IND, &BuildTlvSwiGetResetInfoInd },
        { eTLV_TYPE_INVALID, NULL }  /* Important. Sentinel.
                                      * Signifies last item in map.
                                      */
    };

    enum eQCWWANError eRCode;

    eRCode = qmbuild( pParamField,
                      (BYTE*) pIndicationRegisterReq,
                      map,
                      eQMI_DMS_SWI_INDICATION_REGISTER,
                      pMlength );

    return eRCode;
}

/*
 * This function unpacks the SWI Indication Regster response message to to a
 * user-provided response structure.
 *
 * \param   pMdmResp - Pointer to packed response from the modem.
 *
 * \param   pApiResp - Pointer to storage to unpack into.
 *
 * \return  eQCWWAN_ERR_NONE, on success
 */
enum eQCWWANError UpkQmiDmsIndicationRegister(
    BYTE    *pMdmResp,
    struct  QmiDmsSwiIndicationRegisterResp *pApiResp )
{
    enum eQCWWANError eRCode;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_RESULT_CODE,             &qmUnpackTlvResultCode },
        { eTLV_TYPE_INVALID,            NULL }  /* Important. Sentinel.
                                                 * Signifies last item in map.
                                                 */
    };

    eRCode = qmunpackresp( pMdmResp,
                          (BYTE*)pApiResp,
                          map,
                          eQMI_DMS_SWI_INDICATION_REGISTER );
    return eRCode;
}

