/*
 * \ingroup cbk
 *
 * \file    qaCbkWmsAsyncRawSendInd.c
 *
 * \brief   Contains UnPacking routines for the
 *          QMI_WMS_ASYNC_RAW_SEND_IND message.
 *
 * Copyright: © 2014 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */

#include "SwiDataTypes.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaGobiApiCbk.h"
#include "qaCbkWmsAsyncRawSendInd.h"

/* Functions */

/*
 * Unpacks the Send Status Field
 *
 * \param   pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param   pResp    - Pointer to structure containing destination buffer
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 */
package enum eQCWWANError UnpackCbkSendStatus(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    SMSAsyncRawSend *lResp = (SMSAsyncRawSend *)pResp;

    /* Check For Invalid Parameter */
    if ( NULL == lResp )
    {
        return eQCWWAN_ERR_NONE;
    }

    return GetWord( pTlvData, &(lResp->sendStatus) );
}

/*
 * Unpacks the Message ID Field
 *
 * \param   pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param   pResp    - Pointer to structure containing destination buffer
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 */
package enum eQCWWANError UnpackCbkMsgID(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    SMSAsyncRawSend *lResp = (SMSAsyncRawSend *)pResp;

    /* Check For Invalid Parameter */
    if ( NULL == lResp )
    {
        return eQCWWAN_ERR_NONE;
    }

    return GetWord( pTlvData, &(lResp->messageID) );
}

/*
 * Unpacks the Cause Code field
 *
 * \param   pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param   pResp    - Pointer to structure containing destination buffer
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 */
package enum eQCWWANError UnpackCbkAysncCauseCode(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    SMSAsyncRawSend *lResp = (SMSAsyncRawSend *)pResp;

    /* Check For Invalid Parameter */
    if ( NULL == lResp )
    {
        return eQCWWAN_ERR_NONE;
    }

    return GetWord( pTlvData, &(lResp->causeCode) );
}

/*
 * Unpacks the Error class Information
 *
 * \param   pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param   pResp    - Pointer to structure containing destination buffer
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 */
package enum eQCWWANError UnpackCbkErrorClass(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    SMSAsyncRawSend *lResp = (SMSAsyncRawSend *)pResp;

    /* Check For Invalid Parameter */
    if ( NULL == lResp )
    {
        return eQCWWAN_ERR_NONE;
    }

    return GetByte( pTlvData, &(lResp->errorClass) );
}

/*
 * Unpacks the SMS GW cause Information
 *
 * \param   pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param   pResp    - Pointer to structure containing destination buffer
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 */
package enum eQCWWANError UnpackCbkGWCauseInfo(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    SMSAsyncRawSend *lResp = (SMSAsyncRawSend *)pResp;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Check For Invalid Parameter */
    if ( NULL == lResp )
    {
        return eQCWWAN_ERR_NONE;
    }

    eRCode = GetWord( pTlvData, &(lResp->RPCause) );
    if(eRCode == eQCWWAN_ERR_NONE)
    {
        return GetByte( pTlvData, &(lResp->TPCause));
    }
    return eRCode;
}

/*
 * Unpacks the Message delivery failure type
 *
 * \param   pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param   pResp    - Pointer to structure containing destination buffer
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 */
package enum eQCWWANError UnpackCbkMsgDelFailType(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    SMSAsyncRawSend *lResp = (SMSAsyncRawSend *)pResp;

    /* Check For Invalid Parameter */
    if ( NULL == lResp )
    {
        return eQCWWAN_ERR_NONE;
    }

    return GetByte( pTlvData, &(lResp->msgDelFailureType) );
}

/*
 * Unpacks the Message delivery failure cause
 *
 * \param   pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param   pResp    - Pointer to structure containing destination buffer
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 */
package enum eQCWWANError UnpackCbkMsgDelFailCause(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    SMSAsyncRawSend *lResp = (SMSAsyncRawSend *)pResp;

    /* Check For Invalid Parameter */
    if ( NULL == lResp )
    {
        return eQCWWAN_ERR_NONE;
    }

    return GetByte( pTlvData, &(lResp->msgDelFailureCause) );
}


/*
 * Unpacks the Call Control Modified Information
 *
 * \param   pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param   pResp    - Pointer to structure containing destination buffer
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 */
package enum eQCWWANError UnpackCbkCallCntrlModInfo(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    SMSAsyncRawSend *lResp = (SMSAsyncRawSend *)pResp;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    BYTE lLength;

    /* Check For Invalid Parameter */
    if ( NULL == lResp )
    {
        return eQCWWAN_ERR_NONE;
    }

    eRCode = GetByte( pTlvData, &(lResp->alphaIDLen) );
    if(eRCode == eQCWWAN_ERR_NONE)
    {
        int count = 0;
        lLength = lResp->alphaIDLen;
        for (count = 0; count<=lLength; count++)
        {
            eRCode = GetByte( pTlvData, lResp->pAlphaID );
        }
        return eRCode;
    }
    return eRCode;
}

/*
 * Unpacks the User Data
 *
 * \param   pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param   pResp    - Pointer to structure containing destination buffer
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 */
package enum eQCWWANError UnpackCbkUserData(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    SMSAsyncRawSend *lResp = (SMSAsyncRawSend *)pResp;

    /* Check For Invalid Parameter */
    if ( NULL == lResp )
    {
        return eQCWWAN_ERR_NONE;
    }

    return GetLong( pTlvData, &(lResp->userData) );
}

/*
 * Unpacks the Async Raw Send Indication message to a
 * user-provided response structure.
 *
 * \param   pMdmResp - Pointer to packed response from the modem.
 *
 * \param   pApiResp - Pointer to storage to unpack into.
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 */
package enum eQCWWANError UpkQmiCbkWmsAsyncRawSendInd(
    BYTE            *pMdmResp,
    SMSAsyncRawSend *pApiResp )
{
    enum eQCWWANError eRCode;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_SEND_STATUS,                &UnpackCbkSendStatus },
        { eTLV_MSG_ID,                     &UnpackCbkMsgID },
        { eTLV_ASYNC_CAUSE_CODE,           &UnpackCbkAysncCauseCode },
        { eTLV_ERROR_CLASS,                &UnpackCbkErrorClass },
        { eTLV_GW_CAUSE_INFO,              &UnpackCbkGWCauseInfo },
        { eTLV_MSG_DELIVERY_FAILURE_TYPE,  &UnpackCbkMsgDelFailType },
        { eTLV_MSG_DELIVERY_FAILURE_CAUSE, &UnpackCbkMsgDelFailCause },
        { eTLV_CALL_CNTRL_MOD_INFO,        &UnpackCbkCallCntrlModInfo },
        { eTLV_USER_DATA,                  &UnpackCbkUserData },
        { eTLV_TYPE_INVALID, NULL }  /* Important. Sentinel.
                                      * Signifies last item in map.
                                      */
    };
    eRCode = qmunpackresp( pMdmResp,
                           (BYTE*)pApiResp,
                           map,
                           eQMI_WMS_ASYNC_RAW_SEND_IND );
    return eRCode;
}
