/*
 *
 * \ingroup : cbk
 *
 * \file    : qaCbkWdsSWIDHCPv4ClientLeaseInd.c
 *
 * \brief   : Contains Packing and UnPacking routines for the
 *             eQMI_WDS_SWI_SET_DHCPV4_CLIENT_LEASE_IND message.
 *
 * Copyright: © 2016 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */
#include "SwiDataTypes.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaGobiApiWds.h"
#include "qaGobiApiCbk.h"
#include "qaCbkWdsSWIDHCPv4ClientLeaseInd.h"

/******************************************************************************
 * Response handling
 ******************************************************************************/
/*
 * This function unpacks the Profile Id from the
 * QMI response message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                     to return data for this TLV.
 *
 * \return      eQCWWAN_ERR_NONE, on success
 */
static enum eQCWWANError qmUnpackTlvProfileId(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    DHCPProfileIdTlv *pTlvProfileId =
        &((QmiCbkWdsDHCPv4ClientLeaseInd *)pResp)->ProfileIdTlv;
    enum eQCWWANError eRCode = eQCWWAN_ERR_ENUM_BEGIN;

    eRCode = GetByte( pTlvData, &pTlvProfileId->profileType );
    if ( eRCode != eQCWWAN_ERR_NONE )
        return eRCode;

    eRCode = GetByte( pTlvData, &pTlvProfileId->profileId );
    if ( eRCode != eQCWWAN_ERR_NONE )
        return eRCode;

    pTlvProfileId->TlvPresent = TRUE;
    return eRCode;
}

/*
 * This function unpacks the Lease State from the
 * QMI response message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                     to return data for this TLV.
 *
 * \return      eQCWWAN_ERR_NONE, on success
 */
static enum eQCWWANError qmUnpackTlvLeaseState(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    DHCPLeaseStateTlv *pTlvLeaseState =
        &((QmiCbkWdsDHCPv4ClientLeaseInd *)pResp)->DHCPv4LeaseStateTlv;
    enum eQCWWANError eRCode = eQCWWAN_ERR_ENUM_BEGIN;

    eRCode = GetByte( pTlvData, &pTlvLeaseState->leaseState );
    if ( eRCode != eQCWWAN_ERR_NONE )
        return eRCode;

    pTlvLeaseState->TlvPresent = TRUE;
    return eRCode;
}

/******************************************************************************
 * Response handling
 ******************************************************************************/
/*
 * This function unpacks the IPv4 Address from the
 * QMI response message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                     to return data for this TLV.
 *
 * \return      eQCWWAN_ERR_NONE, on success
 */
static enum eQCWWANError qmUnpackTlvIPAddr(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    IPv4AdTlv *pTlvIPv4Addr =
        &((QmiCbkWdsDHCPv4ClientLeaseInd *)pResp)->IPv4AddrTlv;
    enum eQCWWANError eRCode = eQCWWAN_ERR_ENUM_BEGIN;

    eRCode = GetLong( pTlvData, &pTlvIPv4Addr->IPv4Addr );
    if ( eRCode != eQCWWAN_ERR_NONE )
        return eRCode;

    pTlvIPv4Addr->TlvPresent = TRUE;
    return eRCode;
}

/******************************************************************************
 * Response handling
 ******************************************************************************/
/*
 * This function unpacks the DHCP v4 Lease Options from the
 * QMI response message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                     to return data for this TLV.
 *
 * \return      eQCWWAN_ERR_NONE, on success
 */
static enum eQCWWANError qmUnpackTlvLeaseOpt(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    DHCPLeaseOptTlv *pTlvLeaseOpt =
        &((QmiCbkWdsDHCPv4ClientLeaseInd *)pResp)->DHCPv4LeaseOptTlv;
    enum eQCWWANError eRCode = eQCWWAN_ERR_ENUM_BEGIN;
    swi_uint8 i;
    BYTE dataBufIdx = 0;

    eRCode = GetByte( pTlvData, &pTlvLeaseOpt->numOpt );
    if ( eRCode != eQCWWAN_ERR_NONE )
        return eRCode;

    if (pTlvLeaseOpt->numOpt > DHCP_MAX_NUM_OPTIONS )
        return eQCWWAN_ERR_INVALID_ARG;

    /* Read option data for each option*/
    for (i=0; i < pTlvLeaseOpt->numOpt; i++)
    {
        eRCode = GetByte( pTlvData, &pTlvLeaseOpt->optList[i].optCode );
        if ( eRCode != eQCWWAN_ERR_NONE )
            return eRCode;

        eRCode = GetByte( pTlvData, &pTlvLeaseOpt->optList[i].optValLen );
        if ( eRCode != eQCWWAN_ERR_NONE )
            return eRCode;

        pTlvLeaseOpt->optList[i].pOptVal = NULL;
        if (pTlvLeaseOpt->optList[i].optValLen > 0)
        {
            /* Extract option value into storage buffer and assign TLV pointer to buffer location */
            if (DHCP_OPTION_DATA_BUF_SIZE < dataBufIdx + pTlvLeaseOpt->optList[i].optValLen)
            {
                /* Not enough memory to store full option value, so truncate value */
                pTlvLeaseOpt->optList[i].optValLen = DHCP_OPTION_DATA_BUF_SIZE - dataBufIdx;

                if (pTlvLeaseOpt->optList[i].optValLen == 0)
                    break;
            }

            eRCode = GetStream( pTlvData, pTlvLeaseOpt->optListData + dataBufIdx, pTlvLeaseOpt->optList[i].optValLen );
            if ( eRCode != eQCWWAN_ERR_NONE )
                return eRCode;

            pTlvLeaseOpt->optList[i].pOptVal = pTlvLeaseOpt->optListData + dataBufIdx;
            dataBufIdx += pTlvLeaseOpt->optList[i].optValLen;
        }
    }

    pTlvLeaseOpt->TlvPresent = TRUE;
    return eRCode;
}

/*
 *
 * This function unpacks the WDS_SWI_DHCPV4_CLIENT_LEASE response
 * message to a user-provided response structure.
 *
 * \param     pMdmResp   [IN]  - Pointer to packed response from the modem.
 *
 * \param     pApiResp   [OUT] - Pointer to storage to unpack into.
 *
 * \return    eQCWWAN_ERR_NONE, on success
 *
 */

enum eQCWWANError UpkQmiWdsSWIDHCPv4ClientLeaseInd(
    BYTE                          *pMdmResp,
    QmiCbkWdsDHCPv4ClientLeaseInd *pApiResp )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_DHCP_CLIENT_PROFILE_ID,  &qmUnpackTlvProfileId},
        { eTLV_DHCP_CLIENT_LEASE_STATE, &qmUnpackTlvLeaseState},
        { eTLV_DHCP_CLIENT_IP_ADDRESS,  &qmUnpackTlvIPAddr},
        { eTLV_DHCP_CLIENT_LEASE_OPT,   &qmUnpackTlvLeaseOpt},
        { eTLV_TYPE_INVALID, NULL } /* Important. Sentinel.
                                     * Signifies last item in map.
                                     */
    };

    eRCode = qmunpackresp( pMdmResp,
                           (BYTE *)pApiResp,
                           map,
                           eQMI_WDS_SWI_SET_DHCPV4_CLIENT_LEASE_IND );                           

    return eRCode;
}



