/*
 * \ingroup cbk
 *
 * \file    qaCbkWdsNotify.c
 *
 * \brief   Contains routines for the WDS Notifications.
 *
 * Copyright: © 2011-2014 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */

#include "SwiDataTypes.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaGobiApiCbk.h"
#include "qaCbkWdsEventReportInd.h"
#include "qaCbkWdsGetPktSrvcStatusInd.h"
#include "qaCbkWdsDUNCallInfoInd.h"
#include "qaQmiNotify.h"
#include "sludefs.h"

/* Functions */

/*************
 *
 * Name:    qaQmiWdsPktStatusNotify
 *
 * Purpose: Unpacks the recevied WDS indication and invokes the approriate
 *          callback based on the QMI message type.
 *
 * Parms:   pQmiIndication  [IN] - pointer to structure used to store all QMI
 *                                 Notification parameters.
 *
 * Return:  none
 *
 * Abort:   none
 *
 * Notes:   none
 *
 **************/
package void qaQmiWdsPktStatusNotify(
    struct QmiNotification *pQmiIndication )
{
    void *pCallback;
    enum eQMICallbackIndex CbkIndex;

    CbkIndex  = eQMI_CB_SESSION_STATE;
    pCallback = qaQmiGetCallback( CbkIndex );
    if(pCallback)
    {
        slqsSessionStateInfo sessionStateInfo;
        struct PktSrvStatusTlv *lResp =
            &pQmiIndication->QmiInd.qaQmiCbkWdsSessionStateInd.pktStatTlv;

        struct SessEndReasonTlv *pResp =
            &pQmiIndication->QmiInd.qaQmiCbkWdsSessionStateInd.sERTlv;

        sessionStateInfo.pQmiInterfaceInfo = &pQmiIndication->qmiInterfaceInfo;
        sessionStateInfo.state = lResp->state;
        sessionStateInfo.reconfiguration_required = lResp->reconfiguration_required;
        sessionStateInfo.sessionEndReason = pResp->sessionEndReason;

        /* Invoke the callback */
        ((tFNSLQSSessionState)pCallback)( &sessionStateInfo );
    }

    CbkIndex  = eQMI_CB_PKT_SERV_STATUS;
    pCallback = qaQmiGetCallback( CbkIndex );
    if(pCallback)
    {
        /* Declare the callback parameters */
        packetSrvStatus PacketSrvStatus;

        /* Initialize the data structure */
        slmemset( (char*) &PacketSrvStatus, 0, sizeof( packetSrvStatus) );

        struct PktSrvStatusTlv *pPktStatTlv =
            &pQmiIndication->QmiInd.qaQmiCbkWdsSessionStateInd.pktStatTlv;

        PacketSrvStatus.pQmiInterfaceInfo = &pQmiIndication->qmiInterfaceInfo;

        /* The TLV was present in the Indication - hence process */
        if( TRUE == pPktStatTlv->TlvPresent )
        {
            /* Extract the Parameters */
            PacketSrvStatus.connStatus = pPktStatTlv->state;
            PacketSrvStatus.reconfigReqd =
                pPktStatTlv->reconfiguration_required;
        }
        else
        {
            /* Assign default max values */
            PacketSrvStatus.connStatus = 0xFF;
            PacketSrvStatus.reconfigReqd = 0xFF;
        }

        struct SessEndReasonTlv *pSERTlv =
            &pQmiIndication->QmiInd.qaQmiCbkWdsSessionStateInd.sERTlv;

        /* The TLV was present in the Indication - hence process */
        if( TRUE == pSERTlv->TlvPresent )
        {
            /* Extract the Parameters */
            PacketSrvStatus.sessionEndReason = pSERTlv->sessionEndReason;
        }
        else
        {
            /* Assign default max values */
            PacketSrvStatus.sessionEndReason = 0xFFFF;
        }

        struct VerbSessEndReasonTlv *pVSERTlv =
            &pQmiIndication->QmiInd.qaQmiCbkWdsSessionStateInd.vSERTlv;

        /* The TLV was present in the Indication - hence process */
        if( TRUE == pVSERTlv->TlvPresent )
        {
            /* Extract the Parameters */
            PacketSrvStatus.verboseSessnEndReasonType =
                pVSERTlv->verboseSessnEndReasonType;
            PacketSrvStatus.verboseSessnEndReason =
                pVSERTlv->verboseSessnEndReason;
        }
        else
        {
            /* Assign default max values */
            PacketSrvStatus.verboseSessnEndReasonType = 0xFFFF;
            PacketSrvStatus.verboseSessnEndReason = 0xFFFF;
        }

        struct IPFamilyTlv *pIPFTlv =
            &pQmiIndication->QmiInd.qaQmiCbkWdsSessionStateInd.IPFTlv;

        /* The TLV was present in the Indication - hence process */
        if( TRUE == pIPFTlv->TlvPresent )
        {
            /* Extract the Parameters */
            PacketSrvStatus.ipFamily = pIPFTlv->ipFamily;
        }
        else
        {
            /* Assign default max values */
            PacketSrvStatus.ipFamily = 0xFF;
        }

        struct TechNameTlv *pTNamTlv =
            &pQmiIndication->QmiInd.qaQmiCbkWdsSessionStateInd.tNamTlv;

        /* The TLV was present in the Indication - hence process */
        if( TRUE == pTNamTlv->TlvPresent )
        {
            /* Extract the Parameters */
            PacketSrvStatus.techName = pTNamTlv->techName;
        }
        else
        {
            /* Assign default max values */
            PacketSrvStatus.techName = 0xFFFF;
        }

        struct BearerIdTlv *pBearerIDTlv =
            &pQmiIndication->QmiInd.qaQmiCbkWdsSessionStateInd.bearerIdTlv;

        /* The TLV was present in the Indication - hence process */
        if( TRUE == pBearerIDTlv->TlvPresent )
        {
            /* Extract the Parameters */
            PacketSrvStatus.bearerID= pBearerIDTlv->bearerID;
        }
        else
        {
            /* Assign default max values */
            PacketSrvStatus.bearerID = 0xFF;
        }

        /* Invoke the callback */
        ((tFNPacketSrvState)pCallback)( &PacketSrvStatus );
    }

}

/*************
 *
 * Name:    qaQmiWdsEventNotify
 *
 * Purpose: To notify all the Callbacks associated with the WDS Event Report
 *          Notification.
 *
 * Parms:   pQmiIndication  [IN] - pointer to structure used to store all QMI
 *                                 Notification parameters.
 *
 * Return:  none
 *
 * Abort:   none
 *
 * Notes:   Must be called after qminit() has been called
 *
 **************/
local void qaQmiWdsEventNotify(
    struct QmiNotification *pQmiIndication,
    BYTE                   instance )
{
    void                   *pCallback = NULL;
    void                   *pXferStatsCb = NULL;
    enum eQMICallbackIndex CbkIndex;
    ULONG                  CbkParamOne;
    CurrDataSysStat        CurrDataSystemStats;

    CbkIndex  = eQMI_CB_MOBILE_IP;
    pCallback = qaQmiGetCallback( CbkIndex );
    if ( pCallback )
    {
        struct MobileIPStatusTlv *pResp =
                &pQmiIndication->QmiInd.qaQmiCbkWdsEventStatusReportInd.MSTlv;

        /* The TLV was present in the Indication - hence process */
        /* Included for TLVs which are not implemented */

        if ( pResp->TlvPresent )
        {
            /* Extract the Parameters */
            CbkParamOne = pResp->MipStatus;

            /* Invoke the callback */
            ((tFNMobileIPStatus)pCallback) ( CbkParamOne );
        }
    }

    /* Resolve callback by QMI instance id */
    CbkIndex = qmiInst2WdsEvent(instance);

    if (CbkIndex >= eQMI_CB_END)
    {
        return;
    }

    pCallback = qaQmiGetCallback( CbkIndex );
    if ( pCallback )
    {
        slqsWdsEventInfo WdsEventInfo;
        struct TransferStatsTlv *pResp1 =
               &pQmiIndication->QmiInd.qaQmiCbkWdsEventStatusReportInd.TSTlv;
        WdsEventInfo.pQmiInterfaceInfo = &pQmiIndication->qmiInterfaceInfo;

        memset(&WdsEventInfo, 0, sizeof(WdsEventInfo));

        /* The TLV was present in the Indication - hence process */
        /* Included for TLVs which are not implemented */
        if ( pResp1->TlvPresent )
        {
            /* Extract the Parameters */
            WdsEventInfo.pPacketsCountTX = &pResp1->TxPacketsCount;
            WdsEventInfo.pPacketsCountRX = &pResp1->RxPacketsCount;
            WdsEventInfo.pTotalBytesTX   = &pResp1->TxTotalBytes;
            WdsEventInfo.pTotalBytesRX   = &pResp1->RxTotalBytes;
        }

        struct DormancyStatusTlv *pResp2 =
                &pQmiIndication->QmiInd.qaQmiCbkWdsEventStatusReportInd.DSTlv;

        /* The TLV was present in the Indication - hence process */
        /* Included for TLVs which are not implemented */

        if ( pResp2->TlvPresent )
        {
            /* Extract the Parameters */
            WdsEventInfo.pDormancyStatus = &pResp2->DormancyStatus;
        }

        struct DataBearerTechTlv *pResp3 =
            &pQmiIndication->QmiInd.qaQmiCbkWdsEventStatusReportInd.DBTechTlv;

        struct CurrentDataBearerTechTlv *pResp4 =
            &pQmiIndication->QmiInd.qaQmiCbkWdsEventStatusReportInd.CDBTechTlv;

        /* The TLV was present in the Indication - hence process */
        if( pResp3->TlvPresent || pResp4->TlvPresent )
        {
            /* Extract the Parameters */
            /* If both TLV's are present in the indication use current data
               bearer technology TLV */
            WdsEventInfo.pDataBearer = (pResp4->TlvPresent)?
                &pResp4->RATMask : &pResp3->DataBearerTechnology;

        }

        WdsEventInfo.pQmiInterfaceInfo = &pQmiIndication->qmiInterfaceInfo;

        /* Invoke the callback */
        ((tFNSLQSWDSEvent)pCallback) ( &WdsEventInfo );
    }

    /* report trasfer statistic to another callback function */
    CbkIndex = qmiInst2XferStatsEvent(instance);
    if(eQMI_CB_END > CbkIndex)
    pXferStatsCb = qaQmiGetCallback( CbkIndex );
    if ( pXferStatsCb )
    {
    	  slqsWdsEventInfo WdsEventInfo2;
       struct TransferStatsTlv *pResp2 =
               &pQmiIndication->QmiInd.qaQmiCbkWdsEventStatusReportInd.TSTlv;
        /* The TLV was present in the Indication - hence process */
        /* Included for TLVs which are not implemented */
        if ( pResp2 ->TlvPresent )
        {
            /* Extract the Parameters */
            WdsEventInfo2.pPacketsCountTX = &pResp2->TxPacketsCount;
            WdsEventInfo2.pPacketsCountRX = &pResp2->RxPacketsCount;
            WdsEventInfo2.pTotalBytesTX   = &pResp2->TxTotalBytes;
            WdsEventInfo2.pTotalBytesRX   = &pResp2->RxTotalBytes;
        }
        if ( pResp2->TlvPresent )
        {
            ((tFNSLQSWDSEvent)pXferStatsCb) ( &WdsEventInfo2 );
        }
    }

    CbkIndex  = eQMI_CB_WDS_DATA_SYS_STAT;
    pCallback = qaQmiGetCallback( CbkIndex );
    if ( pCallback )
    {
        struct DataSystemStatTlv *pResp =
                &pQmiIndication->QmiInd.qaQmiCbkWdsEventStatusReportInd.DSStatTlv;

        /* The TLV was present in the Indication - hence process */
        /* Included for TLVs which are not implemented */

        if ( pResp->TlvPresent )
        {
            /* Extract the Parameters */
            CurrDataSystemStats.pPrefNetwork = &(pResp->PrefNetwork);
            CurrDataSystemStats.pNetworkInfoLen = &(pResp->NetInfoLen);
            CurrDataSystemStats.pCurrNetworkInfo = pResp->CurrNWInfo;

            /* Invoke the callback */
            ((tFNDataSysStatus)pCallback) ( &CurrDataSystemStats );
        }
    }
}

/*************
 *
 * Name:    qaQmiWdsDUNCallInfoNotify
 *
 * Purpose: Unpacks the recevied WDS indication and invokes the approriate
 *          callback based on the QMI message type.
 *
 * Parms:   pQmiIndication  [IN] - pointer to structure used to store all QMI
 *                                 Notification parameters.
 *
 * Return:  none
 *
 * Abort:   none
 *
 * Notes:   none
 *
 **************/
package void qaQmiWdsDUNCallInfoNotify(
    struct QmiNotification *pQmiIndication )
{
    void *pCallback;
    enum eQMICallbackIndex CbkIndex;

    CbkIndex  = eQMI_CB_WDS_DUN_CALL_INFO;
    pCallback = qaQmiGetCallback( CbkIndex );
    if(pCallback)
    {
        /* Declare the callback parameters */
        DUNCallInfoInd DUNCallInfo;

        /* Initialize the data structure */
        slmemset( (char*) &DUNCallInfo, 0, sizeof( DUNCallInfo) );

        struct ConnStatusTlv *pConnStatTlv =
            &pQmiIndication->QmiInd.qaQmiCbkWdsDunCallInfo.CSTlv;

        /* The TLV was present in the Indication - hence process */
        if( TRUE == pConnStatTlv->TlvPresent )
        {
            /* Extract the Parameters */
            DUNCallInfo.MdmConnStatus = pConnStatTlv->MDMConnStatus;
        }
        else
        {
            /* Assign default max values */
            DUNCallInfo.MdmConnStatus = 0xFF;
        }

        struct LastMdmCallEndRsnTlv *pLMCERTlv =
            &pQmiIndication->QmiInd.qaQmiCbkWdsDunCallInfo.LMCERTlv;

        /* The TLV was present in the Indication - hence process */
        if( TRUE == pLMCERTlv->TlvPresent )
        {
            /* Extract the Parameters */
            DUNCallInfo.CallEndReason = pLMCERTlv->CallEndReason;
        }
        else
        {
            /* Assign default max values */
            DUNCallInfo.CallEndReason = 0xFFFF;
        }

        struct TXBytesOKTlv *pTBOKTlv =
            &pQmiIndication->QmiInd.qaQmiCbkWdsDunCallInfo.TXBOTlv;

        /* The TLV was present in the Indication - hence process */
        if( TRUE == pTBOKTlv->TlvPresent )
        {
            /* Extract the Parameters */
            DUNCallInfo.TXOKBytesCount = pTBOKTlv->TxByteOKCnt;
        }
        else
        {
            /* Assign default max values */
            DUNCallInfo.TXOKBytesCount = ~0;
        }

        struct RxBytesOKTlv *pRBOKTlv =
            &pQmiIndication->QmiInd.qaQmiCbkWdsDunCallInfo.RXBOTlv;

        /* The TLV was present in the Indication - hence process */
        if( TRUE == pRBOKTlv->TlvPresent )
        {
            /* Extract the Parameters */
            DUNCallInfo.RXOKBytesCount = pRBOKTlv->RxByteOKCnt;
        }
        else
        {
            /* Assign default max values */
            DUNCallInfo.RXOKBytesCount = ~0;
        }

        struct DormStatTlv *pDSTlv =
            &pQmiIndication->QmiInd.qaQmiCbkWdsDunCallInfo.DSTlv;

        /* The TLV was present in the Indication - hence process */
        if( TRUE == pDSTlv->TlvPresent )
        {
            /* Extract the Parameters */
            DUNCallInfo.DormancyStatus = pDSTlv->DormancyStat;
        }
        else
        {
            /* Assign default max values */
            DUNCallInfo.DormancyStatus = 0xFF;
        }

        struct DataBearTechTlv *pDBTlv =
            &pQmiIndication->QmiInd.qaQmiCbkWdsDunCallInfo.DBTTlv;

        /* The TLV was present in the Indication - hence process */
        if( TRUE == pDBTlv->TlvPresent )
        {
            /* Extract the Parameters */
            DUNCallInfo.DataBearerTech = pDBTlv->DataBearerTech;
        }
        else
        {
            /* Assign default max values */
            DUNCallInfo.DataBearerTech = 0xFF;
        }

        struct ChannelRateTlv *pCRTlv =
            &pQmiIndication->QmiInd.qaQmiCbkWdsDunCallInfo.CRTlv;

        /* The TLV was present in the Indication - hence process */
        if( TRUE == pCRTlv->TlvPresent )
        {
            /* Extract the Parameters */
            DUNCallInfo.ChannelRate.CurrChanTxRate =
                pCRTlv->ChannelRate.CurrChanTxRate;
            DUNCallInfo.ChannelRate.CurrChanRxRate =
                pCRTlv->ChannelRate.CurrChanRxRate;
        }
        else
        {
            /* Assign default max values */
            DUNCallInfo.ChannelRate.CurrChanTxRate = 0xFFFFFFFF;
            DUNCallInfo.ChannelRate.CurrChanRxRate = 0xFFFFFFFF;
        }

        /* Invoke the callback */
        ((tFNDUNCallInfo)pCallback)( &DUNCallInfo );
    }

}

/*************
 *
 * Name:    qaQmiWdsRMTransferStatisticNotify
 *
 * Purpose: Unpacks the recevied WDS indication and invokes the approriate
 *          callback based on the QMI message type.
 *
 * Parms:   pQmiIndication  [IN] - pointer to structure used to store all QMI
 *                                 Notification parameters.
 *
 * Return:  none
 *
 * Abort:   none
 *
 * Notes:   none
 *
 **************/
package void qaQmiWdsRMTransferStatisticNotify(
    struct QmiNotification *pQmiIndication )
{
    void *pCallback;
    enum eQMICallbackIndex CbkIndex;
    CbkIndex  = eQMI_CB_RM_TRANSFER_STATISTICS;
    pCallback = qaQmiGetCallback( CbkIndex );
    if(pCallback)
    {
        QmiCbkWdsStatisticsIndState *MsgInfo = &pQmiIndication->QmiInd.qmQmiCbkWdsTransfterStaticsInfo;
        /* Invoke the callback */
        ((tFNNewRMTransferStatistics)pCallback)( MsgInfo );
    }
}

/*************
 *
 * Name:    qaQmiWdsSWIDHCPv4ClientLeaseNotify
 *
 * Purpose: Unpacks the recevied WDS indication and invokes the approriate
 *          callback based on the QMI message type.
 *
 * Parms:   pQmiIndication  [IN] - pointer to structure used to store all QMI
 *                                 Notification parameters.
 *
 * Return:  none
 *
 * Abort:   none
 *
 * Notes:   none
 *
 **************/
package void qaQmiWdsSWIDHCPv4ClientLeaseNotify(
    struct QmiNotification *pQmiIndication,
    BYTE                   instance )
{
    void *pCallback;
    enum eQMICallbackIndex CbkIndex;
    CbkIndex  = qmiInst2DHCPv4ClientLeaseEvent( instance );
    pCallback = qaQmiGetCallback( CbkIndex );
    if(pCallback)
    {
      WdsDHCPv4ClientLeaseInd leaseStatus;
      WdsDHCPv4ProfileId profileId;
      DHCPOptionList optionList;

      /* Initialize the data structure */
      slmemset( (char*) &leaseStatus, 0, sizeof( leaseStatus) );

      if (pQmiIndication->QmiInd.qmQmiCbkWdsSWIDHCPv4ClientLeaseInd.ProfileIdTlv.TlvPresent)
      {
        profileId.profileType = pQmiIndication->QmiInd.qmQmiCbkWdsSWIDHCPv4ClientLeaseInd.ProfileIdTlv.profileType;
        profileId.profileId = pQmiIndication->QmiInd.qmQmiCbkWdsSWIDHCPv4ClientLeaseInd.ProfileIdTlv.profileId;
        leaseStatus.pProfileId = &profileId;
      }
      if (pQmiIndication->QmiInd.qmQmiCbkWdsSWIDHCPv4ClientLeaseInd.DHCPv4LeaseStateTlv.TlvPresent)
      {
        leaseStatus.pLeaseState = &pQmiIndication->QmiInd.qmQmiCbkWdsSWIDHCPv4ClientLeaseInd.DHCPv4LeaseStateTlv.leaseState;
      }
      if (pQmiIndication->QmiInd.qmQmiCbkWdsSWIDHCPv4ClientLeaseInd.IPv4AddrTlv.TlvPresent)
      {
        leaseStatus.pIPv4Addr = &pQmiIndication->QmiInd.qmQmiCbkWdsSWIDHCPv4ClientLeaseInd.IPv4AddrTlv.IPv4Addr;
      }
      if (pQmiIndication->QmiInd.qmQmiCbkWdsSWIDHCPv4ClientLeaseInd.DHCPv4LeaseOptTlv.TlvPresent)
      {
        optionList.numOpt = pQmiIndication->QmiInd.qmQmiCbkWdsSWIDHCPv4ClientLeaseInd.DHCPv4LeaseOptTlv.numOpt;
        optionList.pOptions = (DHCPOption*)&pQmiIndication->QmiInd.qmQmiCbkWdsSWIDHCPv4ClientLeaseInd.DHCPv4LeaseOptTlv.optList[0];
        leaseStatus.pOptList = &optionList;
      }

      /* Invoke the callback */
      ((tFNDHCPv4ClientLeaseStatus)pCallback)( instance, &leaseStatus );
    }
}


/*************
 *
 * Name:    UpkQmiCbkWdsNotification
 *
 * Purpose: Unpacks the recevied WDS indication and invokes the approriate
 *          callback based on the QMI message type.
 *
 * Parms:   QmiMsgID        - QMI Message ID
 *          pMdmResp   [IN] - Pointer to packed response from the modem.
 *          pNotifResp [IN] - Notification Structure to fill in the parameters.
 *
 * Parms:   instance    - PDP instance ID
 *
 * Return:  eQCWWAN_ERR_NONE on success, eQCWWAN_ERR_XXX otherwise
 *
 * Abort:   none
 *
 * Notes:   none
 *
 **************/
package enum eQCWWANError UpkQmiCbkWdsNotification(
    USHORT                  QmiMsgID,
    BYTE                    *pMdmResp,
    struct QmiNotification  *pNotifResp,
    BYTE                    instance )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    switch (QmiMsgID)
    {
        case eQMI_WDS_EVENT_IND:
        {
            struct QmiCbkWdsEventStatusReportInd *pResp =
                &pNotifResp->QmiInd.qaQmiCbkWdsEventStatusReportInd;

            /* Set all the TLVs to be in Not-Present State */
            pResp->DBTechTlv.TlvPresent  = FALSE;
            pResp->DSTlv.TlvPresent      = FALSE;
            pResp->MSTlv.TlvPresent      = FALSE;
            pResp->TSTlv.TlvPresent      = FALSE;
            pResp->CDBTechTlv.TlvPresent = FALSE;
            pResp->DSStatTlv.TlvPresent  = FALSE;

            /* Unpack the WDS Event Indication */
            eRCode = UpkQmiCbkWdsEventReportInd( pMdmResp,
                                                 pResp );
            /* Notify to the Callbacks associated */
            qaQmiWdsEventNotify( pNotifResp, instance );
            break;
        }
        case eQMI_WDS_PKT_STATUS_IND:
        {
            struct QmiCbkWdsSessionStateInd *pResp =
                &pNotifResp->QmiInd.qaQmiCbkWdsSessionStateInd;

            pResp->pktStatTlv.TlvPresent = FALSE;
            pResp->sERTlv.TlvPresent     = FALSE;
            pResp->vSERTlv.TlvPresent    = FALSE;
            pResp->IPFTlv.TlvPresent     = FALSE;
            pResp->tNamTlv.TlvPresent    = FALSE;
            pResp->bearerIdTlv.TlvPresent = FALSE;

            /* Unpack the WDS Packet Status Indication */
            eRCode = UpkQmiCbkWdsGetPktSrvcStatusInd( pMdmResp,
                                                      pResp );
            /* Notify to the Callbacks associated */
            qaQmiWdsPktStatusNotify( pNotifResp );
            break;
        }
        case eQMI_WDS_GET_DUN_CALL_INFO_IND:
        {
            struct QmiCbkWdsDUNCallInfoInd *pResp =
                &pNotifResp->QmiInd.qaQmiCbkWdsDunCallInfo;

            pResp->CSTlv.TlvPresent    = FALSE;
            pResp->LMCERTlv.TlvPresent = FALSE;
            pResp->TXBOTlv.TlvPresent  = FALSE;
            pResp->RXBOTlv.TlvPresent  = FALSE;
            pResp->DSTlv.TlvPresent    = FALSE;
            pResp->DBTTlv.TlvPresent   = FALSE;
            pResp->CRTlv.TlvPresent    = FALSE;

            /* Unpack the WDS DUN Call Info Indication */
            eRCode = UpkQmiCbkWdsDUNCallInfoInd( pMdmResp,
                                                 pResp );

            /* Notify to the Callbacks associated */
            qaQmiWdsDUNCallInfoNotify( pNotifResp );
            break;
        }
        case eQMI_WDS_RM_TRANSFER_STATISTICS_IND:
        {
            QmiCbkWdsStatisticsIndState *pResp =
            &pNotifResp->QmiInd.qmQmiCbkWdsTransfterStaticsInfo;
            
            pResp->RxDropConutTlv.TlvPresent=FALSE;
            pResp->TxDropConutTlv.TlvPresent=FALSE;
            pResp->RxOkConutTlv.TlvPresent=FALSE;
            pResp->TxOkConutTlv.TlvPresent=FALSE;
            pResp->RxOkByteCountTlv.TlvPresent=FALSE;
            pResp->TxOkByteCountTlv.TlvPresent=FALSE;
           
            eRCode = UpkQmiWdsSWIRMTrasnferStaticsInd(pMdmResp,pResp);
            
            qaQmiWdsRMTransferStatisticNotify( pNotifResp );
            break;
        }
        case eQMI_WDS_SWI_SET_DHCPV4_CLIENT_LEASE_IND:
        {
            QmiCbkWdsDHCPv4ClientLeaseInd *pResp =                                   
            &pNotifResp->QmiInd.qmQmiCbkWdsSWIDHCPv4ClientLeaseInd;

            pResp->ProfileIdTlv.TlvPresent = FALSE;
            pResp->DHCPv4LeaseStateTlv.TlvPresent = FALSE;
            pResp->IPv4AddrTlv.TlvPresent = FALSE;
            pResp->DHCPv4LeaseOptTlv.TlvPresent = FALSE;
    
            eRCode = UpkQmiWdsSWIDHCPv4ClientLeaseInd(pMdmResp,pResp);
    
            qaQmiWdsSWIDHCPv4ClientLeaseNotify( pNotifResp, instance );
            break;
        }
        default:
            break;
    }
    return eRCode;
}


