/*
 * \ingroup cbk
 *
 * \file    qaCbkWdsDUNCallInfoInd.c
 *
 * \brief   Contains UnPacking routines for the
 *          QMI_WDS_DUN_CALL_INFO_IND message.
 *
 * Copyright: © 2014 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */

#include "SwiDataTypes.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaCbkWdsDUNCallInfoInd.h"

/* Functions */

/*
 * This function unpacks the connection status from the
 * QMI indication message to a user provided response structure
 *
 * \param   pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param   pResp    - Pointer to structure containing storage
 *                     to return data for this TLV.
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 */
package enum eQCWWANError UnpackCbkTlvConnStatus(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    struct ConnStatusTlv *lResp =
            &((struct QmiCbkWdsDUNCallInfoInd *)pResp)->CSTlv;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Extract the connection status Count */
    eRCode = GetByte( pTlvData, &(lResp->MDMConnStatus) );

    if ( eRCode == eQCWWAN_ERR_NONE )
        lResp->TlvPresent = TRUE;

    return eRCode;
}

/*
 * This function unpacks the call end reason from the
 * QMI indication message to a user provided response structure
 *
 * \param   pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param   pResp    - Pointer to structure containing storage
 *                     to return data for this TLV.
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 */
package enum eQCWWANError UnpackCbkTlvCallEndRsn(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    struct LastMdmCallEndRsnTlv *lResp =
            &((struct QmiCbkWdsDUNCallInfoInd *)pResp)->LMCERTlv;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Extract the call end reason */
    eRCode = GetWord( pTlvData, &(lResp->CallEndReason) );

    if ( eRCode == eQCWWAN_ERR_NONE )
        lResp->TlvPresent = TRUE;

    return eRCode;
}

/*
 * This function unpacks the TX OK Bytes count from the
 * QMI indication message to a user provided response structure
 *
 * \param   pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param   pResp    - Pointer to structure containing storage
 *                     to return data for this TLV.
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 */
package enum eQCWWANError UnpackCbkTlvTxOKBytesCnt(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    struct TXBytesOKTlv *lResp =
            &((struct QmiCbkWdsDUNCallInfoInd *)pResp)->TXBOTlv;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Extract the Tx OK bytes count */
    eRCode = GetLongLong( pTlvData, &(lResp->TxByteOKCnt) );

    if ( eRCode == eQCWWAN_ERR_NONE )
        lResp->TlvPresent = TRUE;

    return eRCode;
}

/*
 * This function unpacks the RX OK Bytes count from the
 * QMI indication message to a user provided response structure
 *
 * \param   pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param   pResp    - Pointer to structure containing storage
 *                     to return data for this TLV.
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 */
package enum eQCWWANError UnpackCbkTlvRxOKBytesCnt(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    struct RxBytesOKTlv *lResp =
            &((struct QmiCbkWdsDUNCallInfoInd *)pResp)->RXBOTlv;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Extract the Rx OK bytes count */
    eRCode = GetLongLong( pTlvData, &(lResp->RxByteOKCnt) );

    if ( eRCode == eQCWWAN_ERR_NONE )
        lResp->TlvPresent = TRUE;

    return eRCode;
}

/*
 * This function unpacks the dormancy status from the
 * QMI indication message to a user provided response structure
 *
 * \param   pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param   pResp    - Pointer to structure containing storage
 *                     to return data for this TLV.
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 */
package enum eQCWWANError UnpackCbkTlvDormStatus(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    struct DormStatTlv *lResp =
            &((struct QmiCbkWdsDUNCallInfoInd *)pResp)->DSTlv;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Extract the dormancy status */
    eRCode = GetByte( pTlvData, &(lResp->DormancyStat) );

    if ( eRCode == eQCWWAN_ERR_NONE )
        lResp->TlvPresent = TRUE;

    return eRCode;
}

/*
 * This function unpacks the data bearer technology from the
 * QMI indication message to a user provided response structure
 *
 * \param   pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param   pResp    - Pointer to structure containing storage
 *                     to return data for this TLV.
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 */
package enum eQCWWANError UnpackCbkTlvDataBearTech(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    struct DataBearTechTlv *lResp =
            &((struct QmiCbkWdsDUNCallInfoInd *)pResp)->DBTTlv;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Extract the data bearer technology */
    eRCode = GetByte( pTlvData, &(lResp->DataBearerTech) );

    if ( eRCode == eQCWWAN_ERR_NONE )
        lResp->TlvPresent = TRUE;

    return eRCode;
}

/*
 * This function unpacks the Channel Rate from the
 * QMI indication message to a user provided response structure
 *
 * \param   pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param   pResp    - Pointer to structure containing storage
 *                     to return data for this TLV.
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 */
package enum eQCWWANError UnpackCbkTlvChanRate(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    struct ChannelRateTlv *lResp =
            &((struct QmiCbkWdsDUNCallInfoInd *)pResp)->CRTlv;
    channelRate *cResp = &lResp->ChannelRate;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Extract the Current Channel TX Rate */
    eRCode = GetLong( pTlvData, &(cResp->CurrChanTxRate) );
    if( eQCWWAN_ERR_NONE != eRCode )
        return eRCode;

    /* Extract the Current Channel RX Rate */
    eRCode = GetLong( pTlvData, &(cResp->CurrChanRxRate) );
    if ( eRCode == eQCWWAN_ERR_NONE )
        lResp->TlvPresent = TRUE;

    return eRCode;
}

/*
 * This function unpacks the DUN Call Info Indication message to a
 * user-provided response structure.
 *
 * \param   pMdmResp - Pointer to packed response from the modem.
 *
 * \param   pApiResp - Pointer to storage to unpack into.
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 */
package enum eQCWWANError UpkQmiCbkWdsDUNCallInfoInd(
    BYTE                           *pMdmResp,
    struct QmiCbkWdsDUNCallInfoInd *pApiResp )
{
    enum eQCWWANError eRCode;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_MDM_CONN_STAT,
            &UnpackCbkTlvConnStatus },
        { eTLV_LAST_CALL_END_RSN,
            &UnpackCbkTlvCallEndRsn },
        { eTLV_TX_OK_BYTES_CNT,
            &UnpackCbkTlvTxOKBytesCnt },
        { eTLV_RX_OK_BYTES_CNT,
            &UnpackCbkTlvRxOKBytesCnt },
        { eTLV_DORMANCY_STAT,
            &UnpackCbkTlvDormStatus },
        { eTLV_DATA_BEAR_TCH,
            &UnpackCbkTlvDataBearTech },
        { eTLV_CHAN_RATE,
            &UnpackCbkTlvChanRate },
        { eTLV_TYPE_INVALID,
            NULL }  /* Important. Sentinel.
                     * Signifies last item in map.
                     */
    };

    eRCode = qmunpackresp( pMdmResp,
                           (BYTE*)pApiResp,
                           map,
                           eQMI_WDS_GET_DUN_CALL_INFO_IND );
    return eRCode;
}
