/*
 * \ingroup cbk
 *
 * \file    qaCbkQosNotify.c
 *
 * \brief   Contains routines for the WDS Notifications.
 *
 * Copyright: © 2013 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */

#include "SwiDataTypes.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaGobiApiQos.h"
#include "qaGobiApiCbk.h"
#include "qaQosCommon.h"
#include "qaCbkQosNetworkStatusInd.h"
#include "qaCbkQosEventReportInd.h"
#include "qaCbkQosEventReportInd.h"
#include "qaQmiNotify.h"
#include <syslog.h>

/* Functions */

/*************
 *
 * Name:    qaQmiQosNetworkStatusNotify
 *
 * Purpose: Unpacks the recevied QOS indication and invokes the approriate
 *          callback based on the QMI message type.
 *
 * Parms:   pQmiIndication  [IN] - pointer to structure used to store all QMI
 *                                 Notification parameters.
 *
 * Return:  none
 *
 * Abort:   none
 *
 * Notes:   none
 *
 **************/
package void qaQmiQosNetworkStatusNotify(
    struct QmiNotification *pQmiIndication )
{
    void *pCallback;
    enum eQMICallbackIndex CbkIndex;
    BYTE status;

    CbkIndex  = eQMI_CB_QOS_NETWORK_STATUS;
    pCallback = qaQmiGetCallback( CbkIndex );
    if(pCallback)
    {
        struct QmiCbkQosNetworkStatusInd *lResp =
                 &pQmiIndication->QmiInd.qaQmiCbkQosNetworkStatusInd;

        status = lResp->Status;
        /* Invoke the callback */
        ((tFNQosNWStatus)pCallback)( status );
    }
}

local void update_filter(
        struct FilterTlv *pFilterTlv,
        QosFlowInfo *pFlowInfo,
        swiQosFilter *pQFilter,
        sFilter *pFilter,
        int is_tx,
        int arrIndex
        )
{

    if( TRUE == pFilterTlv->TlvPresent )
    {
        syslog(LOG_DEBUG, "filter present\n");
        syslog(LOG_DEBUG, "index %d\n", pFilterTlv->index);
        syslog(LOG_DEBUG, "ver %d\n", pFilterTlv->ver);

        if (is_tx)
            pFlowInfo->pTxQFilter[arrIndex] = pQFilter;
        else
            pFlowInfo->pRxQFilter[arrIndex] = pQFilter;

        pQFilter->index = pFilterTlv->index;
        pQFilter->version = pFilterTlv->ver;

        ipV4FilterSrcAddTLV *pIPV4SrcAddTLV = &pFilterTlv->ipV4SrcAddTLV;
        if( TRUE == pIPV4SrcAddTLV->TlvPresent )
        {
            pFilter->ipV4SrcAddr.addr = pIPV4SrcAddTLV->srcAdd;
            pFilter->ipV4SrcAddr.subnetMask = pIPV4SrcAddTLV->srcAddMask;
            pQFilter->pIPv4SrcAddr = &pFilter->ipV4SrcAddr;
            syslog(LOG_DEBUG, "v4 Src Add  %u\n", pIPV4SrcAddTLV->srcAdd);
            syslog(LOG_DEBUG, "v4 Src Mask %u\n", pIPV4SrcAddTLV->srcAddMask);
        }

        ipV4FilterDestAddTLV *pIPV4DestAddTLV = &pFilterTlv->ipV4DestAddTLV;
        if( TRUE == pIPV4DestAddTLV->TlvPresent )
        {
            pFilter->ipV4DstAddr.addr = pIPV4DestAddTLV->destAdd;
            pFilter->ipV4DstAddr.subnetMask = pIPV4DestAddTLV->destAddMask;
            pQFilter->pIPv4DstAddr = &pFilter->ipV4DstAddr;
            syslog(LOG_DEBUG, "v4 Dest Add  %u\n", pIPV4DestAddTLV->destAdd);
            syslog(LOG_DEBUG, "v4 Dest Mask %u\n", pIPV4DestAddTLV->destAddMask);
        }

        ipV6FilterSrcAddTLV *pIPV6SrcAddTLV = &pFilterTlv->ipV6SrcAddTLV;
        if( TRUE == pIPV6SrcAddTLV->TlvPresent )
        {
            memcpy( pFilter->ipV6SrcAddr.addr,
                    pIPV6SrcAddTLV->srcAdd,16);
            pFilter->ipV6SrcAddr.prefixLen = pIPV6SrcAddTLV->srcAddPfxLen;
            pQFilter->pIPv6SrcAddr = &pFilter->ipV6SrcAddr;
            syslog(LOG_DEBUG, "v6 Src Add  %s\n", pIPV6SrcAddTLV->srcAdd);
            syslog(LOG_DEBUG, "v6 Src Pfx Len %d\n", pIPV6SrcAddTLV->srcAddPfxLen);
        }

        ipV6FilterDestAddTLV *pIPV6DestAddTLV = &pFilterTlv->ipV6DestAddTLV;
        if( TRUE == pIPV6DestAddTLV->TlvPresent )
        {
            memcpy( pFilter->ipV6DstAddr.addr,
                    pIPV6DestAddTLV->destAdd,16);
            pFilter->ipV6DstAddr.prefixLen = pIPV6DestAddTLV->destAddPfxLen;
            pQFilter->pIPv6DstAddr = &pFilter->ipV6DstAddr;
            syslog(LOG_DEBUG, "v6 Dest Add  %s\n", pIPV6DestAddTLV->destAdd);
            syslog(LOG_DEBUG, "v6 Dest Pfx Len %d\n", pIPV6DestAddTLV->destAddPfxLen);
        }

        ipV6FilterFlowLblTLV *pIPV6FltrLblTLV = &pFilterTlv->v6FlowLbl;
        if( TRUE == pIPV6FltrLblTLV->TlvPresent )
        {
            pFilter->ipV6Label = pIPV6FltrLblTLV->flowLbl;
            pQFilter->pIPv6Label = &pFilter->ipV6Label;
            syslog(LOG_DEBUG, "Filter Label = %x\n", pIPV6FltrLblTLV->flowLbl);
        }

        filterPortTLV *pTCPSrcPortTLV = &pFilterTlv->tcpSrcPort;
        if( TRUE == pTCPSrcPortTLV->TlvPresent )
        {
            pFilter->tcpSrcPort.port = pTCPSrcPortTLV->port;
            pFilter->tcpSrcPort.range = pTCPSrcPortTLV->portRange;
            pQFilter->pTCPSrcPort = &pFilter->tcpSrcPort;
            syslog(LOG_DEBUG, "TCP Sorce Port Add  %d\n", pTCPSrcPortTLV->port);
            syslog(LOG_DEBUG, "TCP Sorce Port Range %d\n", pTCPSrcPortTLV->portRange);
        }

        filterPortTLV *pTCPDestPortTLV = &pFilterTlv->tcpDestPort;
        if( TRUE == pTCPDestPortTLV->TlvPresent )
        {
            pFilter->tcpDstPort.port = pTCPDestPortTLV->port;
            pFilter->tcpDstPort.range = pTCPDestPortTLV->portRange;
            pQFilter->pTCPDstPort = &pFilter->tcpDstPort;
            syslog(LOG_DEBUG, "TCP Dest Port Add  %d\n", pTCPDestPortTLV->port);
            syslog(LOG_DEBUG, "TCP Dest Port Range %d\n", pTCPDestPortTLV->portRange);
        }

        filterPortTLV *pUDPSrcPortTLV = &pFilterTlv->udpSrcPort;
        if( TRUE == pUDPSrcPortTLV->TlvPresent )
        {
            pFilter->udpSrcPort.port = pUDPSrcPortTLV->port;
            pFilter->udpSrcPort.range = pUDPSrcPortTLV->portRange;
            pQFilter->pUDPSrcPort = &pFilter->udpSrcPort;
            syslog(LOG_DEBUG, "UDP Sorce Port Add  %d\n", pUDPSrcPortTLV->port);
            syslog(LOG_DEBUG, "UDP Sorce Port Range %d\n", pUDPSrcPortTLV->portRange);
        }

        filterPortTLV *pUDPDestPortTLV = &pFilterTlv->udpDestPort;
        if( TRUE == pUDPDestPortTLV->TlvPresent )
        {
            pFilter->udpDstPort.port = pUDPDestPortTLV->port;
            pFilter->udpDstPort.range = pUDPDestPortTLV->portRange;
            pQFilter->pUDPDstPort = &pFilter->udpDstPort;
            syslog(LOG_DEBUG, "UDP Dest Port Add  %d\n", pUDPDestPortTLV->port);
            syslog(LOG_DEBUG, "UDP Dest Port Range %d\n", pUDPDestPortTLV->portRange);
        }

        filterPortTLV *pTPSrcPortTLV = &pFilterTlv->tpSrcPort;
        if( TRUE == pTPSrcPortTLV->TlvPresent )
        {
            pFilter->tranSrcPort.port = pTPSrcPortTLV->port;
            pFilter->tranSrcPort.range = pTPSrcPortTLV->portRange;
            pQFilter->pTranSrcPort = &pFilter->tranSrcPort;
            syslog(LOG_DEBUG, "TP Sorce Port Add  %dn", pTPSrcPortTLV->port);
            syslog(LOG_DEBUG, "TP Sorce Port Range %d\n", pTPSrcPortTLV->portRange);
        }

        filterPortTLV *pTPDestPortTLV = &pFilterTlv->tpDestPort;
        if( TRUE == pTPDestPortTLV->TlvPresent )
        {
            pFilter->tranDstPort.port = pTPDestPortTLV->port;
            pFilter->tranDstPort.range = pTPDestPortTLV->portRange;
            pQFilter->pTranDstPort = &pFilter->tranDstPort;
            syslog(LOG_DEBUG, "TP Dest Port Add  %d\n", pTPDestPortTLV->port);
            syslog(LOG_DEBUG, "TP Dest Port Range %d\n", pTPDestPortTLV->portRange);
        }

        espSPITLV *pEspSPITLV = &pFilterTlv->espSPI;
        if( TRUE == pEspSPITLV->TlvPresent )
        {
            pQFilter->pEspSpi = &pEspSPITLV->val;
            syslog(LOG_DEBUG, "Filter Label = %x\n", pEspSPITLV->val);
        }

        struct V4tosTlv *pV4tosTlv = &pFilterTlv->v4tosTlv;
        if( TRUE == pV4tosTlv->TlvPresent )
        {
            pFilter->tos.val = pV4tosTlv->val;
            pFilter->tos.mask = pV4tosTlv->mask;
            pQFilter->pTos = &pFilter->tos;
            syslog(LOG_DEBUG, "v4 tos val %x\n", pV4tosTlv->val);
            syslog(LOG_DEBUG, "v4 tos mask %x\n", pV4tosTlv->mask);
        }

        struct V6tcTlv *pV6tcTlv = &pFilterTlv->v6tcTlv;
        if( TRUE == pV6tcTlv->TlvPresent )
        {
            pFilter->iPv6TrafCls.val = pV6tcTlv->val;
            pFilter->iPv6TrafCls.mask = pV6tcTlv->mask;
            pQFilter->pIPv6TrafCls = &pFilter->iPv6TrafCls;
            syslog(LOG_DEBUG, "v6 tc val %x\n", pV6tcTlv->val);
            syslog(LOG_DEBUG, "v6 tc mask %x\n", pV6tcTlv->mask);
        }

        sNxtHdrProtoTlv *pNxtHdrProtoTlv = &pFilterTlv->nxtHdrProtoTlv;
        if( TRUE == pNxtHdrProtoTlv->TlvPresent )
        {
            pFilter->nxtHdrProto = pNxtHdrProtoTlv->val;
            pQFilter->pNxtHdrProto = &pFilter->nxtHdrProto;
            syslog(LOG_DEBUG, "nxt hdr proto %x\n", pFilter->nxtHdrProto);
        }

        sPrecedenceTlv *pPrecedenceTlv = &pFilterTlv->precedenceTlv;
        if( TRUE == pPrecedenceTlv->TlvPresent )
        {
            pFilter->precedence = pPrecedenceTlv->val;
            pQFilter->pPrecedence = &pFilter->precedence;
            syslog(LOG_DEBUG, "precedence %x\n", pFilter->precedence);
        }

        sIdTlv *pIdTlv = &pFilterTlv->idTlv;
        if( TRUE == pIdTlv->TlvPresent )
        {
            pFilter->filterId = pIdTlv->val;
            pQFilter->pId = &pFilter->filterId;
            syslog(LOG_DEBUG, "id %x\n", pFilter->filterId);
        }
    }
}

local void update_flow(
        struct FlowGrantedTlv *pFlowTlv,
        QosFlowInfo *pFlowInfo,
        swiQosFlow   *pQFlowGranted,
        sFlow *pFlow,
        int is_tx
        )
{
    if( TRUE == pFlowTlv->TlvPresent )
    {
        if (is_tx)
            pFlowInfo->pTxQFlowGranted = pQFlowGranted;
        else
            pFlowInfo->pRxQFlowGranted = pQFlowGranted;

        syslog(LOG_DEBUG, "flow present\n");
        syslog(LOG_DEBUG, "index %d\n", pFlowTlv->index);
        pQFlowGranted->index = pFlowTlv->index;

        struct DataRateTlv *pDataRateTlv = &pFlowTlv->dataRateTlv;
        if( TRUE == pDataRateTlv->TlvPresent )
        {
            pQFlowGranted->pDataRate = &pFlow->dataRate;
            syslog(LOG_DEBUG, "max %d\n", pDataRateTlv->dataRateMax);
            syslog(LOG_DEBUG, "guaranteed %d\n", pDataRateTlv->guaranteedRate);
            pFlow->dataRate.dataRateMax = pDataRateTlv->dataRateMax;
            pFlow->dataRate.guaranteedRate = pDataRateTlv->guaranteedRate;
        }

        struct LatencyTlv *pLatencyTlv = &pFlowTlv->latencyTlv;
        if( TRUE == pLatencyTlv->TlvPresent )
        {
            pQFlowGranted->pLatency = &pFlow->latency;
            syslog(LOG_DEBUG, "latency %d\n", pLatencyTlv->latency);
            pFlow->latency = pLatencyTlv->latency;
        }

        struct JitterTlv *pJitterTlv = &pFlowTlv->jitterTlv;
        if( TRUE == pJitterTlv->TlvPresent )
        {
            pQFlowGranted->pJitter = &pFlow->jitter;
            syslog(LOG_DEBUG, "jitter %d\n", pJitterTlv->jitter);
            pFlow->jitter = pJitterTlv->jitter;
        }

        struct QciTlv *pQciTlv = &pFlowTlv->qciTlv;
        if( TRUE == pQciTlv->TlvPresent )
        {
            pQFlowGranted->pLteQci = &pFlow->qci;
            syslog(LOG_DEBUG, "qci %d\n", pQciTlv->qci);
            pFlow->qci = pQciTlv->qci;
        }

        sIPTrafficClassTlv *pIPtrafficClass = &pFlowTlv->trafficClass;
        if( TRUE == pIPtrafficClass->TlvPresent )
        {
            pQFlowGranted->pTrafficClass = &pFlow->trafficClass;
            syslog(LOG_DEBUG, "Traffic Class = %d\n", pIPtrafficClass->val);
            pFlow->trafficClass = pIPtrafficClass->val;
        }

        sTokenBucketTlv *pTokenBucket = &pFlowTlv->tokenBucket;
        if( TRUE == pTokenBucket->TlvPresent )
        {
            pQFlowGranted->pTokenBucket = &pFlow->tokenBucket;
            syslog(LOG_DEBUG, "Bucket size = %u\n", pTokenBucket->bucketSize);
            syslog(LOG_DEBUG, "Peak Rate= %u\n", pTokenBucket->peakRate);
            syslog(LOG_DEBUG, "Token Rate = %u\n", pTokenBucket->tokenRate);
            pFlow->tokenBucket.bucketSz = pTokenBucket->bucketSize;
            pFlow->tokenBucket.peakRate = pTokenBucket->peakRate;
            pFlow->tokenBucket.tokenRate = pTokenBucket->tokenRate;
        }

        s3GPP2ProIDTlv *p3GPP2ProID = &pFlowTlv->profID;
        if( TRUE == p3GPP2ProID->TlvPresent )
        {
            pQFlowGranted->pProfileId3GPP2 = &pFlow->profileID;
            syslog(LOG_DEBUG, "Profile ID = %d\n", p3GPP2ProID->val);
            pFlow->profileID = p3GPP2ProID->val;
        }

        sPktErrorRateTlv *pPktErrorRate = &pFlowTlv->pktErrorRate;
        if( TRUE == pPktErrorRate->TlvPresent )
        {
            pQFlowGranted->pPktErrRate = &pFlow->pktErrRate;
            syslog(LOG_DEBUG, "Pkt error rate multiplier = %d\n", pPktErrorRate->multiplier);
            syslog(LOG_DEBUG, "Pkt error rate exponent = %d\n", pPktErrorRate->exponent);
            pFlow->pktErrRate.multiplier = pPktErrorRate->multiplier;
            pFlow->pktErrRate.exponent = pPktErrorRate->exponent;
        }

        sMinPolPktSizeTlv *pMinPolPktSize = &pFlowTlv->minPolPktSize;
        if( TRUE == pMinPolPktSize->TlvPresent )
        {
            pQFlowGranted->pMinPolicedPktSz = &pFlow->minPolicedPktSz;
            syslog(LOG_DEBUG, "Min Policed Pkt Size = %u\n", pMinPolPktSize->val);
            pFlow->minPolicedPktSz = pMinPolPktSize->val;
        }

        sMaxAllowedPktSizeTlv *pMaxAllowedPktSize = &pFlowTlv->maxAllowedpktSize;
        if( TRUE == pMaxAllowedPktSize->TlvPresent )
        {
            pQFlowGranted->pMaxAllowedPktSz = &pFlow->maxAllowedPktSz;
            syslog(LOG_DEBUG, "Max Allowed Pkt Size = %u\n", pMaxAllowedPktSize->val);
            pFlow->maxAllowedPktSz = pMaxAllowedPktSize->val;
        }

        sRBErrorRateTlv *pRBErrorRate = &pFlowTlv->RBErrorRate;
        if( TRUE == pRBErrorRate->TlvPresent )
        {
            pQFlowGranted->p3GPPResResidualBER = &pFlow->RBErrorRate;
            syslog(LOG_DEBUG, "Residual Bit Error Rate = %d\n", pRBErrorRate->val);
            pFlow->RBErrorRate = pRBErrorRate->val;
        }

        sTHPriorityTlv *pTHPriority = &pFlowTlv->thPriority;
        if( TRUE == pTHPriority->TlvPresent )
        {
            pQFlowGranted->p3GPPTraHdlPri = &pFlow->THPriority;
            syslog(LOG_DEBUG, "Traffic Handling Priority = %d\n", pTHPriority->val);
            pFlow->THPriority = pTHPriority->val;
        }

        sFlowPriorityTlv *pFlowPriority = &pFlowTlv->flowPriorityTlv;
        if( TRUE == pFlowPriority->TlvPresent )
        {
            pQFlowGranted->p3GPP2Pri = &pFlow->flowPriority;
            syslog(LOG_DEBUG, "3GPP2 Flow Priority  = %d\n", pFlowPriority->val);
            pFlow->flowPriority = pFlowPriority->val;
        }

        sIMCNFlagTlv *pIMCNFlag = &pFlowTlv->IMCNFlagTlv;
        if( TRUE == pIMCNFlag->TlvPresent )
        {
            pQFlowGranted->p3GPPImCn = &pFlow->ImCnFlag;
            syslog(LOG_DEBUG, "IM CN Flag  = %d\n", pIMCNFlag->val);
            pFlow->ImCnFlag = pIMCNFlag->val;
        }

        sSigIndTlv *pSigInd = &pFlowTlv->sigInd;
        if( TRUE == pSigInd->TlvPresent )
        {
            pQFlowGranted->p3GPPSigInd = &pFlow->SigInd;
            syslog(LOG_DEBUG, "IM CN Flag  = %d\n", pSigInd->val);
            pFlow->SigInd = pSigInd->val;
        }
    }
}

package void qaQmiQosEventNotify(
    struct QmiNotification *pQmiIndication,
    BYTE   instance )
{
    void *pCallback = NULL;
    enum eQMICallbackIndex CbkIndex;

    CbkIndex = qmiInst2QosEvent(instance);
    if(eQMI_CB_END > CbkIndex)
        pCallback = qaQmiGetCallback( CbkIndex );
    if(pCallback)
    {
        sFlow txFlow, rxFlow;
        sFilter txFilter[MAX_QOS_FILTER_TLV], rxFilter[MAX_QOS_FILTER_TLV];
        QosFlowInfoState QFlowState;
        QosFlowInfo flowInfo;
        int index = 0;
        memset(&flowInfo, 0, sizeof(QosFlowInfo));

        swiQosFilter txQFilter[MAX_QOS_FILTER_TLV];
        for(index=0; index < MAX_QOS_FILTER_TLV; index++)
        {
            memset(&txQFilter[index], 0, sizeof(swiQosFilter));
        }

        swiQosFilter rxQFilter[MAX_QOS_FILTER_TLV];
        for(index=0; index < MAX_QOS_FILTER_TLV; index++)
        {
            memset(&rxQFilter[index], 0, sizeof(swiQosFilter));
        }

        swiQosFlow   txQFlowGranted;
        memset(&txQFlowGranted, 0, sizeof(swiQosFlow));

        swiQosFlow   rxQFlowGranted;
        memset(&rxQFlowGranted, 0, sizeof(swiQosFlow));

        struct FlowInfoTlv *pFlowInfoTlv =
            &pQmiIndication->QmiInd.qaQmiCbkQosEventReportInd.flowInfoTlv;

        /* The TLV was present in the Indication - hence process */
        if( TRUE == pFlowInfoTlv->TlvPresent )
        {
            /* Extract the Parameters */

            QFlowState.id = pFlowInfoTlv->id;
            QFlowState.isNewFlow = pFlowInfoTlv->isNew;
            QFlowState.state = pFlowInfoTlv->state;

            flowInfo.pQFlowState = &QFlowState;
            flowInfo.pBearerID   =
                    &pQmiIndication->QmiInd.qaQmiCbkQosEventReportInd.bearerID;
        }

        update_flow(
                &pQmiIndication->QmiInd.qaQmiCbkQosEventReportInd.txFlowTlv,
                &flowInfo, &txQFlowGranted, &txFlow, 1);

        update_flow(
                &pQmiIndication->QmiInd.qaQmiCbkQosEventReportInd.rxFlowTlv,
                &flowInfo, &rxQFlowGranted, &rxFlow, 0);

        for (index=0; index < MAX_QOS_FILTER_TLV; index++)
        {
            update_filter(
                    &pQmiIndication->QmiInd.qaQmiCbkQosEventReportInd.txFilterTlv[index],
                    &flowInfo, &txQFilter[index], &txFilter[index], 1,index);
        }

        for (index=0; index < MAX_QOS_FILTER_TLV; index++)
        {
            update_filter(
                    &pQmiIndication->QmiInd.qaQmiCbkQosEventReportInd.rxFilterTlv[index],
                    &flowInfo, &rxQFilter[index], &rxFilter[index], 0, index);
        }


        /* Invoke the callback */
        ((tFNSLQSQOSEvent)pCallback)(
            instance,
            &flowInfo
            );
    }
}

package void qaQmiQosFlowStatusNotify(
    struct QmiNotification *pQmiIndication,
    BYTE   instance )
{
    enum eQMICallbackIndex CbkIndex;

    CbkIndex = qmiInst2QosFlowStatusEvent(instance);
    if (CbkIndex < eQMI_CB_END)
    {
        void *pCallback;
        pCallback = qaQmiGetCallback( CbkIndex );
        if(pCallback)
        {
            struct QmiCbkQosFlowStatusInd *lResp =
                     &pQmiIndication->QmiInd.qaQmiCbkQosFlowStatusInd;

            /* Invoke the callback */
            ((tFNQosStatus)pCallback)(
                instance,
                lResp->id,
                lResp->status,
                lResp->event,
                lResp->reason
                );
        }
    }
}

package void qaQmiQosPriEventNotify(
    struct QmiNotification *pQmiIndication
    )
{
    void *pCallback;
    pCallback = qaQmiGetCallback( eQMI_CB_QOS_PRI_FLOW );

    if(pCallback)
    {
        struct QmiCbkQosPriEventInd *lResp =
                 &pQmiIndication->QmiInd.qaQmiCbkQosPriEventInd;

        /* Invoke the callback */
        ((tFNQosPriEvent)pCallback)( lResp->event );
    }
}

/*************
 *
 * Name:    UpkQmiCbkQosNotification
 *
 * Purpose: Unpacks the recevied QOS indication and invokes the approriate
 *          callback based on the QMI message type.
 *
 * Parms:   QmiMsgID        - QMI Message ID
 *          pMdmResp   [IN] - Pointer to packed response from the modem.
 *          pNotifResp [IN] - Notification Structure to fill in the parameters.
 *
 * Parms:   instance    - PDP instance ID
 *
 * Return:  eQCWWAN_ERR_NONE on success, eQCWWAN_ERR_XXX otherwise
 *
 * Abort:   none
 *
 * Notes:   none
 *
 **************/
package enum eQCWWANError UpkQmiCbkQosNotification(
    USHORT                  QmiMsgID,
    BYTE                    *pMdmResp,
    struct QmiNotification  *pNotifResp,
    BYTE                    instance )
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    switch (QmiMsgID)
    {
        case eQMI_QOS_EVENT_IND:
        {
            int i;
            struct QmiCbkQosEventReportInd *pResp =
                &pNotifResp->QmiInd.qaQmiCbkQosEventReportInd;

            for(i=0; i<MAX_QOS_FLOW_SUPPORTED; i++)
            {
                memset(pResp, 0, sizeof(struct QmiCbkQosEventReportInd));

                pResp->tlvIdx = i;

                /* Assign default max values */
                pResp->bearerID = 0xFF;

                /* Unpack the QOS Network Support Status Indication */
                eRCode = UpkQmiCbkQosEventInd( pMdmResp, pResp );

                if (
                        (eRCode == eQCWWAN_ERR_NONE) &&
                        (pResp->TlvPresent)
                   )
                {
                    /* Notify to the Callbacks associated */
                    qaQmiQosEventNotify( pNotifResp, instance );
                }
            }

            break;
        }
        case eQMI_QOS_NETWORK_STATUS_IND:
        {
            struct QmiCbkQosNetworkStatusInd *pResp =
                &pNotifResp->QmiInd.qaQmiCbkQosNetworkStatusInd;

            /* Unpack the QOS Network Support Status Indication */
            eRCode = UpkQmiCbkQosNetworkStatusInd( pMdmResp,
                                                      pResp );
            /* Notify to the Callbacks associated */
            qaQmiQosNetworkStatusNotify( pNotifResp );
            break;
        }
        case eQMI_QOS_FLOW_STATUS_IND:
        {
            struct QmiCbkQosFlowStatusInd *pResp =
                &pNotifResp->QmiInd.qaQmiCbkQosFlowStatusInd;

            eRCode = UpkQmiCbkQosFlowStatusInd( pMdmResp, pResp );
            qaQmiQosFlowStatusNotify( pNotifResp, instance );
            break;
        }
        case eQMI_QOS_PRIMARY_QOS_EVENT_IND:
        {
            struct QmiCbkQosPriEventInd *pResp =
                &pNotifResp->QmiInd.qaQmiCbkQosPriEventInd;

            eRCode = UpkQmiCbkQosPriEventInd( pMdmResp, pResp );
            qaQmiQosPriEventNotify( pNotifResp );
            break;
        }
        default:
            break;
    }
    return eRCode;
}

