/*
 * \ingroup cbk
 *
 * \file    qaCbkQosEventReportInd.c
 *
 * \brief   Contains UnPacking routines for the
 *          QMI_QOS_EVENT_REPORT_IND message.
 *
 * Copyright: © 2013 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */

#include "SwiDataTypes.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaGobiApiQos.h"
#include "qaCbkQosEventReportInd.h"
#include "qaQosCommon.h"
#include <syslog.h>

#ifdef QOS_SIMULATE
extern BYTE cached_qos_flow[255];
extern BYTE cached_qos_flow_len;
#endif

static enum eQCWWANError uFlowState(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    struct FlowInfoTlv *pFlowInfoTlv =
        &((struct QmiCbkQosEventReportInd *)pResp)->flowInfoTlv;

    //TODO proper error handling
    eRCode = GetLong( pTlvData, &pFlowInfoTlv->id);
    eRCode = GetByte( pTlvData, &pFlowInfoTlv->isNew);
    eRCode = GetByte( pTlvData, &pFlowInfoTlv->state);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pFlowInfoTlv->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFlowIdx(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    struct FlowGrantedTlv *pTxFlowTlv = (struct FlowGrantedTlv*) pResp;

    eRCode = GetByte( pTlvData, &pTxFlowTlv->index);

    return eRCode;
}

static enum eQCWWANError unpackFlowDataRate(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    struct DataRateTlv *pDataRateTlv =
        &((struct FlowGrantedTlv *)pResp)->dataRateTlv;

    eRCode = GetLong( pTlvData, &pDataRateTlv->dataRateMax);
    if ( eRCode == eQCWWAN_ERR_NONE)
    {
        eRCode = GetLong( pTlvData, &pDataRateTlv->guaranteedRate);
    }

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pDataRateTlv->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFlowLatency(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    struct LatencyTlv *pLatencyTlv =
        &((struct FlowGrantedTlv *)pResp)->latencyTlv;

    eRCode = GetLong( pTlvData, &pLatencyTlv->latency);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pLatencyTlv->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFlowJitter(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    struct JitterTlv *pJitterTlv =
        &((struct FlowGrantedTlv *)pResp)->jitterTlv;

    eRCode = GetLong( pTlvData, &pJitterTlv->jitter);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pJitterTlv->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFlowLteQci(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    struct QciTlv *pQciTlv =
        &((struct FlowGrantedTlv *)pResp)->qciTlv;

    eRCode = GetByte( pTlvData, &pQciTlv->qci);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pQciTlv->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFlowProId(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    s3GPP2ProIDTlv *pProIdTlv =
        &((struct FlowGrantedTlv *)pResp)->profID;

    eRCode = GetWord( pTlvData, &pProIdTlv->val);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pProIdTlv->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFlowTrafficClass(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    sIPTrafficClassTlv *ptrafficClassTlv =
        &((struct FlowGrantedTlv *)pResp)->trafficClass;

    eRCode = GetByte( pTlvData, &ptrafficClassTlv->val);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        ptrafficClassTlv->TlvPresent = TRUE;
    }

    return eRCode;
}
static enum eQCWWANError unpackFlowPktErrRate(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    sPktErrorRateTlv *pPktErrorRateTlv =
        &((struct FlowGrantedTlv *)pResp)->pktErrorRate;

    eRCode = GetWord( pTlvData, &pPktErrorRateTlv->multiplier);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        eRCode = GetWord( pTlvData, &pPktErrorRateTlv->exponent);
    }

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pPktErrorRateTlv->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFlowMinPolPktSize(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    sMinPolPktSizeTlv *pMinPolPktSizeTlv =
        &((struct FlowGrantedTlv *)pResp)->minPolPktSize;

    eRCode = GetLong( pTlvData, &pMinPolPktSizeTlv->val);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pMinPolPktSizeTlv->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFlowMaxAllowedPktSize(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    sMaxAllowedPktSizeTlv *pMaxAllowedPktSizeTlv =
        &((struct FlowGrantedTlv *)pResp)->maxAllowedpktSize;

    eRCode = GetLong( pTlvData, &pMaxAllowedPktSizeTlv->val);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pMaxAllowedPktSizeTlv->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFlowRBErrorRate(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    sRBErrorRateTlv *pRBErrorRateTlv =
        &((struct FlowGrantedTlv *)pResp)->RBErrorRate;

    eRCode = GetByte( pTlvData, &pRBErrorRateTlv->val);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
    	pRBErrorRateTlv->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFlowTHPriority(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    sTHPriorityTlv *pTHPriorityTlv =
        &((struct FlowGrantedTlv *)pResp)->thPriority;

    eRCode = GetByte( pTlvData, &pTHPriorityTlv->val);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pTHPriorityTlv->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFlow3Gpp2FlowPriority(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    sFlowPriorityTlv *pflowPriorityTlv =
        &((struct FlowGrantedTlv *)pResp)->flowPriorityTlv;

    eRCode = GetByte( pTlvData, &pflowPriorityTlv->val);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
    	pflowPriorityTlv->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFlowIMCNFlag(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    sIMCNFlagTlv *pIMCNFlagTlv =
        &((struct FlowGrantedTlv *)pResp)->IMCNFlagTlv;

    eRCode = GetByte( pTlvData, &pIMCNFlagTlv->val);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
    	pIMCNFlagTlv->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFlowSigInd(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    sSigIndTlv *pSigIndTlv =
        &((struct FlowGrantedTlv *)pResp)->sigInd;

    eRCode = GetByte( pTlvData, &pSigIndTlv->val);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
    	pSigIndTlv->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFlowBucket(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    sTokenBucketTlv *pTokenBucketTlv =
        &((struct FlowGrantedTlv *)pResp)->tokenBucket;

    eRCode = GetLong( pTlvData, &pTokenBucketTlv->bucketSize);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
    	eRCode = GetLong( pTlvData, &pTokenBucketTlv->peakRate);
    }

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
    	eRCode = GetLong( pTlvData, &pTokenBucketTlv->tokenRate);
    }

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
    	pTokenBucketTlv->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError uFlow(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    struct qmTBuffer *tbufp = (struct qmTBuffer *)pTlvData;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    struct FlowGrantedTlv *pTxFlowTlv = (struct FlowGrantedTlv *)pResp;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_QOS_FLOW_IDX,                  unpackFlowIdx },
        { eTLV_QOS_FLOW_3GPP2_PRO_ID,         unpackFlowProId },
        { eTLV_QOS_FLOW_TRAFFIC_CLASS,        unpackFlowTrafficClass },
        { eTLV_QOS_FLOW_DATA_RATE,            unpackFlowDataRate },
        { eTLV_QOS_FLOW_BUCKET,               unpackFlowBucket },
        { eTLV_QOS_FLOW_LATAENCY,             unpackFlowLatency },
        { eTLV_QOS_FLOW_JITTER,               unpackFlowJitter },
        { eTLV_QOS_FLOW_PKT_ERR_RATE,         unpackFlowPktErrRate },
        { eTLV_QOS_FLOW_MIN_POL_PKT_SIZE,     unpackFlowMinPolPktSize },
        { eTLV_QOS_FLOW_MAX_ALLOWED_PKT_SIZE, unpackFlowMaxAllowedPktSize },
        { eTLV_QOS_FLOW_RB_ERROR_RATE,        unpackFlowRBErrorRate },
        { eTLV_QOS_FLOW_3GPP_TH_Priority,     unpackFlowTHPriority },
        { eTLV_QOS_FLOW_3GPP2_FLOW_PRIORITY,  unpackFlow3Gpp2FlowPriority },
        { eTLV_QOS_FLOW_3GPP_IM_CN_FLAG,      unpackFlowIMCNFlag },
        { eTLV_QOS_FLOW_3GPP_SIG_IND,         unpackFlowSigInd },
        { eTLV_QOS_FLOW_LTE_QCI,              unpackFlowLteQci },
        { eTLV_TYPE_INVALID,            NULL }  /* Important. Sentinel.
                                                 * Signifies last item in map.
                                                 */
    };

    eRCode = tlv_unpack(
                &tbufp->Datap[tbufp->Index],
                tbufp->tlvLen,
                (BYTE*)(pTxFlowTlv),
                map );

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pTxFlowTlv->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError uFlowSpecTx(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    struct qmTBuffer *tbufp = (struct qmTBuffer *)pTlvData;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    struct FlowGrantedTlv *pTxFlowTlv =
        &((struct QmiCbkQosEventReportInd *)pResp)->txFlowTlv;

    static struct qmTlvUnpackerItem map[] =
    {
        { 0x10,        uFlow },
        { eTLV_TYPE_INVALID,            NULL }  /* Important. Sentinel.
                                                 * Signifies last item in map.
                                                 */
    };

    eRCode = tlv_unpack(
                &tbufp->Datap[tbufp->Index],
                tbufp->tlvLen,
                (BYTE*)(pTxFlowTlv),
                map );

    return eRCode;
}

static enum eQCWWANError unpackFilterIdx(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    struct FilterTlv *pTxFilterTlv = (struct FilterTlv*) pResp;

    eRCode = GetByte( pTlvData, &pTxFilterTlv->index);

    return eRCode;
}

static enum eQCWWANError unpackFilterVer(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    struct FilterTlv *pTxFilterTlv = (struct FilterTlv*) pResp;

    eRCode = GetByte( pTlvData, &pTxFilterTlv->ver);

    return eRCode;
}

static enum eQCWWANError unpackFilterV4tos(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    struct V4tosTlv *pV4tosTlv =
        &((struct FilterTlv *)pResp)->v4tosTlv;

    eRCode = GetByte( pTlvData, &pV4tosTlv->val);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        eRCode = GetByte( pTlvData, &pV4tosTlv->mask);
    }

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pV4tosTlv->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFilterV4SrcAddr(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    ipV4FilterSrcAddTLV *ipV4SrcAddTLV =
            &((struct FilterTlv *)pResp)->ipV4SrcAddTLV;
    ULONG lAddr, lMask;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    eRCode = GetLong( pTlvData, &lAddr);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        ipV4SrcAddTLV->srcAdd = lAddr;
        eRCode = GetLong( pTlvData, &lMask);
    }

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        ipV4SrcAddTLV->srcAddMask = lMask;
        ipV4SrcAddTLV->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFilterV6tc(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    struct V6tcTlv *pV6tcTlv =
        &((struct FilterTlv *)pResp)->v6tcTlv;

    eRCode = GetByte( pTlvData, &pV6tcTlv->val);
    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        eRCode = GetByte( pTlvData, &pV6tcTlv->mask);
    }

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pV6tcTlv->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFilterNxtHdrProto(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    sNxtHdrProtoTlv *pNxtHdrProtoTlv =
        &((struct FilterTlv *)pResp)->nxtHdrProtoTlv;

    eRCode = GetByte( pTlvData, &pNxtHdrProtoTlv->val);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pNxtHdrProtoTlv->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFilterPrecedence(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    sPrecedenceTlv *pPrecedenceTlv =
        &((struct FilterTlv *)pResp)->precedenceTlv;

    eRCode = GetWord( pTlvData, &pPrecedenceTlv->val);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pPrecedenceTlv->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFilterId(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    sIdTlv *pIdTlv =
        &((struct FilterTlv *)pResp)->idTlv;

    eRCode = GetWord( pTlvData, &pIdTlv->val);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pIdTlv->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFilterV4DestAddr(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    ipV4FilterDestAddTLV *ipV4DestAddTLV =
            &((struct FilterTlv *)pResp)->ipV4DestAddTLV;
    ULONG lAddr, lMask;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    eRCode = GetLong( pTlvData, &lAddr);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        ipV4DestAddTLV->destAdd = lAddr;
        eRCode = GetLong( pTlvData, &lMask);
    }

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        ipV4DestAddTLV->destAddMask = lMask;
        ipV4DestAddTLV->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFilterV6SrcAddr(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    ipV6FilterSrcAddTLV *ipV6SrcAddTLV =
            &((struct FilterTlv *)pResp)->ipV6SrcAddTLV;
    BYTE lAddr, lPfxLen, count;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    for(count=0;count<V6_MAX_ADD_LEN;count++)
    {
        eRCode = GetByte( pTlvData, &lAddr);
        if ( eRCode == eQCWWAN_ERR_NONE )
        {
            ipV6SrcAddTLV->srcAdd[count]=lAddr;
        }
    }

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        eRCode = GetByte( pTlvData, &lPfxLen);
    }

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        ipV6SrcAddTLV->srcAddPfxLen = lPfxLen;
        ipV6SrcAddTLV->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFilterV6DestAddr(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    ipV6FilterDestAddTLV *ipV6DestAddTLV =
            &((struct FilterTlv *)pResp)->ipV6DestAddTLV;
    BYTE lAddr, lPfxLen, count;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    for(count=0;count<V6_MAX_ADD_LEN;count++)
    {
        eRCode = GetByte( pTlvData, &lAddr);
        if ( eRCode == eQCWWAN_ERR_NONE )
        {
            ipV6DestAddTLV->destAdd[count]=lAddr;
        }
    }

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        eRCode = GetByte( pTlvData, &lPfxLen);
    }

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        ipV6DestAddTLV->destAddPfxLen = lPfxLen;
        ipV6DestAddTLV->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFilterV6FlowLBL(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    ipV6FilterFlowLblTLV *pv6FlowLbl =
        &((struct FilterTlv *)pResp)->v6FlowLbl;

    eRCode = GetLong( pTlvData, &pv6FlowLbl->flowLbl);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pv6FlowLbl->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFilterTCPSrcPort(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    filterPortTLV *pTCPSrcPortTLV =
            &((struct FilterTlv *)pResp)->tcpSrcPort;
    WORD lPort, lPortRange;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    eRCode = GetWord( pTlvData, &lPort);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pTCPSrcPortTLV->port = lPort;
        eRCode = GetWord( pTlvData, &lPortRange);
    }

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pTCPSrcPortTLV->portRange = lPortRange;
        pTCPSrcPortTLV->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFilterTCPDestPort(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    filterPortTLV *pTCPDestPortTLV =
            &((struct FilterTlv *)pResp)->tcpDestPort;
    WORD lPort, lPortRange;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    eRCode = GetWord( pTlvData, &lPort);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pTCPDestPortTLV->port = lPort;
        eRCode = GetWord( pTlvData, &lPortRange);
    }

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pTCPDestPortTLV->portRange = lPortRange;
        pTCPDestPortTLV->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFilterUDPSrcPort(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    filterPortTLV *pUDPSrcPortTLV =
            &((struct FilterTlv *)pResp)->udpSrcPort;
    WORD lPort, lPortRange;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    eRCode = GetWord( pTlvData, &lPort);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pUDPSrcPortTLV->port = lPort;
        eRCode = GetWord( pTlvData, &lPortRange);
    }

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pUDPSrcPortTLV->portRange = lPortRange;
        pUDPSrcPortTLV->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFilterUDPDestPort(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    filterPortTLV *pUDPDestPortTLV =
            &((struct FilterTlv *)pResp)->udpDestPort;
    WORD lPort, lPortRange;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    eRCode = GetWord( pTlvData, &lPort);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pUDPDestPortTLV->port = lPort;
        eRCode = GetWord( pTlvData, &lPortRange);
    }

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pUDPDestPortTLV->portRange = lPortRange;
        pUDPDestPortTLV->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFilterEspSpIndex(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    espSPITLV *pEspSpiTlv =
        &((struct FilterTlv *)pResp)->espSPI;

    eRCode = GetLong( pTlvData, &pEspSpiTlv->val);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pEspSpiTlv->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFilterTPSrcPort(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    filterPortTLV *pTPSrcPortTLV =
            &((struct FilterTlv *)pResp)->tpSrcPort;
    WORD lPort, lPortRange;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    eRCode = GetWord( pTlvData, &lPort);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pTPSrcPortTLV->port = lPort;
        eRCode = GetWord( pTlvData, &lPortRange);
    }

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pTPSrcPortTLV->portRange = lPortRange;
        pTPSrcPortTLV->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError unpackFilterTPDestPort(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    filterPortTLV *pTPDestPortTLV =
            &((struct FilterTlv *)pResp)->tpDestPort;
    WORD lPort, lPortRange;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    eRCode = GetWord( pTlvData, &lPort);

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pTPDestPortTLV->port = lPort;
        eRCode = GetWord( pTlvData, &lPortRange);
    }

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pTPDestPortTLV->portRange = lPortRange;
        pTPDestPortTLV->TlvPresent = TRUE;
    }

    return eRCode;
}

static enum eQCWWANError uFilter(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    struct qmTBuffer *tbufp = (struct qmTBuffer *)pTlvData;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    struct FilterTlv *pFilterTlv = (struct FilterTlv *)pResp;

    if (tbufp->tlvIdx >= MAX_QOS_FILTER_TLV)
    {
        return eQCWWAN_ERR_BUFFER_SZ;
    }

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_QOS_FILTER_IDX,           unpackFilterIdx },
        { eTLV_QOS_FILTER_VER,           unpackFilterVer },
        { eTLV_QOS_FILTER_V4_SRC_ADDR,   unpackFilterV4SrcAddr },
        { eTLV_QOS_FILTER_V4_DEST_ADDR,  unpackFilterV4DestAddr },
        { eTLV_QOS_FILTER_NXT_HDR_PROTO, unpackFilterNxtHdrProto },
        { eTLV_QOS_FILTER_V4_TOS,        unpackFilterV4tos },
        { eTLV_QOS_FILTER_V6_SRC_ADDR,   unpackFilterV6SrcAddr },
        { eTLV_QOS_FILTER_V6_DEST_ADDR,  unpackFilterV6DestAddr },
        { eTLV_QOS_FILTER_V6_TC,         unpackFilterV6tc },
        { eTLV_QOS_FILTER_V6_FLOW_LBL,   unpackFilterV6FlowLBL },
        { eTLV_QOS_FILTER_TCP_SRC_PORT,  unpackFilterTCPSrcPort },
        { eTLV_QOS_FILTER_TCP_DEST_PORT, unpackFilterTCPDestPort },
        { eTLV_QOS_FILTER_UDP_SRC_PORT,  unpackFilterUDPSrcPort },
        { eTLV_QOS_FILTER_UDP_DEST_PORT, unpackFilterUDPDestPort },
        { eTLV_QOS_FILTER_ESP_SP_INDEX,  unpackFilterEspSpIndex },
        { eTLV_QOS_FILTER_PRECEDENCE,    unpackFilterPrecedence },
        { eTLV_QOS_FILTER_ID,            unpackFilterId },
        { eTLV_QOS_FILTER_TP_SRC_PORT,   unpackFilterTPSrcPort },
        { eTLV_QOS_FILTER_TP_DEST_PORT,  unpackFilterTPDestPort },
        { eTLV_TYPE_INVALID,            NULL }  /* Important. Sentinel.
                                                 * Signifies last item in map.
                                                 */
    };

    eRCode = tlv_unpack(
                &tbufp->Datap[tbufp->Index],
                tbufp->tlvLen,
                (BYTE*)(&pFilterTlv[tbufp->tlvIdx]),
                map );

    if ( eRCode == eQCWWAN_ERR_NONE )
    {
        pFilterTlv[tbufp->tlvIdx].TlvPresent = TRUE;
    }
    tbufp->tlvIdx++;

    return eRCode;
}

static enum eQCWWANError uFlowSpecRx(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    struct qmTBuffer *tbufp = (struct qmTBuffer *)pTlvData;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    struct FlowGrantedTlv *pRxFlowTlv =
        &((struct QmiCbkQosEventReportInd *)pResp)->rxFlowTlv;

    static struct qmTlvUnpackerItem map[] =
    {
        { 0x10,        uFlow },
        { eTLV_TYPE_INVALID,            NULL }  /* Important. Sentinel.
                                                 * Signifies last item in map.
                                                 */
    };

    eRCode = tlv_unpack(
                &tbufp->Datap[tbufp->Index],
                tbufp->tlvLen,
                (BYTE*)(pRxFlowTlv),
                map );

    return eRCode;
}

static enum eQCWWANError uFilterSpecRx(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    struct qmTBuffer *tbufp = (struct qmTBuffer *)pTlvData;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    struct FilterTlv *pRxFilterTlv =
        &((struct QmiCbkQosEventReportInd *)pResp)->rxFilterTlv[0];

    static struct qmTlvUnpackerItem map[] =
    {
        { 0x10,        uFilter },
        { eTLV_TYPE_INVALID,            NULL }  /* Important. Sentinel.
                                                 * Signifies last item in map.
                                                 */
    };

    eRCode = tlv_unpack(
                &tbufp->Datap[tbufp->Index],
                tbufp->tlvLen,
                (BYTE*)(pRxFilterTlv),
                map );

    return eRCode;
}

static enum eQCWWANError uBearerId(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    struct QmiCbkQosEventReportInd *lResp =
        (struct QmiCbkQosEventReportInd *)pResp;

    //Extracting bearer id
    eRCode = GetByte( pTlvData, &(lResp->bearerID));
    return eRCode;
}

static enum eQCWWANError uFilterSpecNI(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    UNUSEDPARAM(pTlvData);
    UNUSEDPARAM(pResp);

    return eRCode;
}

static enum eQCWWANError uFilterSpecTx(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    struct qmTBuffer *tbufp = (struct qmTBuffer *)pTlvData;
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    struct FilterTlv *pTxFilterTlv =
        &((struct QmiCbkQosEventReportInd *)pResp)->txFilterTlv[0];

    static struct qmTlvUnpackerItem map[] =
    {
        { 0x10,        uFilter },
        { eTLV_TYPE_INVALID,            NULL }  /* Important. Sentinel.
                                                 * Signifies last item in map.
                                                 */
    };

    eRCode = tlv_unpack(
                &tbufp->Datap[tbufp->Index],
                tbufp->tlvLen,
                (BYTE*)(pTxFilterTlv),
                map );

    return eRCode;
}

static enum eQCWWANError uFlowInfo(
    swi_uint8 *pTlvData,
    swi_uint8 *pResp)
{
    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    struct qmTBuffer *tbufp = (struct qmTBuffer *)pTlvData;
    syslog(LOG_DEBUG, "tlvIdx %d\n", tbufp->tlvIdx);

    struct QmiCbkQosEventReportInd *lResp =
        (struct QmiCbkQosEventReportInd *)pResp;

    if (lResp->tlvIdx != tbufp->tlvIdx)
    {
        syslog(LOG_DEBUG, "target pResp->tlvIdx %d, skipping tlvIdx %d\n",
                lResp->tlvIdx, tbufp->tlvIdx);
        tbufp->tlvIdx++;
        return eQCWWAN_ERR_NONE;
    }


    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_FLOW_STATE,        uFlowState },
        { eTLV_FLOW_TX,           uFlowSpecTx },
        { eTLV_FLOW_RX,           uFlowSpecRx },
        { eTLV_FILTER_TX,         uFilterSpecTx },
        { eTLV_FILTER_RX,         uFilterSpecRx },
        { eTLV_FILTER_NI,         uFilterSpecNI },
        { eTLV_BEARER_ID,         uBearerId },
        { eTLV_TYPE_INVALID,            NULL }  /* Important. Sentinel.
                                                 * Signifies last item in map.
                                                 */
    };

    eRCode = tlv_unpack(
                &tbufp->Datap[tbufp->Index],
                tbufp->tlvLen,
                (BYTE*)(lResp),
                map );

    tbufp->tlvIdx++;

    lResp->TlvPresent = TRUE;
    return eRCode;
}

package enum eQCWWANError UpkQmiCbkQosEventInd(
    BYTE    *pMdmResp,
    struct QmiCbkQosEventReportInd *pApiResp )
{
    enum eQCWWANError eRCode;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_FLOW_INFO,    &uFlowInfo },
        { eTLV_TYPE_INVALID,             NULL }  /* Important. Sentinel.
                                                  * Signifies last item in map.
                                                  */
    };

#ifdef TEST_NI_INDICATION
    UNUSEDPARAM(pMdmResp);
    BYTE ind_sample[] = {
0x01,0x00,0x5E +22+22+22,0x02,

  0x10,0x55,0x01,

    0x10,0x06,0x00,0x50,0x94,0xDC,0x47,0x01,0x01,

    0x14,0x95,0x00,
      0x10,0x92,0x00,
        0x23,0x02,0x00,0x01,0x02,
        0x22,0x02,0x00,0x21,0x00,
        0x15,0x02,0x00,0x58,0x78,
        0x11,0x01,0x00,0x00,
        0x12,0x08,0x00, 0xde,0xad,0xbe,0xef, 0xff,0xff,0xff,0xff,
        0x13,0x08,0x00, 0xde,0xad,0xbe,0xef, 0xff,0xff,0xff,0xff,
        0x14,0x01,0x00, 0x01,
        0x19,0x02,0x00, 0x01,0x01,
        0x1A,0x04,0x00, 0x21,0x31,0x01,0x01,
        0x25,0x04,0x00, 0x01,0x91,0xab,0x01,
        0x24,0x04,0x00, 0x51,0x82,0x01,0xd1,
        0x1E,0x04,0x00, 0x71,0x01,0xe1,0x01,
        0x21,0x04,0x00, 0x01,0x78,0xc1,0x01,
        0x1C,0x04,0x00, 0x91,0x01,0x31,0xf1,
        0x1D,0x04,0x00, 0x01,0x54,0x01,0xe1,
        0x1B,0x04,0x00, 0x01,0x01,0x41,0x01,
        0x16,0x11,0x00, 0xde,0xad,0xbe,0xef,0xff,0xff,0xff,0xff,0xde,0xad,0xbe,0xef,0xff,0xff,0xff,0xff,0xff,
        0x17,0x11,0x00, 0xde,0xad,0xbe,0xef,0xff,0xff,0xff,0xff,0xde,0xad,0xbe,0xef,0xff,0xff,0xff,0xff,0xff,


    0x13,0x95,0x00,
      0x10,0x92,0x00,
        0x23,0x02,0x00,0x00,0x02,
        0x22,0x02,0x00,0x20,0x00,
        0x15,0x02,0x00,0x58,0x78,
        0x11,0x01,0x00,0x00,
        0x12,0x08,0x00, 0xde,0xad,0xbe,0xef, 0xff,0xff,0xff,0xff,
        0x13,0x08,0x00, 0xde,0xad,0xbe,0xef, 0xff,0xff,0xff,0xff,
        0x14,0x01,0x00, 0x01,
        0x16,0x11,0x00, 0xde,0xad,0xbe,0xef,0xff,0xff,0xff,0xff,0xde,0xad,0xbe,0xef,0xff,0xff,0xff,0xff,0xff,
        0x17,0x11,0x00, 0xde,0xad,0xbe,0xef,0xff,0xff,0xff,0xff,0xde,0xad,0xbe,0xef,0xff,0xff,0xff,0xff,0xff,
        0x19,0x02,0x00, 0x01,0x01,
        0x1A,0x04,0x00, 0x21,0x31,0x01,0x01,
        0x1B,0x04,0x00, 0x01,0x01,0x41,0x01,
        0x1C,0x04,0x00, 0x91,0x01,0x31,0xf1,
        0x1D,0x04,0x00, 0x01,0x54,0x01,0xe1,
        0x1E,0x04,0x00, 0x71,0x01,0xe1,0x01,
        0x21,0x04,0x00, 0x01,0x78,0xc1,0x01,
        0x24,0x04,0x00, 0x51,0x82,0x01,0xd1,
        0x25,0x04,0x00, 0x01,0x91,0xab,0x01,

    0x12,0x07,0x00,
      0x10,0x04,0x00,
        0x1F,0x01,0x00,0x07,

    0x11,0x07,0x00,
      0x10,0x04,0x00,
        0x1F,0x01,0x00,0x07,

    0x15 ,0x01,0x00,0x01,
    /* bearer id */
    0x16, 0x01,0x00, 0xAB,

  0x10,0x07+22+22+22,0x01,
    0x10,0x06,0x00,0xB0,0x56,0xD1,0x43,0x01,0x01,

      0x14,0x16+22+22,0x00,
        0x10,0x13,0x00,
          0x23,0x02,0x00,0x01,0x00,
          0x22,0x02,0x00,0x11,0x00,
          0x15,0x02,0x00,0xA0,0xE0,
          0x11,0x01,0x00,0x04,

        0x10,0x13,0x00,
          0x23,0x02,0x00,0x04,0x00,
          0x22,0x02,0x00,0x04,0x00,
          0x15,0x02,0x00,0x04,0x00,
          0x11,0x01,0x00,0x06,

        0x10,0x13,0x00,
          0x23,0x02,0x00,0x08,0x00,
          0x22,0x02,0x00,0x08,0x00,
          0x15,0x02,0x00,0x08,0x00,
          0x11,0x01,0x00,0x08,

    0x13,0x16+22,0x00,
      0x10,0x13,0x00,
        0x23,0x02,0x00,0x00,0x00,
        0x22,0x02,0x00,0x10,0x00,
        0x15,0x02,0x00,0xA0,0xE0,
        0x11,0x01,0x00,0x04,

      0x10,0x13,0x00,
        0x23,0x02,0x00,0x5,0x00,
        0x22,0x02,0x00,0x5,0x00,
        0x15,0x02,0x00,0x5,0x5,
        0x11,0x01,0x00,0x05,

    0x12,0x61,0x00,
      0x10,0x5E,0x00,
        0x12,0x08,0x00,0x80,0x91,0x07,0x00,0x80,0x91,0x07,0x00,
        0x1F,0x01,0x00,0x01,
        0x18,0x04,0x00,0x01,0x01,0x01,0x01,
        0x1B,0x02,0x00,0x1B,0x1B,
        0x11,0x01,0x00,0x11,
        0x13,0x0C,0x00,0x11,0x11,0x01,0x00,0x11,0x11,0x01,0x00,0x11,0x11,0x01,0x00,
        0x14,0x04,0x00,0x01,0x02,0x03,0x04,
        0x15,0x04,0x00,0x01,0x01,0x01,0x01,
        0x16,0x04,0x00,0x01,0x01,0x01,0x01,
        0x17,0x04,0x00,0x01,0x01,0x01,0x01,
        0x19,0x01,0x00,0x11,
        0x1A,0x01,0x00,0x11,
        0x1C,0x01,0x00,0x11,
        0x1D,0x01,0x00,0x11,
        0x1E,0x01,0x00,0x11,

    0x11,0x61,0x00,
      0x10,0x5E,0x00,
        0x12,0x08,0x00,0x80 ,0x91,0x07,0x00,0x80,0x91,0x07,0x00,
        0x1F,0x01,0x00,0x01,
        0x18,0x04,0x00,0x01,0x01,0x01,0x01,
        0x1B,0x02,0x00,0x1B,0x1B,
        0x11,0x01,0x00,0x11,
        0x13,0x0C,0x00,0x11,0x11,0x01,0x00,0x11,0x11,0x01,0x00,0x11,0x11,0x01,0x00,
        0x14,0x04,0x00,0x01,0x02,0x03,0x04,
        0x15,0x04,0x00,0x01,0x01,0x01,0x01,
        0x16,0x04,0x00,0x01,0x01,0x01,0x01,
        0x17,0x04,0x00,0x01,0x01,0x01,0x01,
        0x19,0x01,0x00,0x11,
        0x1A,0x01,0x00,0x11,
        0x1C,0x01,0x00,0x11,
        0x1D,0x01,0x00,0x11,
        0x1E,0x01,0x00,0x11,

    0x15,0x01,0x00,0x01

};
    eRCode = qmunpackresp( ind_sample,
                          (BYTE*)pApiResp,
                          map,
                          eQMI_QOS_EVENT_IND );
#else
    eRCode = qmunpackresp( pMdmResp,
                          (BYTE*)pApiResp,
                          map,
                          eQMI_QOS_EVENT_IND );
#endif

    return eRCode;
}

