/*
 * \ingroup cbk
 *
 * \file    qaCbkPdsEventReportInd.c
 *
 * \brief   Contains UnPacking routines for the
 *          QMI_PDS_EVENT_REPORT_IND message.
 *
 * Copyright: © 2011 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */

#include "SwiDataTypes.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaCbkPdsEventReportInd.h"
#include "qaGobiApiPds.h"

/* Functions */

/*
 * This function unpacks the Position Data NMEA to a user provided
 * response structure
 *
 * \param   pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param   pResp    - Pointer to structure containing storage
 *                     to return data for this TLV.
 *
 * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 */
package enum eQCWWANError UnpackCbkPositionDataNMEA(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    struct PositionDataNMEATlv *lResp =
            &((struct QmiCbkPdsEventStatusReportInd *)pResp)->PDNMEATlv;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;

    /* Extract the String Parameter */
    /* The size of the string is stored to check for buffer size in API */
    eRCode = qmQmiExtractString ( pTlvData,
                                  lResp->PositionDataNMEA,
                                  QMI_MAX_POSITION_DATA_NMEA_BUFFER );
    lResp->TlvPresent = TRUE;
    return eRCode;
}

/*****************************************************************************
 * Response handling
 *****************************************************************************/
/*
 * This function unpacks the Parased Position Data from the QMI response
 * message to a user provided response structure
 *
 * \param pTlvData - Pointer to TLV data from which to unpack.
 *
 * \param pResp    - Pointer to structure containing storage
 *                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE
 */

enum eQCWWANError UnpackStatus(BYTE *pTlvData, BYTE *pResp)
{
    struct PositionDataNMEATlv *lResp =
        (struct PositionDataNMEATlv *)pResp;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    eRCode = GetByte(pTlvData, &lResp->session_status);
    return eRCode;
}

enum eQCWWANError UnpackSrc(BYTE *pTlvData, BYTE *pResp)
{
    struct PositionDataNMEATlv *lResp =
        (struct PositionDataNMEATlv *)pResp;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    eRCode = GetLong(pTlvData, &lResp->pos_src);
    return eRCode;
}

union {
    double d;
    
    unsigned long long ull;
} u;
enum eQCWWANError UnpackQmiCbkPdsSetEventReportParsedPositionDataResp(BYTE *pTlvData, BYTE *pResp)
{
    struct PositionDataNMEATlv *lResp =
        (struct PositionDataNMEATlv *)pResp;

    enum eQCWWANError eRCode = eQCWWAN_ERR_NONE;
    int i = 0;  
    unsigned long long ulTemp = 0;
    unsigned int uTemp = 0;
    WORD wTemp = 0;
    BYTE bTemp = 0;

    lResp->iIsPositionDataValid = 0;
    
    eRCode = GetLong(pTlvData,&uTemp);//Valid Mask
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }
    if( (0x30 & uTemp) == 0x30)
    {
        lResp->iIsPositionDataValid = 1;
    }
    eRCode = GetWord(pTlvData,&wTemp);//Calendar year
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }

    eRCode = GetByte(pTlvData,&bTemp);//Calendar Month
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }

    eRCode = GetByte(pTlvData,&bTemp);//Calendar week
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }

    eRCode = GetByte(pTlvData,&bTemp);//Calendar Day of Month
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }

    eRCode = GetByte(pTlvData,&bTemp);//Calendar hour
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }

    eRCode = GetByte(pTlvData,&bTemp);//Calendar minute
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }

    eRCode = GetByte(pTlvData,&bTemp);//Calendar second
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }

    eRCode = GetWord(pTlvData,&wTemp);//Calendar millisecond
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }
    
    eRCode = GetByte(pTlvData,&bTemp);//Number of leap seconds(in second)
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }

    eRCode = GetLongLong(pTlvData,&ulTemp);//Timestamp utc
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }

    /*
         * One-sided time uncertainty for the calendar and
         * UTC timestamp (in milliseconds). This
         * uncertainty is provided at 99 percent
         * confidence.
         */
    eRCode = GetLong(pTlvData,&uTemp);
    if (eQCWWAN_ERR_NONE != eRCode)
    {
       return eRCode;
    }


    eRCode = GetLongLong(pTlvData,&ulTemp);//Latitude
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }
    if(lResp->iIsPositionDataValid)
    {
        u.ull = ulTemp;
        lResp->dLatitude = u.d ;
        //SetQmiPdsLatitude(lResp->dLatitude);
        
    }
    eRCode = GetLongLong(pTlvData,&ulTemp);//Longitude
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }
    if(lResp->iIsPositionDataValid)
    {
        u.ull = ulTemp;
        lResp->dLongitude = u.d;
        //SetQmiPdsLongitude(lResp->dLongitude);
    }
    
    /*
         * 0 : Height above the WGS-84 reference ellipsoid.
         *      Value conveys height (in meters) plus 500 m. 
         * 1 : Height of the MS above the mean sea level (in
         *      meters).
         * 2 : Horizontal speed of the MS (in meters/sec).
         * 3 : Vertical speed of the MS (in meters/sec).
         * 4 : Direction of the MS, counting degrees from 0
         *      in the north and increasing clockwise.
         *            Units: Decimal degrees
         *            Range: 0 to 359.99
         * 5 : Circular horizontal uncertainty (in meters).
         *      Error circular confidence is specified by the
         *      horizontal_confidence field.
         * 6 : Size of the axis along the major angle specified
         *      for location uncertainty (in meters) for the error
         *      ellipse. Error ellipse confidence is specified by
         *      the horizontal_confidence field.
         * 7 : Size of the axis along the minor angle specified
         *     for location uncertainty (in meters) for the error
         *     ellipse. Error ellipse confidence is specified by
         *     the horizontal_confidence field.
         * 8 : Azimuth of the major for the location
         *       uncertainty ellipse.
         *             Units: Decimal degrees
         *             Range: 1 to 180
         * 9 :  Vertical uncertainty (in meters) provided at a
         *       1-sigma (68 percent) confidence.
         *10:  Horizontal velocity uncertainty (in meters/sec).
         *       Up to QMI_PDS version 1.5, this represents a
         *       3-D uncertainty. In QMI_PDS version 1.6 and
         *       above, this represents a 2-D uncertainty. The
         *       2-D uncertainty is provided at 63 percent
         *       confidence.
         *11:  Vertical velocity uncertainty (in meters/sec).
         *       Up until QMI_PDS version 1.5, this represents
         *       a 3-D uncertainty. In QMI_PDS version 1.6 and
         *       above, this represents a 12-D uncertainty. The
         *       1-D uncertainty is provided at 68 percent
         *       confidence.
         *
         *      Note Values are in single float format
         */
    for (i=0;i<12;i++)
    {
        eRCode = GetLong(pTlvData,&uTemp);
        if (eQCWWAN_ERR_NONE != eRCode)
        {
            return eRCode;
        }
    }
    /*
         *   Confidence value of the location horizontal
         *   uncertainty specified as 0 percent to 99 percent.
         */
    eRCode = GetByte(pTlvData,&bTemp);//
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }
    /*
         * 0:    Position dilution of position. Range: 1.0
         *        (highest accuracy) to 50.0 (lowest accuracy).
         *        PDOP = sqrt(HDOPˆ2 +VDOPˆ2)
         * 1:    Horizontal dilution of position. Range: 1.0
         *        (highest accuracy) to 50.0 (lowest accuracy).
         * 2:    Vertical dilution of position. Range: 1.0
         *        (highest accuracy) to 50.0 (lowest accuracy).
         *       
         *        Note: Value is in single float format
         */
    for(i=0;i<3;i++)
    {
        eRCode = GetLong(pTlvData,&uTemp);
        if (eQCWWAN_ERR_NONE != eRCode)
        {
            return eRCode;
        }
    }
    /*
         * Operating mode used to calculate a fix. Values:
         *     0x00 Standalone
         *     0x01 MS-based
         *     0x02 MS-assisted
         *     -1 – Unknown (final fix determination is
         *       pending or cannot be determined
         */
    eRCode = GetByte(pTlvData,&bTemp);//
    if (eQCWWAN_ERR_NONE != eRCode)
    {
        return eRCode;
    }
    lResp->TlvPresent = TRUE;
    return eRCode;
}


/*
  * This function unpacks the Event Report Indication message to a
  * user-provided response structure.
  *
  * \param   pMdmResp - Pointer to packed response from the modem.
  *
  * \param   pApiResp - Pointer to storage to unpack into.
  *
  * \return: eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
  *
  */
package enum eQCWWANError UpkQmiCbkPdsEventReportInd(
    BYTE   *pMdmResp,
    struct QmiCbkPdsEventStatusReportInd *pApiResp )
{
    enum eQCWWANError eRCode;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_POSITION_DATA_NMEA, &UnpackCbkPositionDataNMEA },
        { eTLV_POSITION_SESSION_STATUS, &UnpackStatus },
        { eTLV_PARSED_POSITION_DATA_RESP,&UnpackQmiCbkPdsSetEventReportParsedPositionDataResp },
        { eTLV_POSITION_SOURCE, &UnpackSrc},
        { eTLV_TYPE_INVALID,       NULL }  /* Important. Sentinel.
                                            * Signifies last item in map.
                                            */
    };
    eRCode = qmunpackresp( pMdmResp,
                           (BYTE*)pApiResp,
                           map,
                           eQMI_PDS_EVENT_IND );
    return eRCode;
}
