/*
 * \ingroup nas
 *
 * \file    qaCbkNasLteCPHYCAInd.c
 *
 * \brief   Contains Packing and UnPacking routines for the
 *          eQMI_NAS_LTE_CPHY_CA_IND message.
 *
 * Copyright: © 2013 Sierra Wireless, Inc. all rights reserved
 *
 */

/* include files */
#include "SwiDataTypes.h"
#include "qmudefs.h"
#include "qmerrno.h"
#include "qaCbkNasLteCPHYCAInd.h"
#include "qaGobiApiNasCommon.h"


/* Functions */
/*
 * This function unpacks the Physical Carrier Aggregation of Scell Indicator from the QMI
 * indication message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UnpackTlvIndPhyCaScellInd(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    QmiCbkNasLTECphyCaInfo *lResp = ( QmiCbkNasLTECphyCaInfo *)pResp;
    enum eQCWWANError eRCode = eQCWWAN_ERR_ENUM_BEGIN;
    swi_uint16 iTemp = -1;
    swi_uint8 bTemp = 0;

    if(!lResp)
        return eQCWWAN_ERR_MEMORY;

    lResp->sPhyCaAggScellIndType.TlvPresent = FALSE;
    /* Extract the Physical cell ID of the Scell Range*/
    eRCode = GetWord( pTlvData, &iTemp);
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggScellIndType.pci = iTemp;
    /* Extract the Frequency of the absolute cell Range*/
    eRCode = GetWord( pTlvData,  &iTemp);
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggScellIndType.freq = iTemp;
    /* Extract the Scell state values*/
    eRCode = GetByte( pTlvData, &bTemp);
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggScellIndType.scell_state = bTemp;
        lResp->sPhyCaAggScellIndType.TlvPresent = TRUE;
    return eRCode;
}

/* Functions */
/*
 * This function unpacks the Physical Carrier Aggregation Dwonlonk Bandwidth Type from the QMI
 * indication message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UnpackTlvIndPhyCaScellDlBw(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    QmiCbkNasLTECphyCaInfo *lResp = ( QmiCbkNasLTECphyCaInfo *)pResp;
    enum eQCWWANError eRCode = eQCWWAN_ERR_ENUM_BEGIN;
    ULONG uTemp = -1;

    if(!lResp)
        return eQCWWAN_ERR_MEMORY;

    lResp->sPhyCaAggScellDlBw.TlvPresent = FALSE;
    /* Extract the Donwlink Bandwidth Values*/
    eRCode = GetLong( pTlvData, &uTemp);
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggScellDlBw.dl_bw_value = uTemp;
    lResp->sPhyCaAggScellDlBw.TlvPresent = TRUE;
    return eRCode;
}

/* Functions */
/*
 * This function unpacks the Scell Information from the QMI
 * indication message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UnpackTlvIndPhyCaScellInfo(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    QmiCbkNasLTECphyCaInfo *lResp = ( QmiCbkNasLTECphyCaInfo *)pResp;
    enum eQCWWANError eRCode = eQCWWAN_ERR_ENUM_BEGIN;
    swi_uint16 iTemp = -1;
    swi_uint32 uTemp = -1;

    if(!lResp)
        return eQCWWAN_ERR_MEMORY;

    lResp->sPhyCaAggScellInfo.TlvPresent = FALSE;
    /* Extract the Physical cell ID of the Scell Range*/
    eRCode = GetWord( pTlvData, &iTemp);
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggScellInfo.pci = iTemp;
    /* Extract the Frequency of the absolute Scell Range*/
    eRCode = GetWord( pTlvData, &iTemp);
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggScellInfo.freq = iTemp;

    /* Extract the Scell Dwonlink Bandwidth values*/
    eRCode = GetLong( pTlvData, &uTemp);
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggScellInfo.dl_bw_value = uTemp;
    
    /* Extract the Scell Dwonlink Bandwidth values*/
    eRCode = GetWord( pTlvData, &iTemp);
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggScellInfo.iLTEbandValue = iTemp;

    eRCode = GetLong( pTlvData, &uTemp);
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggScellInfo.scell_state = uTemp;
        lResp->sPhyCaAggScellInfo.TlvPresent = TRUE;
    return eRCode;
}

/* Functions */
/*
 * This function unpacks the Pcell Information from the QMI
 * indication message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UnpackTlvIndPhyCaPcellInfo(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    QmiCbkNasLTECphyCaInfo *lResp = ( QmiCbkNasLTECphyCaInfo *)pResp;
    enum eQCWWANError eRCode = eQCWWAN_ERR_ENUM_BEGIN;
    swi_uint16 iTemp = -1;
    swi_uint32 uTemp = -1;

    if(!lResp)
        return eQCWWAN_ERR_MEMORY;

    lResp->sPhyCaAggPcellInfo.TlvPresent = FALSE;
    /* Extract the Physical cell ID of the Pcell Range*/
    eRCode = GetWord( pTlvData, &iTemp);
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggPcellInfo.pci = iTemp;
    /* Extract the Frequency of the absolute Pcell Range*/
    eRCode = GetWord( pTlvData, &iTemp);
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggPcellInfo.freq = iTemp;

    /* Extract the Pcell Dwonlink Bandwidth values*/
    eRCode = GetLong( pTlvData, &uTemp);
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggPcellInfo.dl_bw_value = uTemp;

    /* Extract the Pcell Dwonlink Bandwidth values*/
    eRCode = GetWord( pTlvData, &iTemp);
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggPcellInfo.iLTEbandValue = iTemp;
    lResp->sPhyCaAggPcellInfo.TlvPresent = TRUE;
    return eRCode;
}

/* Functions */
/*
 * This function unpacks the Scell Index from the QMI
 * indication message to a user provided response structure
 *
 * \param       pTlvData [IN/OUT]  - Pointer to TLV data from which to unpack.
 *
 * \param       pResp    [OUT]     - Pointer to structure containing storage
 *                                   to return data for this TLV.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UnpackTlvIndPhyCaScellIndex(
    BYTE *pTlvData,
    BYTE *pResp )
{
    /* Get the required TLV structure to the local pointer */
    QmiCbkNasLTECphyCaInfo *lResp = ( QmiCbkNasLTECphyCaInfo *)pResp;
    enum eQCWWANError eRCode = eQCWWAN_ERR_ENUM_BEGIN;
    swi_uint8 bTemp = -1;

    if(!lResp)
        return eQCWWAN_ERR_MEMORY;

    lResp->sPhyCaAggScellIndex.TlvPresent = FALSE;
    /* Extract the Scell index*/
    eRCode = GetByte( pTlvData, &bTemp);
    if(eQCWWAN_ERR_NONE != eRCode)
        return eRCode;
    lResp->sPhyCaAggScellIndex.scell_idx = bTemp;
    lResp->sPhyCaAggScellIndex.TlvPresent = TRUE;
    return eRCode;
}


/*
 * This function unpacks the NAS LTE Carrier Aggreation event to a
 * user-provided response structure.
 *
 * \param     pMdmResp   [IN]  - Pointer to packed response from the modem.
 *
 * \param     pApiResp  [OUT]  - Pointer to storage to unpack into.
 *
 * \return eQCWWAN_ERR_NONE, on success, eQCWWAN_ERR_XXX otherwise
 *
 */
enum eQCWWANError UpkQmiCbkNasLteCphyCaInd(
    BYTE       *pMdmResp,
    QmiCbkNasLTECphyCaInfo *pApiResp )
{
    enum eQCWWANError eRCode;

    static struct qmTlvUnpackerItem map[] =
    {
        { eTLV_IND_NAS_LTE_CPHY_CA_SCELL_IND_TYPE,
            &UnpackTlvIndPhyCaScellInd },
        { eTLV_IND_NAS_LTE_CPHY_CA_SCELL_DL_BW,
            &UnpackTlvIndPhyCaScellDlBw },
        { eTLV_IND_NAS_LTE_CPHY_CA_SCELL_INFO,
            &UnpackTlvIndPhyCaScellInfo },
        { eTLV_IND_NAS_LTE_CPHY_CA_PCELL_INFO,
            &UnpackTlvIndPhyCaPcellInfo },
        { eTLV_IND_NAS_LTE_CPHY_CA_SCELL_INDEX,
            &UnpackTlvIndPhyCaScellIndex },
        { eTLV_TYPE_INVALID,              NULL }  /* Important. Sentinel.
                                                   * Signifies last item in map.
                                                   */
    };
    eRCode = qmunpackresp( pMdmResp,
                           (BYTE*)pApiResp,
                           map,
                           eQMI_NAS_LTE_CPHY_CA_IND );
    return eRCode;
}
