/*************
 *
 * Filename:    piLe - network byte order put and get functions.
 *
 * Purpose:     This file contains all functions for reading and
 *              writing packets in network byte order, or in little
 *              Endian byte order( QMI protocol is packed in little
 *              Endian format). If SLQS is defined the packing is
 *              assumed to be little Endian.
 *
 * Copyright:     2011 Sierra Wireless, All rights reserved
 *
 **************/
#include "aa/aaglobal.h"
#include "pi/piudefs.h"

/*************
 *
 * Name:        piput16 - put a 16-bit value into a packet
 *
 * Purpose:     To put a 16-bit value into a packet that is in network
 *              byte order and increment the packet pointer passed the
 *              written field.
 *
 * Parms:       field - 16-bit field value
 *              packetpp - memory location of the pointer to the packet
 *              where the 16-bit field will be written to
 *
 * Return:      no value
 *
 * Abort:       none
 *
 * Notes:       This function performs no pointer validation.
 *
 **************/
global void piput16(swi_uint16 field, swi_uint8 **packetpp)
{
    swi_uint8 *packetp;

    packetp = *packetpp;
#ifndef SLQS
    *packetp++ = field >> 8;
    *packetp++ = field;
#else
    *packetp++ = field;
    *packetp++ = field >> 8;
#endif /* SLQS */
    *packetpp = packetp;
}

/*************
 *
 * Name:     piput32 - put a 32-bit value into a packet
 *
 * Purpose:  To put a 32-bit value into a packet that is in network
 *           byte order and increment the packet pointer passed the
 *           written field.
 *
 * Parms:    field     - 32-bit field value
 *           bufferpp  - memory location of the pointer to the packet
 *                       where the 32-bit value will be put.
 *
 * Return:   no value
 *
 * Abort:    none
 *
 * Notes:    This function performs no pointer validation.
 *
 **************/
global void piput32(swi_ulong field, swi_uint8 **packetpp)
{
    swi_uint8 *packetp;

    packetp = *packetpp;
#ifndef SLQS
    *packetp++ = field >> 24;
    *packetp++ = field >> 16;
    *packetp++ = field >> 8;
    *packetp++ = field;
#else
    *packetp++ = field;
    *packetp++ = field >> 8;
    *packetp++ = field >> 16;
    *packetp++ = field >> 24;
#endif /* SLQS */
    *packetpp = packetp;
}
/*************
 *
 * Name:     piput64 - put a 64 bit value into a packet
 *
 * Purpose:  To put a 64 bit value into a packet that is in network
 *           byte order and increment the packet pointer passed the
 *           written field.
 *
 * Parms:    field     - 64 bit field value
 *           bufferpp  - memory location of the pointer to the packet
 *                       where the 64 bit value will be put.
 *
 * Return:   no value
 *
 * Abort:    none
 *
 * Notes:    This function performs no pointer validation.
 *
 **************/
global void piput64(swi_uint64 field, swi_uint8 **packetpp)
{
    swi_uint8 *packetp;

    packetp = *packetpp;

#ifndef SLQS
    *packetp++ = field >> 56;
    *packetp++ = field >> 48;
    *packetp++ = field >> 40;
    *packetp++ = field >> 32;
    *packetp++ = field >> 24;
    *packetp++ = field >> 16;
    *packetp++ = field >> 8;
    *packetp++ = field;
#else
    *packetp++ = field;
    *packetp++ = field >> 8;
    *packetp++ = field >> 16;
    *packetp++ = field >> 24;
    *packetp++ = field >> 32;
    *packetp++ = field >> 40;
    *packetp++ = field >> 48;
    *packetp++ = field >> 56;
#endif /* SLQS */

    *packetpp = packetp;
}

/*************
 *
 * Name:     piget16 - get a 16-bit value from a packet
 *
 * Purpose:  To get a 16-bit value from a packet that is in network byte
 *           order and increment the packet pointer passed the read field.
 *
 * Parms:    packetpp  - memory location of the pointer to the buffer where
 *                       the 16-bit value will be read from.
 *
 * Return:   16-bit value
 *
 * Abort:    none
 *
 * Notes:    This function performs no pointer validation.
 *
 **************/
global swi_uint16 piget16(swi_uint8 **packetpp)
{
    swi_uint16 field;
    swi_uint8 *packetp;

    packetp = *packetpp;

#ifndef SLQS
    field = *packetp++;
    field <<= 8;
    field += *packetp++;
#else
    field = *(packetp + 1);
    field <<= 8;
    field += *packetp;
    packetp += 2;
#endif /* SLQS */

    *packetpp = packetp;

    return(field);

}

/*************
*
* Name:     piget32 - get a 32-bit value from a packet
*
* Purpose:  To get a 32-bit value from a packet that is in network
*           byte order and increment the packet pointer passed the
*           read field.
*
* Parms:    packetpp - memory location of the pointer to the packet
*                      where the 32-bit value will be read from.
*
* Return:   32-bit value
*
* Abort:    none
*
* Notes:    This function performs no pointer validation.
*
**************/
global swi_ulong piget32(swi_uint8 **packetpp)
{
    swi_ulong field;
    swi_uint8 *packetp;

    packetp = *packetpp;
#ifndef SLQS
    field = *packetp++;
    field <<= 8;
    field += *packetp++;
    field <<= 8;
    field += *packetp++;
    field <<= 8;
    field += *packetp++;
#else
    field =  ((swi_ulong)*(packetp + 3)) << 24;
    field += ((swi_ulong)*(packetp + 2)) << 16;
    field += ((swi_ulong)*(packetp + 1)) << 8;
    field += *packetp;
    packetp += 4;
#endif /* SLQS */
    *packetpp = packetp;

    return(field);

}
/*************
*
* Name:     piget64 - get a 64 bit value from a packet
*
* Purpose:  To get a 64 bit value from a packet that is in network
*           byte order and increment the packet pointer passed the
*           read field.
*
* Parms:    packetpp - memory location of the pointer to the packet
*                      where the 64 bit value will be read from.
*
* Return:   64 bit value
*
* Abort:    none
*
* Notes:    This function performs no pointer validation.
*           Due to a problem we have noted with the gcc compiler this function
*           should not be called. Instead one should duplicate the code inline
*           where ever the function needs to be called as shown in
*           SwiIntUpkCmBasic.c, function SwiUpkRadioBandCfg().
*
**************/
global swi_uint64 piget64(swi_uint8 **packetpp)
{
    swi_uint64 field;

    swi_uint8 *packetp;

    packetp = *packetpp;

#ifndef SLQS
    field = *packetp++;
    field <<= 8;
    field += *packetp++;
    field <<= 8;
    field += *packetp++;
    field <<= 8;
    field += *packetp++;
    field <<= 8;
    field += *packetp++;
    field <<= 8;
    field += *packetp++;
    field <<= 8;
    field += *packetp++;
    field <<= 8;
    field += *packetp++;
#else
    field =  ((swi_uint64)*(packetp + 7)) << 8*7;
    field += ((swi_uint64)*(packetp + 6)) << 8*6;
    field += ((swi_uint64)*(packetp + 5)) << 8*5;
    field += ((swi_uint64)*(packetp + 4)) << 8*4;
    field += ((swi_uint64)*(packetp + 3)) << 8*3;
    field += ((swi_uint64)*(packetp + 2)) << 8*2;
    field += ((swi_uint64)*(packetp + 1)) << 8*1;
    field += *packetp;
    packetp += 8;
#endif /* SLQS */

    *packetpp = packetp;

    return(field);

}

/*************
*
* Name:     piget32_noinc - get a 32-bit value from a packet
*
* Purpose:  To get a 32-bit value from a packet that is in network byte order
*           and not increment the packet pointer
*
* Parms:    packetpp - memory location of the pointer to the packet where the 32
*                      bit value will be read from.
*
* Return:   32-bit value
*
* Abort:    none
*
* Notes:    This function performs no pointer validation. This function is
*           declared inline for speed. This may not work with other compilers.
*
**************/
global swi_ulong piget32_noinc ( swi_uint8 **packetpp )
{
   swi_ulong field;
   swi_uint8 *packetp;

   packetp = *packetpp;

#ifndef SLQS
   field = *packetp++;
   field <<= 8;
   field += *packetp++;
   field <<= 8;
   field += *packetp++;
   field <<= 8;
   field += *packetp++;
#else
   field =  ((swi_ulong)*(packetp + 3)) << 24;
   field += ((swi_ulong)*(packetp + 2)) << 16;
   field += ((swi_ulong)*(packetp + 1)) << 8;
   field += *packetp;
#endif /* SLQS */

   return field;
}

